/*
 *  Midi synth routines for the emu8000 (AWE32/64)
 *
 *  Copyright (c) 1999 by Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "emu8000.h"
#include "emu8000_port.h"
#include "emu8000_equalizer.h"
#include "soundfont.h"

/*
 * compile flags
 */
#define USE_AWE_NRPN
#define USE_GS_NRPN
#define USE_XG_CONTROL


/*
 * module parameters
 */
#ifdef USE_AWE_NRPN
int emu8000_nrpn = 1;
#ifdef MODULE_PARM
MODULE_PARM(emu8000_nrpn, "i");
MODULE_PARM_DESC(emu8000_nrpn, "accept AWE32/64 NRPN");
#endif
#endif

#ifdef USE_GS_NRPN
int emu8000_gs = 1;
#ifdef MODULE_PARM
MODULE_PARM(emu8000_gs, "i");
MODULE_PARM_DESC(emu8000_gs, "accept GS NRPN");
#endif
#endif

#ifdef USE_XG_CONTROL
int emu8000_xg = 1;
#ifdef MODULE_PARM
MODULE_PARM(emu8000_xg, "i");
MODULE_PARM_DESC(emu8000_xg, "accept XG controls");
#endif
#endif


/*
 * conversion from NRPN/control parameters to Emu8000 raw parameters
 */

#if (defined(USE_AWE_NRPN) || defined(USE_GS_NRPN) || defined(USE_XG_CONTROL))

/* NRPN / CC -> Emu8000 parameter converter */
typedef struct {
	int control;
	int effect;
	int (*convert)(int val);
} nrpn_conv_table;

/* effect sensitivity */

#define FX_CUTOFF	0
#define FX_RESONANCE	1
#define FX_ATTACK	2
#define FX_RELEASE	3
#define FX_VIBRATE	4
#define FX_VIBDEPTH	5
#define FX_VIBDELAY	6
#define FX_NUMS		7

/*
 * convert NRPN/control values
 */

static int send_converted_effect(nrpn_conv_table *table, int num_tables,
				 emu8000_t *emu, snd_midi_channel_t *chan,
				 int type, int val, int mode)
{
	int i, cval;
	for (i = 0; i < num_tables; i++) {
		if (table[i].control == type) {
			cval = table[i].convert(val);
			snd_emu8000_send_effect(emu, chan, table[i].effect,
						cval, mode);
			return 1;
		}
	}
	return 0;
}

#endif /* defined */

#define DEF_FX_CUTOFF		170
#define DEF_FX_RESONANCE	6
#define DEF_FX_ATTACK		50
#define DEF_FX_RELEASE		50
#define DEF_FX_VIBRATE		30
#define DEF_FX_VIBDEPTH		4
#define DEF_FX_VIBDELAY		1500

#ifdef USE_GS_NRPN
/* effect sensitivities for GS NRPN:
 *  adjusted for chaos 8MB soundfonts
 */
static int gs_sense[] = 
{
	DEF_FX_CUTOFF, DEF_FX_RESONANCE, DEF_FX_ATTACK, DEF_FX_RELEASE,
	DEF_FX_VIBRATE, DEF_FX_VIBDEPTH, DEF_FX_VIBDELAY
};
#endif /* USE_GS_NRPN */

#ifdef USE_XG_CONTROL
/* effect sensitivies for XG controls:
 * adjusted for chaos 8MB soundfonts
 */
static int xg_sense[] = 
{
	DEF_FX_CUTOFF, DEF_FX_RESONANCE, DEF_FX_ATTACK, DEF_FX_RELEASE,
	DEF_FX_VIBRATE, DEF_FX_VIBDEPTH, DEF_FX_VIBDELAY
};
#endif /* USE_XG_CONTROL */


/*
 * AWE32 NRPN effects
 */

#ifdef USE_AWE_NRPN
static int fx_delay(int val);
static int fx_attack(int val);
static int fx_hold(int val);
static int fx_decay(int val);
static int fx_the_value(int val);
static int fx_twice_value(int val);
static int fx_conv_pitch(int val);
static int fx_conv_Q(int val);

/* function for each NRPN */		/* [range]  units */
#define fx_env1_delay	fx_delay	/* [0,5900] 4msec */
#define fx_env1_attack	fx_attack	/* [0,5940] 1msec */
#define fx_env1_hold	fx_hold		/* [0,8191] 1msec */
#define fx_env1_decay	fx_decay	/* [0,5940] 4msec */
#define fx_env1_release	fx_decay	/* [0,5940] 4msec */
#define fx_env1_sustain	fx_the_value	/* [0,127] 0.75dB */
#define fx_env1_pitch	fx_the_value	/* [-127,127] 9.375cents */
#define fx_env1_cutoff	fx_the_value	/* [-127,127] 56.25cents */

#define fx_env2_delay	fx_delay	/* [0,5900] 4msec */
#define fx_env2_attack	fx_attack	/* [0,5940] 1msec */
#define fx_env2_hold	fx_hold		/* [0,8191] 1msec */
#define fx_env2_decay	fx_decay	/* [0,5940] 4msec */
#define fx_env2_release	fx_decay	/* [0,5940] 4msec */
#define fx_env2_sustain	fx_the_value	/* [0,127] 0.75dB */

#define fx_lfo1_delay	fx_delay	/* [0,5900] 4msec */
#define fx_lfo1_freq	fx_twice_value	/* [0,127] 84mHz */
#define fx_lfo1_volume	fx_twice_value	/* [0,127] 0.1875dB */
#define fx_lfo1_pitch	fx_the_value	/* [-127,127] 9.375cents */
#define fx_lfo1_cutoff	fx_twice_value	/* [-64,63] 56.25cents */

#define fx_lfo2_delay	fx_delay	/* [0,5900] 4msec */
#define fx_lfo2_freq	fx_twice_value	/* [0,127] 84mHz */
#define fx_lfo2_pitch	fx_the_value	/* [-127,127] 9.375cents */

#define fx_init_pitch	fx_conv_pitch	/* [-8192,8192] cents */
#define fx_chorus	fx_the_value	/* [0,255] -- */
#define fx_reverb	fx_the_value	/* [0,255] -- */
#define fx_cutoff	fx_twice_value	/* [0,127] 62Hz */
#define fx_filterQ	fx_conv_Q	/* [0,127] -- */

static int fx_delay(int val)
{
	return (unsigned short)snd_sf_calc_parm_delay(val);
}

static int fx_attack(int val)
{
	return (unsigned short)snd_sf_calc_parm_attack(val);
}

static int fx_hold(int val)
{
	return (unsigned short)snd_sf_calc_parm_hold(val);
}

static int fx_decay(int val)
{
	return (unsigned short)snd_sf_calc_parm_decay(val);
}

static int fx_the_value(int val)
{
	return (unsigned short)(val & 0xff);
}

static int fx_twice_value(int val)
{
	return (unsigned short)((val * 2) & 0xff);
}

static int fx_conv_pitch(int val)
{
	return (short)(val * 4096 / 1200);
}

static int fx_conv_Q(int val)
{
	return (unsigned short)((val / 8) & 0xff);
}


static nrpn_conv_table awe_effects[] =
{
	{ 0, EMU8000_FX_LFO1_DELAY,	fx_lfo1_delay},
	{ 1, EMU8000_FX_LFO1_FREQ,	fx_lfo1_freq},
	{ 2, EMU8000_FX_LFO2_DELAY,	fx_lfo2_delay},
	{ 3, EMU8000_FX_LFO2_FREQ,	fx_lfo2_freq},

	{ 4, EMU8000_FX_ENV1_DELAY,	fx_env1_delay},
	{ 5, EMU8000_FX_ENV1_ATTACK,fx_env1_attack},
	{ 6, EMU8000_FX_ENV1_HOLD,	fx_env1_hold},
	{ 7, EMU8000_FX_ENV1_DECAY,	fx_env1_decay},
	{ 8, EMU8000_FX_ENV1_SUSTAIN,	fx_env1_sustain},
	{ 9, EMU8000_FX_ENV1_RELEASE,	fx_env1_release},

	{10, EMU8000_FX_ENV2_DELAY,	fx_env2_delay},
	{11, EMU8000_FX_ENV2_ATTACK,	fx_env2_attack},
	{12, EMU8000_FX_ENV2_HOLD,	fx_env2_hold},
	{13, EMU8000_FX_ENV2_DECAY,	fx_env2_decay},
	{14, EMU8000_FX_ENV2_SUSTAIN,	fx_env2_sustain},
	{15, EMU8000_FX_ENV2_RELEASE,	fx_env2_release},

	{16, EMU8000_FX_INIT_PITCH,	fx_init_pitch},
	{17, EMU8000_FX_LFO1_PITCH,	fx_lfo1_pitch},
	{18, EMU8000_FX_LFO2_PITCH,	fx_lfo2_pitch},
	{19, EMU8000_FX_ENV1_PITCH,	fx_env1_pitch},
	{20, EMU8000_FX_LFO1_VOLUME,	fx_lfo1_volume},
	{21, EMU8000_FX_CUTOFF,		fx_cutoff},
	{22, EMU8000_FX_FILTERQ,	fx_filterQ},
	{23, EMU8000_FX_LFO1_CUTOFF,	fx_lfo1_cutoff},
	{24, EMU8000_FX_ENV1_CUTOFF,	fx_env1_cutoff},
	{25, EMU8000_FX_CHORUS,		fx_chorus},
	{26, EMU8000_FX_REVERB,		fx_reverb},
};

static int num_awe_effects = NELEM(awe_effects);
#endif /* USE_AWE_NRPN */


#ifdef USE_GS_NRPN
/*
 * GS(SC88) NRPN effects; still experimental
 */

/* cutoff: quarter semitone step, max=255 */
static int gs_cutoff(int val)
{
	return (val - 64) * gs_sense[FX_CUTOFF] / 50;
}

/* resonance: 0 to 15(max) */
static int gs_filterQ(int val)
{
	return (val - 64) * gs_sense[FX_RESONANCE] / 50;
}

/* attack: */
static int gs_attack(int val)
{
	return -(val - 64) * gs_sense[FX_ATTACK] / 50;
}

/* decay: */
static int gs_decay(int val)
{
	return -(val - 64) * gs_sense[FX_RELEASE] / 50;
}

/* release: */
static int gs_release(int val)
{
	return -(val - 64) * gs_sense[FX_RELEASE] / 50;
}

/* vibrato freq: 0.042Hz step, max=255 */
static int gs_vib_rate(int val)
{
	return (val - 64) * gs_sense[FX_VIBRATE] / 50;
}

/* vibrato depth: max=127, 1 octave */
static int gs_vib_depth(int val)
{
	return (val - 64) * gs_sense[FX_VIBDEPTH] / 50;
}

/* vibrato delay: -0.725msec step */
static int gs_vib_delay(int val)
{
	return -(val - 64) * gs_sense[FX_VIBDELAY] / 50;
}

static nrpn_conv_table gs_effects[] =
{
	{32, EMU8000_FX_CUTOFF,	gs_cutoff},
	{33, EMU8000_FX_FILTERQ,	gs_filterQ},
	{99, EMU8000_FX_ENV2_ATTACK, gs_attack},
	{100, EMU8000_FX_ENV2_DECAY, gs_decay},
	{102, EMU8000_FX_ENV2_RELEASE, gs_release},
	{8, EMU8000_FX_LFO1_FREQ, gs_vib_rate},
	{9, EMU8000_FX_LFO1_VOLUME, gs_vib_depth},
	{10, EMU8000_FX_LFO1_DELAY, gs_vib_delay},
};

static int num_gs_effects = NELEM(gs_effects);
#endif /* USE_GS_NRPN */


/*
 * NRPN events
 */
void
snd_emu8000_nrpn(void *p, snd_midi_channel_t *chan, snd_midi_channel_set_t *chset)
{
	emu8000_port_t *port;
	emu8000_t *emu;

	if (!(port = p) || !chan)
		return;
	emu = port->emu;
	if (EMU8000_CHECK(emu) < 0)
		return;

#ifdef USE_AWE_NRPN
	if (chan->control[SND_MCTL_NONREG_PARM_NUM_MSB] == 127 &&
	    chan->control[SND_MCTL_NONREG_PARM_NUM_LSB] <= 26) {
		int val;
		/* Win/DOS AWE32 specific NRPNs */
		/* both MSB/LSB necessary */
		val = (chan->control[SND_MCTL_MSB_DATA_ENTRY] << 7) |
			chan->control[SND_MCTL_LSB_DATA_ENTRY]; 
		val -= 8192;
		send_converted_effect
			(awe_effects, num_awe_effects,
			 emu, chan, chan->control[SND_MCTL_NONREG_PARM_NUM_LSB],
			 val, EMU8000_FX_FLAG_SET);
		return;
	}
#endif /* USE_AWE_NRPN */

#ifdef USE_GS_NRPN
	if (port->chset.midi_mode == SND_MIDI_MODE_GS &&
	    chan->control[SND_MCTL_NONREG_PARM_NUM_MSB] == 1) {
		int val;
		/* GS specific NRPNs */
		/* only MSB is valid */
		val = chan->control[SND_MCTL_MSB_DATA_ENTRY];
		send_converted_effect
			(gs_effects, num_gs_effects,
			 emu, chan, chan->control[SND_MCTL_NONREG_PARM_NUM_LSB],
			 val, EMU8000_FX_FLAG_ADD);
		return;
	}
#endif /* USE_GS_NRPN */
}


/*
 * XG control effects; still experimental
 */

#ifdef USE_XG_CONTROL
/* cutoff: quarter semitone step, max=255 */
static int xg_cutoff(int val)
{
	return (val - 64) * xg_sense[FX_CUTOFF] / 64;
}

/* resonance: 0(open) to 15(most nasal) */
static int xg_filterQ(int val)
{
	return (val - 64) * xg_sense[FX_RESONANCE] / 64;
}

/* attack: */
static int xg_attack(int val)
{
	return -(val - 64) * xg_sense[FX_ATTACK] / 64;
}

/* release: */
static int xg_release(int val)
{
	return -(val - 64) * xg_sense[FX_RELEASE] / 64;
}

static nrpn_conv_table xg_effects[] =
{
	{71, EMU8000_FX_CUTOFF,	xg_cutoff},
	{74, EMU8000_FX_FILTERQ,	xg_filterQ},
	{72, EMU8000_FX_ENV2_RELEASE, xg_release},
	{73, EMU8000_FX_ENV2_ATTACK, xg_attack},
};

static int num_xg_effects = NELEM(xg_effects);
#endif /* USE_XG_CONTROL */

int
snd_emu8000_xg_control(emu8000_t *emu, emu8000_port_t *port,
		       snd_midi_channel_t *chan, int param)
{
#ifdef USE_XG_CONTROL
	return send_converted_effect(xg_effects, num_xg_effects,
				     emu, chan, param,
				     chan->control[param],
				     EMU8000_FX_FLAG_ADD);
#else
	return 0;
#endif /* XG */
}

/*
 * receive sysex
 */
void
snd_emu8000_sysex(void *p, unsigned char *buf, int len, int parsed, snd_midi_channel_set_t *chset)
{
	emu8000_port_t *port;
	emu8000_t *emu;

	if (!(port = p) || !chset)
		return;
	emu = port->emu;
	if (EMU8000_CHECK(emu) < 0)
		return;

	switch (parsed) {
	case SND_MIDI_SYSEX_GS_CHORUS_MODE:
		emu->chorus_mode = chset->gs_chorus_mode;
		snd_emu8000_update_chorus_mode(emu);
		break;

	case SND_MIDI_SYSEX_GS_REVERB_MODE:
		emu->reverb_mode = chset->gs_reverb_mode;
		snd_emu8000_update_reverb_mode(emu);
		break;

	case SND_MIDI_SYSEX_GS_MASTER_VOLUME:
		snd_emu8000_update_port(emu, port, EMU8000_UPDATE_VOLUME);
		break;
	}
}

