/*
 *  Advanced Linux Sound Architecture
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"

#ifdef CONFIG_SND_OSSEMUL

#include "minors.h"
#include "info.h"
#ifndef LINUX_2_2
#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif
#ifdef CONFIG_KERNELD
#include <linux/kerneld.h>
#endif
#else
#include <linux/sound.h>
#endif /* LINUX_2_2 */

#ifndef LINUX_2_2
extern int snd_oss_major;
#endif

#if 0
#define DEBUG_ACCESS( d, m ) printk( ">>" d "[%i] - begin\n", m )
#define DEBUG_RETURN( d, m, f ) \
	do { long result = f; \
	     printk( ">>" d "[%i] - end (result = %li)\n", m, result ); \
	     return result; } while( 0 )
#else
#define DEBUG_ACCESS( d, m )	/* nothing */
#define DEBUG_RETURN( d, m, f ) return(f)
#endif

#define SND_OS_MINORS		256

static int snd_oss_minors_device[SND_OS_MINORS] = {
	0,		/* SND_MINOR_OSS_MIXER */
	-1,		/* SND_MINOR_OSS_SEQUENCER */
	0,		/* SND_MINOR_OSS_MIDI */
	0,		/* SND_MINOR_OSS_PCM_8 */
	0,		/* SND_MINOR_OSS_AUDIO */
	0,		/* SND_MINOR_OSS_PCM_16 */
	-1,		/* SND_MINOR_OSS_SNDSTAT */
	-1,		/* SND_MINOR_OSS_RESERVED7 */
	-1,		/* SND_MINOR_OSS_MUSIC */
	0,		/* SND_MINOR_OSS_DMMIDI */
	0,		/* SND_MINOR_OSS_DMFM */
	1,		/* SND_MINOR_OSS_MIXER1 */
	1, 		/* SND_MINOR_OSS_PCM1 */
	1,		/* SND_MINOR_OSS_MIDI1 */
	1,		/* SND_MINOR_OSS_DMMIDI1 */
	-1,		/* SND_MINOR_OSS_RESERVED15 */
};
static snd_minor_t *snd_oss_minors[SND_OS_MINORS] = {[0 ... (SND_OS_MINORS - 1)] = NULL};

snd_mutex_define_static(sound_oss);

static inline snd_minor_t *snd_oss_verify_card(unsigned short minor)
{
	int dev;

	dev = minor & SND_MINOR_OSS_MASK;
	if (dev != SND_MINOR_OSS_SNDSTAT &&
	    dev != SND_MINOR_OSS_SEQUENCER &&
	    dev != SND_MINOR_OSS_MUSIC) {
		if (!(snd_cards_bitmap & (1 << (minor >> 4))))
			return NULL;
	}
	return snd_oss_minors[minor];
}

#if !defined(LINUX_2_2) && (defined(CONFIG_KERNELD) || defined(CONFIG_KMOD))

static void snd_oss_request_card(int card)
{
	char str[32];

	if (card < 0 || card >= snd_ecards_limit)
		return;
	sprintf(str, "sound-slot-%i", card);
	request_module(str);
}

static void snd_oss_request_minor(int minor)
{
	char str[32];

	sprintf(str, "sound-service-%i-%i", minor >> 4, minor & SND_MINOR_OSS_MASK);
	request_module(str);
}

#endif				/* request_module support */

static inline snd_minor_t *snd_oss_verify_card_open(unsigned short minor)
{
	int dev;
	snd_minor_t *mptr;

	dev = minor & SND_MINOR_OSS_MASK;
	if (dev != SND_MINOR_OSS_SNDSTAT &&
	    dev != SND_MINOR_OSS_SEQUENCER &&
	    dev != SND_MINOR_OSS_MUSIC) {
		if (!(snd_cards_bitmap & (1 << (minor >> 4)))) {
#if !defined(LINUX_2_2) && (defined(CONFIG_KERNELD) || defined(CONFIG_KMOD))
			snd_oss_request_card(minor >> 4);
			if (!(snd_cards_bitmap & (1 << (minor >> 4))))
#endif
				return NULL;
		}
	}
	mptr = snd_oss_minors[minor];
#if !defined(LINUX_2_2) && (defined(CONFIG_KERNELD) || defined(CONFIG_KMOD))
	/* only OSS minors should be add-on modules */
	if (!mptr) {
		snd_oss_request_minor(minor);
		mptr = snd_oss_minors[minor];
	}
#endif
	return mptr;
}

#ifdef LINUX_2_1
static loff_t snd_oss_lseek(struct file *file, loff_t offset, int orig)
#else
static int snd_oss_lseek(struct inode *inode, struct file *file,
			 off_t offset, int orig)
#endif				/* LINUX_2_1 */
{
#ifdef LINUX_2_1
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
#else
	unsigned short minor = MINOR(inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_lseek_t *ptr;

	DEBUG_ACCESS("lseek", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->lseek) == NULL)
		return -ESPIPE;
	DEBUG_RETURN("lseek", minor, ptr(file, offset, orig));
}

#ifdef LINUX_2_1
static ssize_t snd_oss_read(struct file *file,
			    char *buf, size_t count,
			    loff_t * offset)
#else
static int snd_oss_read(struct inode *inode, struct file *file,
			char *buf, int count)
#endif
{
#ifdef LINUX_2_1
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
#else
	unsigned short minor = MINOR(inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_read_t *ptr;

	DEBUG_ACCESS("read", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->read) == NULL)
		return -ENXIO;
	DEBUG_RETURN("read", minor, ptr(file, buf, count));
}

#ifdef LINUX_2_1
static ssize_t snd_oss_write(struct file *file,
			     const char *buf, size_t count,
			     loff_t * offset)
#else
static int snd_oss_write(struct inode *inode, struct file *file,
			 const char *buf, int count)
#endif
{
#ifdef LINUX_2_1
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
#else
	unsigned short minor = MINOR(inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_write_t *ptr;

	DEBUG_ACCESS("write", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->write) == NULL)
		return -ENXIO;
	DEBUG_RETURN("write", minor, ptr(file, buf, count));
}

static int snd_oss_open(struct inode *inode, struct file *file)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_open_t *ptr;

	DEBUG_ACCESS("open", minor);
	if ((mptr = snd_oss_verify_card_open(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->open) == NULL)
		return -ENXIO;
	DEBUG_RETURN("open", minor, ptr(minor, minor >> 4,
		     snd_oss_minors_device[minor & SND_MINOR_OSS_MASK], file));
}

static int snd_oss_release(struct inode *inode, struct file *file)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_release_t *ptr;

	DEBUG_ACCESS("release", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->release) == NULL)
		return -ENXIO;
	DEBUG_RETURN("release", minor, ptr(minor, minor >> 4,
		     snd_oss_minors_device[minor & SND_MINOR_OSS_MASK], file));
}

#ifdef SND_POLL
static unsigned int snd_oss_poll(struct file *file, poll_table * wait)
{
#ifdef LINUX_2_1
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
#else
	unsigned short minor = MINOR(file->f_inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_poll_t *ptr;

	DEBUG_ACCESS("poll", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->poll) == NULL)
		return -ENODEV;
	DEBUG_RETURN("poll", minor, ptr(file, wait));
}
#else
static int snd_oss_select(struct inode *inode, struct file *file,
			  int sel_type, select_table * wait)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_select_t *ptr;

	DEBUG_ACCESS("select", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->select) == NULL)
		return -ENXIO;
	DEBUG_RETURN("select", minor, ptr(file, sel_type, wait));
}
#endif

static int snd_oss_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_ioctl_t *ptr;

	DEBUG_ACCESS("ioctl", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->ioctl) == NULL)
		return -ENXIO;
	DEBUG_RETURN("ioctl", minor, ptr(file, cmd, arg));
}

#ifdef LINUX_2_1
static int snd_oss_mmap(struct file *file, struct vm_area_struct *vma)
#else
static int snd_oss_mmap(struct inode *inode, struct file *file,
			struct vm_area_struct *vma)
#endif
{
#ifdef LINUX_2_1
	struct inode *inode = file->f_dentry->d_inode;
	unsigned short minor = MINOR(inode->i_rdev);
#else
	unsigned short minor = MINOR(file->f_inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_mmap_t *ptr;

	DEBUG_ACCESS("mmap", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->mmap) == NULL)
		return -ENXIO;
	DEBUG_RETURN("mmap", minor, ptr(inode, file, vma));
}

struct file_operations snd_oss_fops =
{
	snd_oss_lseek,		/* (l)lseek */
	snd_oss_read,		/* read */
	snd_oss_write,		/* write */
	NULL,			/* readdir */
#ifdef SND_POLL
	snd_oss_poll,		/* poll */
#else
	snd_oss_select,		/* select */
#endif
	snd_oss_ioctl,		/* ioctl */
	snd_oss_mmap,		/* mmap */
	snd_oss_open,		/* open */
#ifdef SND_FOPS_FLUSH
	NULL,			/* flush */
#endif
#ifdef LINUX_2_1
	snd_oss_release,	/* release */
#else
	(void (*)(struct inode * inode, struct file * file)) snd_oss_release,
#endif
	NULL,			/* fsync */
#ifdef LINUX_2_1
	NULL,			/* fasync */
	NULL,			/* check_media_change */
	NULL,			/* revalidate */
	NULL,			/* lock */
#endif
};

static int snd_kernel_minor(int type, snd_card_t * card, int dev)
{
	int minor;

	switch (type) {
	case SND_OSS_DEVICE_TYPE_MIXER:
		if (!card || dev > 1)
			return -EINVAL;
		minor = (card->number * (SND_MINOR_OSS_MASK + 1)) +
			(dev ? SND_MINOR_OSS_MIXER1 : SND_MINOR_OSS_MIXER);
		break;
	case SND_OSS_DEVICE_TYPE_SEQUENCER:
		minor = SND_MINOR_OSS_SEQUENCER;
		break;
	case SND_OSS_DEVICE_TYPE_PCM:
		if (!card || dev > 1)
			return -EINVAL;
		minor = (card->number * (SND_MINOR_OSS_MASK + 1)) +
			(dev ? SND_MINOR_OSS_PCM1 : SND_MINOR_OSS_PCM);
		break;
	case SND_OSS_DEVICE_TYPE_MIDI:
		if (!card || dev > 1)
			return -EINVAL;
		minor = (card->number * (SND_MINOR_OSS_MASK + 1)) +
			(dev ? SND_MINOR_OSS_MIDI1 : SND_MINOR_OSS_MIDI);
		break;
	case SND_OSS_DEVICE_TYPE_DMFM:
		minor = SND_MINOR_OSS_DMFM;
		break;
	case SND_OSS_DEVICE_TYPE_SNDSTAT:
		minor = SND_MINOR_OSS_SNDSTAT;
		break;
	default:
		return -EINVAL;
	}
	if (minor < 0 || minor > SND_OS_MINORS)
		return -EINVAL;
	return minor;
}

int snd_register_oss_device(int type, snd_card_t * card, int dev, snd_minor_t * reg, const char *name)
{
	int minor = snd_kernel_minor(type, card, dev);
	int xcard = minor & ~SND_MINOR_OSS_MASK;
#ifdef LINUX_2_2
	int err;
#endif
	int track1 = -1, track2 = -1, track3 = -1;
	int register1 = 1, register2 = -1, register3 = -1;

	if (minor < 0)
		return minor;
	snd_mutex_down_static(sound_oss);
	if (snd_oss_minors[minor]) {
		snd_mutex_up_static(sound_oss);
		return -EBUSY;
	}
	if ((minor & SND_MINOR_OSS_MASK) == SND_MINOR_OSS_PCM) {
		if (snd_oss_minors[minor+1] ||
		    snd_oss_minors[minor+2]) {
			snd_mutex_up_static(sound_oss);
			return -EBUSY;
		}
		snd_oss_minors[track2 = minor+1] =
		snd_oss_minors[track3 = minor+2] = reg;
	} else if ((minor & SND_MINOR_OSS_MASK) == SND_MINOR_OSS_MIDI) {
		if (snd_oss_minors[xcard + SND_MINOR_OSS_DMMIDI]) {
			snd_mutex_up_static(sound_oss);
			return -EBUSY;
		}
		snd_oss_minors[track2 = xcard + SND_MINOR_OSS_DMMIDI] = reg;
	} else if ((minor & SND_MINOR_OSS_MASK) == SND_MINOR_OSS_MIDI1) {
		if (snd_oss_minors[xcard + SND_MINOR_OSS_DMMIDI1]) {
			snd_mutex_up_static(sound_oss);
			return -EBUSY;
		}
		snd_oss_minors[track2 = xcard + SND_MINOR_OSS_DMMIDI1] = reg;
	}
	snd_oss_minors[track1 = minor] = reg;
#ifdef LINUX_2_2
	if ((minor & SND_MINOR_OSS_MASK) == SND_MINOR_OSS_PCM) {
		if (register_sound_dsp(&snd_oss_fops, card->number)<0)
			goto __end;
	} else {
		if (track1 >= 0 && (err = register_sound_special(&snd_oss_fops, track1)) < 0)
			goto __end;
		register1 = track1;
		if (track2 >= 0 && register_sound_special(&snd_oss_fops, track2) < 0)
			goto __end;
		register2 = track2;
		if (track3 >= 0 && register_sound_special(&snd_oss_fops, track3) < 0)
			goto __end;
		register3 = track3;
	}
#else
	track1 = track2; track2 = track3;	/* to make GCC happy */
	register1 = register2; register2 = register3;
#endif
	snd_mutex_up_static(sound_oss);
	return 0;

#ifdef LINUX_2_2
      __end:
      	if (track1 >= 0)
      		snd_oss_minors[track1] = NULL;
      	if (track2 >= 0)
      		snd_oss_minors[track2] = NULL;
      	if (track3 >= 0)
      		snd_oss_minors[track3] = NULL;
	snd_mutex_up_static(sound_oss);
      	return -EBUSY;
#endif
}

int snd_unregister_oss_device(int type, snd_card_t * card, int dev)
{
	int minor = snd_kernel_minor(type, card, dev);
	int xcard = minor & ~SND_MINOR_OSS_MASK;

	if (minor < 0)
		return minor;
	snd_mutex_down_static(sound_oss);
	if (snd_oss_minors[minor] == NULL) {
		snd_mutex_up_static(sound_oss);
		return -EINVAL;
	}
	if ((minor & SND_MINOR_OSS_MASK) == SND_MINOR_OSS_PCM) {
		snd_oss_minors[minor + 1] =
		snd_oss_minors[minor + 2] = NULL;
#ifdef LINUX_2_2
		unregister_sound_dsp(minor);
#endif
	} else if ((minor & SND_MINOR_OSS_MASK) == SND_MINOR_OSS_MIDI) {
		snd_oss_minors[xcard + SND_MINOR_OSS_DMMIDI] = NULL;
#ifdef LINUX_2_2
		unregister_sound_special(xcard + SND_MINOR_OSS_DMMIDI);
#endif
	} else if ((minor & SND_MINOR_OSS_MASK) == SND_MINOR_OSS_MIDI1) {
		snd_oss_minors[xcard + SND_MINOR_OSS_DMMIDI1] = NULL;
#ifdef LINUX_2_2
		unregister_sound_special(xcard + SND_MINOR_OSS_DMMIDI1);
#endif
	}
	snd_oss_minors[minor] = NULL;
#ifdef LINUX_2_2
	if ((minor & SND_MINOR_OSS_MASK) != SND_MINOR_OSS_PCM)
		unregister_sound_special(minor);
#endif
	snd_mutex_up_static(sound_oss);
	return 0;
}

/*
 *  INFO PART
 */

static snd_info_entry_t *snd_minor_info_oss_entry = NULL;

static void snd_minor_info_oss_read(snd_info_buffer_t * buffer, void *private_data)
{
	int idx, dev;
	snd_minor_t *mptr;

	snd_mutex_down_static(sound_oss);
	for (idx = 0; idx < SND_OS_MINORS; idx++) {
		if ((mptr = snd_oss_minors[idx]) == NULL)
			continue;
		dev = idx & SND_MINOR_OSS_MASK;
	        if (dev != SND_MINOR_OSS_SNDSTAT &&
		    dev != SND_MINOR_OSS_SEQUENCER &&
		    dev != SND_MINOR_OSS_MUSIC)
			snd_iprintf(buffer, "%3i: [%i-%2i]: %s\n", idx, idx >> 4, dev, mptr->comment);
		else
			snd_iprintf(buffer, "%3i:       : %s\n", idx, mptr->comment);
	}
	snd_mutex_up_static(sound_oss);
}

int snd_minor_info_oss_init(void)
{
	snd_info_entry_t *oss_entry;

	oss_entry = snd_info_create_entry(NULL, "oss-devices");
	if (oss_entry) {
		oss_entry->t.text.read_size = PAGE_SIZE;
		oss_entry->t.text.read = snd_minor_info_oss_read;
		if (snd_info_register(oss_entry) < 0) {
			snd_info_free_entry(oss_entry);
			oss_entry = NULL;
		}
	}
	snd_minor_info_oss_entry = oss_entry;
	return 0;
}

int snd_minor_info_oss_done(void)
{
	if (snd_minor_info_oss_entry)
		snd_info_unregister(snd_minor_info_oss_entry);
	return 0;
}

int snd_oss_init_module(void)
{
#if !defined(LINUX_2_2) && defined(CONFIG_SND_OSSEMUL)
	if (register_chrdev(snd_oss_major, "snd", &snd_oss_fops)) {
		snd_printk("unable to register OSS/Lite major device number %d\n", snd_oss_major);
		return -EIO;
	}
#endif
	return 0;
}

void snd_oss_cleanup_module(void)
{
	int idx;
	snd_minor_t *minor;

	for (idx = 0; idx < SND_OS_MINORS; idx++) {
		minor = snd_oss_minors[idx];
		if (minor)
			snd_printk("OSS device minor %i not unregistered!!!\n", idx);
	}
#if !defined(LINUX_2_2)
	if (unregister_chrdev(snd_oss_major, "snd") != 0)
		snd_printk("unable to unregister major device number %d\n", snd_oss_major);
#endif
}

#endif /* CONFIG_SND_OSSEMUL */
