/*
 * OSS compatible sequencer driver
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __SEQ_OSS_DEVICE_H
#define __SEQ_OSS_DEVICE_H

#include "driver.h"
#include "seq_oss.h"
#include "midi.h"
#include "seq_kernel.h"
#include "info.h"

/* enable debug print */
//#define SND_SEQ_OSS_DEBUG

/* max. applications */
#define SND_SEQ_OSS_MAX_CLIENTS		16
#define SND_SEQ_OSS_MAX_SYNTH_DEVS	16
#define SND_SEQ_OSS_MAX_MIDI_DEVS	32

/* version */
#define SND_SEQ_OSS_MAJOR_VERSION	0
#define SND_SEQ_OSS_MINOR_VERSION	1
#define SND_SEQ_OSS_TINY_VERSION	5
#define SND_SEQ_OSS_VERSION_STR		"0.1.5"

/* device and proc interface name */
#define SND_SEQ_OSS_DEVNAME	"seq_oss"
#define SND_SEQ_OSS_PROCNAME	"oss"


/*
 * type definitions
 */

typedef struct seq_oss_devinfo_t seq_oss_devinfo_t;
typedef struct seq_oss_writeq_t seq_oss_writeq_t;
typedef struct seq_oss_readq_t seq_oss_readq_t;
typedef struct seq_oss_timer_t seq_oss_timer_t;
typedef struct seq_oss_synthinfo_t seq_oss_synthinfo_t;
typedef struct seq_oss_chinfo_t seq_oss_chinfo_t;
typedef unsigned int reltime_t;
typedef unsigned int abstime_t;
typedef union evrec_t evrec_t;


/*
 * synthesizer channel information
 */
struct seq_oss_chinfo_t {
	int note, vel;
};

/*
 * synthesizer information
 */
struct seq_oss_synthinfo_t {
	snd_seq_oss_arg_t arg;
	seq_oss_chinfo_t *ch;
	int nr_voices;
	int opened;
};


/*
 * sequencer client information
 */

struct seq_oss_devinfo_t {

	int index;	/* application index */
	int cseq;	/* sequencer client number */
	int port;	/* sequencer port number */

	snd_seq_addr_t addr;	/* address of this device */

	int seq_mode;	/* sequencer mode */
	int file_mode;	/* file access */

	/* midi device table */
	int max_mididev;

	/* synth device table */
	int max_synthdev;
	seq_oss_synthinfo_t synths[SND_SEQ_OSS_MAX_SYNTH_DEVS];
	int synth_opened;

	/* output queue */
	seq_oss_writeq_t *writeq;

	/* midi input queue */
	seq_oss_readq_t *readq;

	/* timer */
	seq_oss_timer_t *timer;
};


/*
 * function prototypes
 */

/* create/delete OSS sequencer client */
int snd_seq_oss_create_client(void);
int snd_seq_oss_delete_client(void);

/* device file interface */
int snd_seq_oss_open(struct file *file, int level);
void snd_seq_oss_release(seq_oss_devinfo_t *dp);
int snd_seq_oss_ioctl(seq_oss_devinfo_t *dp, unsigned int cmd, unsigned long arg);
int snd_seq_oss_read(seq_oss_devinfo_t *dev, char *buf, int count);
int snd_seq_oss_write(seq_oss_devinfo_t *dp, const char *buf, int count, struct file *opt);
#ifdef SND_POLL
unsigned int snd_seq_oss_poll(seq_oss_devinfo_t *dp, struct file *file, poll_table * wait);
#else
int snd_seq_oss_select(seq_oss_devinfo_t *dp, int sel_type, select_table * wait);
#endif

void snd_seq_oss_reset(seq_oss_devinfo_t *dp);
void snd_seq_oss_flush_write(seq_oss_devinfo_t *dp);

/* */
void snd_seq_oss_process_queue(seq_oss_devinfo_t *dp, abstime_t time);


/* proc interface */
void snd_seq_oss_system_info_read(snd_info_buffer_t *buf);
void snd_seq_oss_midi_info_read(snd_info_buffer_t *buf);
void snd_seq_oss_synth_info_read(snd_info_buffer_t *buf);
void snd_seq_oss_readq_info_read(seq_oss_readq_t *q, snd_info_buffer_t *buf);
void snd_seq_oss_writeq_info_read(seq_oss_writeq_t *q, snd_info_buffer_t *buf);
void snd_seq_oss_timer_info_read(seq_oss_timer_t *q, snd_info_buffer_t *buf);

/* file mode macros */
#define is_read_mode(mode)	((mode) & SND_SEQ_OSS_FILE_READ)
#define is_write_mode(mode)	((mode) & SND_SEQ_OSS_FILE_WRITE)
#define is_nonblock_mode(mode)	((mode) & SND_SEQ_OSS_FILE_NONBLOCK)

/* send event - the headers must be filled before calling this */
inline static int
snd_seq_oss_send_event(seq_oss_devinfo_t *dp, snd_seq_event_t *ev, int atomic)
{
	ev->time.tick = 0;
	return snd_seq_kernel_client_dispatch(dp->cseq, ev, atomic);
}

/* fill the addresses in header */
inline static void
snd_seq_oss_fill_addr(seq_oss_devinfo_t *dp, snd_seq_event_t *ev,
		     int dest_client, int dest_port)
{
	ev->source = dp->addr;
	ev->dest.client = dest_client;
	ev->dest.port = dest_port;
	ev->dest.queue = 0;
	/* ev->dest.channel is not written here */
}


/* misc. functions for proc interface */
char *enabled_str(int bool);
char *filemode_str(int fmode);


/* for debug */
#ifdef SND_SEQ_OSS_DEBUG
extern int seq_oss_debug;
#define debug_printk(x)	do { if (seq_oss_debug > 0) snd_printk x; } while (0)
#else
#define debug_printk(x)	/**/
#endif


/*
 * copy from/to user-space
 */
#ifndef LINUX_2_1

#undef copy_from_user
#undef copy_to_user

inline static int
copy_from_user(void *dst, const void *src, int size)
{
	if (verify_area(VERIFY_READ, src, size))
		return -EFAULT;
	memcpy_fromfs(dst, src, size);
	return 0;
}

inline static int
copy_to_user(void *dst, const void *src, int size)
{
	if (verify_area(VERIFY_WRITE, dst, size))
		return -EFAULT;
	memcpy_tofs(dst, src, size);
	return 0;
}

#endif



#endif /* __SEQ_OSS_DEVICE_H */
