/*
 *   IWFFFF - AMD InterWave (tm) - Instrument routines
 *   Copyright (c) 1990 by Jaroslav Kysela <perex@suse.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
 
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "ainstr_iw.h"

static void snd_seq_iwffff_copy_lfo_from_stream(iwffff_lfo_t *fp,
						iwffff_xlfo_t *fx)
{
	fp->freq = snd_itoh_16(fx->freq);
	fp->depth = snd_itoh_16(fx->depth);
	fp->sweep = snd_itoh_16(fx->sweep);
	fp->shape = fx->shape;
	fp->delay = fx->delay;
}

static int snd_seq_iwffff_copy_env_from_stream(__u32 req_stype,
					       iwffff_layer_t *lp,
					       iwffff_env_t *ep,
					       iwffff_xenv_t *ex,
					       char **data,
					       long *len)
{
	__u32 stype;
	iwffff_env_record_t *rp, *rp_last;
	iwffff_xenv_record_t rx;
	iwffff_env_point_t *pp;
	iwffff_xenv_point_t px;
	int points_size, idx;

	ep->flags = ex->flags;
	ep->mode = ex->mode;
	ep->index = ex->index;
	rp_last = NULL;
	while (1) {
		if (*len < sizeof(__u32))
			return -EINVAL;
		copy_from_user(&stype, data, sizeof(stype));
		if (stype == IWFFFF_STRU_WAVE)
			return 0;
		if (req_stype != stype) {
			if (stype == IWFFFF_STRU_ENV_RECP ||
			    stype == IWFFFF_STRU_ENV_RECV)
				return 0;
		}
		if (*len < sizeof(rx))
			return -EINVAL;
		copy_from_user(&rx, *data, sizeof(rx));
		*data += sizeof(rx);
		*len -= sizeof(rx);
		points_size = (snd_itoh_16(rx.nattack) + snd_itoh_16(rx.nrelease)) * 2 * sizeof(__u16);
		if (points_size > *len)
			return -EINVAL;
		rp = (iwffff_env_record_t *)snd_calloc(sizeof(*rp) + points_size);
		if (rp == NULL)
			return -ENOMEM;
		rp->nattack = snd_itoh_16(rx.nattack);
		rp->nrelease = snd_itoh_16(rx.nrelease);
		rp->sustain_offset = snd_itoh_16(rx.sustain_offset);
		rp->sustain_rate = snd_itoh_16(rx.sustain_rate);
		rp->release_rate = snd_itoh_16(rx.release_rate);
		rp->hirange = rx.hirange;
		pp = (iwffff_env_point_t *)(rp + 1);
		for (idx = 0; idx < rp->nattack + rp->nrelease; idx++) {
			copy_from_user(&px, *data, sizeof(px));
			*data += sizeof(px);
			*len -= sizeof(px);
			pp->offset = snd_itoh_16(px.offset);
			pp->rate = snd_itoh_16(px.rate);
		}
		if (ep->record == NULL) {
			ep->record = rp;
		} else {
			rp_last = rp;
		}
		rp_last = rp;
	}
	return 0;
}

static int snd_seq_iwffff_copy_wave_from_stream(snd_iwffff_ops_t *ops,
						iwffff_layer_t *lp,
					        char **data,
					        long *len,
					        int atomic)
{
	iwffff_wave_t *wp, *prev;
	iwffff_xwave_t xp;
	int err;
	
	if (*len < sizeof(xp))
		return -EINVAL;
	copy_from_user(&xp, *data, sizeof(xp));
	*data += sizeof(xp);
	*len -= sizeof(xp);
	wp = (iwffff_wave_t *)snd_calloc(sizeof(*wp));
	if (wp == NULL)
		return -ENOMEM;
	wp->share_id[0] = snd_itoh_32(xp.share_id[0]);
	wp->share_id[1] = snd_itoh_32(xp.share_id[1]);
	wp->share_id[2] = snd_itoh_32(xp.share_id[2]);
	wp->share_id[3] = snd_itoh_32(xp.share_id[3]);
	wp->format = snd_itoh_32(xp.format);
	wp->address.memory = snd_itoh_32(xp.offset);
	wp->size = snd_itoh_32(xp.size);
	wp->start = snd_itoh_32(xp.start);
	wp->loop_start = snd_itoh_32(xp.loop_start);
	wp->loop_end = snd_itoh_32(xp.loop_end);
	wp->loop_repeat = snd_itoh_16(xp.loop_repeat);
	wp->sample_ratio = snd_itoh_32(xp.sample_ratio);
	wp->attenuation = xp.attenuation;
	wp->low_note = xp.low_note;
	wp->high_note = xp.high_note;
	if (!(wp->format & IWFFFF_WAVE_ROM)) {
		if (wp->size > *len) {
			snd_free(wp, sizeof(*wp));
			return -ENOMEM;
		}
	}
	if (ops->put_sample) {
		err = ops->put_sample(ops->private_data, wp,
				      *data, wp->size, atomic);
		if (err < 0) {
			snd_free(wp, sizeof(*wp));
			return err;
		}
	}
	prev = lp->wave;
	if (prev) {
		while (prev->next) prev = prev->next;
		prev->next = wp;
	} else {
		lp->wave = wp;
	}
	return 0;
}

static void snd_seq_iwffff_env_free(snd_iwffff_ops_t *ops,
				    iwffff_env_t *env,
				    int atomic)
{
	iwffff_env_record_t *rec;
	
	while ((rec = env->record) != NULL) {
		env->record = rec->next;
		snd_free(rec, sizeof(*rec) + 2 * sizeof(short) * rec->nattack + rec->nrelease);
	}
}
				    
static void snd_seq_iwffff_wave_free(snd_iwffff_ops_t *ops,
				     iwffff_wave_t *wave,
				     int atomic)
{
	if (ops->remove_sample)
		ops->remove_sample(ops->private_data, wave, atomic);
	snd_free(wave, sizeof(*wave));
}

static void snd_seq_iwffff_instr_free(snd_iwffff_ops_t *ops,
                                      iwffff_instrument_t *ip,
                                      int atomic)
{
	iwffff_layer_t *layer;
	iwffff_wave_t *wave;
	
	while ((layer = ip->layer) != NULL) {
		ip->layer = layer->next;
		snd_seq_iwffff_env_free(ops, &layer->penv, atomic);
		snd_seq_iwffff_env_free(ops, &layer->venv, atomic);
		while ((wave = layer->wave) != NULL) {
			layer->wave = wave->next;
			snd_seq_iwffff_wave_free(ops, wave, atomic);
		}
		snd_free(layer, sizeof(*layer));
	}
}

static int snd_seq_iwffff_put(void *private_data, snd_seq_kinstr_t *instr,
			      char *instr_data, long len, int atomic)
{
	snd_iwffff_ops_t *ops = (snd_iwffff_ops_t *)private_data;
	iwffff_instrument_t *ip;
	iwffff_xinstrument_t ix;
	iwffff_layer_t *lp;
	iwffff_xlayer_t lx;
	int err;

	if (atomic)
		return -EINVAL;		/* not supported yet */
	/* copy instrument data */
	if (len < sizeof(ix))
		return -EINVAL;
	copy_from_user(&ix, instr_data, sizeof(ix));
	if (ix.stype != IWFFFF_STRU_INSTR)
		return -EINVAL;
	instr_data += sizeof(ix);
	len -= sizeof(ix);
	ip = (iwffff_instrument_t *)KINSTR_DATA(instr);
	ip->exclusion = snd_itoh_16(ix.exclusion);
	ip->layer_type = snd_itoh_16(ix.layer_type);
	ip->exclusion_group = snd_itoh_16(ix.exclusion_group);
	ip->effect1 = ix.effect1;
	ip->effect1_depth = ix.effect1_depth;
	ip->effect2 = ix.effect2;
	ip->effect2_depth = ix.effect2_depth;
	/* copy layers */
	while (len > 0) {
		if (len < sizeof(iwffff_xlayer_t)) {
			snd_seq_iwffff_instr_free(ops, ip, atomic);
			return -EINVAL;
		}
		copy_from_user(&lx, instr_data, sizeof(lx));
		instr_data += sizeof(lx);
		len -= sizeof(lx);
		if (lx.stype != IWFFFF_STRU_LAYER) {
			snd_seq_iwffff_instr_free(ops, ip, atomic);
			return -EINVAL;
		}
		lp = (iwffff_layer_t *)snd_calloc(sizeof(*lp));
		if (lp == NULL) {
			snd_seq_iwffff_instr_free(ops, ip, atomic);
			return -ENOMEM;
		}
		lp->flags = lx.flags;
		lp->velocity_mode = lx.velocity_mode;
		lp->layer_event = lx.layer_event;
		lp->low_range = lx.low_range;
		lp->high_range = lx.high_range;
		lp->pan = lx.pan;
		lp->pan_freq_scale = lx.pan_freq_scale;
		lp->attenuation = lx.attenuation;
		snd_seq_iwffff_copy_lfo_from_stream(&lp->tremolo, &lx.tremolo);
		snd_seq_iwffff_copy_lfo_from_stream(&lp->vibrato, &lx.vibrato);
		lp->freq_scale = snd_itoh_16(lx.freq_scale);
		lp->freq_center = lx.freq_center;
		err = snd_seq_iwffff_copy_env_from_stream(IWFFFF_STRU_ENV_RECP,
							  lp,
							  &lp->penv, &lx.penv,
						          &instr_data, &len);
		if (err < 0) {
			snd_seq_iwffff_instr_free(ops, ip, atomic);
			return err;
		}
		err = snd_seq_iwffff_copy_env_from_stream(IWFFFF_STRU_ENV_RECV,
							  lp,
							  &lp->venv, &lx.venv,
						          &instr_data, &len);
		if (err < 0) {
			snd_seq_iwffff_instr_free(ops, ip, atomic);
			return err;
		}
		while (len > sizeof(__u32)) {
			__u32 stype;

			copy_from_user(&stype, instr_data, sizeof(stype));
			if (stype != IWFFFF_STRU_WAVE)
				break;
			err = snd_seq_iwffff_copy_wave_from_stream(ops,
								   lp,
							    	   &instr_data,
								   &len,
								   atomic);
			if (err < 0) {
				snd_seq_iwffff_instr_free(ops, ip, atomic);
				return err;
			}
		}
	}
	return 0;
}

static int snd_seq_iwffff_get(void *private_data, snd_seq_kinstr_t *instr,
			      char *instr_data, long len, int atomic)
{
	return -ENXIO;
}

static int snd_seq_iwffff_remove(void *private_data,
				 snd_seq_kinstr_t *instr,
                                 int atomic)
{
	snd_iwffff_ops_t *ops = (snd_iwffff_ops_t *)private_data;
	iwffff_instrument_t *ip;

	ip = (iwffff_instrument_t *)KINSTR_DATA(instr);
	snd_seq_iwffff_instr_free(ops, ip, atomic);
	return 0;
}

int snd_seq_iwffff_init(snd_iwffff_ops_t *ops,
			void *private_data,
			unsigned int flags,
			snd_seq_kinstr_ops_t *next)
{
	memset(ops, 0, sizeof(*ops));
	ops->private_data = private_data;
	ops->kops.private_data = ops;
	ops->kops.flags = flags;
	ops->kops.add_len = sizeof(iwffff_instrument_t);
	ops->kops.instr_type = SND_SEQ_INSTR_ID_INTERWAVE;
	ops->kops.put = snd_seq_iwffff_put;
	ops->kops.get = snd_seq_iwffff_get;
	ops->kops.remove = snd_seq_iwffff_remove;
	ops->kops.next = next;
	return 0;
}

/*
 *  Init part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_ainstr_iw;
#endif

int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_ainstr_iw) < 0)
		return -ENOMEM;
#endif
	return 0;
}

void cleanup_module(void)
{
}

#ifdef MODULE_PARM		/* hey - we have new 2.1.18+ kernel... */
MODULE_AUTHOR("Jaroslav Kysela <perex@suse.cz>");
MODULE_DESCRIPTION("Advanced Linux Sound Architecture IWFFFF support.");
MODULE_SUPPORTED_DEVICE("sound");
#endif
