/*
 *  Abstract routines for MIXER control
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "minors.h"
#include "mixer.h"
#include "info.h"
#include "control.h"
#include "sndpersist.h"

#define SND_MIXERS		(SND_CARDS * SND_MINOR_MIXERS)
#define SND_ATOMIC_POOL_SIZE	50		/* should be enough */

snd_kmixer_t *snd_mixers[SND_MIXERS] = {[0 ... (SND_MIXERS - 1)] = NULL};
static struct snd_stru_mixer_notify * snd_mixer_notify_first = NULL;

snd_mutex_define_static(register);

static void snd_mixer_prepare_read_queue(snd_kmixer_file_t * mfile);

int snd_mixer_lock(snd_kmixer_t *mixer, int xup)
{
	if (!mixer)
		return -EINVAL;
	if (!xup) {
		snd_mutex_down(mixer, lock);
	} else {
		snd_mutex_up(mixer, lock);
	}
	return 0;
}

static int snd_mixer_open(unsigned short minor, int cardnum, int device,
                          struct file *file)
{
	snd_kmixer_file_t *mfile, *mfile1;
	snd_kmixer_t *mixer;

	if (!(mixer = snd_mixers[(cardnum * SND_MINOR_MIXERS) + device]))
		return -ENODEV;
	mfile = (snd_kmixer_file_t *) snd_calloc(sizeof(*mfile));
	if (!mfile)
		return -ENOMEM;
	snd_spin_prepare(mfile, change_lock);
	snd_sleep_prepare(mfile, change);
	mfile->mixer = mixer;
	file->private_data = mfile;
	MOD_INC_USE_COUNT;
	mixer->card->use_inc(mixer->card);
	snd_mixer_prepare_read_queue(mfile);
	snd_mixer_lock(mixer, 0);
	if (mixer->ffile) {
		for (mfile1 = mixer->ffile;
		     mfile1->next;
		     mfile1 = mfile1->next);
		mfile1->next = mfile;
	} else {
		mixer->ffile = mfile;
	}
	snd_mixer_lock(mixer, 1);
	return 0;
}

static int snd_mixer_release(unsigned short minor, int cardnum, int device,
                             struct file *file)
{
	snd_kmixer_file_t *mfile, *mfile1;
	snd_kmixer_t *mixer;
	snd_kmixer_read_t *mread;

	if (file->private_data) {
		mfile = (snd_kmixer_file_t *) file->private_data;
		file->private_data = NULL;
		mixer = mfile->mixer;
		snd_mixer_lock(mixer, 0);
		if (mixer->ffile == mfile) {
			mixer->ffile = mfile->next;
		} else {
			for (mfile1 = mixer->ffile;
			     mfile1->next != mfile;
			     mfile1 = mfile1->next);
			mfile1->next = mfile->next;
		}
		snd_mixer_lock(mixer, 1);
		while (mfile->pool) {
			mread = mfile->pool;
			mfile->pool = mread->next;
			snd_free(mread, sizeof(*mread));
		}
		while (mfile->first_item) {
			mread = mfile->first_item;
			mfile->first_item = mread->next;
			snd_free(mread, sizeof(*mread));
		}
		snd_free(mfile, sizeof(snd_kmixer_file_t));
		mixer->card->use_dec(mixer->card);
		MOD_DEC_USE_COUNT;
	}
	return 0;
}

int snd_mixer_busy(snd_kmixer_file_t * mfile)
{
	if (mfile->rebuild || mfile->first_item)
		return 1;
	return 0;
}

static void snd_mixer_prepare_read_queue(snd_kmixer_file_t * mfile)
{
	unsigned long flags;
	snd_kmixer_read_t *item;
	
	while (mfile->pool_size < SND_ATOMIC_POOL_SIZE) {		
		item = (snd_kmixer_read_t *)snd_malloc(sizeof(*item));
		if (!item) {
			snd_printk("mixer - atomic pool initialization failure\n");
			break;
		}
		snd_spin_lock(mfile, read_lock, &flags);
		item->next = mfile->pool;
		mfile->pool = item;
		mfile->pool_size++;
		snd_spin_unlock(mfile, read_lock, &flags);
	}
	while (mfile->pool_size > SND_ATOMIC_POOL_SIZE) {
		snd_spin_lock(mfile, read_lock, &flags);
		item = mfile->pool;
		if (item)
			mfile->pool = item->next;
		snd_spin_unlock(mfile, read_lock, &flags);
		if (item)
			snd_free(item, sizeof(*item));
	}
}

static snd_kmixer_read_t *snd_mixer_alloc_read(snd_kmixer_file_t * mfile, int atomic)
{
	unsigned long flags;
	snd_kmixer_read_t *item;

	if (atomic) {
		snd_spin_lock(mfile, read_lock, &flags);
		item = mfile->pool;
		if (item) {
			mfile->pool = item->next;
			memset(item, 0, sizeof(*item));
		}
		snd_spin_unlock(mfile, read_lock, &flags);
		if (!item)
			snd_printk("mixer - atomic allocation failure\n");
	} else {
		item = (snd_kmixer_read_t *)snd_calloc(sizeof(*item));
	}
	return item;
}

static void snd_mixer_empty_read_queue(snd_kmixer_file_t * mfile, int rebuild, int atomic)
{
	unsigned long flags;
	snd_kmixer_read_t *read1, *read2;

	snd_spin_lock(mfile, read_lock, &flags);
	read1 = mfile->first_item;
	mfile->first_item = mfile->last_item = NULL;
	mfile->rebuild = rebuild ? 1 : 0;
	snd_spin_unlock(mfile, read_lock, &flags);
	while (read1) {
		read2 = read1->next;
		if (atomic) {
			snd_spin_lock(mfile, read_lock, &flags);
			read1->next = mfile->pool;
			mfile->pool = read1;
			mfile->pool_size++;
			snd_spin_unlock(mfile, read_lock, &flags);
		} else {
			snd_free(read1, sizeof(snd_kmixer_read_t));
		}
		read1 = read2;
	}
}

static void snd_mixer_notify_change(snd_kmixer_file_t * mfile, snd_mixer_read_t * mread, int all, int atomic)
{
	unsigned long flags;
	snd_kmixer_file_t *mfile1;
	snd_kmixer_t *mixer;
	snd_kmixer_read_t *nread;

	if (!mfile || !mread)
		return;
	mixer = mfile->mixer;
	/* TODO: lock mixer->ffile list */
	for (mfile1 = mixer->ffile; mfile1; mfile1 = mfile1->next) {
		if (all || mfile1 != mfile) {
			if (!mfile1->read_active) {
				mfile1->rebuild = 1;
				continue;
			}
			nread = snd_mixer_alloc_read(mfile1, atomic);
			if (!nread) {
				snd_mixer_empty_read_queue(mfile1, 1, atomic);
			} else {
				memcpy(&nread->data, mread, sizeof(nread->data));
				snd_spin_lock(mfile1, read_lock, &flags);
				if (!mfile1->first_item) {
					mfile1->first_item = mfile1->last_item = nread;
				} else {
					mfile1->last_item->next = nread;
				}
				snd_spin_unlock(mfile1, read_lock, &flags);
			}
			snd_spin_lock(mfile1, change_lock, &flags);
			if (snd_getlock(mfile1, change) & SND_WK_SLEEP) {
				snd_getlock(mfile1, change) &= ~SND_WK_SLEEP;
				snd_wakeup(mfile1, change);
			}
			snd_spin_unlock(mfile1, change_lock, &flags);
		}
	}
#ifdef CONFIG_SND_OSSEMUL
	mixer->oss_change_count++;
#endif
	if (!atomic)
		snd_mixer_prepare_read_queue(mfile);
}

void snd_mixer_structure_change(snd_kmixer_file_t * mfile, snd_mixer_read_t * read)
{
	snd_mixer_notify_change(mfile, read, 1, 0);
}

void snd_mixer_value_change(snd_kmixer_file_t * mfile, snd_mixer_read_t * read, int atomic)
{
	snd_mixer_notify_change(mfile, read, 0, atomic);
}

void snd_mixer_element_free(snd_kmixer_element_t *element)
{
	if (element) {
		if (element->routes_next.size > 0)
			snd_free(element->routes_next.routes, element->routes_next.size * sizeof(void *));
		if (element->routes_prev.size > 0)
			snd_free(element->routes_prev.routes, element->routes_prev.size * sizeof(void *));
		if (element->private_free)
			element->private_free(element->private_data);
		snd_free_str(element->name);
		snd_free(element, sizeof(*element));
	}
}

void snd_mixer_element_new_free(snd_kmixer_element_new_t *nelement)
{
	if (nelement) {
		if (nelement->private_free)
			nelement->private_free(nelement->private_data);
	}
}

snd_kmixer_element_t *snd_mixer_element_find(snd_kmixer_t * mixer,
						char *name, int index,
						int type)
{
	snd_kmixer_element_t *element;
	
	if (!mixer || !name)
		return NULL;
	for (element = mixer->elements; element; element = element->next)
		if (element->name && !strcmp(element->name, name) &&
		    element->index == index && element->type == type)
			return element;
	return NULL;
}

static void snd_mixer_element_notify(snd_kmixer_t * mixer, snd_kmixer_element_t *element, unsigned int cmd)
{
	snd_mixer_read_t r;
	
	memset(&r, 0, sizeof(r));
	r.cmd = cmd;
	strncpy(r.data.eid.name, element->name, sizeof(r.data.eid.name));
	r.data.eid.index = element->index;
	snd_mixer_structure_change(mixer->ffile, &r);
}

int snd_mixer_element_add(snd_kmixer_t * mixer, snd_kmixer_element_t *element)
{
	if (!mixer || !element)
		return -EINVAL;
	mixer->elements_count++;
	element->next = mixer->elements;
	mixer->elements = element;
	snd_mixer_element_notify(mixer, element, SND_MIXER_READ_ELEMENT_ADD);
	return 0;
}

int snd_mixer_element_remove(snd_kmixer_t * mixer, snd_kmixer_element_t * element)
{
	snd_kmixer_element_t *prev;

	if (!mixer || !element)
		return -EINVAL;
	for (prev = mixer->elements; prev; prev = prev->next) {
		snd_mixer_element_route_remove(mixer, element, prev);
		snd_mixer_element_route_remove(mixer, prev, element);
	}
	if (mixer->elements == element) {
		mixer->elements_count--;
		mixer->elements = element->next;
	} else {
		for (prev = mixer->elements; prev && prev->next != element; prev = prev->next);
		if (prev) {
			mixer->elements_count--;
			prev->next = element->next;
		}
	}
	snd_mixer_element_free(element);
	snd_mixer_element_notify(mixer, element, SND_MIXER_READ_ELEMENT_REMOVE);
	return 0;
}

int snd_mixer_element_rename(snd_kmixer_t * mixer,
			     char *name, int index, int type,
			     char *nname, int nindex)
{
	snd_kmixer_element_t *element;
	char *nname1;
	
	element = snd_mixer_element_find(mixer, name, index, type);
	if (element == NULL)
		return -ENOENT;
	nname1 = snd_malloc_strdup(nname);
	if (nname1 == NULL)
		return -ENOMEM;
	snd_free_str(element->name);
	element->name = nname1;
	element->index = nindex;
	return 0;
}

snd_kmixer_element_t *snd_mixer_element_new(snd_kmixer_t * mixer, snd_kmixer_element_new_t * nelement)
{
	snd_kmixer_element_t *element;
	
	if (!mixer || !nelement) {
		snd_mixer_element_new_free(nelement);
		return NULL;
	}
	element = (snd_kmixer_element_t *)snd_calloc(sizeof(*element));
	if (!element) {
		snd_mixer_element_new_free(nelement);
		return NULL;
	}
	element->name = snd_malloc_strdup(nelement->name);
	if (!element->name) {
		snd_mixer_element_new_free(nelement);
		snd_free(element, sizeof(*element));
		return NULL;
	}
	element->index = nelement->index;
	element->type = nelement->type;
	element->info = nelement->info;
	element->control = nelement->control;
	element->private_data = nelement->private_data;
	element->private_free = nelement->private_free;
	if (snd_mixer_element_add(mixer, element) < 0) {
		snd_mixer_element_free(element);
		return NULL;
	}
	return element;
}

int snd_mixer_element_change(snd_kmixer_t * mixer, snd_kmixer_element_t * element)
{
	if (!mixer || !element)
		return -EINVAL;
	snd_mixer_element_notify(mixer, element, SND_MIXER_READ_ELEMENT_CHANGE);
	return 0;
}

int snd_mixer_element_value_change(snd_kmixer_file_t * mfile, snd_kmixer_element_t * element, int atomic)
{
	snd_mixer_read_t r;
	
	if (!mfile || !element)
		return -EINVAL;
	memset(&r, 0, sizeof(r));
	r.cmd = SND_MIXER_READ_ELEMENT_VALUE;
	strncpy(r.data.eid.name, element->name, sizeof(r.data.eid.name));
	r.data.eid.index = element->index;
	snd_mixer_value_change(mfile, &r, atomic);
	return 0;
}

static int snd_mixer_element_list_add(snd_kmixer_element_route_t *route, snd_kmixer_element_t *element)
{
	snd_kmixer_element_t **pelement;

	if (!route || !element)
		return -EINVAL;
	if (route->count + 1 > route->size) {
		pelement = snd_malloc((route->size + 4) * sizeof(snd_kmixer_element_t *));
		if (!pelement)
			return -ENOMEM;
		route->size += 4;
		memcpy(pelement, route->routes, route->count * sizeof(snd_kmixer_element_t *));
		snd_free(route->routes, route->count * sizeof(snd_kmixer_element_t *));
		route->routes = pelement;
	}
	route->routes[route->count++] = element;
	return 0;
}

static int snd_mixer_element_list_remove(snd_kmixer_element_route_t *route, snd_kmixer_element_t *element)
{
	int idx;

	if (!route || !element)
		return -EINVAL;
	for (idx = 0; idx < route->count; idx++) {
		if (route->routes[idx] == element) {
			if (idx + 1 < route->count)
				memmove(&route->routes[idx],
					&route->routes[idx+1],
					(route->count - idx - 1) * sizeof(snd_kmixer_element_t *));
			route->count--;
			return 0;
		}
	}
	return -ENOENT;
}

int snd_mixer_element_route_add(snd_kmixer_t * mixer,
				snd_kmixer_element_t *src_element,
				snd_kmixer_element_t *dst_element)
{
	int err;

	if (!mixer || !src_element || !dst_element)
		return -EINVAL;
	err = snd_mixer_element_list_add(&src_element->routes_next, dst_element);
	if (err < 0)
		return err;
	err = snd_mixer_element_list_add(&dst_element->routes_prev, src_element);
	if (err < 0) {
		snd_mixer_element_list_remove(&src_element->routes_next, dst_element);
		return err;
	}
	return 0;
}

int snd_mixer_element_route_remove(snd_kmixer_t * mixer,
				   snd_kmixer_element_t *src_element,
				   snd_kmixer_element_t *dst_element)
{
	if (!mixer || !src_element || !dst_element)
		return -EINVAL;
	if (snd_mixer_element_list_remove(&src_element->routes_next, dst_element) < 0)
		snd_printd("mixer - element list remove failed (1), dst = '%s', index = %i, type = %i\n", dst_element->name, dst_element->index, dst_element->type);
	if (snd_mixer_element_list_remove(&dst_element->routes_prev, src_element) < 0)
		snd_printd("mixer - element list remove failed (2), src = '%s', index = %i, type = %i\n", src_element->name, src_element->index, src_element->type);
	return 0;
}

void snd_mixer_group_free(snd_kmixer_group_t * group)
{
	if (group) {
		if (group->elements_size)
			snd_free(group->elements, group->elements_size * sizeof(void *));
		snd_free_str(group->name);
		snd_free(group, sizeof(*group));
	}
}

snd_kmixer_group_t *snd_mixer_group_find(snd_kmixer_t * mixer,
						char *name, int index)
{
	snd_kmixer_group_t *group;
	
	if (!mixer || !name)
		return NULL;
	for (group = mixer->groups; group; group = group->next)
		if (group->name && !strcmp(group->name, name) && group->index == index)
			return group;
	return NULL;
}

static void snd_mixer_group_notify(snd_kmixer_t * mixer, snd_kmixer_group_t *group, unsigned int cmd)
{
	snd_mixer_read_t r;
	
	memset(&r, 0, sizeof(r));
	r.cmd = cmd;
	strncpy(r.data.gid.name, group->name, sizeof(r.data.gid.name));
	r.data.gid.index = group->index;
	snd_mixer_structure_change(mixer->ffile, &r);
}

int snd_mixer_group_add(snd_kmixer_t * mixer, snd_kmixer_group_t * group)
{
	if (!mixer || !group)
		return -EINVAL;
	mixer->groups_count++;
	group->next = mixer->groups;
	mixer->groups = group;
	snd_mixer_group_notify(mixer, group, SND_MIXER_READ_GROUP_ADD);
	return 0;
}

int snd_mixer_group_remove(snd_kmixer_t * mixer, snd_kmixer_group_t * group)
{
	snd_kmixer_group_t *prev;

	if (!mixer || !group)
		return -EINVAL;
	if (mixer->groups == group) {
		mixer->groups_count--;
		mixer->groups = group->next;
	} else {
		for (prev = mixer->groups; prev && prev->next != group; prev = prev->next);
		if (prev) {
			mixer->groups_count--;
			prev->next = group->next;
		}
	}
	snd_mixer_group_free(group);
	snd_mixer_group_notify(mixer, group, SND_MIXER_READ_GROUP_REMOVE);
	return 0;
}

int snd_mixer_group_rename(snd_kmixer_t * mixer,
			     char *name, int index,
			     char *nname, int nindex)
{
	snd_kmixer_group_t *group;
	char *nname1;
	
	group = snd_mixer_group_find(mixer, name, index);
	if (group == NULL)
		return -ENOENT;
	nname1 = snd_malloc_strdup(nname);
	if (nname1 == NULL)
		return -ENOMEM;
	snd_free_str(group->name);
	group->name = nname1;
	group->index = nindex;
	return 0;
}

snd_kmixer_group_t *snd_mixer_group_new(snd_kmixer_t * mixer, snd_kmixer_group_new_t * ngroup)
{
	snd_kmixer_group_t *group;
	
	if (!mixer || !ngroup)
		return NULL;
	group = (snd_kmixer_group_t *)snd_calloc(sizeof(*group));
	if (!group) {
		return NULL;
	}
	group->name = snd_malloc_strdup(ngroup->name);
	if (!group->name) {
		snd_free(group, sizeof(*group));
		return NULL;
	}
	group->index = ngroup->index;
	if (snd_mixer_group_add(mixer, group) < 0) {
		snd_mixer_group_free(group);
		return NULL;
	}
	return group;
}

int snd_mixer_group_change(snd_kmixer_t * mixer, snd_kmixer_group_t * group)
{
	if (!mixer || !group)
		return -EINVAL;
	snd_mixer_group_notify(mixer, group, SND_MIXER_READ_GROUP_CHANGE);
	return 0;
}

int snd_mixer_group_element_add(snd_kmixer_t * mixer,
				snd_kmixer_group_t * group,
				snd_kmixer_element_t * element)
{
	snd_kmixer_element_t **pelement;

	if (group->elements_count + 1 > group->elements_size) {
		pelement = snd_malloc((group->elements_size + 4) * sizeof(snd_kmixer_element_t *));
		if (!pelement)
			return -ENOMEM;
		memcpy(pelement, group->elements, group->elements_count * sizeof(snd_kmixer_element_t *));
		snd_free(group->elements, group->elements_size * sizeof(snd_kmixer_element_t *));
		group->elements_size += 4;
		group->elements = pelement;
	}
	group->elements[group->elements_count++] = element;
	snd_mixer_group_change(mixer, group);
	return 0;
}

int snd_mixer_group_element_remove(snd_kmixer_t * mixer,
				   snd_kmixer_group_t * group,
				   snd_kmixer_element_t * element)
{
	int idx;

	for (idx = 0; idx < group->elements_count; idx++) {
		if (group->elements[idx] == element) {
			if (idx + 1 < group->elements_count)
				memmove(&group->elements[idx],
					&group->elements[idx+1],
					(group->elements_count - idx - 1) * sizeof(snd_kmixer_element_t *));
			group->elements_count--;
			return 0;
		}
	}
	return -ENOENT;
}

static int snd_mixer_switch_read(snd_kmixer_t * mixer,
				 snd_switch_t * _uswitch)
{
	return snd_switch_read(&mixer->switches, mixer, _uswitch);
}

static int snd_mixer_switch_write(snd_control_t * control,
				  snd_kmixer_t * mixer,
				  snd_switch_t * _uswitch)
{
	int result;

	if (snd_control_busy(control))
		return -EBUSY;
	result = snd_switch_write(&mixer->switches, mixer, _uswitch);
	if (result > 0) {	/* change */
		snd_switch_t uswitch;

		copy_from_user(&uswitch, _uswitch, sizeof(uswitch));
		snd_control_notify_switch_value_change(control,
						       SND_CTL_IFACE_MIXER,
						       uswitch.name,
						       0);
	}
	return result;
}

static int snd_mixer_info(snd_kmixer_t * mixer, snd_mixer_info_t * _info)
{
	snd_mixer_info_t info;

	if (verify_area(VERIFY_WRITE, _info, sizeof(*_info)))
		return -EFAULT;
	memset(&info, 0, sizeof(info));
	info.type = mixer->card->type;
	info.attrib = mixer->attrib;
	info.elements = mixer->elements_count;
	info.groups = mixer->groups_count;
	strncpy(info.id, mixer->id, sizeof(info.id));
	strncpy(info.name, mixer->name, sizeof(info.name));
	info.switches = snd_switch_count(&mixer->switches);
	copy_to_user(_info, &info, sizeof(info));
	return 0;
}

static int snd_mixer_elements(snd_kmixer_t * mixer, snd_mixer_elements_t * _elements)
{
	snd_mixer_elements_t elements;
	snd_kmixer_element_t *kelement;
	int idx;

	if (verify_area(VERIFY_READ, _elements, sizeof(*_elements)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _elements, sizeof(*_elements)))
		return -EFAULT;
	copy_from_user(&elements, _elements, sizeof(elements));
	if (elements.pelements) {
		if (!elements.elements_size)
			return -EINVAL;
		if (verify_area(VERIFY_WRITE, elements.pelements, elements.elements_size * sizeof(void *)))
			return -EFAULT;
	} else {
		if (elements.elements_size)
			return -EINVAL;
	}
	snd_mixer_lock(mixer, 0);
	elements.elements = 0;
	for (idx = 0, kelement = mixer->elements;
	     kelement && idx < elements.elements_size &&
					     idx < mixer->elements_count;
	     idx++, kelement = kelement->next) {
		snd_mixer_eid_t eid;
		
		memset(&eid, 0, sizeof(eid));
		strncpy(eid.name, kelement->name, sizeof(eid.name));
		eid.index = kelement->index;
		eid.type = kelement->type;
		copy_to_user(&elements.pelements[idx], &eid, sizeof(eid));
		elements.elements++;
	}
	elements.elements_over = mixer->elements_count - idx;
	snd_mixer_lock(mixer, 1);
	copy_to_user(_elements, &elements, sizeof(*_elements));
	return 0;
}

static int snd_mixer_routes(snd_kmixer_t * mixer, snd_mixer_routes_t * _routes)
{
	snd_mixer_routes_t routes;
	snd_kmixer_element_t *kelement, *kelement1;
	int idx;

	if (verify_area(VERIFY_READ, _routes, sizeof(*_routes)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _routes, sizeof(*_routes)))
		return -EFAULT;
	copy_from_user(&routes, _routes, sizeof(routes));
	if (routes.proutes) {
		if (!routes.routes_size)
			return -EINVAL;
		if (verify_area(VERIFY_WRITE, routes.proutes, routes.routes_size * sizeof(snd_mixer_eid_t)))
			return -EFAULT;
	} else {
		if (routes.routes_size)
			return -EINVAL;
	}
	snd_mixer_lock(mixer, 0);
	kelement = snd_mixer_element_find(mixer, routes.eid.name, routes.eid.index, routes.eid.type);
	if (!kelement) {
		snd_mixer_lock(mixer, 1);
		return -ENOENT;
	}
	routes.routes = 0;
	for (idx = 0; idx < routes.routes_size &&
		      idx < kelement->routes_next.count; idx++) {
		snd_mixer_eid_t eid;
		
		kelement1 = kelement->routes_next.routes[idx];
		memset(&eid, 0, sizeof(eid));
		strncpy(eid.name, kelement1->name, sizeof(eid.name));
		eid.index = kelement1->index;
		eid.type = kelement1->type;
		copy_to_user(&routes.proutes[idx], &eid, sizeof(eid));
		routes.routes++;
	}
	routes.routes_over = kelement->routes_next.count - idx;
	snd_mixer_lock(mixer, 1);
	copy_to_user(_routes, &routes, sizeof(*_routes));
	return 0;
}

static int snd_mixer_groups(snd_kmixer_t * mixer, snd_mixer_groups_t * _groups)
{
	snd_mixer_groups_t groups;
	snd_kmixer_group_t *kgroup;
	int idx;

	if (verify_area(VERIFY_READ, _groups, sizeof(*_groups)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _groups, sizeof(*_groups)))
		return -EFAULT;
	copy_from_user(&groups, _groups, sizeof(groups));
	if (groups.pgroups) {
		if (!groups.groups_size)
			return -EINVAL;
		if (verify_area(VERIFY_WRITE, groups.pgroups, groups.groups_size * sizeof(snd_mixer_gid_t)))
			return -EFAULT;
	} else {
		if (groups.groups_size)
			return -EINVAL;
	}
	snd_mixer_lock(mixer, 0);
	groups.groups = 0;
	for (idx = 0, kgroup = mixer->groups;
	     kgroup && idx < groups.groups_size &&
					     idx < mixer->groups_count;
	     idx++, kgroup = kgroup->next) {
		snd_mixer_gid_t gid;
		
		memset(&gid, 0, sizeof(gid));
		strncpy(gid.name, kgroup->name, sizeof(gid.name));
		gid.index = kgroup->index;
		copy_to_user(&groups.pgroups[idx], &gid, sizeof(gid));
		groups.groups++;
	}
	groups.groups_over = mixer->groups_count - idx;
	snd_mixer_lock(mixer, 1);
	copy_to_user(_groups, &groups, sizeof(*_groups));
	return 0;
}

static int snd_mixer_group(snd_kmixer_t * mixer, snd_mixer_group_t * _group)
{
	snd_mixer_group_t group;
	snd_kmixer_group_t *kgroup;
	snd_kmixer_element_t *kelement;
	int idx;

	if (verify_area(VERIFY_READ, _group, sizeof(*_group)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _group, sizeof(*_group)))
		return -EFAULT;
	copy_from_user(&group, _group, sizeof(group));
	if (group.pelements) {
		if (!group.elements_size)
			return -EINVAL;
		if (verify_area(VERIFY_WRITE, group.pelements, group.elements_size * sizeof(snd_mixer_eid_t)))
			return -EFAULT;
	} else {
		if (group.elements_size)
			return -EINVAL;
	}
	snd_mixer_lock(mixer, 0);
	kgroup = snd_mixer_group_find(mixer, group.gid.name, group.gid.index);
	if (!kgroup) {
		snd_mixer_lock(mixer, 1);
		return -ENOENT;
	}
	group.elements = 0;
	for (idx = 0; idx < group.elements_size &&
				     idx < kgroup->elements_count; idx++) {
		snd_mixer_eid_t eid;
		
		kelement = kgroup->elements[idx];
		memset(&eid, 0, sizeof(eid));
		strncpy(eid.name, kelement->name, sizeof(eid.name));
		eid.index = kelement->index;
		eid.type = kelement->type;
		copy_to_user(&group.pelements[idx], &eid, sizeof(eid));
		group.elements++;
	}
	group.elements_over = kgroup->elements_count - idx;
	snd_mixer_lock(mixer, 1);
	copy_to_user(_group, &group, sizeof(*_group));
	return 0;
}

static int snd_mixer_element_info(snd_kmixer_file_t * mfile, snd_mixer_element_info_t * _info)
{
	snd_kmixer_t *mixer;
	snd_mixer_element_info_t info;
	snd_kmixer_element_t *kelement;
	int err;

	mixer = mfile->mixer;
	if (verify_area(VERIFY_READ, _info, sizeof(*_info)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _info, sizeof(*_info)))
		return -EFAULT;
	copy_from_user(&info, _info, sizeof(info));
	snd_mixer_lock(mixer, 0);
	kelement = snd_mixer_element_find(mixer, info.eid.name, info.eid.index, info.eid.type);
	if (!kelement) {
		snd_mixer_lock(mixer, 1);
		return -ENOENT;
	}
	if (!kelement->info) {
		snd_mixer_lock(mixer, 1);
		return -EIO;
	}
	err = kelement->info(mfile, &info, kelement->private_data);
	snd_mixer_lock(mixer, 1);
	copy_to_user(_info, &info, sizeof(*_info));
	return err;
}

static int snd_mixer_element_rw(snd_kmixer_file_t * mfile, snd_mixer_element_t * _element, int w_flag)
{
	snd_kmixer_t *mixer;
	snd_mixer_element_t element;
	snd_kmixer_element_t *kelement;
	int err;

	mixer = mfile->mixer;
	if (verify_area(VERIFY_READ, _element, sizeof(*_element)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _element, sizeof(*_element)))
		return -EFAULT;
	copy_from_user(&element, _element, sizeof(element));
	snd_mixer_lock(mixer, 0);
	if (w_flag && snd_mixer_busy(mfile)) {
		snd_mixer_lock(mixer, 1);
		return -EBUSY;
	}
	kelement = snd_mixer_element_find(mixer, element.eid.name, element.eid.index, element.eid.type);
	if (!kelement) {
		snd_mixer_lock(mixer, 1);
		return -ENOENT;
	}
	if (!kelement->control) {
		snd_mixer_lock(mixer, 1);
		return -EIO;
	}
	err = kelement->control(mfile, w_flag ? SND_MIXER_CMD_WRITE : SND_MIXER_CMD_READ,
				&element, sizeof(element), kelement->private_data);
	if (err > 0 && w_flag)
		snd_mixer_element_value_change(mfile, kelement, 0);
	snd_mixer_lock(mixer, 1);
	copy_to_user(_element, &element, sizeof(*_element));
	return err;
}

int snd_mixer_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	snd_kmixer_file_t *mfile;
	snd_kmixer_t *mixer;

	mfile = (snd_kmixer_file_t *) file->private_data;
	if (!mfile)
		return -EIO;
	mixer = mfile->mixer;
	if (((cmd >> 8) & 0xff) == 'R') {
		switch (cmd) {
		case SND_MIXER_IOCTL_PVERSION:
			return snd_ioctl_out((long *) arg, SND_MIXER_VERSION);
		case SND_MIXER_IOCTL_INFO:
			return snd_mixer_info(mixer, (snd_mixer_info_t *) arg);
		case SND_MIXER_IOCTL_ELEMENTS:
			return snd_mixer_elements(mixer, (snd_mixer_elements_t *) arg);
		case SND_MIXER_IOCTL_ROUTES:
			return snd_mixer_routes(mixer, (snd_mixer_routes_t *) arg);
		case SND_MIXER_IOCTL_GROUPS:
			return snd_mixer_groups(mixer, (snd_mixer_groups_t *) arg);
		case SND_MIXER_IOCTL_GROUP:
			return snd_mixer_group(mixer, (snd_mixer_group_t *) arg);
		case SND_MIXER_IOCTL_ELEMENT_INFO:
			return snd_mixer_element_info(mfile, (snd_mixer_element_info_t *) arg);
		case SND_MIXER_IOCTL_ELEMENT_READ:
			return snd_mixer_element_rw(mfile, (snd_mixer_element_t *) arg, 0);
		case SND_MIXER_IOCTL_ELEMENT_WRITE:
			return snd_mixer_element_rw(mfile, (snd_mixer_element_t *) arg, 1);
		}
	}
	return -ENXIO;
}

static int snd_mixer_control_ioctl(snd_card_t * card, snd_control_t * control, unsigned int cmd, unsigned long arg)
{
	snd_kmixer_t *mixer;
	unsigned int tmp = card->number * SND_MINOR_MIXERS;
	int idx;

	switch (cmd) {
	case SND_CTL_IOCTL_HW_INFO:
		{
			struct snd_ctl_hw_info *ptr = (struct snd_ctl_hw_info *) arg;

			ptr->mixerdevs = 0;
			for (idx = SND_MINOR_MIXERS - 1; idx >= 0; idx--) {
				if (snd_mixers[tmp + idx]) {
					ptr->mixerdevs = idx + 1;
					break;
				}
			}
			return 0;
		}
	case SND_CTL_IOCTL_MIXER_DEVICE:
		{
			int val = snd_ioctl_in((long *) arg);
			
			if (val < 0 || val >= SND_MINOR_MIXERS)
				return -EINVAL;
			if (!snd_mixers[tmp + val])
				return -EINVAL;
			control->mixer_device = val;
			return 0;
		}
	case SND_CTL_IOCTL_MIXER_INFO:
	case SND_CTL_IOCTL_MIXER_SWITCH_LIST:
	case SND_CTL_IOCTL_MIXER_SWITCH_READ:
	case SND_CTL_IOCTL_MIXER_SWITCH_WRITE:
		mixer = snd_mixers[tmp + control->mixer_device];
		if (!mixer)
			return -ENOENT;
		switch (cmd) {
		case SND_CTL_IOCTL_MIXER_INFO:
			return snd_mixer_info(mixer, (snd_mixer_info_t *) arg);
		case SND_CTL_IOCTL_MIXER_SWITCH_LIST:
			return snd_switch_list(&mixer->switches, (snd_switch_list_t *) arg);
		case SND_CTL_IOCTL_MIXER_SWITCH_READ:
			return snd_mixer_switch_read(mixer, (snd_switch_t *) arg);
		case SND_CTL_IOCTL_MIXER_SWITCH_WRITE:
			return snd_mixer_switch_write(control, mixer, (snd_switch_t *) arg);
		}
		break;
	}
	return -EAGAIN;
}

static long snd_mixer_read(struct file *file, char *buffer, long count)
{
	unsigned long flags;
	snd_kmixer_file_t *mfile;
	snd_kmixer_read_t *mread;
	snd_mixer_read_t rebuild;
	long result;

	mfile = (snd_kmixer_file_t *) file->private_data;
	if (!mfile || !mfile->mixer)
		return -ENOENT;
	snd_spin_lock(mfile, read_lock, &flags);
	mfile->read_active = 1;
	snd_spin_unlock(mfile, read_lock, &flags);
	result = 0;
	while (count >= sizeof(snd_mixer_read_t)) {
		snd_spin_lock(mfile, read_lock, &flags);
		if (mfile->rebuild) {
			mfile->rebuild = 0;
			snd_spin_unlock(mfile, read_lock, &flags);
			memset(&rebuild, 0, sizeof(rebuild));
			rebuild.cmd = SND_MIXER_READ_REBUILD;
			copy_to_user(buffer, &rebuild, sizeof(rebuild));
			snd_spin_unlock(mfile, read_lock, &flags);
			goto __move;
		} else {
			mread = mfile->first_item;
			if (mread) {
				mfile->first_item = mread->next;
				if (!mfile->first_item)
					mfile->last_item = NULL;
			}
		}
		snd_spin_unlock(mfile, read_lock, &flags);
		if (mread) {
			copy_to_user(buffer, &mread->data, sizeof(snd_mixer_read_t));
			snd_free(mread, sizeof(*mread));
		      __move:
			buffer += sizeof(snd_mixer_read_t);
			count -= sizeof(snd_mixer_read_t);
			result += sizeof(snd_mixer_read_t);
		} else {
			break;
		}
	}
	return result;
}

#ifdef SND_POLL
static unsigned int snd_mixer_poll(struct file *file, poll_table * wait)
{
	unsigned long flags;
	unsigned int mask;
	snd_kmixer_file_t *mfile;
	snd_kmixer_t *mixer;

	mfile = (snd_kmixer_file_t *) file->private_data;
	if (!mfile)
		return -EIO;
	mixer = mfile->mixer;

	snd_spin_lock(mfile, read_lock, &flags);
	mfile->read_active = 1;
	snd_spin_unlock(mfile, read_lock, &flags);

	snd_spin_lock(mfile, change_lock, &flags);
	snd_getlock(mfile, change) |= SND_WK_SLEEP;
	snd_spin_unlock(mfile, change_lock, &flags);
	snd_poll_wait(file, mfile, change, wait);

	mask = 0;
	if (mfile->first_item)
		mask |= POLLIN | POLLRDNORM;

	return mask;
}
#else
static int snd_mixer_select(struct file *file,
                            int sel_type,
                            select_table * wait)
{
	unsigned long flags;
	snd_kmixer_file_t *mfile;
	snd_kmixer_t *mixer;

	mfile = (snd_kmixer_file_t *) file->private_data;
	if (!mfile)
		return -EIO;
	mixer = mfile->mixer;

	switch (sel_type) {
	case SEL_IN:
		snd_spin_lock(mfile, read_lock, &flags);
		mfile->read_active = 1;
		snd_spin_unlock(mfile, read_lock, &flags);
		snd_spin_lock(mfile, change_lock, &flags);
		if (!mfile->first_item) {
			snd_getlock(mfile, change) |= SND_WK_SLEEP;
			snd_spin_unlock(mfile, change_lock, &flags);
			snd_select_wait(mfile, change, wait);
			return 0;
		}
		snd_getlock(mfile, change) &= ~SND_WK_SLEEP;
		snd_spin_unlock(mfile, change_lock, &flags);
		return 1;
	case SEL_OUT:
		break;
	case SEL_EX:
		break;
	}
	return 0;
}
#endif

/*
 *  /proc interface
 */

static void snd_mixer_proc_read(snd_info_buffer_t * buffer, void *private_data)
{
	snd_kmixer_t *mixer;

	mixer = (snd_kmixer_t *) private_data;
	snd_iprintf(buffer, "Mixer '%s' '%s'\n", mixer->id, mixer->name);
	snd_iprintf(buffer, "Elements %i\n", mixer->elements_count);
	snd_iprintf(buffer, "Groups %i\n", mixer->groups_count);
	snd_iprintf(buffer, "Switches %i\n", snd_switch_count(&mixer->switches));
}

/*
 *  REGISTRATION PART
 */

static snd_minor_t snd_mixer_reg =
{
	"mixer",		/* comment */
	NULL,			/* reserved */

	NULL,			/* unregister */

	NULL,			/* lseek */
	snd_mixer_read,		/* read */
	NULL,			/* write */
	snd_mixer_open,		/* open */
	snd_mixer_release,	/* release */
#ifdef SND_POLL
	snd_mixer_poll,		/* poll */
#else
	snd_mixer_select,	/* select */
#endif
	snd_mixer_ioctl,	/* ioctl */
	NULL			/* mmap */
};

snd_kmixer_t *snd_mixer_new(snd_card_t * card, char *id)
{
	snd_kmixer_t *mixer;

	mixer = (snd_kmixer_t *) snd_calloc(sizeof(*mixer));
	if (!mixer)
		return NULL;
	mixer->card = card;
	if (id) {
		strncpy(mixer->id, id, sizeof(mixer->id) - 1);
	}
	snd_mutex_prepare(mixer, lock);
	snd_switch_prepare(&mixer->switches);
	return mixer;
}

int snd_mixer_free(snd_kmixer_t * mixer)
{
	snd_kmixer_element_t *element;
	snd_kmixer_group_t *group;

	if (!mixer)
		return -EINVAL;
	while (mixer->elements) {
		element = mixer->elements;
		mixer->elements = element->next;
		snd_mixer_element_free(element);
	}
	while (mixer->groups) {
		group = mixer->groups;
		mixer->groups = group->next;
		snd_mixer_group_free(group);
	}
	snd_switch_free(&mixer->switches);
	if (mixer->private_data && mixer->private_free)
		mixer->private_free(mixer->private_data);
	snd_free(mixer, sizeof(*mixer));
	return 0;
}

int snd_mixer_switch_add(snd_kmixer_t * mixer, snd_kswitch_t * kswitch)
{
	int err = snd_switch_add(&mixer->switches, kswitch);
	if (err >= 0)
		snd_control_notify_switch_change(mixer->card,
						 SND_CTL_READ_SWITCH_ADD,
						 SND_CTL_IFACE_MIXER,
						 kswitch->name);
	return err;
}

int snd_mixer_switch_remove(snd_kmixer_t * mixer, snd_kswitch_t * kswitch)
{
	int err;

	err = snd_switch_remove(&mixer->switches, kswitch);
	if (err >= 0)
		snd_control_notify_switch_change(mixer->card,
						 SND_CTL_READ_SWITCH_REMOVE,
						 SND_CTL_IFACE_MIXER,
						 kswitch->name);
	return err;
}

snd_kswitch_t *snd_mixer_switch_new(snd_kmixer_t * mixer, snd_kswitch_t * ksw, void *private_data)
{
	snd_kswitch_t *sw;
	
	sw = snd_switch_new(ksw);
	if (!sw)
		return NULL;
	if (snd_mixer_switch_add(mixer, sw) < 0) {
		snd_switch_free_one(sw);
		return NULL;
	}
	sw->private_data = private_data;
	return sw;
}

int snd_mixer_switch_change(snd_kmixer_t * mixer, snd_kswitch_t * kswitch)
{
	if (!mixer || !kswitch)
		return -EINVAL;
	snd_control_notify_switch_change(mixer->card,
					 SND_CTL_READ_SWITCH_CHANGE,
					 SND_CTL_IFACE_MIXER,
					 kswitch->name);
	return 0;
}

static void snd_mixer_store(snd_kmixer_t * mixer)
{
	int size;
	long err;
	unsigned int pos;
	char *buffer;
	snd_kswitch_list_t *lswitch;
	char key[32];
	
	lswitch = &mixer->switches;
	snd_switch_lock(lswitch, 0);
	size = sizeof(mixer->id) + sizeof(mixer->name) +
	       snd_switch_size(lswitch);
	buffer = (char *) vmalloc(size);
	if (!buffer) {
		snd_printk("cannot store mixer settings - no enough memory\n" );
		snd_switch_lock(lswitch, 1);
		return;
	}
	pos = 0;
	memcpy(buffer + pos, mixer->id, sizeof(mixer->id));
	pos += sizeof(mixer->id);
	memcpy(buffer + pos, mixer->name, sizeof(mixer->name));
	pos += sizeof(mixer->name);
	err = snd_switch_store(lswitch, mixer, buffer + pos, size - pos);
	snd_switch_lock(lswitch, 1);
	if (err < 0) {
		snd_printd("cannot store mixer switch settings\n");
		vfree(buffer);
		return;
	}
	sprintf(key, "snd:%i/mixer:%i", mixer->card->number, mixer->device);
	snd_persist_store(key, buffer, size);
	vfree(buffer);
}

static void snd_mixer_restore(snd_kmixer_t * mixer)
{
	int size;
	long err;
	unsigned int pos;
	char *buffer;
	snd_kswitch_list_t *lswitch;
	char key[32];
	
	lswitch = &mixer->switches;
	sprintf(key, "snd:%i/mixer:%i", mixer->card->number, mixer->device);
	size = snd_persist_length(key);
	if (size <= 0)
		return;
	buffer = (char *) vmalloc(size);
	if (!buffer) {
		snd_printk("cannot restore mixer settings: no enough memory\n" );
		return;
	}
	size = snd_persist_restore(key, buffer, size);
	/* ok. I'm very paranoid in the code below */
	if (size < (int)sizeof(unsigned int)) {
		snd_printd("cannot restore mixer settings\n");
		vfree(buffer);
		return;
	}
	pos = 0;
	if (strcmp(buffer + pos, mixer->id)) {
		snd_printd("cannot restore mixer settings: discording id\n");
		vfree(buffer);
		return;
	}
	pos += sizeof(mixer->id);
	if (strcmp(buffer + pos, mixer->name)) {
		snd_printd("cannot restore mixer settings: discording name\n");
		vfree(buffer);
		return;
	}
	pos += sizeof(mixer->name);
	err = snd_switch_restore(lswitch, mixer, buffer + pos, size - pos);
	if (err < 0)
		snd_printd("cannot restore mixer switch settings\n");
	vfree(buffer);
}

int snd_mixer_register(snd_kmixer_t * mixer, int device)
{
	char name[16];
	int idx, err, cardnum;
	snd_info_entry_t *entry;
	struct snd_stru_mixer_notify *notify;

	if (device < 0 || device > 1)
		return -EINVAL;
	if (!mixer->card)
		return -EINVAL;
	idx = ((cardnum = mixer->card->number) * SND_MINOR_MIXERS) + device;
	snd_mutex_down_static(register);
	if (snd_mixers[idx]) {
		snd_mutex_up_static(register);
		return -EBUSY;
	}
	mixer->device = device;
	snd_mixers[idx] = mixer;
	sprintf(name, "mixerC%iD%i", mixer->card->number, device);
	if ((err = snd_register_device(SND_DEVICE_TYPE_MIXER, mixer->card, device, &snd_mixer_reg, name)) < 0) {
		snd_mutex_up_static(register);
		snd_mixers[idx] = NULL;
		return err;
	}
	for (notify = snd_mixer_notify_first; notify; notify=notify->next) {
		if (notify->n_register)
			notify->n_register(SND_MINOR_MIXER + idx, mixer);
	}
	snd_mutex_up_static(register);
	sprintf(name, "mixer%i", device);
	if ((entry = snd_info_create_entry(mixer->card, name)) != NULL) {
		entry->private_data = mixer;
		entry->mode = S_IFREG | S_IRUGO | S_IWUSR;
		entry->t.text.read_size = 256;
		entry->t.text.read = snd_mixer_proc_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	mixer->proc_entry = entry;
	snd_mixer_restore(mixer);
	return 0;
}

int snd_mixer_unregister(snd_kmixer_t * mixer)
{
	int idx;
	struct snd_stru_mixer_notify *notify;

	if (!mixer)
		return -EINVAL;
	if (mixer->proc_entry) {
		snd_info_unregister(mixer->proc_entry);
		mixer->proc_entry = NULL;
	}
	snd_mixer_store(mixer);
	snd_mutex_down_static(register);
	idx = (mixer->card->number * SND_MINOR_MIXERS) + mixer->device;
	if (snd_mixers[idx] != mixer) {
		snd_mutex_up_static(register);
		return -EINVAL;
	}
	snd_unregister_device(SND_DEVICE_TYPE_MIXER, mixer->card, mixer->device);
	for (notify = snd_mixer_notify_first; notify; notify=notify->next) {
		if (notify->n_unregister)
			notify->n_unregister(SND_MINOR_MIXER + idx, mixer);
	}
	snd_mixers[idx] = NULL;
	snd_mutex_up_static(register);
	return snd_mixer_free(mixer);
}

int snd_mixer_notify(struct snd_stru_mixer_notify *notify, int nfree)
{
	int idx;
	struct snd_stru_mixer_notify *tnotify;

	if (!notify || !notify->n_register || !notify->n_unregister)
		return -EINVAL;
	snd_mutex_down_static(register);
	if (nfree) {
		tnotify = snd_mixer_notify_first;
		if (tnotify == notify) {
			snd_mixer_notify_first = notify->next;
		} else {
			while (tnotify && tnotify->next != notify)
				tnotify = tnotify->next;
			if (!tnotify) {
				snd_mutex_up_static(register);
				return -ENOENT;
			}
			tnotify->next = tnotify->next->next;
		}
		for (idx = 0; idx < SND_MIXERS; idx++) {
			if (snd_mixers[idx] == NULL)
				continue;
			notify->n_unregister(idx + SND_MINOR_MIXER,
				             snd_mixers[idx]);
		}
	} else {
		notify->next = NULL;
		if (!snd_mixer_notify_first) {
			snd_mixer_notify_first = notify;
		} else {
			for (tnotify = snd_mixer_notify_first;
			     tnotify->next;
			     tnotify = tnotify->next);
			tnotify->next = notify;
		}
		for (idx = 0; idx < SND_MIXERS; idx++) {
			if (snd_mixers[idx] == NULL)
				continue;
			notify->n_register(idx + SND_MINOR_MIXER,
				           snd_mixers[idx]);
		}
	}
	snd_mutex_up_static(register);
	return 0;
}

/*
 *  INIT PART
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_mixer_export;
#endif

int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_mixer_export) < 0)
		return -ENOMEM;
#endif
	snd_control_register_ioctl(snd_mixer_control_ioctl);
	return 0;
}

void cleanup_module(void)
{
	snd_control_unregister_ioctl(snd_mixer_control_ioctl);
}
