/*
 *  ISA DMA support functions
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"

/*
 * DMA isn't disabled now when residue is read. Looks that this causes
 * problems with some DMA controllers. Maybe old controllers may give
 * wrong values, so if you have this problem - try to change the line bellow.
 */

#if LinuxVersionCode(2, 2, 0) > LINUX_VERSION_CODE
#define isa_dma_bridge_buggy (0)
#define isa_dma_lock(f) snd_cli(&f)
#define isa_dma_unlock(f) snd_sti(&f)
#else
#define isa_dma_lock(f) f = claim_dma_lock()
#define isa_dma_unlock(f) release_dma_lock(f)
#endif

#undef SND_DISABLE_DMA_WHEN_RESIDUE

/*
 * I added some delay here [Jaroslav]. Maybe this can help for some broken
 * ISA DMA controllers.
 */

#define SND_DELAY_BY_DMA

/*
 *
 */
 
#ifdef SND_DELAY_BY_DMA
#define snd_dma_delay() snd_delay(3)
#else
#define snd_dma_delay() do { ; } while(0)
#endif

void snd_dma_program(int dma,
		     const void *buf, unsigned int size,
                     unsigned short mode)
{
	unsigned long flags;

	/* we don't have a DMA spinlock... */
	isa_dma_lock(flags);
	disable_dma(dma);
	snd_dma_delay();
	clear_dma_ff(dma);
	snd_dma_delay();
	set_dma_mode(dma, mode);
	set_dma_addr(dma, virt_to_bus((void *) buf));
	set_dma_count(dma, size);
	snd_dma_delay();
	if (!(mode & DMA_MODE_NO_ENABLE))
		enable_dma(dma);
	isa_dma_unlock(flags);
}

unsigned int snd_dma_residue(int dma)
{
	unsigned long flags;
	unsigned int result;

	/* we don't have a DMA spinlock... */
	isa_dma_lock(flags);
	if (!isa_dma_bridge_buggy) {
		disable_dma(dma);
		snd_dma_delay();
	}
	clear_dma_ff(dma);
	snd_dma_delay();
	result = get_dma_residue(dma);
	if (!isa_dma_bridge_buggy)
		enable_dma(dma);
	isa_dma_unlock(flags);
	return result;
}
