/*
 *  Driver for Yamaha OPL3-SA[2,3] soundcards
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "cs4231.h"
#include "mpu401.h"
#include "opl3.h"
#include "initval.h"

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_port[SND_CARDS] = SND_DEFAULT_PORT;	/* 0xf86,0x370,0x100 */
int snd_sb_port[SND_CARDS] = SND_DEFAULT_PORT;	/* 0x220,0x240,0x260 */
int snd_wss_port[SND_CARDS] = SND_DEFAULT_PORT;	/* 0x530,0xe80,0xf40,0x604 */
int snd_fm_port[SND_CARDS] = SND_DEFAULT_PORT;	/* 0x388 */
int snd_midi_port[SND_CARDS] = SND_DEFAULT_PORT;/* 0x330,0x300 */
int snd_irq[SND_CARDS] = SND_DEFAULT_IRQ;	/* 0,1,3,5,9,11,12,15 */
int snd_dma1[SND_CARDS] = SND_DEFAULT_DMA;	/* 1,3,5,6,7 */
int snd_dma2[SND_CARDS] = SND_DEFAULT_DMA;	/* 1,3,5,6,7 */
int snd_dma1_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
int snd_dma2_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
#ifdef CONFIG_ISAPNP
int snd_isapnp[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 1};
#endif
#ifdef MODULE_PARM
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for OPL3-SA soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for OPL3-SA soundcard.");
MODULE_PARM(snd_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_port, "Port # for OPL3-SA driver.");
MODULE_PARM(snd_sb_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_sb_port, "SB port # for OPL3-SA driver.");
MODULE_PARM(snd_wss_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_wss_port, "WSS port # for OPL3-SA driver.");
MODULE_PARM(snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_fm_port, "FM port # for OPL3-SA driver.");
MODULE_PARM(snd_midi_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_midi_port, "MIDI port # for OPL3-SA driver.");
MODULE_PARM(snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_irq, "IRQ # for OPL3-SA driver.");
MODULE_PARM(snd_dma1, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1, "DMA1 # for OPL3-SA driver.");
MODULE_PARM(snd_dma2, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma2, "DMA2 # for OPL3-SA driver.");
MODULE_PARM(snd_dma1_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1_size, "DMA1 size in kB for OPL3-SA driver.");
MODULE_PARM(snd_dma2_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma2_size, "DMA2 size in kB for OPL3-SA driver.");
MODULE_PARM(snd_isapnp, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_isapnp, "ISA PnP detection for specified soundcard.");
#endif

struct snd_opl3sa {
	int version;		/* 2 or 3 */
	snd_irq_t *irqptr;
	snd_dma_t *dma1ptr;
	snd_dma_t *dma2ptr;
	unsigned short port;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rawmidi;
	snd_synth_t *synth;
	unsigned short pcm_status_reg;
	snd_spin_define(reg);
	snd_kmixer_element_t *mvolume;
	snd_kmixer_element_t *mswitch;
#ifdef CONFIG_ISAPNP
	struct isapnp_logdev *logdev;
#endif
};

static struct snd_opl3sa *snd_opl3sa_cards[SND_CARDS] = SND_DEFAULT_PTR;

#ifdef CONFIG_ISAPNP
static unsigned int snd_opl3sa_pnpids[] = {
	/* Yamaha YMF719E-S (Genius Sound Maker 3DX) */
	(ISAPNP_VENDOR('Y','M','H')<<16)|ISAPNP_DEVICE(0x0020),   /* DEVICE */
	(ISAPNP_VENDOR('Y','M','H')<<16)|ISAPNP_FUNCTION(0x0021), /* all */
	/* Yamaha OPL3-SA3 (integrated on Intel's Pentium II AL440LX motherboard) */
	(ISAPNP_VENDOR('Y','M','H')<<16)|ISAPNP_DEVICE(0x0030),   /* DEVICE */
	(ISAPNP_VENDOR('Y','M','H')<<16)|ISAPNP_FUNCTION(0x0021), /* all */
	/* ??? */
	(ISAPNP_VENDOR('Y','M','H')<<16)|ISAPNP_DEVICE(0x0800),   /* DEVICE */
	(ISAPNP_VENDOR('Y','M','H')<<16)|ISAPNP_FUNCTION(0x0021), /* all */
	/* --- */
	0       /* end */
};
#endif

static void snd_opl3sa_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_opl3sa_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static unsigned char snd_opl3sa_read(unsigned short port, unsigned char reg)
{
	unsigned long flags;
	unsigned char result;

	snd_cli(&flags);
#if 0
	outb(0x1d, port);	/* password */
	printk("read [0x%x] = 0x%x\n", port, inb(port));
#endif
	outb(reg, port);	/* register */
	result = inb(port + 1);
	snd_sti(&flags);
#if 0
	printk("read [0x%x] = 0x%x [0x%x]\n", port, result, inb(port));
#endif
	return result;
}

static void snd_opl3sa_write(unsigned short port,
			     unsigned char reg, unsigned char value)
{
	unsigned long flags;

	snd_cli(&flags);
#if 0
	outb(0x1d, port);	/* password */
#endif
	outb(reg, port);	/* register */
	outb(value, port + 1);
	snd_sti(&flags);
}

static int snd_opl3sa_detect(struct snd_opl3sa *oplcard,
			     unsigned short port,
			     unsigned short wss_port,
			     unsigned short sb_port,
			     unsigned short fm_port,
			     unsigned short midi_port)
{
	snd_card_t *card;
	unsigned char tmp, tmp1;
	char str[2];

	card = oplcard->card;
	if (snd_register_ioport(card, port, 2, "OPL3-SA control", NULL) < 0)
		return -EBUSY;
	if (snd_register_ioport(card, wss_port, 8, "OPL3-SA WSS", NULL) < 0)
		goto __nodev;
	if (sb_port >= 0x200 && sb_port < 0x300)
		if (snd_register_ioport(card, sb_port, 16, "OPL3-SA SB", NULL) < 0)
			goto __nodev;
	if (fm_port >= 0x340 && fm_port < 0x400)
		if (snd_register_ioport(card, fm_port, 8, "OPL3-SA AdLib FM", NULL) < 0)
			goto __nodev;
	if (midi_port >= 0x300 && midi_port < 0x340)
		if (snd_register_ioport(card, midi_port, 2, "OPL3-SA MPU-401", NULL) < 0)
			goto __nodev;
#if 0
	snd_printk("REG 0A = 0x%x\n", snd_opl3sa_read(port, 0x0a));
#endif
	oplcard->port = port;
	oplcard->version = 0;
	tmp = snd_opl3sa_read(port, 0x0a);
	if (tmp == 0xff) {
		snd_printd("OPL3-SA [0x%x] detect = 0x%x\n", port, tmp);
		goto __nodev;
	}
	switch (tmp & 0x07) {
	case 0x01:
		oplcard->version = 2;
		break;
	default:
		oplcard->version = 3;
		/* 0x02 - standard */
		/* 0x03 - YM715B */
		/* 0x04 - YM719 - OPL-SA4? */
		/* 0x05 - OPL3-SA3 - Libretto 100 */
		break;
#if 0
	default:
		snd_printd("OPL3-SA [0x%x] detect (0) = 0x%x\n", port, tmp);
		goto __nodev;
#endif
	}
	str[0] = oplcard->version + '0';
	str[1] = 0;
	strcat(card->shortname, str);
	snd_opl3sa_write(port, 0x0a, tmp ^ 7);
	if ((tmp1 = snd_opl3sa_read(port, 0x0a)) != tmp) {
		snd_printd("OPL3-SA [0x%x] detect (1) = 0x%x (0x%x)\n", port, tmp, tmp1);
		goto __nodev;
	}
	/* try if the MIC register is accesible */
	tmp = snd_opl3sa_read(port, 0x09);
	snd_opl3sa_write(port, 0x09, 0x8a);
	if (((tmp1 = snd_opl3sa_read(port, 0x09)) & 0x9f) != 0x8a) {
		snd_printd("OPL3-SA [0x%x] detect (2) = 0x%x (0x%x)\n", port, tmp, tmp1);
		goto __nodev;
	}
	snd_opl3sa_write(port, 0x09, 0x9f);
	/* initialization */
	snd_opl3sa_write(port, 0x01, 0x00);	/* Power Management - default */
	snd_opl3sa_write(port, 0x02, 0x00);	/* System Control - default */
	snd_opl3sa_write(port, 0x03, 0x0d);	/* Interrupt Channel Configuration - IRQ A = OPL3 + MPU + WSS */
	if (oplcard->dma2ptr == NULL) {
		snd_opl3sa_write(port, 0x06, 0x03);	/* DMA Configuration - DMA A = WSS-R + WSS-P */
	} else {
		snd_opl3sa_write(port, 0x06, 0x21);	/* DMA Configuration - DMA B = WSS-R, DMA A = WSS-P */
	}
	snd_opl3sa_write(port, 0x0a, 0x80 | (tmp & 7));	/* Miscellaneous - default */
	if (oplcard->version > 2) {
		snd_opl3sa_write(port, 0x12, 0x00);	/* Digital Block Partial Power Down - default */
		snd_opl3sa_write(port, 0x13, 0x00);	/* Analog Block Partial Power Down - default */
	}
	return 0;

      __nodev:
	snd_unregister_ioports(card);
	return -ENODEV;
}

static void snd_opl3sa_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned short status;
	unsigned long flags;
	struct snd_opl3sa *oplcard = (struct snd_opl3sa *) dev_id;

	if (!oplcard || !oplcard->card)
		return;

	snd_spin_lock(oplcard, reg, &flags);
	outb(0x04, oplcard->port);	/* register - Interrupt IRQ-A status */
	status = inb(oplcard->port + 1);
	snd_spin_unlock(oplcard, reg, &flags);

	if (oplcard->synth && (status & 0x20))
		snd_opl3_interrupt(oplcard->synth);

	if (oplcard->rawmidi && (status & 0x10))
		snd_mpu401_uart_interrupt(oplcard->rawmidi);

	if (oplcard->pcm && (status & 0x07)) {	/* TI,CI,PI */
		status = inb(oplcard->pcm_status_reg);
		if (status & 0x01)
			snd_cs4231_interrupt(oplcard->pcm, status);
	}

	if (status & 0x40) {
		/* reading from Master Lch register at 0x07 clears this bit */
		snd_opl3sa_read(oplcard->port, 0x08);
		snd_opl3sa_read(oplcard->port, 0x07);
		snd_mixer_element_value_change(NULL, oplcard->mvolume, 1);
		snd_mixer_element_value_change(NULL, oplcard->mswitch, 1);
	}
}

static int snd_opl3sa_resources(int dev, struct snd_opl3sa *oplcard, snd_card_t * card)
{
	static int possible_irqs[] = {-1}; /* must be specified by user */
	static int possible_dmas[] = {-1}; /* must be specified by user */
	int err;

	if ((err = snd_register_interrupt(card,
			"OPL3-SA", snd_irq[dev], SND_IRQ_TYPE_ISA,
			snd_opl3sa_interrupt, oplcard,
			possible_irqs, &oplcard->irqptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"OPL3-SA playback", snd_dma1[dev], SND_DMA_TYPE_ISA,
			snd_dma1_size[dev], possible_dmas, &oplcard->dma1ptr)) < 0)
		return err;
	if (snd_dma2[dev] >= 0) {
		if ((err = snd_register_dma_channel(card,
				"OPL3-SA record", snd_dma2[dev],
				SND_DMA_TYPE_ISA, snd_dma2_size[dev],
				possible_dmas, &oplcard->dma2ptr)) < 0)
			return err;
	} else {
		oplcard->dma2ptr = NULL;
	}
	return 0;
}

static int snd_opl3sa_master_volume(int w_flag, int *voices,
				      struct snd_opl3sa *oplcard)
{
	unsigned long flags;
	unsigned char lreg, rreg, oleft, oright;
	int change = 0;
	
	snd_spin_lock(oplcard, reg, &flags);
	oleft = 15 - ((lreg = snd_opl3sa_read(oplcard->port, 0x07)) & 15);
	oright = 15 - ((rreg = snd_opl3sa_read(oplcard->port, 0x08)) & 15);
	if (!w_flag) {
		voices[0] = oleft;
		voices[1] = oright;
	} else {
		change = oleft != voices[0] || oright != voices[1];
		lreg &= ~15;
		lreg |= 15 - voices[0];
		rreg &= ~15;
		rreg |= 15 - voices[1];
		snd_opl3sa_write(oplcard->port, 0x07, lreg);
		snd_opl3sa_write(oplcard->port, 0x08, rreg);
	}
	snd_spin_unlock(oplcard, reg, &flags);
	return change;
	
}

static int snd_opl3sa_master_switch(int w_flag, unsigned int *bitmap,
				      struct snd_opl3sa *oplcard)
{
	unsigned long flags;
	unsigned char lreg, rreg, oleft, oright;
	int change = 0;
	
	snd_spin_lock(oplcard, reg, &flags);
	oleft = (lreg = snd_opl3sa_read(oplcard->port, 0x07)) & 0x80 ? 0 : 1;
	oright = (rreg = snd_opl3sa_read(oplcard->port, 0x08)) & 0x80 ? 0 : 1;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, oleft);
		snd_mixer_set_bit(bitmap, 0, oright);
	} else {
		change = oleft != snd_mixer_get_bit(bitmap, 0) ||
		         oright != snd_mixer_get_bit(bitmap, 1);
		lreg &= 0x7f;
		if (!snd_mixer_get_bit(bitmap, 0))
			lreg |= 0x80;
		rreg &= 0x7f;
		if (!snd_mixer_get_bit(bitmap, 1))
			rreg |= 0x80;
		snd_opl3sa_write(oplcard->port, 0x07, lreg);
		snd_opl3sa_write(oplcard->port, 0x08, rreg);
	}
	snd_spin_unlock(oplcard, reg, &flags);
	return change;
	
}

static int snd_opl3sa_mic_volume(int w_flag, int *voices,
				      struct snd_opl3sa *oplcard)
{
	unsigned long flags;
	unsigned char reg, oval;
	int change = 0;
	
	snd_spin_lock(oplcard, reg, &flags);
	oval = 31 - ((reg = snd_opl3sa_read(oplcard->port, 0x09)) & 31);
	if (!w_flag) {
		voices[0] = oval;
	} else {
		change = oval != voices[0];
		reg &= ~31;
		reg |= 31 - voices[0];
		snd_opl3sa_write(oplcard->port, 0x09, reg);
	}
	snd_spin_unlock(oplcard, reg, &flags);
	return change;
	
}

static int snd_opl3sa_mic_switch(int w_flag, int *value,
				      struct snd_opl3sa *oplcard)
{
	unsigned long flags;
	unsigned char reg, oval;
	int change = 0;
	
	snd_spin_lock(oplcard, reg, &flags);
	oval = (reg = snd_opl3sa_read(oplcard->port, 0x09)) & 0x80 ? 0 : 1;
	if (!w_flag) {
		*value = oval;
	} else {
		change = oval != *value;
		reg &= 0x7f;
		if (!*value)
			reg |= 0x80;
		snd_opl3sa_write(oplcard->port, 0x09, reg);
	}
	snd_spin_unlock(oplcard, reg, &flags);
	return change;
	
}

static int snd_opl3sa_tone_control(int w_flag,
			struct snd_mixer_element_tone_control1 *tc1,
			struct snd_opl3sa *oplcard)
{
	unsigned long flags;
	unsigned char obass, otreble;
	int change = 0;
	
	snd_spin_lock(oplcard, reg, &flags);
	obass = snd_opl3sa_read(oplcard->port, 0x15) & 7;
	otreble = snd_opl3sa_read(oplcard->port, 0x16) & 7;
	if (w_flag) {
		if (tc1->tc & SND_MIXER_TC1_BASS) {
			change |= obass != tc1->bass;
			obass = ((tc1->bass & 7) << 4) | (tc1->bass & 7);
			snd_opl3sa_write(oplcard->port, 0x15, obass);
		}
		if (tc1->tc & SND_MIXER_TC1_TREBLE) {
			change |= otreble != tc1->treble;
			otreble = ((tc1->treble & 7) << 4) | (tc1->treble & 7);
			snd_opl3sa_write(oplcard->port, 0x16, otreble);
		}
	} else {
		tc1->tc = SND_MIXER_TC1_BASS | SND_MIXER_TC1_TREBLE;
		tc1->bass = obass;
		tc1->treble = otreble;
	}
	snd_spin_unlock(oplcard, reg, &flags);
	return change;
}

static int snd_opl3sa_3d_control(int w_flag,
			struct snd_mixer_element_3d_effect1 *eff,
			struct snd_opl3sa *oplcard)
{
	unsigned long flags;
	unsigned char owide;
	int change = 0;
	
	snd_spin_lock(oplcard, reg, &flags);
	owide = snd_opl3sa_read(oplcard->port, 0x14) & 7;
	if (w_flag) {
		if (eff->effect & SND_MIXER_EFF1_WIDE) {
			change |= owide != eff->wide;
			owide = ((eff->wide & 7) << 4) | (eff->wide & 7);
			snd_opl3sa_write(oplcard->port, 0x14, owide);
		}
	} else {
		eff->effect = SND_MIXER_EFF1_WIDE;
		eff->wide = owide;
	}
	snd_spin_unlock(oplcard, reg, &flags);
	return change;
}

static int snd_opl3sa_mixer(struct snd_opl3sa *oplcard, snd_kmixer_t * mixer)
{
	snd_kmixer_group_t *group, *ogroup;
	snd_kmixer_element_t *oelement1, *oelement2, *oelement3;
	snd_kmixer_element_t *element1, *element2;
	static struct snd_mixer_element_tone_control1_info tinfo;
	static struct snd_mixer_element_3d_effect1_info einfo;
	static struct snd_mixer_element_volume1_range mic_range[2] = {
		{0, 31, -3450, 1200}
	};
	static struct snd_mixer_element_volume1_range master_range[2] = {
		{0, 15, -3000, 0},
		{0, 15, -3000, 0}
	};

	/* reassign AUX1 to CD */
	if (snd_mixer_group_rename(mixer,
				SND_MIXER_IN_AUX, 0,
				SND_MIXER_IN_CD, 0) < 0)
		goto __error;
	if (snd_mixer_element_rename(mixer,
				SND_MIXER_IN_AUX, 0, SND_MIXER_ETYPE_INPUT,
				SND_MIXER_IN_CD, 0) < 0)
		goto __error;
	if (snd_mixer_element_rename(mixer,
				"Aux Input Volume", 0, SND_MIXER_ETYPE_VOLUME1,
				"CD Input Volume", 0) < 0)
		goto __error;
	if (snd_mixer_element_rename(mixer,
				"Aux Input Switch", 0, SND_MIXER_ETYPE_SWITCH1,
				"CD Input Switch", 0) < 0)
		goto __error;
	/* reassign AUX2 to FM */
	if (snd_mixer_group_rename(mixer,
				SND_MIXER_IN_AUX, 1,
				SND_MIXER_IN_FM, 0) < 0)
		goto __error;
	if (snd_mixer_element_rename(mixer,
				SND_MIXER_IN_AUX, 1, SND_MIXER_ETYPE_INPUT,
				SND_MIXER_IN_FM, 0) < 0)
		goto __error;
	if (snd_mixer_element_rename(mixer,
				"Aux Input Volume", 1, SND_MIXER_ETYPE_VOLUME1,
				"FM Input Volume", 0) < 0)
		goto __error;
	if (snd_mixer_element_rename(mixer,
				"Aux Input Switch", 1, SND_MIXER_ETYPE_SWITCH1,
				"FM Input Switch", 0) < 0)
		goto __error;
	/* reassign MIC Volume to MIC Gain */
	if (snd_mixer_element_rename(mixer,
				"MIC Volume", 0, SND_MIXER_ETYPE_VOLUME1,
				"MIC Gain Volume", 0) < 0)
		goto __error;
	if ((ogroup = snd_mixer_group_find(mixer, SND_MIXER_IN_MIC, 0)) == NULL)
		goto __error;
	if ((oelement1 = snd_mixer_element_find(mixer, "MIC Gain Volume", 0, SND_MIXER_ETYPE_VOLUME1)) == NULL)
		goto __error;
	if ((oelement2 = snd_mixer_element_find(mixer, SND_MIXER_ELEMENT_INPUT_MUX, 0, SND_MIXER_ETYPE_MUX1)) == NULL)
		goto __error;
	if ((oelement3 = snd_mixer_element_find(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, SND_MIXER_ETYPE_ACCU1)) == NULL)
		goto __error;
	if (snd_mixer_element_route_remove(mixer, oelement1, oelement3) < 0)
		goto __error;
	if ((element1 = snd_mixer_lib_volume1(mixer, "MIC Volume", 0, 1, mic_range, (snd_mixer_volume1_control_t *)snd_opl3sa_mic_volume, oplcard)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, ogroup, element1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, oelement1, element1) < 0)
		goto __error;
	if ((element2 = snd_mixer_lib_sw2(mixer, "MIC Switch", 0, (snd_mixer_sw2_control_t *)snd_opl3sa_mic_switch, oplcard)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, ogroup, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, oelement3) < 0)
		goto __error;
	/* master volume */
	if ((oelement1 = snd_mixer_element_find(mixer, SND_MIXER_OUT_MASTER, 0, SND_MIXER_ETYPE_OUTPUT)) == NULL)
		goto __error;
	if (snd_mixer_element_route_remove(mixer, oelement3, oelement1) < 0)
		goto __error;
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_OUT_MASTER, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_volume1(mixer, "Master Volume", 0, 2, master_range, (snd_mixer_volume1_control_t *)snd_opl3sa_master_volume, oplcard)) == NULL)
		goto __error;
	oplcard->mvolume = element1;
	if (snd_mixer_group_element_add(mixer, group, element1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, oelement3, element1) < 0)
		goto __error;
	if ((element2 = snd_mixer_lib_sw1(mixer, "Master Switch", 0, 2, (snd_mixer_sw1_control_t *)snd_opl3sa_master_switch, oplcard)) == NULL)
		goto __error;
	oplcard->mswitch = element2;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (oplcard->version > 2) {
		if ((group = snd_mixer_lib_group(mixer, SND_MIXER_GRP_TONE_CONTROL, 0)) == NULL)
			goto __error;
		memset(&tinfo, 0, sizeof(tinfo));
		tinfo.tc = SND_MIXER_TC1_BASS | SND_MIXER_TC1_TREBLE;
		tinfo.max_bass = tinfo.max_treble = 7;
		tinfo.max_bass_dB = tinfo.max_treble_dB = 1050;
		if ((element1 = snd_mixer_lib_tone_control1(mixer, SND_MIXER_GRP_TONE_CONTROL, 0, &tinfo, (snd_mixer_tone_control1_control_t *)snd_opl3sa_tone_control, oplcard)) == NULL)
			goto __error;
		if (snd_mixer_group_element_add(mixer, group, element1) < 0)
			goto __error;
		if (snd_mixer_element_route_add(mixer, element2, element1) < 0)
			goto __error;
		if ((group = snd_mixer_lib_group(mixer, SND_MIXER_GRP_EFFECT_3D, 0)) == NULL)
			goto __error;
		memset(&einfo, 0, sizeof(einfo));
		einfo.effect = SND_MIXER_EFF1_WIDE;
		einfo.max_wide = 7;
		if ((element2 = snd_mixer_lib_3d_effect1(mixer, SND_MIXER_GRP_EFFECT_3D, 0, &einfo, (snd_mixer_3d_effect1_control_t *)snd_opl3sa_3d_control, oplcard)) == NULL)
			goto __error;
		if (snd_mixer_group_element_add(mixer, group, element2) < 0)
			goto __error;
		if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
			goto __error;
		if (snd_mixer_element_route_add(mixer, element2, oelement1) < 0)
			goto __error;		
	} else {
		if (snd_mixer_element_route_add(mixer, element2, oelement1) < 0)
			goto __error;
	}
	return 0;

      __error:
      	oplcard->mvolume = NULL;
      	oplcard->mswitch = NULL;
      	return -ENOMEM;
}

#ifdef CONFIG_ISAPNP
static int snd_opl3sa_isapnp(int dev, struct snd_opl3sa *oplcard)
{
        int idx, step;
        unsigned int tmp;
	struct isapnp_dev *pdev;
	struct isapnp_logdev *logdev;
	struct isapnp_config cfg;

	step = 2;
	idx = 0;
      __again:
      	pdev = NULL;
	while ((tmp = snd_opl3sa_pnpids[idx]) != 0) {
		pdev = isapnp_find_device(tmp >> 16, tmp & 0xffff, dev);
		if (pdev)
			break;
		idx += step;
	}
	if (!pdev) {
		snd_printdd("isapnp failed for OPL3-SA\n");
		return -ENODEV;
	}
	tmp = snd_opl3sa_pnpids[idx+1];
	oplcard->logdev = logdev = isapnp_find_logdev(pdev, tmp >> 16, tmp & 0xffff, 0);
	if (!oplcard->logdev) {
                snd_printdd("isapnp failed for OPL3-SA - %i\n", idx);
                idx += step;
                goto __again;           /* maybe we have another config */
        }
	if (isapnp_cfg_begin(logdev->dev->csn, logdev->number))
		return -EAGAIN;
	/* PnP initialization */
	if (isapnp_config_init(&cfg, logdev)) {
		isapnp_cfg_end();
		return -EAGAIN;
	}
	if (snd_sb_port[dev] != SND_AUTO_PORT)
		cfg.port[0] = snd_sb_port[dev];
	if (snd_wss_port[dev] != SND_AUTO_PORT)
		cfg.port[1] = snd_wss_port[dev];
	if (snd_fm_port[dev] != SND_AUTO_PORT)
		cfg.port[2] = snd_fm_port[dev];
	if (snd_midi_port[dev] != SND_AUTO_PORT)
		cfg.port[3] = snd_midi_port[dev];
	if (snd_port[dev] != SND_AUTO_PORT)
		cfg.port[4] = snd_port[dev];
	if (snd_dma1[dev] != SND_AUTO_DMA)
		cfg.dma[0] = snd_dma1[dev];
	if (snd_dma2[dev] != SND_AUTO_DMA)
		cfg.dma[1] = snd_dma2[dev];
	if (snd_irq[dev] != SND_AUTO_IRQ)
		cfg.irq[0] = snd_irq[dev];
	if (isapnp_configure(&cfg)<0) {
		snd_printk("OPL3-SA isapnp configure failure (out of resources?)\n");
		isapnp_cfg_end();
		return -EBUSY;
	}
	snd_printdd("isapnp OPL3-SA: sb port=0x%x, wss port=0x%x, fm port=0x%x, midi port=0x%x\n", cfg.port[0], cfg.port[1], cfg.port[2], cfg.port[3]);
	snd_printdd("isapnp OPL3-SA: control port=0x%x, dma1=%i, dma2=%i, irq=%i\n", cfg.port[4], cfg.dma[0], cfg.dma[1], cfg.irq[0]);
	snd_sb_port[dev] = cfg.port[0];
	snd_wss_port[dev] = cfg.port[1];
	snd_fm_port[dev] = cfg.port[2];
	snd_midi_port[dev] = cfg.port[3];
	snd_port[dev] = cfg.port[4];
	snd_dma1[dev] = cfg.dma[0];
	snd_dma2[dev] = cfg.dma[1];
	snd_irq[dev] = cfg.irq[0];
        isapnp_activate(logdev->number);
	isapnp_cfg_end();
	return 0;
}

static void snd_opl3sa_deactivate(struct snd_opl3sa *oplcard)
{
	if (!oplcard->logdev)
		return;
	if (isapnp_cfg_begin(oplcard->logdev->dev->csn, oplcard->logdev->number)<0)
		return;
	isapnp_deactivate(oplcard->logdev->number);
	isapnp_cfg_end();
}
#endif /* CONFIG_ISAPNP */

static int snd_opl3sa_probe(int dev, struct snd_opl3sa *oplcard)
{
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer = NULL;
	snd_rawmidi_t *rmidi = NULL;
	snd_synth_t *synth = NULL;

#ifdef CONFIG_ISAPNP
	if (!snd_isapnp[dev]) {
#endif
		if (snd_port[dev] == SND_AUTO_PORT ||
		    snd_wss_port[dev] == SND_AUTO_PORT ||
		    snd_fm_port[dev] == SND_AUTO_PORT ||
		    snd_midi_port[dev] == SND_AUTO_PORT) {
			snd_printk("probing for Yamaha OPL3-SA isn't supported\n");
			snd_printk("port = 0x%x, wss_port = 0x%x, fm_port = 0x%x, midi_port = 0x%x\n",
					   snd_port[dev],
					   snd_wss_port[dev],
					   snd_fm_port[dev],
					   snd_midi_port[dev]);
			return -EINVAL;
		}
#ifdef CONFIG_ISAPNP
	}
#endif
	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_opl3sa_use_inc, snd_opl3sa_use_dec);
	if (!card)
		return -ENOMEM;
	card->type = SND_CARD_TYPE_OPL3_SA;
	strcpy(card->abbreviation, "OPL3SA");
	strcpy(card->shortname, "Yamaha OPL3-SA");
#ifdef CONFIG_ISAPNP
	if (snd_isapnp[dev] && snd_opl3sa_isapnp(dev, oplcard) < 0) {
		snd_card_free(card);
		return -EBUSY;
	}
#endif
	if (snd_opl3sa_resources(dev, oplcard, card) < 0) {
		snd_card_free(card);
		return -EBUSY;
	}
	oplcard->card = card;
	if (snd_opl3sa_detect(oplcard, snd_port[dev], snd_wss_port[dev],
			      snd_sb_port[dev], snd_fm_port[dev],
			      snd_midi_port[dev])) {
		snd_card_free(card);
		oplcard->card = NULL;
		return -ENODEV;
	}
	oplcard->port = snd_port[dev];
	if (snd_card_register(card)) {
		snd_card_free(card);
		return -ENOMEM;
	}
	pcm = snd_cs4231_new_device(card,
				    snd_wss_port[dev] + 4,
				    oplcard->irqptr,
				    oplcard->dma1ptr,
				    oplcard->dma2ptr == NULL ?
				    	oplcard->dma1ptr : oplcard->dma2ptr,
				    CS4231_HW_OPL3SA,
				    0);
	if (!pcm) {
		snd_printd("Oops, WSS not detected at 0x%x\n", snd_wss_port[dev] + 4);
		goto __nodev;
	}
	oplcard->pcm_status_reg = snd_wss_port[dev] + 4 + 2;
	mixer = snd_cs4231_new_mixer(pcm, 0);
	if (!mixer)
		goto __nodev;
	if (snd_opl3sa_mixer(oplcard, mixer) < 0)
		goto __nodev;
	if (snd_fm_port[dev] >= 0x340 && snd_fm_port[dev] < 0x400) {
		synth = snd_opl3_new_device(card, snd_fm_port[dev],
					    snd_fm_port[dev] + 2,
					    OPL3_HW_OPL3, 1);
		if (!synth)
			goto __nodev;
	}
	if (snd_midi_port[dev] >= 0x300 && snd_midi_port[dev] < 0x340) {
		rmidi = snd_mpu401_uart_new_device(card, MPU401_HW_OPL3SA,
						   snd_midi_port[dev],
						   oplcard->irqptr->irq);
		if (!rmidi)
			goto __nodev;
	}
	if (synth)
		if (snd_synth_register(synth) < 0)
			goto __nodev;
	if (rmidi) {
		if (snd_rawmidi_register(rmidi, 0) < 0) {
			if (synth)
				snd_synth_unregister(synth);
			synth = NULL;
			goto __nodev;
		}
	}
	if (snd_pcm_register(pcm, 0) < 0) {
		if (synth)
			snd_synth_unregister(synth);
		synth = NULL;
		if (rmidi)
			snd_rawmidi_unregister(rmidi);
		rmidi = NULL;
		goto __nodev;
	}
	if (snd_mixer_register(mixer, 0) < 0) {
		if (synth)
			snd_synth_unregister(synth);
		synth = NULL;
		if (rmidi)
			snd_rawmidi_unregister(rmidi);
		rmidi = NULL;
		snd_pcm_unregister(pcm);
		pcm = NULL;
		goto __nodev;
	}
	sprintf(card->longname, "%s at 0x%x, irq %i, dma %i",
		card->shortname,
		oplcard->port,
		oplcard->irqptr->irq,
		oplcard->dma1ptr->dma);
	if (oplcard->dma2ptr)
		sprintf(card->longname + strlen(card->longname), "&%i",
			oplcard->dma2ptr->dma);

	if (!snd_card_register(card)) {
		oplcard->pcm = pcm;
		oplcard->mixer = mixer;
		oplcard->rawmidi = rmidi;
		oplcard->synth = synth;
		return 0;
	}
	snd_mixer_unregister(mixer);
	mixer = NULL;
	snd_pcm_unregister(pcm);
	pcm = NULL;

      __nodev:
	if (rmidi)
		snd_rawmidi_free(rmidi);
	if (mixer)
		snd_mixer_free(mixer);
	if (pcm)
		snd_pcm_free(pcm);
	oplcard->card = NULL;
	snd_card_free(card);
	return -ENXIO;
}

int init_module(void)
{
	int dev, cards;
	struct snd_opl3sa *oplcard;

#ifndef LINUX_2_1
	register_symtab(NULL);
#endif
	for (dev = cards = 0; dev < SND_CARDS && snd_port[dev] > 0; dev++) {
		oplcard = (struct snd_opl3sa *)
				snd_malloc(sizeof(struct snd_opl3sa));
		if (!oplcard)
			continue;
		snd_spin_prepare(oplcard, reg);
		memset(oplcard, 0, sizeof(struct snd_opl3sa));
		if (snd_opl3sa_probe(dev, oplcard) < 0) {
#ifdef CONFIG_ISAPNP
			snd_opl3sa_deactivate(oplcard);
#endif
			snd_free(oplcard, sizeof(struct snd_opl3sa));
			if (snd_port[dev] == SND_AUTO_PORT)
				break;
			snd_printk("Yamaha OPL3-SA soundcard #%i not found at 0x%x or device busy\n", dev + 1, snd_port[dev]);
			continue;
		}
		snd_opl3sa_cards[dev] = oplcard;
		cards++;
	}
	if (!cards) {
		snd_printk("Yamaha OPL3-SA soundcard not found or device busy\n");
		return -ENODEV;
	}
	return 0;
}

void cleanup_module(void)
{
	int idx;
	struct snd_opl3sa *oplcard;
	snd_synth_t *synth;
	snd_rawmidi_t *rawmidi;
	snd_kmixer_t *mixer;
	snd_pcm_t *pcm;

	for (idx = 0; idx < SND_CARDS; idx++) {
		oplcard = snd_opl3sa_cards[idx];
		if (oplcard) {
			snd_card_unregister(oplcard->card);
			if (oplcard->synth) {
				synth = oplcard->synth;
				oplcard->synth = NULL;
				snd_synth_unregister(synth);
			}
			if (oplcard->rawmidi) {
				rawmidi = oplcard->rawmidi;
				oplcard->rawmidi = NULL;
				snd_rawmidi_unregister(rawmidi);
			}
			if (oplcard->mixer) {
				mixer = oplcard->mixer;
				oplcard->mixer = NULL;
				snd_mixer_unregister(mixer);
			}
			if (oplcard->pcm) {
				pcm = oplcard->pcm;
				oplcard->pcm = NULL;	/* turn off interrupts */
				snd_pcm_unregister(pcm);
			}
			snd_card_free(oplcard->card);
			oplcard->card = NULL;
#ifdef CONFIG_ISAPNP
			snd_opl3sa_deactivate(oplcard);
#endif
			snd_free(oplcard, sizeof(struct snd_opl3sa));
		}
	}
}
