/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of Sound Blaster CD/Pro mixer
 *  TODO: Routines for CTL1335 mixer (Sound Blaster 2.0 CD).
 *        If you have this soundcard, mail me!!!
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#include "driver.h"
#include "sb.h"

void snd_sb8mixer_write(sbmixer_t * mixer,
		        unsigned char reg, unsigned char data)
{
	outb(reg, SBP1(mixer->port, MIXER_ADDR));
	snd_delay(1);
	outb(data, SBP1(mixer->port, MIXER_DATA));
	snd_delay(1);
}

unsigned char snd_sb8mixer_read(sbmixer_t * mixer, unsigned char reg)
{
	unsigned char result;

	outb(reg, SBP1(mixer->port, MIXER_ADDR));
	snd_delay(1);
	result = inb(SBP1(mixer->port, MIXER_DATA));
	snd_delay(1);
	return result;
}

static int snd_sb8mixer_input_mux(int w_flag,
				  snd_kmixer_element_t **element,
				  sbmixer_t *sbmix)
{
	unsigned long flags;
	unsigned char reg, oval, nval;
	int change = 0;

	snd_spin_lock(sbmix, mixer, &flags);
	reg = snd_sb8mixer_read(sbmix, SB_DSP_RECORD_SOURCE);
	oval = reg & 6;
	if (w_flag) {
		nval = 0;
		if (*element == sbmix->me_mux_mic) {
			nval = SB_DSP_MIXS_MIC;
		} else if (*element == sbmix->me_mux_line) {
			nval = SB_DSP_MIXS_LINE;
		} else if (*element == sbmix->me_mux_cd) {
			nval = SB_DSP_MIXS_CD;
		} else {
			snd_spin_unlock(sbmix, mixer, &flags);
			return -EINVAL;
		}
		change = nval != oval;
		snd_sb8mixer_write(sbmix, SB_DSP_RECORD_SOURCE, (oval & ~6) | nval);
	} else {
		switch (oval) {
		case SB_DSP_MIXS_LINE:
			*element = sbmix->me_mux_line;
			break;
		case SB_DSP_MIXS_CD:
			*element = sbmix->me_mux_cd;
			break;
		case SB_DSP_MIXS_MIC:
		default:
			*element = sbmix->me_mux_mic;
			break;
		}
	}	
	snd_spin_unlock(sbmix, mixer, &flags);
	return change;
}

static int snd_sb8mixer_stereo_volume_level(int w_flag,
					    int *voices,
					    sbmixer_t *sbmix,
					    unsigned char reg,
					    unsigned char max,
					    unsigned char left_shift,
					    unsigned char right_shift)
{
	unsigned long flags;
	int change = 0;
	unsigned char oval;
	int left, right;

	snd_spin_lock(sbmix, mixer, &flags);
	oval = snd_sb8mixer_read(sbmix, reg);
	left = (oval >> left_shift) & max;
	right = (oval >> right_shift) & max;
	if (!w_flag) {
		voices[0] = left;
		voices[1] = right;
	} else {
		change = voices[0] != left && voices[1] != left;
		oval &= ~(max << left_shift);
		oval &= ~(max << right_shift);
		oval |= voices[0] << left_shift;
		oval |= voices[1] << right_shift;
		snd_sb8mixer_write(sbmix, reg, oval);
	}
	snd_spin_unlock(sbmix, mixer, &flags);
	return change;
}

static int snd_sb8mixer_mono_volume_level(int w_flag,
					  int *voices,
					  sbmixer_t *sbmix,
					  unsigned char reg,
					  unsigned char max,
					  unsigned char shift)
{
	unsigned long flags;
	int change = 0;
	unsigned char oval;
	int val;

	snd_spin_lock(sbmix, mixer, &flags);
	oval = snd_sb8mixer_read(sbmix, reg);
	val = (oval >> shift) & max;
	if (!w_flag) {
		voices[0] = val;
	} else {
		change = voices[0] != val;
		oval &= ~(max << shift);
		oval |= voices[0] << shift;
		snd_sb8mixer_write(sbmix, reg, oval);
	}
	snd_spin_unlock(sbmix, mixer, &flags);
	return change;
}

static int snd_sb8mixer_master_volume(int w_flag, int *voices, sbmixer_t *sbmix)
{
	return snd_sb8mixer_stereo_volume_level(w_flag, voices, sbmix,
						SB_DSP_MASTER_DEV, 7, 5, 1);
}

static int snd_sb8mixer_pcm_volume(int w_flag, int *voices, sbmixer_t *sbmix)
{
	return snd_sb8mixer_stereo_volume_level(w_flag, voices, sbmix,
						SB_DSP_PCM_DEV, 7, 5, 1);
}

static int snd_sb8mixer_line_volume(int w_flag, int *voices, sbmixer_t *sbmix)
{
	return snd_sb8mixer_stereo_volume_level(w_flag, voices, sbmix,
						SB_DSP_LINE_DEV, 7, 5, 1);
}

static int snd_sb8mixer_cd_volume(int w_flag, int *voices, sbmixer_t *sbmix)
{
	return snd_sb8mixer_stereo_volume_level(w_flag, voices, sbmix,
						SB_DSP_CD_DEV, 7, 5, 1);
}

static int snd_sb8mixer_fm_volume(int w_flag, int *voices, sbmixer_t *sbmix)
{
	return snd_sb8mixer_stereo_volume_level(w_flag, voices, sbmix,
						SB_DSP_FM_DEV, 7, 5, 1);
}

static int snd_sb8mixer_mic_volume(int w_flag, int *voices, sbmixer_t *sbmix)
{
	return snd_sb8mixer_mono_volume_level(w_flag, voices, sbmix,
					      SB_DSP_MIC_DEV, 3, 1);
}

snd_kmixer_t *snd_sb8dsp_new_mixer(snd_card_t * card,
				   sbmixer_t * sbmix,
				   unsigned short hardware,
				   int pcm_dev)
{
	snd_kmixer_t *mixer;
	unsigned long flags;
	snd_kmixer_group_t *group;
	snd_kmixer_element_t *mux;
	snd_kmixer_element_t *accu;
	snd_kmixer_element_t *element1, *element2;
	static struct snd_mixer_element_volume1_range db_range1[2] = {
		{0, 7, -4600, 0},
		{0, 7, -4600, 0}
	};
	static struct snd_mixer_element_volume1_range db_range2[1] = {
		{0, 3, -4600, 0},
	};

	if (!card)
		return NULL;
	mixer = snd_mixer_new(card, "CTL1345");
	if (!mixer)
		return NULL;
	strcpy(mixer->name, mixer->id);

	snd_spin_lock(sbmix, mixer, &flags);
	snd_sb8mixer_write(sbmix, 0x00, 0x00);	/* mixer reset */
	snd_spin_unlock(sbmix, mixer, &flags);

	/* build input mux and accumulator */
	if ((mux = snd_mixer_lib_mux1(mixer, SND_MIXER_ELEMENT_INPUT_MUX, 0, 0, 2, (snd_mixer_mux1_control_t *)snd_sb8mixer_input_mux, sbmix)) == NULL)
		goto __error;
	if ((accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0)) == NULL)
		goto __error;
	/* buid capture endpoint */
 	if ((element1 = snd_mixer_lib_pcm(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE, 1, &pcm_dev)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, mux, element1) < 0)
		goto __error;
	/* build master volume control */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_OUT_MASTER, 0)) == NULL)
		goto __error;
 	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_OUT_MASTER, 0, SND_MIXER_ETYPE_OUTPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "Master Volume", 0, 2, db_range1, (snd_mixer_volume1_control_t *)snd_sb8mixer_master_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, accu, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element1) < 0)
		goto __error;
	/* build PCM volume control */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_PCM, 0)) == NULL)
		goto __error;
 	if ((element1 = snd_mixer_lib_pcm(mixer, SND_MIXER_ELEMENT_PLAYBACK, 0, SND_MIXER_ETYPE_PLAYBACK, 1, &pcm_dev)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "PCM Volume", 0, 2, db_range1, (snd_mixer_volume1_control_t *)snd_sb8mixer_pcm_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, accu) < 0)
		goto __error;
	/* build LINE volume control */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_LINE, 0)) == NULL)
		goto __error;
 	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "Line Volume", 0, 2, db_range1, (snd_mixer_volume1_control_t *)snd_sb8mixer_line_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, accu) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, mux) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	sbmix->me_mux_line = element2;
	/* build CD volume control */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_CD, 0)) == NULL)
		goto __error;
 	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_CD, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "CD Volume", 0, 2, db_range1, (snd_mixer_volume1_control_t *)snd_sb8mixer_cd_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, accu) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, mux) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	sbmix->me_mux_cd = element2;
	/* build FM volume control */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_FM, 0)) == NULL)
		goto __error;
 	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_FM, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "FM Volume", 0, 2, db_range1, (snd_mixer_volume1_control_t *)snd_sb8mixer_fm_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, accu) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, mux) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	/* build MIC volume control */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_MIC, 0)) == NULL)
		goto __error;
 	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_MIC, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "MIC Volume", 0, 1, db_range2, (snd_mixer_volume1_control_t *)snd_sb8mixer_mic_volume, sbmix)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, accu) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, mux) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	sbmix->me_mux_mic = element2;
	
	return mixer;

      __error:
      	snd_mixer_free(mixer);
      	return NULL;
}
