/*
 * OSS compatible sequencer driver
 *
 * synth device handlers
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "seq_oss_synth.h"

/*
 * constants
 */
#define SND_SEQ_OSS_MAX_SYNTH_NAME	30
#define MAX_SYSEX_BUFLEN	128


/*
 * definition of synth info records
 */

/* sysex buffer */
struct seq_oss_synth_sysex_t {
	int len;
	int skip;
	unsigned char buf[MAX_SYSEX_BUFLEN];
};

/* synth info */
struct seq_oss_synth_t {
	int seq_device;

	/* for synth_info */
	int synth_type;
	int synth_subtype;
	int nr_voices;

	snd_synth_t *synth;
	char name[SND_SEQ_OSS_MAX_SYNTH_NAME];
	snd_seq_oss_callback_t oper;

	int opened;

	void *private;

	seq_oss_synth_sysex_t sysex;
};


/*
 * device table
 */
static int max_synth_devs = 0;
static seq_oss_synth_t *synth_devs[SND_SEQ_OSS_MAX_SYNTH_DEVS];

snd_mutex_define_static(register);

/*
 * prototypes
 */
static seq_oss_synth_t *get_synthdev(seq_oss_devinfo_t *dp, int dev);
static void reset_channels(seq_oss_synthinfo_t *info);


/*
 * registration of the synth device
 */
int
snd_seq_oss_synth_register(char *name, int type, int subtype, int nvoices,
			  snd_seq_oss_callback_t *oper, void *private)
{
	int i;
	seq_oss_synth_t *rec;

	if ((rec = snd_calloc(sizeof(*rec))) == NULL) {
		snd_printk("sequencer: can't malloc synth info\n");
		return -ENOMEM;
	}
	rec->seq_device = -1;
	rec->synth_type = type;
	rec->synth_subtype = subtype;
	rec->nr_voices = nvoices;
	rec->oper = *oper;
	rec->private = private;
	rec->opened = 0;

	/* copy and truncate the name of synth device */
	strncpy(rec->name, name, sizeof(rec->name));
	rec->name[sizeof(rec->name)-1] = 0;

	/* registration */
	snd_mutex_down_static(register);
	for (i = 0; i < max_synth_devs; i++) {
		if (synth_devs[i] == NULL)
			break;
	}
	if (i >= max_synth_devs) {
		if (max_synth_devs >= SND_SEQ_OSS_MAX_SYNTH_DEVS) {
			snd_printk("sequencer: no more synth slot\n");
			snd_mutex_up_static(register);
			snd_free(rec, sizeof(*rec));
			return -ENOMEM;
		}
		max_synth_devs++;
	}
	rec->seq_device = i;
	synth_devs[i] = rec;
	debug_printk(("seq_oss: synth %s registered %d\n", rec->name, i));
	snd_mutex_up_static(register);
	return i;
}


int
snd_seq_oss_synth_unregister(int index)
{
	seq_oss_synth_t *rec;

	snd_mutex_down_static(register);
	if (index < 0 || index >= max_synth_devs ||
	    (rec = synth_devs[index]) == NULL) {
		snd_printk("sequencer: can't unregister synth\n");
		snd_mutex_up_static(register);
		return -EINVAL;
	}

	snd_free(rec, sizeof(*rec));
	synth_devs[index] = NULL;

	if (index == max_synth_devs - 1) {
		for (index--; index >= 0; index--) {
			if (synth_devs[index])
				break;
		}
		max_synth_devs = index + 1;
	}
	snd_mutex_up_static(register);
	return 0;
}


/*
 * set up synth tables
 */

void
snd_seq_oss_synth_setup(seq_oss_devinfo_t *dp)
{
	int i;
	seq_oss_synth_t *rec;
	seq_oss_synthinfo_t *info;

	snd_mutex_down_static(register);
	dp->max_synthdev = max_synth_devs;
	dp->synth_opened = 0;
	memset(dp->synths, 0, sizeof(dp->synths));
	for (i = 0; i < dp->max_synthdev; i++) {
		if ((rec = synth_devs[i]) == NULL)
			continue;
		if (! rec->oper.open || ! rec->oper.close)
			continue;
		info = &dp->synths[i];
		info->arg.app_index = dp->port;
		info->arg.file_mode = dp->file_mode;
		info->arg.seq_mode = dp->seq_mode;
		info->arg.event_passing = SND_SEQ_OSS_PROCESS_EVENTS;
		info->opened = 0;
		if (rec->oper.open(&info->arg, rec->private) < 0)
			continue;
		info->nr_voices = rec->nr_voices;
		if (info->nr_voices > 0) {
			info->ch = snd_malloc(sizeof(seq_oss_chinfo_t) * info->nr_voices);
			reset_channels(info);
		}
		debug_printk(("seq_oss: synth %d assigned\n", i));
		info->opened++;
		rec->opened++;

		/* reset sysex buffer */
		rec->sysex.len = 0;
		rec->sysex.skip = 0;

		dp->synth_opened++;
	}
	snd_mutex_up_static(register);
}


void
snd_seq_oss_synth_cleanup(seq_oss_devinfo_t *dp)
{
	int i;
	seq_oss_synth_t *rec;
	seq_oss_synthinfo_t *info;

	snd_mutex_down_static(register);
	for (i = 0; i < dp->max_synthdev; i++) {
		info = &dp->synths[i];
		if (! info->opened || (rec = synth_devs[i]) == NULL ||
		    ! rec->opened)
			continue;
		debug_printk(("seq_oss: synth %d closed\n", i));
		rec->oper.close(&info->arg);
		if (info->ch)
			snd_free(info->ch, sizeof(seq_oss_chinfo_t) * info->nr_voices);
		rec->opened--;
	}
	snd_mutex_up_static(register);
	dp->synth_opened = 0;
	dp->max_synthdev = 0;
}

/*
 * return synth device information pointer
 */
static seq_oss_synth_t *
get_synthdev(seq_oss_devinfo_t *dp, int dev)
{
	seq_oss_synth_t *rec;
	if (dev < 0 || dev >= dp->max_synthdev)
		return NULL;
	if (! dp->synths[dev].opened || (rec = synth_devs[dev]) == NULL)
		return NULL;
	if (! rec->opened)
		return NULL;
	return rec;
}


/*
 * reset note and velocity on each channel.
 */
static void
reset_channels(seq_oss_synthinfo_t *info)
{
	int i;
	if (! info->ch || ! info->nr_voices)
		return;
	for (i = 0; i < info->nr_voices; i++) {
		info->ch[i].note = -1;
		info->ch[i].vel = 0;
	}
}


/*
 * reset synth device:
 * call reset callback.  if no callback is defined, send a heartbeat
 * event to the corresponding port.
 */
void
snd_seq_oss_synth_reset(seq_oss_devinfo_t *dp, int dev)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return;
	if (rec->oper.reset) {
		rec->oper.reset(&dp->synths[dev].arg);
	} else {
		snd_seq_event_t ev;
		memset(&ev, 0, sizeof(ev));
		snd_seq_oss_fill_addr(dp, &ev, dp->synths[dev].arg.addr.client,
				     dp->synths[dev].arg.addr.port);
		ev.type = SND_SEQ_EVENT_HEARTBEAT;
		snd_seq_oss_send_event(dp, &ev, 0);
	}
	reset_channels(&dp->synths[dev]);
}


/*
 * load a patch record:
 * call load_patch callback function
 */
int
snd_seq_oss_synth_load_patch(seq_oss_devinfo_t *dp, int dev, int fmt,
			    const char *buf, int p, int c)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;

	if (! rec->oper.load_patch)
		return -ENXIO;

	return rec->oper.load_patch(&dp->synths[dev].arg, fmt, buf, p, c);
}

/*
 * check if the device is valid synth device
 */
int
snd_seq_oss_synth_is_valid(seq_oss_devinfo_t *dp, int dev)
{
	if (get_synthdev(dp, dev))
		return 1;
	return 0;
}


/*
 * receive OSS 6 byte sysex packet:
 * the full sysex message will be sent if it reaches to the end of data
 * (0xff).
 */
int
snd_seq_oss_synth_send_sysex(seq_oss_devinfo_t *dp, int dev, unsigned char *buf, int atomic)
{
	int i, send;
	unsigned char *dest;
	seq_oss_synth_t *rec;
	seq_oss_synth_sysex_t *sysex;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;

	sysex = &rec->sysex;

	send = 0;
	dest = sysex->buf + sysex->len;
	/* copy 6 byte packet to the buffer */
	for (i = 0; i < 6; i++) {
		if (buf[i] == 0xff) {
			send = 1;
			break;
		}
		dest[i] = buf[i];
		sysex->len++;
		if (sysex->len >= MAX_SYSEX_BUFLEN) {
			sysex->len = 0;
			sysex->skip = 1;
			break;
		}
	}

	if (sysex->len && send) {
		if (! sysex->skip) {
			/* copy the data to event record and send it */
			snd_seq_event_t ev;
			ev.flags = SND_SEQ_EVENT_LENGTH_VARIABLE;
			snd_seq_oss_synth_addr(dp, dev, &ev);
			ev.data.ext.len = sysex->len;
			ev.data.ext.ptr = snd_seq_ext_malloc(sysex->len, 0);
			if (ev.data.ext.ptr) {
				memcpy(ev.data.ext.ptr, sysex->buf, sysex->len);
				snd_seq_oss_send_event(dp, &ev, atomic);
			}
		}
		sysex->skip = 0;
		sysex->len = 0;
	}

	return 0;
}

/*
 * fill the event source/destination addresses
 */
void
snd_seq_oss_synth_addr(seq_oss_devinfo_t *dp, int dev, snd_seq_event_t *ev)
{
	seq_oss_synth_t *rec;
	if ((rec = get_synthdev(dp, dev)) == NULL) {
		/* this should not happen.. */
		snd_seq_oss_fill_addr(dp, ev, dp->cseq, dp->port);
	} else {
		snd_seq_oss_fill_addr(dp, ev, dp->synths[dev].arg.addr.client,
				     dp->synths[dev].arg.addr.port);
	}
}


/*
 * OSS compatible ioctl
 */
int
snd_seq_oss_synth_ioctl(seq_oss_devinfo_t *dp, int dev, unsigned int cmd, unsigned long addr)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;
	if (! rec->oper.ioctl)
		return -ENXIO;
	return rec->oper.ioctl(&dp->synths[dev].arg, cmd, addr);
}


/*
 * send OSS raw events - SEQ_PRIVATE and SEQ_VOLUME
 */
int snd_seq_oss_synth_send_raw_event(seq_oss_devinfo_t *dp, int dev, unsigned char *data, int atomic)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;
	if (! rec->oper.raw_event)
		return -ENXIO;
	return rec->oper.raw_event(&dp->synths[dev].arg, data);
}


/*
 * create OSS compatible synth_info record
 */
int
snd_seq_oss_synth_make_info(seq_oss_devinfo_t *dp, int dev, oss_synth_info_t *inf)
{
	seq_oss_synth_t *rec;

	if ((rec = get_synthdev(dp, dev)) == NULL)
		return -ENXIO;
	inf->synth_type = rec->synth_type;
	inf->synth_subtype = rec->synth_subtype;
	inf->nr_voices = rec->nr_voices;
	inf->device = dev;
	strncpy(inf->name, rec->name, sizeof(inf->name));
	
	return 0;
}

/*
 * proc interface
 */
void
snd_seq_oss_synth_info_read(snd_info_buffer_t *buf)
{
	int i;
	seq_oss_synth_t *rec;

	snd_mutex_down_static(register);
	snd_iprintf(buf, "\nNumber of synth devices: %d\n", max_synth_devs);
	for (i = 0; i < max_synth_devs; i++) {
		snd_iprintf(buf, "\nsynth %d: ", i);
		if ((rec = synth_devs[i]) == NULL) {
			snd_iprintf(buf, "*empty*\n");
			continue;
		}
		snd_iprintf(buf, "%s\n", rec->name);
		snd_iprintf(buf, "  type 0x%x\n", rec->synth_type);
		snd_iprintf(buf, "  subtype 0x%x\n", rec->synth_subtype);
		snd_iprintf(buf, "  voices %d\n", rec->nr_voices);
		snd_iprintf(buf, "  ioctl %s\n", enabled_str((long)rec->oper.ioctl));
		snd_iprintf(buf, "  load_patch %s\n", enabled_str((long)rec->oper.load_patch));
	}
	snd_mutex_up_static(register);
}

