/*
 * OSS compatible sequencer driver
 *
 * read/write/select interface to device file
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "seq_oss_device.h"
#include "seq_oss_readq.h"
#include "seq_oss_writeq.h"
#include "seq_oss_synth.h"
#include "seq_oss_legacy.h"


/*
 * prototypes
 */
static int insert_event(seq_oss_devinfo_t *dp, evrec_t *rec, struct file *opt);


/*
 * read interface
 */

int
snd_seq_oss_read(seq_oss_devinfo_t *dp, char *buf, int count)
{
	seq_oss_readq_t *readq = dp->readq;
	int cnt, pos;

	if (readq == NULL || ! is_read_mode(dp->file_mode))
		return -EIO;

	if (snd_seq_oss_readq_is_empty(readq)) {
		/* no input events in queue */
 		if (is_nonblock_mode(dp->file_mode)) {
  			return -EAGAIN;
  		}

		/* wait for any input */
		snd_seq_oss_readq_sleep(readq);
		if (snd_seq_oss_readq_is_empty(readq)) /* interrupted */
			return 0;
	}

	/* copy queued events to read buffer */
	cnt = count;
	pos = 0;
	while (! snd_seq_oss_readq_is_empty(readq)) {
		evrec_t *q = snd_seq_oss_readq_pick(readq);
		int ev_len;

		/* tansfer the data */
		ev_len = ev_length(q);
		if (copy_to_user(buf + pos, q, ev_len))
			break;
		snd_seq_oss_readq_free(readq, q);
		pos += ev_len;
		cnt -= ev_len;
		if (cnt < ev_len)
			break;
	}

	return count - cnt;
}


/*
 * write interface
 */

int
snd_seq_oss_write(seq_oss_devinfo_t *dp, const char *buf, int count, struct file *opt)
{
	int rc, c, p, ev_size;
	evrec_t rec;

	if (! is_write_mode(dp->file_mode) || ! dp->writeq)
		return -EIO;

	c = count;
	p = 0;
	while (c >= SHORT_EVENT_SIZE) {
		if (copy_from_user(rec.c, buf + p, SHORT_EVENT_SIZE))
			break;
		p += SHORT_EVENT_SIZE;

		if (rec.s.code == SND_OSS_SEQ_FULLSIZE) {
			/* load patch */
			int fmt = (*(unsigned short *)rec.c) & 0xffff;
			return snd_seq_oss_synth_load_patch(dp, rec.s.dev, fmt, buf, p, c);

		}
		if (ev_is_long(&rec)) {
			/* extended code */
			ev_size = LONG_EVENT_SIZE;
			if (c < ev_size)
				break;
			/* copy the reset 4 bytes */
			if (copy_from_user(rec.c + SHORT_EVENT_SIZE, buf + p,
					   LONG_EVENT_SIZE - SHORT_EVENT_SIZE))
				break;
			p += LONG_EVENT_SIZE - SHORT_EVENT_SIZE;

		} else {
			/* old-type code */
			ev_size = SHORT_EVENT_SIZE;
		}

		/* insert queue */
		if ((rc = insert_event(dp, &rec, opt)) < 0)
			break;

		c -= ev_size;
	}

	if (count == c && is_nonblock_mode(dp->file_mode))
		return -EAGAIN;
	return count - c;
}


/*
 * add the event to write queue
 * return: 0 = OK, non-zero = NG
 */
static int
insert_event(seq_oss_devinfo_t *dp, evrec_t *rec, struct file *opt)
{
	seq_oss_writeq_t *q = dp->writeq;
	if (q->maxlen - q->qlen <= 1) {
		/* no space on queue */
		if (is_nonblock_mode(dp->file_mode))
			return -EBUSY;
		/* sleep until enouch space is assigned */
		snd_seq_oss_wait_room(dp->writeq, opt);
		if (! snd_seq_oss_writeq_has_room(dp->writeq))
			/* interrupted */
			return -EINTR;
	}

	/* insert a queue */
	return snd_seq_oss_insert_queue(dp, rec);
}


/*
 * select / poll
 */
  
#ifdef SND_POLL
unsigned int
snd_seq_oss_poll(seq_oss_devinfo_t *dp, struct file *file, poll_table * wait)
{
	unsigned int mask = 0;

	/* input */
	if (dp->readq && is_read_mode(dp->file_mode)) {
		if (snd_seq_oss_readq_poll(dp->readq, file, wait))
			mask |= POLLIN | POLLRDNORM;
	}

	/* output */
	if (dp->writeq && is_write_mode(dp->file_mode)) {
		if (snd_seq_oss_writeq_poll(dp->writeq, file, wait))
			mask |= POLLOUT | POLLWRNORM;
	}
	return mask;
}
#else
int
snd_seq_oss_select(seq_oss_devinfo_t *dp, int sel_type, select_table * wait)
{
	switch (sel_type) {
	case SEL_IN:
		if (dp->readq && is_read_mode(dp->file_mode)) {
			if (snd_seq_oss_readq_select(dp->readq, wait))
			    return 1;
		}
		break;

	case SEL_OUT:
		if (dp->writeq && is_write_mode(dp->file_mode)) {
			if (snd_seq_oss_writeq_select(dp->writeq, wait))
				return 1;
		}
		break;
	}

	return 0;
}
#endif
