/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of AD1848/AD1847/CS4248
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "ad1848.h"
#include "ulaw.h"

#if 0
#define SND_DEBUG_MCE
#endif

/*
 *  Some variables
 */

static struct snd_stru_ad1848_freq snd_ad1848_freq[14] =
{
	{5, 5510, 0x00 | AD1848_XTAL2},
	{6, 6620, 0x0E | AD1848_XTAL2},
	{8, 8000, 0x00 | AD1848_XTAL1},
	{9, 9600, 0x0E | AD1848_XTAL1},
	{11, 11025, 0x02 | AD1848_XTAL2},
	{16, 16000, 0x02 | AD1848_XTAL1},
	{18, 18900, 0x04 | AD1848_XTAL2},
	{22, 22050, 0x06 | AD1848_XTAL2},
	{27, 27042, 0x04 | AD1848_XTAL1},
	{32, 32000, 0x06 | AD1848_XTAL1},
	{33, 33075, 0x0C | AD1848_XTAL2},
	{37, 37800, 0x08 | AD1848_XTAL2},
	{44, 44100, 0x0A | AD1848_XTAL2},
	{48, 48000, 0x0C | AD1848_XTAL1}
};

static unsigned char snd_ad1848_original_image[16] =
{
	0x00,			/* 00 - lic */
	0x00,			/* 01 - ric */
	0x80,			/* 02 - la1ic */
	0x80,			/* 03 - ra1ic */
	0x80,			/* 04 - la2ic */
	0x80,			/* 05 - ra2ic */
	0x80,			/* 06 - loc */
	0x80,			/* 07 - roc */
	0x20,			/* 08 - dfr */
	AD1848_AUTOCALIB,	/* 09 - ic */
	0x00,			/* 0a - pc */
	0x00,			/* 0b - ti */
	0x00,			/* 0c - mi */
	0x00,			/* 0d - lbc */
	0x00,			/* 0e - dru */
	0x00,			/* 0f - drl */
};

/*
 *  Basic I/O functions
 */

static void snd_ad1848_outm(ad1848_t * codec, unsigned char reg,
			    unsigned char mask, unsigned char value)
{
	int timeout;
	unsigned char tmp;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		snd_delay(1);
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("snd_ad1848_outm: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	if (codec->calibrate_mute) {
		codec->image[reg] &= mask;
		codec->image[reg] |= value;
	} else {
		outb(codec->mce_bit | reg, AD1848P(codec, REGSEL));
		mb();
		tmp = (codec->image[reg] & mask) | value;
		outb(tmp, AD1848P(codec, REG));
		codec->image[reg] = tmp;
		mb();
	}
}

static void snd_ad1848_out(ad1848_t * codec,
			   unsigned char reg,
			   unsigned char value)
{
	int timeout;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		snd_delay(1);
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("snd_ad1848_out: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	outb(codec->mce_bit | reg, AD1848P(codec, REGSEL));
	outb(codec->image[reg] = value, AD1848P(codec, REG));
	mb();
#if 0
	printk("codec out - reg 0x%x = 0x%x\n", codec->mce_bit | reg, value);
#endif
}

static void snd_ad1848_dout(ad1848_t * codec,
			    unsigned char reg,
			    unsigned char value)
{
	int timeout;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		snd_delay(1);
	outb(codec->mce_bit | reg, AD1848P(codec, REGSEL));
	outb(value, AD1848P(codec, REG));
	mb();
}

static unsigned char snd_ad1848_in(ad1848_t * codec, unsigned char reg)
{
	int timeout;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		snd_delay(1);
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("snd_ad1848_in: auto calibration time out - reg = 0x%x\n", reg);
#endif
	outb(codec->mce_bit | reg, AD1848P(codec, REGSEL));
	mb();
	return inb(AD1848P(codec, REG));
}

#ifdef CONFIG_SND_DEBUG

void snd_ad1848_debug(ad1848_t * codec)
{
	printk("AD1848 REGS:      INDEX = 0x%02x  ", inb(AD1848P(codec, REGSEL)));
	printk("                 STATUS = 0x%02x\n", inb(AD1848P(codec, STATUS)));
	printk("  0x00: left input      = 0x%02x  ", snd_ad1848_in(codec, 0x00));
	printk("  0x08: playback format = 0x%02x\n", snd_ad1848_in(codec, 0x08));
	printk("  0x01: right input     = 0x%02x  ", snd_ad1848_in(codec, 0x01));
	printk("  0x09: iface (CFIG 1)  = 0x%02x\n", snd_ad1848_in(codec, 0x09));
	printk("  0x02: AUXA left       = 0x%02x  ", snd_ad1848_in(codec, 0x02));
	printk("  0x0a: pin control     = 0x%02x\n", snd_ad1848_in(codec, 0x0a));
	printk("  0x03: AUXA right      = 0x%02x  ", snd_ad1848_in(codec, 0x03));
	printk("  0x0b: init & status   = 0x%02x\n", snd_ad1848_in(codec, 0x0b));
	printk("  0x04: AUXB left       = 0x%02x  ", snd_ad1848_in(codec, 0x04));
	printk("  0x0c: revision & mode = 0x%02x\n", snd_ad1848_in(codec, 0x0c));
	printk("  0x05: AUXB right      = 0x%02x  ", snd_ad1848_in(codec, 0x05));
	printk("  0x0d: loopback        = 0x%02x\n", snd_ad1848_in(codec, 0x0d));
	printk("  0x06: left output     = 0x%02x  ", snd_ad1848_in(codec, 0x06));
	printk("  0x0e: data upr count  = 0x%02x\n", snd_ad1848_in(codec, 0x0e));
	printk("  0x07: right output    = 0x%02x  ", snd_ad1848_in(codec, 0x07));
	printk("  0x0f: data lwr count  = 0x%02x\n", snd_ad1848_in(codec, 0x0f));
}

#endif

/*
 *  AD1848 detection / MCE routines
 */

static void snd_ad1848_mce_up(ad1848_t * codec)
{
	unsigned long flags;
	int timeout;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		snd_delay(1);
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("ad1848_mce_up - auto calibration time out (0)\n");
#endif
	snd_spin_lock(codec, reg, &flags);
	codec->mce_bit |= AD1848_MCE;
	timeout = inb(AD1848P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_ad1848_up [0x%x]: serious init problem - codec still busy\n", codec->port);
	if (!(timeout & AD1848_MCE))
		outb(codec->mce_bit | (timeout & 0x1f), AD1848P(codec, REGSEL));
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_ad1848_mce_down(ad1848_t * codec)
{
	unsigned long flags;
	int timeout;
	signed long time;

	snd_spin_lock(codec, reg, &flags);
	for (timeout = 5; timeout > 0; timeout--)
		inb(AD1848P(codec, REGSEL));
	/* end of cleanup sequence */
	for (timeout = 12000; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		snd_delay(1);
#if 0
	printk("(1) timeout = %i\n", timeout);
#endif
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("ad1848_mce_down [0x%x] - auto calibration time out (0)\n", AD1848P(codec, REGSEL));
#endif
	codec->mce_bit &= ~AD1848_MCE;
	timeout = inb(AD1848P(codec, REGSEL));
	outb(codec->mce_bit | (timeout & 0x1f), AD1848P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_ad1848_down [0x%x]: serious init problem - codec still busy\n", codec->port);
	if ((timeout & AD1848_MCE) == 0) {
		snd_spin_unlock(codec, reg, &flags);
		return;
	}
	/* calibration process */

	for (timeout = 500; timeout > 0 && (snd_ad1848_in(codec, AD1848_TEST_INIT) & AD1848_CALIB_IN_PROGRESS) == 0; timeout--);
	if ((snd_ad1848_in(codec, AD1848_TEST_INIT) & AD1848_CALIB_IN_PROGRESS) == 0) {
		snd_printd("ad1848_mce_down - auto calibration time out (1)\n");
		snd_spin_unlock(codec, reg, &flags);
		return;
	}
#if 0
	printk("(2) timeout = %i, jiffies = %li\n", timeout, jiffies);
#endif
	time = HZ / 4;
	while (snd_ad1848_in(codec, AD1848_TEST_INIT) & AD1848_CALIB_IN_PROGRESS) {
		snd_spin_unlock(codec, reg, &flags);
		if (time <= 0) {
			snd_printk("ad1848_mce_down - auto calibration time out (2)\n");
			return;
		}
		snd_schedule(codec, mce, time);
		time = snd_timevalue(codec, mce);
		snd_spin_lock(codec, reg, &flags);
	}
#if 0
	printk("(3) jiffies = %li\n", jiffies);
#endif
	time = HZ / 10;
	while (inb(AD1848P(codec, REGSEL)) & AD1848_INIT) {
		snd_spin_unlock(codec, reg, &flags);
		if (time <= 0) {
			snd_printk("ad1848_mce_down - auto calibration time out (3)\n");
			return;
		}
		snd_schedule(codec, mce, time);
		time = snd_timevalue(codec, mce);
		snd_spin_lock(codec, reg, &flags);
	}
	snd_spin_unlock(codec, reg, &flags);
#if 0
	printk("(4) jiffies = %li\n", jiffies);
	snd_printk("ad1848_mce_down - exit = 0x%x\n", inb(AD1848P(codec, REGSEL)));
#endif
}

static unsigned int snd_ad1848_get_count(unsigned char format,
				         unsigned int size)
{
	switch (format & 0xe0) {
	case AD1848_LINEAR_16:
		size >>= 1;
		break;
	}
	if (format & AD1848_STEREO)
		size >>= 1;
	return size;
}

static void snd_ad1848_trigger(ad1848_t * codec,
			       unsigned char what, int enable)
{
	unsigned long flags;

#if 0
	printk("codec trigger!!! - what = %i, enable = %i, status = 0x%x\n", what, enable, inb(AD1848P(card, STATUS)));
#endif
	snd_spin_lock(codec, reg, &flags);
	if (enable) {
		if (codec->image[AD1848_IFACE_CTRL] & what) {
			snd_spin_unlock(codec, reg, &flags);
			return;
		}
		snd_ad1848_out(codec, AD1848_IFACE_CTRL, codec->image[AD1848_IFACE_CTRL] |= what);
	} else {
		if (!(codec->image[AD1848_IFACE_CTRL] & what)) {
			snd_spin_unlock(codec, reg, &flags);
			return;
		}
		snd_ad1848_out(codec, AD1848_IFACE_CTRL, codec->image[AD1848_IFACE_CTRL] &= ~what);
	}
	snd_spin_unlock(codec, reg, &flags);
#if 0
	snd_ad1848_debug(codec);
	snd_delay(100);
	printk("residue = 0x%x\n", get_dma_residue(codec->dma));
#endif
}

/*
 *  CODEC I/O
 */

static unsigned char snd_ad1848_get_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_ad1848_freq[i].hertz)
			return snd_ad1848_freq[i].bits;
	return snd_ad1848_freq[13].bits;
}

static unsigned int snd_ad1848_get_real_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_ad1848_freq[i].hertz)
			return snd_ad1848_freq[i].rate;
	return snd_ad1848_freq[13].rate;
}

static int snd_ad1848_playback_ioctl(snd_pcm1_t * pcm1, unsigned int cmd, unsigned long *arg)
{
	switch (cmd) {
	case SND_PCM1_IOCTL_RATE:
		pcm1->playback.real_rate = snd_ad1848_get_real_freq(pcm1->playback.rate);
		return 0;
	}
	return -ENXIO;
}

static int snd_ad1848_record_ioctl(snd_pcm1_t * pcm1, unsigned int cmd, unsigned long *arg)
{
	switch (cmd) {
	case SND_PCM1_IOCTL_RATE:
		pcm1->record.real_rate = snd_ad1848_get_real_freq(pcm1->record.rate);
		return 0;
	}
	return -ENXIO;
}

static unsigned char snd_ad1848_get_format(unsigned int mode, int voices)
{
	unsigned char format;

	format = 0;
	if (mode & SND_PCM1_MODE_16) {
		format |= AD1848_LINEAR_16;
	} else {
		if (mode & SND_PCM1_MODE_ALAW)
			format |= AD1848_ALAW_8;
		else if (mode & SND_PCM1_MODE_ULAW)
			format |= AD1848_ULAW_8;
#if 0
		else
			format |= AD1848_LINEAR_8;	/* I know, OR with 0 */
#endif
	}
	if (voices > 1)
		format |= AD1848_STEREO;
#if 0
	snd_printk("snd_ad1848_get_format: 0x%x (mode=0x%x)\n", format, mode);
#endif
	return format;
}

static void snd_ad1848_calibrate_mute(ad1848_t * codec, int mute)
{
	unsigned long flags;
	
	mute = mute ? 1 : 0;
	snd_spin_lock(codec, reg, &flags);
	if (codec->calibrate_mute == mute) {
		snd_spin_unlock(codec, reg, &flags);
		return;
	}
	if (!mute) {
		snd_ad1848_dout(codec, AD1848_LEFT_INPUT, codec->image[AD1848_LEFT_INPUT]);
		snd_ad1848_dout(codec, AD1848_RIGHT_INPUT, codec->image[AD1848_RIGHT_INPUT]);
	}
	snd_ad1848_dout(codec, AD1848_AUX1_LEFT_INPUT, mute ? 0x80 : codec->image[AD1848_AUX1_LEFT_INPUT]);
	snd_ad1848_dout(codec, AD1848_AUX1_RIGHT_INPUT, mute ? 0x80 : codec->image[AD1848_AUX1_RIGHT_INPUT]);
	snd_ad1848_dout(codec, AD1848_AUX2_LEFT_INPUT, mute ? 0x80 : codec->image[AD1848_AUX2_LEFT_INPUT]);
	snd_ad1848_dout(codec, AD1848_AUX2_RIGHT_INPUT, mute ? 0x80 : codec->image[AD1848_AUX2_RIGHT_INPUT]);
	snd_ad1848_dout(codec, AD1848_LEFT_OUTPUT, mute ? 0x80 : codec->image[AD1848_LEFT_OUTPUT]);
	snd_ad1848_dout(codec, AD1848_RIGHT_OUTPUT, mute ? 0x80 : codec->image[AD1848_RIGHT_OUTPUT]);
	codec->calibrate_mute = mute;
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_ad1848_set_data_format(ad1848_t * codec, int direction)
{
	snd_pcm1_t *pcm1;
	snd_pcm1_channel_t *pchn;

	pcm1 = (snd_pcm1_t *) codec->pcm->private_data;
	pchn = direction == SND_PCM1_PLAYBACK ? &pcm1->playback : &pcm1->record;
	codec->image[AD1848_DATA_FORMAT] =
	    snd_ad1848_get_format(pchn->mode, pchn->voices) |
	    snd_ad1848_get_freq(pchn->real_rate);
#if 0
	snd_printk(">>> pmode = 0x%x, dfr = 0x%x\n", pchn->mode, codec->image[AD1848_DATA_FORMAT]);
#endif
}

static int snd_ad1848_open(ad1848_t * codec, unsigned int mode)
{
	unsigned long flags;

	snd_mutex_down(codec, open);
	if (codec->mode & AD1848_MODE_OPEN) {
		snd_mutex_up(codec, open);
		return -1;
	}
	snd_ad1848_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_ad1848_open: (1)\n");
#endif
	snd_ad1848_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	codec->image[AD1848_IFACE_CTRL] &= ~(AD1848_PLAYBACK_ENABLE | AD1848_PLAYBACK_PIO |
			     AD1848_RECORD_ENABLE | AD1848_RECORD_PIO |
			     AD1848_CALIB_MODE);
	codec->image[AD1848_IFACE_CTRL] |= AD1848_AUTOCALIB;
	snd_ad1848_out(codec, AD1848_IFACE_CTRL, codec->image[AD1848_IFACE_CTRL]);
	snd_spin_unlock(codec, reg, &flags);
	snd_ad1848_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_ad1848_open: (2)\n");
#endif

	snd_ad1848_set_data_format(codec, SND_PCM1_PLAYBACK);

	snd_ad1848_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_ad1848_out(codec, AD1848_DATA_FORMAT, codec->image[AD1848_DATA_FORMAT]);
	snd_spin_unlock(codec, reg, &flags);
	snd_ad1848_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_ad1848_open: (3)\n");
#endif

	/* ok. now enable and ack CODEC IRQ */
	snd_spin_lock(codec, reg, &flags);
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	codec->image[AD1848_PIN_CTRL] |= AD1848_IRQ_ENABLE;
	snd_ad1848_out(codec, AD1848_PIN_CTRL, codec->image[AD1848_PIN_CTRL]);
	snd_spin_unlock(codec, reg, &flags);

	codec->mode = mode;
	snd_mutex_up(codec, open);

	return 0;
}

static void snd_ad1848_close(ad1848_t * codec)
{
	unsigned long flags;

	snd_mutex_down(codec, open);
	if (!codec->mode) {
		snd_mutex_up(codec, open);
		return;
	}
	/* disable IRQ */
	snd_spin_lock(codec, reg, &flags);
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	codec->image[AD1848_PIN_CTRL] &= ~AD1848_IRQ_ENABLE;
	snd_ad1848_out(codec, AD1848_PIN_CTRL, codec->image[AD1848_PIN_CTRL]);
	snd_spin_unlock(codec, reg, &flags);

	/* now disable record & playback */

	snd_ad1848_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	codec->image[AD1848_IFACE_CTRL] &= ~(AD1848_PLAYBACK_ENABLE | AD1848_PLAYBACK_PIO |
			     AD1848_RECORD_ENABLE | AD1848_RECORD_PIO);
	snd_ad1848_out(codec, AD1848_IFACE_CTRL, codec->image[AD1848_IFACE_CTRL]);
	snd_spin_unlock(codec, reg, &flags);
	snd_ad1848_mce_down(codec);

	/* clear IRQ again */
	snd_spin_lock(codec, reg, &flags);
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	snd_spin_unlock(codec, reg, &flags);

	codec->mode = 0;
	snd_mutex_up(codec, open);
}

/*
 *  ok.. exported functions..
 */

static void snd_ad1848_playback_trigger(snd_pcm1_t * pcm1, int up)
{
	ad1848_t *codec;

	codec = (ad1848_t *) pcm1->private_data;

	snd_ad1848_trigger(codec, AD1848_PLAYBACK_ENABLE, up);

#if 0
	if (up)
		snd_ad1848_debug(codec);
#endif
}

static void snd_ad1848_record_trigger(snd_pcm1_t * pcm1, int up)
{
	ad1848_t *codec;

	codec = (ad1848_t *) pcm1->private_data;
	snd_ad1848_trigger(codec, AD1848_RECORD_ENABLE, up);
}

static void snd_ad1848_playback_prepare(snd_pcm1_t * pcm1,
					unsigned char *buffer,
					unsigned int size,
					unsigned int offset,
					unsigned int count)
{
	ad1848_t *codec;
	unsigned long flags;

	codec = (ad1848_t *) pcm1->private_data;
	/* Note: offset is always 0 if AUTO DMA */
	snd_ad1848_calibrate_mute(codec, 1);
	snd_ad1848_set_data_format(codec, SND_PCM1_PLAYBACK);
#if 0
	snd_printk("ad1848_start_playback - dma = %i, buffer = 0x%lx, size = 0x%x, offset = 0x%x, count = 0x%x\n", codec->dma1, (long) buffer, size, offset, count);
#endif
	codec->image[AD1848_IFACE_CTRL] &= ~(AD1848_PLAYBACK_ENABLE | AD1848_PLAYBACK_PIO);
	snd_dma_program(codec->dma, buffer, size, DMA_MODE_WRITE | DMA_MODE_AUTOINIT);
	count = snd_ad1848_get_count(codec->image[AD1848_DATA_FORMAT], count) - 1;
	snd_ad1848_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_ad1848_out(codec, AD1848_DATA_FORMAT, codec->image[AD1848_DATA_FORMAT]);
	snd_spin_unlock(codec, reg, &flags);
	snd_ad1848_mce_down(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_ad1848_out(codec, AD1848_DATA_LWR_CNT, (unsigned char) count);
	snd_ad1848_out(codec, AD1848_DATA_UPR_CNT, (unsigned char) (count >> 8));
	snd_spin_unlock(codec, reg, &flags);
	snd_ad1848_calibrate_mute(codec, 0);
#if 0
	snd_ad1848_debug(codec);
#endif
}

static void snd_ad1848_record_prepare(snd_pcm1_t * pcm1,
				      unsigned char *buffer,
				      unsigned int size,
				      unsigned int offset,
				      unsigned int count)
{
	ad1848_t *codec;
	unsigned long flags;

	codec = (ad1848_t *) pcm1->private_data;
	snd_ad1848_calibrate_mute(codec, 1);
	snd_ad1848_set_data_format(codec, SND_PCM1_RECORD);
#if 0
	snd_printk("ad1848_start_record: start - buffer = 0x%lx, offset = 0x%x, size = 0x%x, count = 0x%x\n", (long) buffer, offset, size, count);
#endif
	codec->image[AD1848_IFACE_CTRL] &= ~(AD1848_RECORD_ENABLE | AD1848_RECORD_PIO);
	snd_dma_program(codec->dma, buffer, size, DMA_MODE_READ | DMA_MODE_AUTOINIT);
	count = snd_ad1848_get_count(codec->image[AD1848_DATA_FORMAT], count) - 1;
	snd_ad1848_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_ad1848_out(codec, AD1848_DATA_FORMAT, codec->image[AD1848_DATA_FORMAT]);
	snd_spin_unlock(codec, reg, &flags);
	snd_ad1848_mce_down(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_ad1848_out(codec, AD1848_DATA_LWR_CNT, (unsigned char) count);
	snd_ad1848_out(codec, AD1848_DATA_UPR_CNT, (unsigned char) (count >> 8));
	snd_spin_unlock(codec, reg, &flags);
	snd_ad1848_calibrate_mute(codec, 0);
}

void snd_ad1848_interrupt(snd_pcm_t * pcm, unsigned char status)
{
	unsigned long flags;
	ad1848_t *codec;
	snd_pcm1_t *pcm1;

#if 0
	snd_printk("snd_ad1848_interrupt: status=0x%x\n", status);
#endif
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = pcm1->private_data;
	if (!codec)
		return;

	if ((codec->mode & AD1848_MODE_PLAY) && pcm1->playback.ack)
		pcm1->playback.ack(pcm1);
	if ((codec->mode & AD1848_MODE_RECORD) && pcm1->record.ack)
		pcm1->record.ack(pcm1);

	snd_spin_lock(codec, reg, &flags);
	outb(0, AD1848P(codec, STATUS));	/* clear global interrupt bit */
	snd_spin_unlock(codec, reg, &flags);
}

/*

 */

static int snd_ad1848_playback_open(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	ad1848_t *codec;
	int err;

	card = pcm1->card;
	codec = (ad1848_t *) pcm1->private_data;
	if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_PLAYBACK, codec->dmaptr, "AD1848 (playback)")) < 0)
		return err;
	if (snd_ad1848_open(codec, AD1848_MODE_PLAY) < 0) {
		snd_pcm1_dma_free(pcm1, SND_PCM1_PLAYBACK, codec->dmaptr);
		return -EBUSY;
	}
	return 0;
}

static int snd_ad1848_record_open(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	ad1848_t *codec;
	int err;

	card = pcm1->card;
	codec = (ad1848_t *) pcm1->private_data;
	if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_RECORD, codec->dmaptr, "AD1848 (record)")) < 0)
		return err;
	if (snd_ad1848_open(codec, AD1848_MODE_RECORD) < 0) {
		snd_pcm1_dma_free(pcm1, SND_PCM1_RECORD, codec->dmaptr);
		return -EBUSY;
	}
	return 0;
}

static void snd_ad1848_playback_close(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	ad1848_t *codec;

	card = pcm1->card;
	codec = (ad1848_t *) pcm1->private_data;
	codec->mode &= ~AD1848_MODE_PLAY;
	snd_ad1848_close(codec);
	snd_pcm1_dma_free(pcm1, SND_PCM1_PLAYBACK, codec->dmaptr);
}

static void snd_ad1848_record_close(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	ad1848_t *codec;

	card = pcm1->card;
	codec = (ad1848_t *) pcm1->private_data;
	codec->mode &= ~AD1848_MODE_RECORD;
	snd_ad1848_close(codec);
	snd_pcm1_dma_free(pcm1, SND_PCM1_RECORD, codec->dmaptr);
}

static unsigned int snd_ad1848_playback_pointer(snd_pcm1_t * pcm1,
					        unsigned int used_size)
{
	ad1848_t *codec;

	codec = (ad1848_t *) pcm1->private_data;
	if (!(codec->image[AD1848_IFACE_CTRL] & AD1848_PLAYBACK_ENABLE))
		return 0;
	return used_size - snd_dma_residue(codec->dma);
}

static unsigned int snd_ad1848_record_pointer(snd_pcm1_t * pcm1,
					      unsigned int used_size)
{
	ad1848_t *codec;

	codec = (ad1848_t *) pcm1->private_data;
	if (!(codec->image[AD1848_IFACE_CTRL] & AD1848_RECORD_ENABLE))
		return 0;
	return used_size - snd_dma_residue(codec->dma);
}

/*

 */

static int snd_ad1848_probe(snd_pcm1_t * pcm1)
{
	unsigned long flags;
	snd_pcm_t *pcm;
	ad1848_t *codec;
	int i, id, rev, ad1847;
	unsigned char *ptr;

	pcm = pcm1->pcm;
	codec = (ad1848_t *) pcm1->private_data;
	if (!codec)
		return -EINVAL;

#if 0
	snd_ad1848_debug(codec);
#endif
	id = ad1847 = 0;
	for (i = 0; i < 1000; i++) {
		mb();
		if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
			snd_delay(50 * 8);
		else {
			snd_spin_lock(codec, reg, &flags);
			snd_ad1848_out(codec, AD1848_MISC_INFO, 0x00);
			snd_ad1848_out(codec, AD1848_LEFT_INPUT, 0xaa);
			snd_ad1848_out(codec, AD1848_RIGHT_INPUT, 0x45);
			rev = snd_ad1848_in(codec, AD1848_RIGHT_INPUT);
			if (rev == 0x65) {
				id = 1;
				ad1847 = 1;
				break;
			}
			if (snd_ad1848_in(codec, AD1848_LEFT_INPUT) == 0xaa && rev == 0x45) {
				id = 1;
				break;
			}
		}
	}
	if (id != 1)
		return -ENODEV;	/* no valid device found */
	if (codec->hardware == AD1848_HW_DETECT) {
		if (ad1847) {
			codec->hardware = AD1848_HW_AD1847;
			strcpy(pcm->name, "AD1847");
		} else {
			codec->hardware = AD1848_HW_AD1848;
			strcpy(pcm->name, "AD1848");
			rev = snd_ad1848_in(codec, AD1848_MISC_INFO);
			if (rev & 0x80) {
				codec->hardware = AD1848_HW_CS4248;
				strcpy(pcm->name, "CS4248");
			}
		}
	}
	snd_spin_lock(codec, reg, &flags);
	inb(AD1848P(codec, STATUS));	/* clear any pendings IRQ */
	outb(0, AD1848P(codec, STATUS));
	mb();
	snd_spin_unlock(codec, reg, &flags);

	codec->image[AD1848_MISC_INFO] = 0x00;
	codec->image[AD1848_IFACE_CTRL] =
	    (codec->image[AD1848_IFACE_CTRL] & ~AD1848_SINGLE_DMA) | AD1848_SINGLE_DMA;
	ptr = (unsigned char *) &codec->image;
	snd_ad1848_mce_down(codec);
	snd_spin_lock(codec, reg, &flags);
	for (i = 0; i < 16; i++)	/* ok.. fill all AD1848 registers */
		snd_ad1848_out(codec, i, *ptr++);
	snd_spin_unlock(codec, reg, &flags);
	snd_ad1848_mce_up(codec);
	snd_ad1848_mce_down(codec);
	return 0;		/* all things are ok.. */
}

/*

 */

static struct snd_stru_pcm1_hardware snd_ad1848_playback =
{
	NULL,			/* private data */
	NULL,			/* private_free */
	SND_PCM1_HW_AUTODMA,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* formats */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* hardware formats */
	0,			/* align value */
	6,			/* minimal fragment */
	5510,			/* min. rate */
	48000,			/* max. rate */
	2,			/* max. voices */
	snd_ad1848_playback_open,
	snd_ad1848_playback_close,
	snd_ad1848_playback_ioctl,
	snd_ad1848_playback_prepare,
	snd_ad1848_playback_trigger,
	snd_ad1848_playback_pointer,
	snd_pcm1_playback_dma,
	snd_pcm1_dma_move,
	snd_pcm1_playback_dma_neutral
};

static struct snd_stru_pcm1_hardware snd_ad1848_record =
{
	NULL,			/* private data */
	NULL,			/* private free */
	SND_PCM1_HW_AUTODMA,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* formats */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* hardware formats */
	0,			/* align value */
	6,			/* minimal fragment */
	5510,			/* min. rate */
	48000,			/* max. rate */
	2,			/* max. voices */
	snd_ad1848_record_open,
	snd_ad1848_record_close,
	snd_ad1848_record_ioctl,
	snd_ad1848_record_prepare,
	snd_ad1848_record_trigger,
	snd_ad1848_record_pointer,
	snd_pcm1_record_dma,
	snd_pcm1_dma_move,
	NULL
};

static void snd_ad1848_free(void *private_data)
{
	snd_free(private_data, sizeof(ad1848_t));
}

snd_pcm_t *snd_ad1848_new_device(snd_card_t * card,
				 unsigned short port,
				 snd_irq_t * irqptr,
				 snd_dma_t * dmaptr,
				 unsigned short hardware)
{
	snd_pcm_t *pcm;
	snd_pcm1_t *pcm1;
	ad1848_t *codec;
	char *str;

	pcm = snd_pcm1_new_device(card, "AD1848");
	if (!pcm)
		return NULL;
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = (ad1848_t *) snd_calloc(sizeof(ad1848_t));
	if (!codec)
		return NULL;
	snd_spin_prepare(codec, reg);
	snd_mutex_prepare(codec, open);
	snd_sleep_prepare(codec, mce);
	codec->pcm = pcm;
	codec->card = pcm1->card;
	codec->port = port;
	codec->irq = irqptr->irq;
	codec->irqptr = irqptr;
	codec->dmaptr = dmaptr;
	codec->dma = dmaptr->dma;
	codec->hardware = hardware;
	memcpy(&codec->image, &snd_ad1848_original_image, sizeof(snd_ad1848_original_image));
	memcpy(&pcm1->playback.hw, &snd_ad1848_playback, sizeof(snd_ad1848_playback));
	memcpy(&pcm1->record.hw, &snd_ad1848_record, sizeof(snd_ad1848_record));
	pcm1->private_data = codec;
	pcm1->private_free = snd_ad1848_free;
	pcm->info_flags = SND_PCM_INFO_CODEC | SND_PCM_INFO_MMAP |
	    SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_RECORD;
	switch (codec->hardware) {
	case AD1848_HW_AD1847:
		str = "AD1847";
		break;
	case AD1848_HW_AD1848:
		str = "AD1848";
		break;
	case AD1848_HW_CS4248:
		str = "CS4248";
		break;
	default:
		str = "???";
	}
	strcpy(pcm->name, str);
	if (snd_ad1848_probe(pcm1) < 0) {
		snd_pcm_free(pcm);
		return NULL;
	}
	return pcm;
}

/*
 *  MIXER part
 */

static int snd_ad1848_mixer_get_recsrc(ad1848_t *codec, snd_kmixer_element_t *element)
{
	if (element == codec->me_mux_line1)
		return AD1848_MIXS_LINE1;
	if (element == codec->me_mux_aux1)
		return AD1848_MIXS_AUX1;
	if (element == codec->me_mux_line2)
		return AD1848_MIXS_LINE2;
	if (element == codec->me_mux_mix)
		return AD1848_MIXS_ALL;
	return -EINVAL;
}

static snd_kmixer_element_t *snd_ad1848_mixer_get_recsrc_element(ad1848_t *codec, int value)
{
	value &= AD1848_MIXS_ALL;
	switch (value) {
	case AD1848_MIXS_LINE1:
		return codec->me_mux_line1;
	case AD1848_MIXS_AUX1:
		return codec->me_mux_aux1;
	case AD1848_MIXS_LINE2:
		return codec->me_mux_line2;
	case AD1848_MIXS_ALL:
		return codec->me_mux_mix;
	}
	return NULL;
}

static int snd_ad1848_mixer_mux(int w_flag, snd_kmixer_element_t **elements, ad1848_t *codec)
{
	unsigned long flags;
	int change = 0;
	int left, right, nleft, nright;

	snd_spin_lock(codec, reg, &flags);
	left = codec->image[AD1848_LEFT_INPUT] & AD1848_MIXS_ALL;
	right = codec->image[AD1848_RIGHT_INPUT] & AD1848_MIXS_ALL;
	if (w_flag) {
		nleft = snd_ad1848_mixer_get_recsrc(codec, elements[0]);
		nright = snd_ad1848_mixer_get_recsrc(codec, elements[1]);
		if (nleft >= 0 && nright >= 0) {
			change = nleft != left || nright != right;
			snd_ad1848_outm(codec, AD1848_LEFT_INPUT, (unsigned char)~AD1848_MIXS_ALL, (unsigned char)nleft);
			snd_ad1848_outm(codec, AD1848_RIGHT_INPUT, (unsigned char)~AD1848_MIXS_ALL, (unsigned char)nright);
		} else {
			change = -EINVAL;
		}
	} else {
		elements[0] = snd_ad1848_mixer_get_recsrc_element(codec, left);
		elements[1] = snd_ad1848_mixer_get_recsrc_element(codec, right);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;
}

int snd_ad1848_mixer_stereo_volume(int w_flag, int *voices, ad1848_t *codec,
					  int max, int invert, int shift,
					  unsigned char left_reg,
					  unsigned char right_reg)
{
	int change = 0, left, right;
	unsigned long flags;

	snd_spin_lock(codec, reg, &flags);
	left = (codec->image[left_reg] >> shift) & max;
	right = (codec->image[right_reg] >> shift) & max;
	if (!w_flag) {
		if (invert) {
			voices[0] = max - left;
			voices[1] = max - right;
		} else {
			voices[0] = left;
			voices[1] = right;
		}
	} else {
		if (invert) {
			change = max - left != voices[0] || max - right != voices[1];
		} else {
			change = left != voices[0] || right != voices[1];
		}
		left = voices[0]; right = voices[1];
		if (invert) {
			left = max - left;
			right = max - right;
		}
		snd_ad1848_outm(codec, left_reg, ~(max << shift), left << shift);
		snd_ad1848_outm(codec, right_reg, ~(max << shift), right << shift);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;	
}

int snd_ad1848_mixer_mono_volume(int w_flag, int *voices, ad1848_t *codec,
					  int max, int invert, int shift,
					  unsigned char reg)
{
	int change = 0, val;
	unsigned long flags;

	snd_spin_lock(codec, reg, &flags);
	val = (codec->image[reg] >> shift) & max;
	if (!w_flag) {
		voices[0] = invert ? max - val : val;
	} else {
		change = (invert ? max - val : val) != voices[0];
		val = voices[0];
		if (invert)
			val = max - val;
		snd_ad1848_outm(codec, reg, ~(max << shift), val << shift);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;	
}

int snd_ad1848_mixer_stereo_switch(int w_flag, unsigned int *bitmap, ad1848_t *codec,
					  int bit, int invert,
					  unsigned char left_reg,
					  unsigned char right_reg)
{
	int change = 0, left, right;
	unsigned long flags;

	snd_spin_lock(codec, reg, &flags);
	left = (codec->image[left_reg] >> bit) & 1;
	right = (codec->image[right_reg] >> bit) & 1;
	if (!w_flag) {
		if (invert) {
			snd_mixer_set_bit(bitmap, 0, left ^ 1);
			snd_mixer_set_bit(bitmap, 1, right ^ 1);
		} else {
			snd_mixer_set_bit(bitmap, 0, left);
			snd_mixer_set_bit(bitmap, 1, right);
		}
	} else {
		if (invert) {
			change = (left ^ 1) != snd_mixer_get_bit(bitmap, 0) ||
			         (right ^ 1) != snd_mixer_get_bit(bitmap, 1);
		} else {
			change = left != snd_mixer_get_bit(bitmap, 0) ||
			         right != snd_mixer_get_bit(bitmap, 1);
		}
		left = snd_mixer_get_bit(bitmap, 0);
		right = snd_mixer_get_bit(bitmap, 1);
		if (invert) {
			left ^= 1;
			right ^= 1;
		}
		snd_ad1848_outm(codec, left_reg, ~(1 << bit), left << bit);
		snd_ad1848_outm(codec, right_reg, ~(1 << bit), right << bit);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;	
}

int snd_ad1848_mixer_mono_switch(int w_flag, unsigned int *bitmap, ad1848_t *codec,
					  int bit, int invert,
					  unsigned char reg)
{
	int change = 0, val;
	unsigned long flags;

	snd_spin_lock(codec, reg, &flags);
	val = (codec->image[reg] >> bit) & 1;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, invert ? val ^ 1 : val);
	} else {
		change = (invert ? val ^ 1 : val) != snd_mixer_get_bit(bitmap, 0);
		val = snd_mixer_get_bit(bitmap, 0);
		if (invert)
			val ^= 1;
		snd_ad1848_outm(codec, reg, ~(1 << bit), val << bit);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;	
}

static int snd_ad1848_mixer_aux1_volume(int w_flag, int *voices, ad1848_t *codec)
{
	return snd_ad1848_mixer_stereo_volume(w_flag, voices, codec,
					      31, 1, 0,
					      AD1848_AUX1_LEFT_INPUT,
					      AD1848_AUX1_RIGHT_INPUT);
}

static int snd_ad1848_mixer_aux1_switch(int w_flag, unsigned int *bitmap, ad1848_t *codec)
{
	return snd_ad1848_mixer_stereo_switch(w_flag, bitmap, codec,
					      7, 1,
					      AD1848_AUX1_LEFT_INPUT,
					      AD1848_AUX1_RIGHT_INPUT);
}

static int snd_ad1848_mixer_aux2_volume(int w_flag, int *voices, ad1848_t *codec)
{
	return snd_ad1848_mixer_stereo_volume(w_flag, voices, codec,
					      31, 1, 0,
					      AD1848_AUX2_LEFT_INPUT,
					      AD1848_AUX2_RIGHT_INPUT);
}

static int snd_ad1848_mixer_aux2_switch(int w_flag, unsigned int *bitmap, ad1848_t *codec)
{
	return snd_ad1848_mixer_stereo_switch(w_flag, bitmap, codec,
					      7, 1,
					      AD1848_AUX2_LEFT_INPUT,
					      AD1848_AUX2_RIGHT_INPUT);
}

static int snd_ad1848_mixer_igain_volume(int w_flag, int *voices, ad1848_t *codec)
{
	return snd_ad1848_mixer_stereo_volume(w_flag, voices, codec,
					      15, 0, 0,
					      AD1848_LEFT_INPUT,
					      AD1848_RIGHT_INPUT);
}

static int snd_ad1848_mixer_loopback_volume(int w_flag, int *voices, ad1848_t *codec)
{
	return snd_ad1848_mixer_mono_volume(w_flag, voices, codec,
					      63, 1, 2,
					      AD1848_LOOPBACK);
}

static int snd_ad1848_mixer_loopback_switch(int w_flag, unsigned int *bitmap, ad1848_t *codec)
{
	return snd_ad1848_mixer_mono_switch(w_flag, bitmap, codec,
					      0, 0,
					      AD1848_LOOPBACK);
}

static int snd_ad1848_mixer_dac_volume(int w_flag, int *voices, ad1848_t *codec)
{
	return snd_ad1848_mixer_stereo_volume(w_flag, voices, codec,
					      63, 1, 0,
					      AD1848_LEFT_OUTPUT,
					      AD1848_RIGHT_OUTPUT);
}

static int snd_ad1848_mixer_dac_switch(int w_flag, unsigned int *bitmap, ad1848_t *codec)
{
	return snd_ad1848_mixer_stereo_switch(w_flag, bitmap, codec,
					      7, 1,
					      AD1848_LEFT_OUTPUT,
					      AD1848_RIGHT_OUTPUT);
}

snd_kmixer_t *snd_ad1848_new_mixer(snd_pcm_t * pcm, int pcm_dev)
{
	ad1848_t *codec;
	snd_pcm1_t *pcm1;
	snd_kmixer_t *mixer;
	snd_kmixer_element_t *mux;		/* input MUX */
	snd_kmixer_element_t *accu;		/* output accumulator */
	snd_kmixer_element_t *dig_accu;		/* digital accumulator */
	snd_kmixer_element_t *element1, *element2, *element3, *element4;
	snd_kmixer_group_t *group;
	static struct snd_mixer_element_volume1_range igain_range[2] = {
		{0, 15, 0, 2250},
		{0, 15, 0, 2250}
	};
	static struct snd_mixer_element_volume1_range aux_range[2] = {
		{0, 31, -3450, 1200},
		{0, 31, -3450, 1200}
	};
	static struct snd_mixer_element_volume1_range pcm_range[2] = {
		{0, 63, -9450, 0},
		{0, 63, -9450, 0}
	};

	if (!pcm || !pcm->card)
		return NULL;
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	if (!pcm1)
		return NULL;
	codec = (ad1848_t *) pcm1->private_data;
	if (!codec)
		return NULL;
	mixer = snd_mixer_new(pcm->card, pcm->id);
	if (!mixer)
		return NULL;
	strcpy(mixer->name, pcm->name);
	
	/* build input MUX and accumulators */
	if ((mux = snd_mixer_lib_mux1(mixer, SND_MIXER_ELEMENT_INPUT_MUX, 0, 0, 2, (snd_mixer_mux1_control_t *)snd_ad1848_mixer_mux, codec)) == NULL)
		goto __error;
	if ((accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, accu, mux) < 0)
		goto __error;
	codec->me_mux_mix = accu;
	if ((dig_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_DIGITAL_ACCU, 0, 0)) == NULL)
		goto __error;
	/* build Line1 */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_LINE, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, mux) < 0)
		goto __error;
	codec->me_mux_line1 = element1;
	/* build Line2 */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_LINE, 1)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 1, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, mux) < 0)
		goto __error;
	codec->me_mux_line2 = element1;
	/* build AUX1 */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_AUX, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, mux) < 0)
		goto __error;
	codec->me_mux_aux1 = element1;
	if ((element2 = snd_mixer_lib_volume1(mixer, "Aux Input Volume", 0, 2, aux_range, (snd_mixer_volume1_control_t *)snd_ad1848_mixer_aux1_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if ((element3 = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 0, 2, (snd_mixer_sw1_control_t *)snd_ad1848_mixer_aux1_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, accu) < 0)
		goto __error;
	/* build AUX2 */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_AUX, 1)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 1, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "Aux Input Volume", 1, 2, aux_range, (snd_mixer_volume1_control_t *)snd_ad1848_mixer_aux2_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if ((element3 = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 1, 2, (snd_mixer_sw1_control_t *)snd_ad1848_mixer_aux2_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, accu) < 0)
		goto __error;
	/* build input GAIN */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_GRP_IGAIN, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_volume1(mixer, "Input Gain Volume", 0, 2, igain_range, (snd_mixer_volume1_control_t *)snd_ad1848_mixer_igain_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, mux, element1) < 0)
		goto __error;
	/* build ADC */
	if ((element2 = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_ADC, 0, SND_MIXER_ETYPE_ADC, 16)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	/* build capture endpoint */
	if ((element3 = snd_mixer_lib_pcm(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE, 1, &pcm_dev)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	/* build loopback */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_GRP_DIGITAL_LOOPBACK, 0)) == NULL)
		goto __error;
	if ((element3 = snd_mixer_lib_volume1(mixer, "Loopback Volume", 0, 1, pcm_range, (snd_mixer_volume1_control_t *)snd_ad1848_mixer_loopback_volume, codec)) == NULL)
		goto __error;	
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if ((element4 = snd_mixer_lib_sw1(mixer, "Loopback Switch", 0, 1, (snd_mixer_sw1_control_t *)snd_ad1848_mixer_loopback_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element4) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, element4) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element4, dig_accu) < 0)
		goto __error;
	/* build playback endpoint */
	if ((element1 = snd_mixer_lib_pcm(mixer, SND_MIXER_ELEMENT_PLAYBACK, 0, SND_MIXER_ETYPE_PLAYBACK, 1, &pcm_dev)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, dig_accu) < 0)
		goto __error;
	/* build DAC */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_PCM, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_DAC, 0, SND_MIXER_ETYPE_DAC, 16)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, dig_accu, element1) < 0)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "PCM Volume", 0, 2, pcm_range, (snd_mixer_volume1_control_t *)snd_ad1848_mixer_dac_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if ((element3 = snd_mixer_lib_sw1(mixer, "PCM Switch", 0, 2, (snd_mixer_sw1_control_t *)snd_ad1848_mixer_dac_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, accu) < 0)
		goto __error;	

	codec->mixer = mixer;
	return mixer;

      __error:
      	snd_mixer_free(mixer);
      	return NULL;
}

/*
 *  INIT part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_ad1848_export;
#endif

int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_ad1848_export) < 0)
		return -ENOMEM;
#endif
	return 0;
}

void cleanup_module(void)
{
}
