/*
 * OSS compatible sequencer driver
 *
 * open/close and reset interface
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "seq_oss_device.h"
#include "seq_oss_synth.h"
#include "seq_oss_midi.h"
#include "seq_oss_writeq.h"
#include "seq_oss_readq.h"
#include "seq_oss_timer.h"

/*
 * common variables
 */
#ifdef MODULE_PARM
MODULE_PARM(maxqlen, "i");
MODULE_PARM_DESC(maxqlen, "maximum queue length");
MODULE_PARM(oss_queue, "i");
MODULE_PARM_DESC(oss_queue, "ALSA sequencer queue index");
#endif

static int system_client = -1; /* ALSA sequencer client number */
static int system_port = -1;
int oss_queue = 7;

int maxqlen = SND_SEQ_OSS_MAX_QLEN;
static int num_clients = 0;
static seq_oss_devinfo_t *client_table[SND_SEQ_OSS_MAX_CLIENTS];


/*
 * prototypes
 */
static int receive_announce(snd_seq_event_t *ev, int direct, void *private);
static int translate_mode(struct file *file);
static int create_port(seq_oss_devinfo_t *dp);
static int delete_port(seq_oss_devinfo_t *dp);
static void free_devinfo(void *private);

#define call_ctl(type,rec) snd_seq_kernel_client_ctl(system_client, type, rec)


/*
 * create sequencer client for OSS sequencer
 */
int
snd_seq_oss_create_client(void)
{
	int rc;
	snd_seq_client_callback_t callback;
	snd_seq_client_info_t info;
	snd_seq_port_info_t port;
	snd_seq_port_callback_t port_callback;

	/* create ALSA client */
	memset(&callback, 0, sizeof(callback));

	callback.private_data = NULL;
	callback.allow_input = 1;
	callback.allow_output = 1;

	rc = snd_seq_create_kernel_client(NULL, SND_SEQ_CLIENT_OSS, &callback);
	if (rc < 0)
		return rc;

	system_client = rc;
	debug_printk(("seq_oss: new client = %d\n", rc));

	/* set client information */
	memset(&info, 0, sizeof(info));
	info.client = system_client;
	info.type = KERNEL_CLIENT;
	strcpy(info.name, "OSS sequencer");

	rc = call_ctl(SND_SEQ_IOCTL_SET_CLIENT_INFO, &info);

	/* look up midi devices */
	snd_seq_oss_midi_lookup_ports(system_client);

	/* create annoucement receiver port */
	memset(&port, 0, sizeof(port));
	strcpy(port.name, "Receiver");
	port.client = system_client;
	port.capability = SND_SEQ_PORT_CAP_OUT;
	port.type = 0;

	memset(&port_callback, 0, sizeof(port_callback));
	port_callback.event_input = receive_announce;
	port.kernel = &port_callback;
	
	call_ctl(SND_SEQ_IOCTL_CREATE_PORT, &port);
	if ((system_port = port.port) >= 0) {
		snd_seq_port_subscribe_t subs;
		snd_seq_queue_client_t qclt;

		memset(&qclt, 0, sizeof(qclt));
		qclt.queue = oss_queue;
		qclt.client = system_client;
		qclt.used = 1;
		call_ctl(SND_SEQ_IOCTL_SET_QUEUE_CLIENT, &qclt);

		memset(&subs, 0, sizeof(subs));
		subs.sender.client = SND_SEQ_CLIENT_SYSTEM;
		subs.sender.port = SND_SEQ_PORT_SYSTEM_ANNOUNCE;
		subs.sender.queue = oss_queue;
		subs.dest.client = system_client;
		subs.dest.port = system_port;
		subs.dest.queue = oss_queue;
		call_ctl(SND_SEQ_IOCTL_SUBSCRIBE_PORT, &subs);
	}


	return 0;
}


/*
 * receive annoucement from system port, and check the midi device
 */
static int
receive_announce(snd_seq_event_t *ev, int direct, void *private)
{
	snd_seq_port_info_t pinfo;

	switch (ev->type) {
	case SND_SEQ_EVENT_PORT_START:
	case SND_SEQ_EVENT_PORT_CHANGE:
		memset(&pinfo, 0, sizeof(pinfo));
		pinfo.client = ev->data.addr.client;
		pinfo.port = ev->data.addr.port;
		if (call_ctl(SND_SEQ_IOCTL_GET_PORT_INFO, &pinfo) >= 0)
			snd_seq_oss_midi_check_new_port(&pinfo);
		break;

	case SND_SEQ_EVENT_PORT_EXIT:
		snd_seq_oss_midi_check_exit_port(ev->data.addr.client,
						ev->data.addr.port);
		break;
	}
	return 0;
}


/*
 * delete OSS sequencer client
 */
int
snd_seq_oss_delete_client(void)
{
	if (system_client >= 0)
		snd_seq_delete_kernel_client(system_client);
	return 0;
}


/*
 * open sequencer device
 */
int
snd_seq_oss_open(struct file *file, int level)
{
	int i, rc;
	seq_oss_devinfo_t *dp;

	if ((dp = snd_malloc(sizeof(*dp))) == NULL) {
		snd_printk("sequencer: can't malloc device info\n");
		return -ENOMEM;
	}

	for (i = 0; i < SND_SEQ_OSS_MAX_CLIENTS; i++) {
		if (! client_table[i])
			break;
	}
	if (i >= SND_SEQ_OSS_MAX_CLIENTS) {
		snd_printk("sequencer: too many applications\n");
		return -ENOMEM;
	}

	dp->index = i;
	dp->cseq = system_client;
	dp->port = -1;
	dp->readq = NULL;
	dp->writeq = NULL;

	/* look up synth and midi devices */
	snd_seq_oss_synth_setup(dp);
	snd_seq_oss_midi_setup(dp);

	if (dp->synth_opened == 0 && dp->max_mididev == 0) {
		snd_printk("sequencer: no device found\n");
		snd_free(dp, sizeof(*dp));
		return -ENODEV;
	}

	/* create port */
	if ((rc = create_port(dp)) < 0) {
		snd_printk("sequencer: can't create port\n");
		free_devinfo(dp);
		return rc;
	}

	/* set address */
	dp->addr.client = dp->cseq;
	dp->addr.port = dp->port;
	dp->addr.queue = 0;
	dp->addr.channel = 0;

	dp->seq_mode = SND_SEQ_OSS_MODE_SYNTH;

	/* set up file mode */
	dp->file_mode = translate_mode(file);

	/* initialize read queue */
	if (is_read_mode(dp->file_mode)) {
		if ((dp->readq = snd_seq_oss_readq_new(dp, maxqlen)) == NULL) {
			delete_port(dp);
			return -ENOMEM;
		}
	}

	/* initialize write queue */
	if (is_write_mode(dp->file_mode)) {
		dp->writeq = snd_seq_oss_writeq_new(dp, maxqlen);
		if (dp->writeq == NULL) {
			delete_port(dp);
			return -ENOMEM;
		}
	}

	/* initialize timer */
	if ((dp->timer = snd_seq_oss_timer_new(dp)) == NULL) {
		snd_printk("sequencer: can't alloc timer\n");
		delete_port(dp);
		return -ENOMEM;
	}

	/* set private data pointer */
	file->private_data = dp;

	client_table[dp->index] = dp;
	num_clients++;
	MOD_INC_USE_COUNT;

	debug_printk(("seq_oss: open done\n"));

	return 0;
}

/*
 * translate file flags to private mode
 */
static int
translate_mode(struct file *file)
{
	int file_mode = 0;
	if ((file->f_flags & O_ACCMODE) != O_RDONLY)
		file_mode |= SND_SEQ_OSS_FILE_WRITE;
	if ((file->f_flags & O_ACCMODE) != O_WRONLY)
		file_mode |= SND_SEQ_OSS_FILE_READ;
	if (file->f_flags & O_NONBLOCK)
		file_mode |= SND_SEQ_OSS_FILE_NONBLOCK;
	return file_mode;
}


/*
 * create sequencer port
 */
static int
create_port(seq_oss_devinfo_t *dp)
{
	int rc;
	snd_seq_port_info_t port;
	snd_seq_port_callback_t callback;

	memset(&port, 0, sizeof(port));
	memset(&callback, 0, sizeof(callback));

	port.client = dp->cseq;
	sprintf(port.name, "Sequencer");
	port.capability = SND_SEQ_PORT_CAP_IN|SND_SEQ_PORT_CAP_OUT;
	//port.capability |= SND_SEQ_PORT_CAP_SYNC_IN|SND_SEQ_PORT_CAP_SYNC_OUT;
	port.type = SND_SEQ_PORT_TYPE_SPECIFIC;
	port.midi_channels = 128;
	port.synth_voices = 128;

	callback.private_data = dp;
	callback.subscribe = NULL;
	callback.unsubscribe = NULL;
	callback.use = NULL;
	callback.unuse = NULL;
	callback.event_input = snd_seq_oss_midi_input;
	callback.private_free = free_devinfo;
	port.kernel = &callback;

	rc = call_ctl(SND_SEQ_IOCTL_CREATE_PORT, &port);
	if (rc < 0)
		return rc;

	dp->port = port.port;
	debug_printk(("seq_oss: new port = %d\n", port.port));

	return 0;
}

/*
 * delete ALSA port
 */
static int
delete_port(seq_oss_devinfo_t *dp)
{
	snd_seq_port_info_t port_info;

	if (dp->port < 0)
		return 0;

	memset(&port_info, 0, sizeof(port_info));
	port_info.client = dp->cseq;
	port_info.port = dp->port;
	return snd_seq_kernel_client_ctl(dp->cseq,
					 SND_SEQ_IOCTL_DELETE_PORT,
					 &port_info);
}


/*
 * free device informations - private_free callback of port
 */
static void
free_devinfo(void *private)
{
	seq_oss_devinfo_t *dp = (seq_oss_devinfo_t *)private;

	if (dp->timer)
		snd_seq_oss_timer_delete(dp->timer);
		
	if (dp->writeq)
		snd_seq_oss_writeq_delete(dp->writeq);

	if (dp->readq)
		snd_seq_oss_readq_delete(dp->readq);
	
	snd_free(dp, sizeof(*dp));
}


/*
 * close sequencer device
 */
void
snd_seq_oss_release(seq_oss_devinfo_t *dp)
{
	debug_printk(("seq_oss: resetting..\n"));
	snd_seq_oss_reset(dp);

	debug_printk(("seq_oss: cleaning up..\n"));
	snd_seq_oss_synth_cleanup(dp);
	snd_seq_oss_midi_cleanup(dp);

	/* clear slot */
	debug_printk(("seq_oss: releasing resource..\n"));
	if (dp->port >= 0)
		delete_port(dp);

	client_table[dp->index] = NULL;
	num_clients--;

	MOD_DEC_USE_COUNT;
	debug_printk(("seq_oss: release done\n"));
}


/*
 * Wait until the queue is empty (if we don't have nonblock)
 */
void
snd_seq_oss_flush_write(seq_oss_devinfo_t *dp)
{
	if (is_write_mode(dp->file_mode) && !is_nonblock_mode(dp->file_mode) &&
	    dp->writeq) {
		debug_printk(("seq_oss: syncing..\n"));
		while (snd_seq_oss_writeq_sync(dp->writeq))
			;
	}
}


/*
 * reset sequencer devices
 */
void
snd_seq_oss_reset(seq_oss_devinfo_t *dp)
{
	int i;

	/* remove queues */
	if (dp->readq)
		snd_seq_oss_readq_clear(dp->readq);
	if (dp->writeq)
		snd_seq_oss_writeq_clear(dp->writeq);

	/* reset all synth devices */
	for (i = 0; i < dp->max_synthdev; i++)
		snd_seq_oss_synth_reset(dp, i);

	/* reset all midi devices */
	for (i = 0; i < dp->max_mididev; i++)
		snd_seq_oss_midi_reset(dp, i);

	/* reset timer */
	snd_seq_oss_timer_init(dp->timer);
}

/*
 * proc interface
 */
void
snd_seq_oss_system_info_read(snd_info_buffer_t *buf)
{
	int i;
	seq_oss_devinfo_t *dp;

	snd_iprintf(buf, "ALSA client number %d\n", system_client);
	snd_iprintf(buf, "ALSA receiver port %d\n", system_port);

	snd_iprintf(buf, "\nNumber of applications: %d\n", num_clients);
	for (i = 0; i < num_clients; i++) {
		snd_iprintf(buf, "\nApplication %d: ", i);
		if ((dp = client_table[i]) == NULL) {
			snd_iprintf(buf, "*empty*\n");
			continue;
		}
		snd_iprintf(buf, "\n");
		snd_iprintf(buf, "  port %d\n", dp->port);
		snd_iprintf(buf, "  seq mode %s\n",
			    (dp->seq_mode ? "music" : "synth"));
		snd_iprintf(buf, "  file mode %s\n", filemode_str(dp->file_mode));
		snd_iprintf(buf, "  max queue length %d\n", maxqlen);
		if (is_read_mode(dp->file_mode) && dp->readq)
			snd_seq_oss_readq_info_read(dp->readq, buf);
		if (is_write_mode(dp->file_mode) && dp->writeq)
			snd_seq_oss_writeq_info_read(dp->writeq, buf);
		snd_seq_oss_timer_info_read(dp->timer, buf);
	}
}

/*
 * misc. functions for proc interface
 */
char *
enabled_str(int bool)
{
	return bool ? "enabled" : "disabled";
}

char *
filemode_str(int val)
{
	static char *str[] = {
		"none", "read", "write", "read/write",
	};
	return str[val & SND_SEQ_OSS_FILE_ACMODE];
}


