/*
 *  Copyright (c) by Christian Fischbach
 *  <fishbach@pool.informatik.rwth-aachen.de>
 *  Copyright (c) by Abramo Bagnara
 *  <abbagnara@racine.ra.it>
 *  Routines for control of ESS ES1869 chip
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * NOTES:
 * - there are a first full duplex pcm and a second playback only pcm
 *   (incompatible with first pcm record)
 * - there is support for the record volume and ESS Spatializer 3D effect.
 *
 * BUGS:
 * - the delays to avoid pops on playback and recording are wrong
 * - contrary to some pages in DS_1869.PDF the rates can be set
 *   independently.
 * - There is a trouble (maybe correlated) I noted:
 *   using both channel at high rate (say 44100) the first pcm slows
 *   and the sound is somewhat garbled 
 *   (I suppose this is a limitation of ES1869, but I'm not sure)
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "es1869.h"


#define DAC1 0x01
#define ADC1 0x02
#define DAC2 0x04

static int snd_es1869_dsp_command(es1869_t * codec, unsigned char val)
{
        int i;

        for(i = 10000; i; i--)
                if ((inb(codec->port + 0x0C) & 0x80) == 0) {
                        outb(val, codec->port + 0x0C);
                        return 1;
                }
#ifdef SNDCFG_DEBUG
        printk("snd_es1869_dsp_command: timeout (0x%x)\n", val);
#endif
        return 0;
}

static int snd_es1869_dsp_get_byte(es1869_t * codec)
{
        int i;

        for(i = 1000; i; i--)
                if (inb(codec->port + 0x0C) & 0x40)
                        return inb(codec->port + 0x0A);
        snd_printd("es1869 get byte failed: 0x%x = 0x%x!!!\n", codec->port + 0x0A, inb(codec->port + 0x0A));
        return -ENODEV;
}

static int snd_es1869_write(es1869_t * codec,
			    unsigned char reg, unsigned char data)
{
        if (!snd_es1869_dsp_command(codec, reg))
                return 0;
        return snd_es1869_dsp_command(codec, data);
}

static int snd_es1869_read(es1869_t *codec, unsigned char reg)
{
        /* Read a byte from an extended mode register of ES1869 */
        if (!snd_es1869_dsp_command(codec, 0xC0))
                return -1;
        if (!snd_es1869_dsp_command(codec, reg))
                return -1;
        return snd_es1869_dsp_get_byte(codec);
}

static int snd_es1869_bits(es1869_t *codec, unsigned char reg,
			   unsigned char set, unsigned char unset)
{
        unsigned char c = snd_es1869_read(codec, reg);

        if (!c)
                return -1;
        return snd_es1869_write(codec, reg, (c & ~unset) | set);
}

void snd_es1869_mixer_write(es1869_t * codec,
			    unsigned char reg, unsigned char data)
{
        outb(reg, codec->port + 0x04);
        outb(data, codec->port + 0x05);
}

unsigned char snd_es1869_mixer_read(es1869_t * codec, unsigned char reg)
{
        outb(reg, codec->port + 0x04);
        return inb(codec->port + 0x05);
}

static void snd_es1869_mixer_bits(es1869_t *codec, unsigned char reg,
                                  unsigned char set, unsigned char unset)
{
        snd_es1869_mixer_write(codec, reg,
                               (snd_es1869_mixer_read(codec, reg) & ~unset) | set);
}

static int snd_es1869_reset(es1869_t * codec)
{
        int i;

        outb(0x03, codec->port + 0x06);
        inb(codec->port + 0x06);
        outb(0x00, codec->port + 0x06);
        for(i = 0; i < 10000 && !(inb(codec->port + 0x0E) & 0x80); i++);
        if (inb(codec->port + 0x0A) != 0xAA) {
                snd_printd("ess_reset at 0x%x: failed!!!\n", codec->port);
                return -ENODEV;
        }
        /* enable extended mode */
        snd_es1869_dsp_command(codec, 0xC6);
	/* Enable Audio 2 interrupt */
        snd_es1869_mixer_bits(codec, 0x7A, 0x40, 0x40);
	/* Enable MPU interrupt */
        if (codec->mpu_port >= 0x300 && codec->mpu_port < 0x400)
        	snd_es1869_mixer_bits(codec, 0x64, 0x40, 0x40);
        /* Change behaviour of register A1
           2nd channel DAC asynchronous */
        snd_es1869_mixer_bits(codec, 0x71, 0x22, 0x22);
        return 0;
}

static int snd_es1869_probe(es1869_t *codec)
{
        unsigned long flags;
        unsigned short port;

        port = codec->port;

        /* I am not sure if this is nessesary */
        snd_spin_lock(codec, reg, &flags);

        /*
         *  identifying the ES1869
         */

        outb(0x40, port + 0x04);
        codec->version =  inb(port + 0x05) << 8;
        codec->version += inb(port + 0x05);
        codec->ctrl_port =  inb(port + 0x05) << 8;
        codec->ctrl_port += inb(port + 0x05);
        if (codec->version != 0x1869) {
                snd_printk("[0x%x] ESS: unknown AudioDrive chip ES%x\n",
                           port, codec->version);
                snd_spin_unlock(codec, reg, &flags);
                return -ENODEV;
        }

        /*
         *  reset the ES1869
         */

        if (snd_es1869_reset(codec) < 0) {
                snd_printdd("ESS: [0x%x] reset failed... 0x%x\n",
                            port, inb(port + 0x0A));
                snd_spin_unlock(codec, reg, &flags);
                return -ENODEV;
        }

        snd_spin_unlock(codec, reg, &flags);

        snd_printdd("ESS: [0x%x] found.. version = 0x%x, ctrl port = 0x%x\n", codec->port, codec->version, codec->ctrl_port);

        return 0;
}

static void snd_es1869_set_rate(es1869_t *codec, int rate, int dma)
{
        int div0, div1, diff0, diff1;
        unsigned char bits;

#if 0
        /* Useless: middle level checks for this */
        if (rate < 4000) {
                snd_printd("Wrong rate: %d\n", rate);
                rate = 4000;
        }
        if (rate > 48000) {
                snd_printd("Wrong rate: %d\n", rate);
                rate = 48000;
        }
#endif

        /* let us calculate the best clock choice */
        div0 = (793800 + (rate >> 1)) / rate;
        div1 = (768000 + (rate >> 1)) / rate;
        diff0 = 793800 / div0 - rate;
        diff0 = (diff0 < 0) ? (-diff0) : diff0;
        diff1 = 768000 / div1 - rate;
        diff1 = (diff1 < 0) ? (-diff1) : diff1;
        if (diff0 < diff1)
                bits = 128 - div0;
        else
                bits = 256 - div1;

        /* set filter register */
        rate = (rate * 9) / 20;
        div0 = 256 - 7160000 / (rate * 82);

        if (dma == 1) {
                snd_es1869_write(codec, 0xA1, bits);
                snd_es1869_write(codec, 0xA2, div0);
        }
        else {
                snd_es1869_mixer_write(codec, 0x70, bits);
                snd_es1869_mixer_write(codec, 0x72, div0);
        }
}

static int snd_es1869_playback_ioctl(snd_pcm1_t * pcm1,
				     unsigned int cmd, unsigned long *arg)
{
        switch(cmd) {
        case SND_PCM1_IOCTL_RATE:
                pcm1->playback.real_rate = pcm1->playback.rate;
#if 0
                /* Useless: middle level check for this */
                if (pcm1->playback.real_rate < 4000)
                        pcm1->playback.real_rate = 4000;
                if (pcm1->playback.real_rate > 48000)
                        pcm1->playback.real_rate = 48000;
#endif
                return 0;
        }
        return -ENXIO;
}

static int snd_es1869_record_ioctl(snd_pcm1_t * pcm1,
				   unsigned int cmd, unsigned long *arg)
{
        switch(cmd) {
        case SND_PCM1_IOCTL_RATE:
                pcm1->record.real_rate = pcm1->record.rate;
#if 0
                /* Useless: middle level check for this */
                if (pcm1->record.real_rate < 4000)
                        pcm1->record.real_rate = 4000;
                if (pcm1->record.real_rate > 48000)
                        pcm1->record.real_rate = 48000;
#endif
                return 0;
        }
        return -ENXIO;
}

static void snd_es1869_playback_prepare(snd_pcm1_t *pcm1,
					unsigned char *buffer,
					unsigned int size,
					unsigned int offset,
					unsigned int count)
{
        unsigned long flags;
        es1869_t *codec;
        snd_pcm1_channel_t *pchn;

        pchn = &pcm1->playback;
        codec = (es1869_t *) pcm1->private_data;

        snd_spin_lock(codec, mixer, &flags);

        /* only do reset if not recording is active */
        if (codec->active == 0x00)
                snd_es1869_reset(codec);


        /* Auto-Initialize DMA mode + demand mode (4 bytes/request) */
        snd_es1869_mixer_write(codec, 0x78, 0x90);

        snd_es1869_set_rate(codec, pchn->rate, 2);

        /* Transfer Count Reload */
        count = -count;
        snd_es1869_mixer_write(codec, 0x74, (unsigned char) count);
        snd_es1869_mixer_write(codec, 0x76, (unsigned char) (count >> 8));

        snd_es1869_mixer_bits(codec, 0x7A, 
			      ((pchn->voices == 1) ? 0x00 : 0x02) |
			      ((pchn->mode & SND_PCM1_MODE_16) ? 0x01 : 0x00) |
			      ((pchn->mode & SND_PCM1_MODE_U) ? 0x00 : 0x04),
			      0x07);


        snd_spin_unlock(codec, mixer, &flags);

        /* Set DMA controler */
        snd_dma_program(codec->dma2, buffer, size, DMA_MODE_WRITE | DMA_MODE_AUTOINIT);

        snd_printd("Playback prepare! Voices = 0x%x\n", pchn->voices);
}

static void snd_es1869_playback_trigger(snd_pcm1_t * pcm1, int up)
{
        es1869_t *codec;
        unsigned long flags;

        codec = (es1869_t *) pcm1->private_data;
        if (up)
                codec->active |= DAC2;
        else
                codec->active &= ~DAC2;

        snd_spin_lock(codec, reg, &flags);

        if (up) {
                /* Start DMA */
                snd_es1869_mixer_bits(codec, 0x78, 0x03, 0x03);
	  
                /* avoid pops */
                snd_delay(10);

                /* Enable Audio 2 (this register is set to 0 by reset) */
                snd_es1869_mixer_write(codec, 0x7C, codec->audio2_vol);
        }
        else {
                /* Stop DMA */
                snd_es1869_mixer_bits(codec, 0x78, 0x00, 0x03);

                snd_delay(3);

                /* Disable Audio 2 (this register is set to 0 by reset) */
                snd_es1869_mixer_write(codec, 0x7C, 0);
        }

        snd_spin_unlock(codec, reg, &flags);

        snd_printd("Playback trigger! Up = %d\n", up);
}

static void snd_es1869_record_prepare(snd_pcm1_t *pcm1,
				      unsigned char *buffer,
				      unsigned int size,
				      unsigned int offset,
				      unsigned int count)
{
        unsigned long flags;
        es1869_t *codec;
        snd_pcm1_channel_t *pchn;

        pchn = &pcm1->record;
        codec = (es1869_t *) pcm1->private_data;

        snd_spin_lock(codec, reg, &flags);

        /* only do reset if not playback is active */
        if (codec->active == 0x00)
                snd_es1869_reset(codec);

        /* Auto-Initialize DMA mode */
        snd_es1869_write(codec, 0xB8, 0x0E);

        /* Set stereo/mono */
        snd_es1869_bits(codec, 0xA8, 3 - pchn->voices, 0x03);

        /* demand mode (4 bytes/request) */
        snd_es1869_write(codec, 0xB9, 2);

        snd_es1869_set_rate(codec, pchn->rate, 1);

        /* Transfer Count Reload */
        count = -count;
        snd_es1869_write(codec, 0xA4, (unsigned char) count);
        snd_es1869_write(codec, 0xA5, (unsigned char) (count >> 8));

        /* Init DACs */
        snd_es1869_write(codec, 0xB7, 
                         (pchn->mode & SND_PCM1_MODE_U) ? 0x51 : 0x71);
        snd_es1869_write(codec, 0xB7, 0x90 |
                         ((pchn->voices == 1) ? 0x40 : 0x08) |
                         ((pchn->mode & SND_PCM1_MODE_16) ? 0x04 : 0x00) |
                         ((pchn->mode & SND_PCM1_MODE_U) ? 0x00 : 0x20));

        /* Enable DMA and IRQ */
        snd_es1869_bits(codec, 0xB1, 0x50, 0xf0);
        snd_es1869_bits(codec, 0xB2, 0x50, 0xf0);
        snd_spin_unlock(codec, reg, &flags);

        /* Set DMA controler */
        snd_dma_program(codec->dma1, buffer, size, DMA_MODE_READ | DMA_MODE_AUTOINIT);

        snd_printd("Record prepare! Voices = 0x%x\n", pchn->voices);
}

static void snd_es1869_record_trigger(snd_pcm1_t *pcm1, int up)
{
        es1869_t *codec;
        unsigned long flags;

        codec = (es1869_t *) pcm1->private_data;
        if (up)
                codec->active |= ADC1;
        else
                codec->active &= ~ADC1;

        snd_spin_lock(codec, reg, &flags);

        if (up) {
                /* Start DMA */
                snd_es1869_bits(codec, 0xB8, 0x01, 0x01);
        }
        else {
                /* Stop DMA */
                snd_es1869_bits(codec, 0xB8, 0x00, 0x01);
        }

        snd_spin_unlock(codec, reg, &flags);

        snd_printd("Record trigger! Up = %d\n", up);
}

static void snd_es1869_playback2_prepare(snd_pcm1_t *pcm1,
                                         unsigned char *buffer,
                                         unsigned int size,
                                         unsigned int offset,
                                         unsigned int count)
{
        unsigned long flags;
        es1869_t *codec;
        snd_pcm1_channel_t *pchn;

        pchn = &pcm1->playback;
        codec = (es1869_t *) pcm1->private_data;

        snd_spin_lock(codec, reg, &flags);

        /* only do reset if not playback is active */
        if (codec->active == 0x00)
                snd_es1869_reset(codec);

        /* Auto-Initialize DMA mode */
        snd_es1869_write(codec, 0xB8, 0x04);

        /* Set stereo/mono */
        snd_es1869_bits(codec, 0xA8, 3 - pchn->voices, 0x03);

        /* demand mode (4 bytes/request) */
        snd_es1869_write(codec, 0xB9, 2);

        snd_es1869_set_rate(codec, pchn->rate, 1);

        /* Transfer Count Reload */
        count = -count;
        snd_es1869_write(codec, 0xA4, (unsigned char) count);
        snd_es1869_write(codec, 0xA5, (unsigned char) (count >> 8));

        /* Init DACs */
        snd_es1869_write(codec, 0xB6,
                         (pchn->mode & SND_PCM1_MODE_U) ? 0x80 : 0x00);
        snd_es1869_write(codec, 0xB7, 
                         (pchn->mode & SND_PCM1_MODE_U) ? 0x51 : 0x71);
        snd_es1869_write(codec, 0xB7, 0x90 |
                         ((pchn->voices == 1) ? 0x40 : 0x08) |
                         ((pchn->mode & SND_PCM1_MODE_16) ? 0x04 : 0x00) |
                         ((pchn->mode & SND_PCM1_MODE_U) ? 0x00 : 0x20));


        /* Enable DMA and IRQ */
        snd_es1869_bits(codec, 0xB1, 0x50, 0xF0);
        snd_es1869_bits(codec, 0xB2, 0x50, 0xF0);
        snd_spin_unlock(codec, reg, &flags);

        /* Set DMA controler */
        snd_dma_program(codec->dma1, buffer, size, DMA_MODE_WRITE | DMA_MODE_AUTOINIT);

        snd_printd("Playback2 prepare! Voices = 0x%x\n", pchn->voices);
}

static void snd_es1869_playback2_trigger(snd_pcm1_t *pcm1, int up)
{
        es1869_t *codec;
        unsigned long flags;

        codec = (es1869_t *) pcm1->private_data;
        if (up)
                codec->active |= DAC1;
        else
                codec->active &= ~DAC1;

        snd_spin_lock(codec, reg, &flags);

        if (up) {
                /* Start DMA */
                snd_es1869_bits(codec, 0xB8, 0x01, 0x01);
                snd_delay(10);
                /* Enable Audio 1 */
                snd_es1869_dsp_command(codec, 0xD1);
        }
        else {
                /* Stop DMA */
                snd_es1869_bits(codec, 0xB8, 0x00, 0x01);
                snd_delay(3);
                /* Disable Audio 1 */
                snd_es1869_dsp_command(codec, 0xD3);
        }

        snd_spin_unlock(codec, reg, &flags);

        snd_printd("Playback2 trigger! Up = %d\n", up);
}

void snd_es1869_interrupt(es1869_t *codec, unsigned char status)
{
        if (status & 0x01) {
                /* ok.. record is active */
                if (codec->active & ADC1) {
                        snd_pcm1_t *pcm1=(snd_pcm1_t *) codec->pcm->private_data;
                        pcm1->record.ack(pcm1);
                }
                /* ok.. playback2 is active */
                if (codec->active & DAC1) {
                        snd_pcm1_t *pcm1=(snd_pcm1_t *) codec->pcm2->private_data;
                        pcm1->playback.ack(pcm1);
                }
        }

        /* ok.. playback is active */
        if (status & 0x02) {
                if (codec->active & DAC2) {
                        snd_pcm1_t *pcm1=(snd_pcm1_t *) codec->pcm->private_data;
                        pcm1->playback.ack(pcm1);
                }
        }

#if 0
	/* For undocumented reason reset is needed for both 
	   interrupt source */
        if (status & 0x02)
#endif
                snd_es1869_mixer_bits(codec, 0x7A, 0x00, 0x80);
}

static int snd_es1869_playback_open(snd_pcm1_t * pcm1)
{
        es1869_t *codec;
        int err;

        codec = (es1869_t *) pcm1->private_data;
        if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_PLAYBACK, codec->dma2ptr, "ES1869 (playback)")) < 0)
                return err;
        codec->open |= DAC2;
        return 0;
}

static int snd_es1869_playback2_open(snd_pcm1_t * pcm1)
{
        es1869_t *codec;
        int err;

        codec = (es1869_t *) pcm1->private_data;
        if (codec->open & ADC1)
                return -EBUSY;
        if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_PLAYBACK, codec->dma1ptr, "ES1869 (playback2)")) < 0)
                return err;
        codec->open |= DAC1;
        return 0;
}

static int snd_es1869_record_open(snd_pcm1_t * pcm1)
{
        es1869_t *codec;
        int err;

        codec = (es1869_t *) pcm1->private_data;
        if (codec->open & DAC1)
                return -EBUSY;
        if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_RECORD, codec->dma1ptr, "ES1869 (record)")) < 0)
                return err;
        codec->open |= ADC1;
        return 0;
}

static void snd_es1869_playback_close(snd_pcm1_t * pcm1)
{
        es1869_t *codec;

        codec = (es1869_t *) pcm1->private_data;
        snd_pcm1_dma_free(pcm1, SND_PCM1_PLAYBACK, codec->dma2ptr);
        codec->open &= ~DAC2;
}

static void snd_es1869_playback2_close(snd_pcm1_t * pcm1)
{
        es1869_t *codec;

        codec = (es1869_t *) pcm1->private_data;
        snd_pcm1_dma_free(pcm1, SND_PCM1_PLAYBACK, codec->dma1ptr);
        codec->open &= ~DAC1;
}

static void snd_es1869_record_close(snd_pcm1_t * pcm1)
{
        es1869_t *codec;

        codec = (es1869_t *) pcm1->private_data;
        snd_pcm1_dma_free(pcm1, SND_PCM1_RECORD, codec->dma1ptr);
        codec->open &= ~ADC1;
}

static unsigned int snd_es1869_playback_pointer(snd_pcm1_t * pcm1, unsigned int used_size)
{
        es1869_t *codec;

        codec = (es1869_t *) pcm1->private_data;
        if (!(codec->active & DAC2))
                return 0;
        return used_size - snd_dma_residue(codec->dma2);
}

static unsigned int snd_es1869_playback2_pointer(snd_pcm1_t * pcm1, unsigned int used_size)
{
        es1869_t *codec;

        codec = (es1869_t *) pcm1->private_data;
        if (!(codec->active & DAC1))
                return 0;
        return used_size - snd_dma_residue(codec->dma1);
}

static unsigned int snd_es1869_record_pointer(snd_pcm1_t * pcm1, unsigned int used_size)
{
        es1869_t *codec;

        codec = (es1869_t *) pcm1->private_data;
        if (!(codec->active & ADC1))
                return 0;
        return used_size - snd_dma_residue(codec->dma1);
}

static struct snd_stru_pcm1_hardware snd_es1869_playback =
{
        NULL,			/* private data */
        NULL,			/* private_free */
        SND_PCM1_HW_AUTODMA,	/* flags */
        SND_PCM_FMT_MU_LAW |
        SND_PCM_FMT_U8 | SND_PCM_FMT_S8 | 
        SND_PCM_FMT_S16_LE | SND_PCM_FMT_U16_LE,	/* formats */
        SND_PCM_FMT_U8 | SND_PCM_FMT_S8 | 
        SND_PCM_FMT_S16_LE | SND_PCM_FMT_U16_LE,	/* hardware formats */
        0,			/* align value */
        6,			/* minimal fragment */
        4000,			/* min. rate */
        48000,			/* max. rate */
        2,			/* max. voices */
        snd_es1869_playback_open,
        snd_es1869_playback_close,
        snd_es1869_playback_ioctl,
        snd_es1869_playback_prepare,
        snd_es1869_playback_trigger,
        snd_es1869_playback_pointer,
        snd_pcm1_playback_dma_ulaw,
        snd_pcm1_dma_move,
        snd_pcm1_playback_dma_neutral
};

static struct snd_stru_pcm1_hardware snd_es1869_playback2 =
{
        NULL,			/* private data */
        NULL,			/* private_free */
        SND_PCM1_HW_AUTODMA,	/* flags */
        SND_PCM_FMT_MU_LAW |
        SND_PCM_FMT_U8 | SND_PCM_FMT_S8 | 
        SND_PCM_FMT_S16_LE | SND_PCM_FMT_U16_LE,	/* formats */
        SND_PCM_FMT_U8 | SND_PCM_FMT_S8 | 
        SND_PCM_FMT_S16_LE | SND_PCM_FMT_U16_LE,	/* hardware formats */
        0,			/* align value */
        6,			/* minimal fragment */
        4000,			/* min. rate */
        48000,			/* max. rate */
        2,			/* max. voices */
        snd_es1869_playback2_open,
        snd_es1869_playback2_close,
        snd_es1869_playback_ioctl,
        snd_es1869_playback2_prepare,
        snd_es1869_playback2_trigger,
        snd_es1869_playback2_pointer,
        snd_pcm1_playback_dma_ulaw,
        snd_pcm1_dma_move,
        snd_pcm1_playback_dma_neutral
};

static struct snd_stru_pcm1_hardware snd_es1869_record =
{
        NULL,			/* private data */
        NULL,			/* private free */
        SND_PCM1_HW_AUTODMA,	/* flags */
        SND_PCM_FMT_MU_LAW |
        SND_PCM_FMT_U8 | SND_PCM_FMT_S8 | 
        SND_PCM_FMT_S16_LE | SND_PCM_FMT_U16_LE,	/* formats */
        SND_PCM_FMT_U8 | SND_PCM_FMT_S8 | 
        SND_PCM_FMT_S16_LE | SND_PCM_FMT_U16_LE,	/* hardware formats */
        0,			/* align value */
        6,			/* minimal fragment */
        4000,			/* min. rate */
        48000,			/* max. rate */
        2,			/* max. voices */
        snd_es1869_record_open,
        snd_es1869_record_close,
        snd_es1869_record_ioctl,
        snd_es1869_record_prepare,
        snd_es1869_record_trigger,
        snd_es1869_record_pointer,
        snd_pcm1_record_dma_ulaw,
        snd_pcm1_dma_move,
        NULL
};

static void snd_es1869_free(void *private_data)
{
}

snd_pcm_t *snd_es1869_pcm(es1869_t * codec)
{
        snd_pcm_t *pcm;
        snd_pcm1_t *pcm1;
        pcm = snd_pcm1_new_device(codec->card, "ES1869 DAC2/ADC");
        if (!pcm)
                return NULL;
        pcm1 = (snd_pcm1_t *) pcm->private_data;
        memcpy(&pcm1->playback.hw, &snd_es1869_playback, sizeof(snd_es1869_playback));
        memcpy(&pcm1->record.hw, &snd_es1869_record, sizeof(snd_es1869_record));
        pcm1->private_data = codec;
        pcm1->private_free = snd_es1869_free;
        pcm->info_flags = SND_PCM_INFO_CODEC | SND_PCM_INFO_MMAP |
                SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_RECORD |
                SND_PCM_INFO_DUPLEX;
        strcpy(pcm->name, "ESS AudioDrive ES1869 DAC2/ADC");
        return codec->pcm = pcm;
}

snd_pcm_t *snd_es1869_pcm2(es1869_t * codec)
{
        snd_pcm_t *pcm;
        snd_pcm1_t *pcm1;
        pcm = snd_pcm1_new_device(codec->card, "ES1869 DAC1");
        if (!pcm)
                return NULL;
        pcm1 = (snd_pcm1_t *) pcm->private_data;
        memcpy(&pcm1->playback.hw, &snd_es1869_playback2, sizeof(snd_es1869_playback2));
        pcm1->private_data = codec;
        pcm1->private_free = snd_es1869_free;
        pcm->info_flags = SND_PCM_INFO_CODEC | SND_PCM_INFO_MMAP |
                SND_PCM_INFO_PLAYBACK;
        strcpy(pcm->name, "ESS AudioDrive ES1869 DAC1");
        return codec->pcm2 = pcm;
}


es1869_t *snd_es1869_new_device(snd_card_t * card,
                                unsigned short port,
                                unsigned short mpu_port,
                                unsigned short fm_port,
                                snd_irq_t * irqptr,
                                snd_dma_t * dma1ptr,
                                snd_dma_t * dma2ptr)
{
        es1869_t *codec;

        codec = (es1869_t *) snd_calloc(sizeof(es1869_t));
        if (!codec)
                return NULL;
        snd_spin_prepare(codec, reg);
        snd_spin_prepare(codec, mixer);
        snd_spin_prepare(codec, ctrl);
        codec->card = card;
        codec->port = port;
        codec->mpu_port = mpu_port;
        codec->fm_port = fm_port;
        codec->irqptr = irqptr;
        codec->irq = irqptr->irq;
        codec->dma1ptr = dma1ptr;
        codec->dma1 = dma1ptr->dma;
        codec->dma2ptr = dma2ptr;
        codec->dma2 = dma2ptr->dma;
        codec->audio2_vol = 0x00;

        if (snd_es1869_probe(codec) < 0) {
                snd_free(codec, sizeof(es1869_t));
                return NULL;
        }
        return codec;
}

/*
 *  MIXER part
 */

static void snd_es1869_record_volume_level(snd_kmixer_t * mixer,
					   snd_kmixer_channel_t * channel,
					   int l, int r)
{
        unsigned long flags;
        es1869_t *codec;
        int left = (channel->record & SND_MIX_REC_LEFT) ? l : 0;
        int right = (channel->record & SND_MIX_REC_RIGHT) ? r : 0;
        codec = (es1869_t *) mixer->private_data;
        snd_spin_lock(codec, mixer, &flags);
        switch (channel->hw.priority) {
        case SND_MIXER_PRI_PCM:
                snd_es1869_mixer_write(codec, 0x69, left << 4 | right);
                break;
        case SND_MIXER_PRI_MIC:
                snd_es1869_mixer_write(codec, 0x68, left << 4 | right);
                break;
        case SND_MIXER_PRI_FM:
                snd_es1869_mixer_write(codec, 0x6b, left << 4 | right);
                break;
        case SND_MIXER_PRI_CD:
                snd_es1869_mixer_write(codec, 0x6a, left << 4 | right);
                break;
        case SND_MIXER_PRI_AUXA:
                snd_es1869_mixer_write(codec, 0x6c, left << 4 | right);
                break;
        case SND_MIXER_PRI_LINE:
                snd_es1869_mixer_write(codec, 0x6e, left << 4 | right);
                break;
        default:
                snd_spin_unlock(codec, mixer, &flags);
                snd_printd("Invalid channel priority: %x\n", channel->hw.priority);
                return;
        }
        snd_spin_unlock(codec, mixer, &flags);
}


static void snd_es1869_record_source(snd_kmixer_t * mixer,
				     snd_kmixer_channel_t *channel,
				     unsigned int rec)
{
        unsigned long flags;
        es1869_t *codec;
        int left = (rec & SND_MIX_REC_LEFT) ? channel->input.left : 0;
        int right = (rec & SND_MIX_REC_RIGHT) ? channel->input.right : 0;
        codec = (es1869_t *) mixer->private_data;
        snd_spin_lock(codec, mixer, &flags);
        switch (channel->hw.priority) {
        case SND_MIXER_PRI_PCM:
                snd_es1869_mixer_write(codec, 0x1c, 0x05);
                snd_es1869_mixer_write(codec, 0x69, left << 4 | right);
                break;
        case SND_MIXER_PRI_MIC:
                snd_es1869_mixer_write(codec, 0x1c, 0x05);
                snd_es1869_mixer_write(codec, 0x68, left << 4 | right);
                break;
#if 0
        case SND_MIXER_PRI_MASTER:
                snd_es1869_mixer_write(codec, 0x1c, 0x07 | (rec ? 0x00 | 0x10));
                break;
#endif
        case SND_MIXER_PRI_FM:
                snd_es1869_mixer_write(codec, 0x1c, 0x05);
                snd_es1869_mixer_write(codec, 0x6b, left << 4 | right);
                break;
        case SND_MIXER_PRI_CD:
                snd_es1869_mixer_write(codec, 0x1c, 0x05);
                snd_es1869_mixer_write(codec, 0x6a, left << 4 | right);
                break;
        case SND_MIXER_PRI_AUXA:
                snd_es1869_mixer_write(codec, 0x1c, 0x05);
                snd_es1869_mixer_write(codec, 0x6c, left << 4 | right);
                break;
        case SND_MIXER_PRI_LINE:
                snd_es1869_mixer_write(codec, 0x1c, 0x05);
                snd_es1869_mixer_write(codec, 0x6e, left << 4 | right);
                break;
        default:
                snd_spin_unlock(codec, mixer, &flags);
                snd_printd("Invalid channel priority: %x\n", channel->hw.priority);
                return;
        }
        snd_spin_unlock(codec, mixer, &flags);
}


static void snd_es1869_volume_level(snd_kmixer_t * mixer,
                                    snd_kmixer_channel_t * channel,
                                    int left, int right)
{
        unsigned long flags;
        es1869_t *codec;
	/* Hoping that right volume has the same flag set */
	if (left & SND_MIX_RECORD_VOLUME) {
	  snd_es1869_record_volume_level(mixer, channel,
					 SND_MIX_VOLUME(left),
					 SND_MIX_VOLUME(right));
	  return;
	}

        codec = (es1869_t *) mixer->private_data;
        snd_spin_lock(codec, mixer, &flags);
        switch (channel->hw.priority) {
        case SND_MIXER_PRI_PCM:
                snd_es1869_mixer_write(codec, 0x7c, (left << 4) | right);
                break;
        case SND_MIXER_PRI_PCM1:
                snd_es1869_mixer_write(codec, 0x14, (left << 4) | right);
                codec->audio2_vol = (left << 4) | right;
                break;
        case SND_MIXER_PRI_MIC:
                snd_es1869_mixer_write(codec, 0x1a, (left << 4) | right);
                break;
        case SND_MIXER_PRI_MASTER:
                snd_es1869_mixer_bits(codec, 0x60, left, 0x3f);
                snd_es1869_mixer_bits(codec, 0x62, right, 0x3f);
                break;
        case SND_MIXER_PRI_FM:
                snd_es1869_mixer_write(codec, 0x36, (left << 4) | right);
                break;
        case SND_MIXER_PRI_CD:
                snd_es1869_mixer_write(codec, 0x38, (left << 4) | right);
                break;
        case SND_MIXER_PRI_AUXA:
                snd_es1869_mixer_write(codec, 0x3a, (left << 4) | right);
                break;
        case SND_MIXER_PRI_SPEAKER:
                snd_es1869_mixer_write(codec, 0x3c, left);
                break;
        case SND_MIXER_PRI_LINE:
                snd_es1869_mixer_write(codec, 0x3e, (left << 4) | right);
                break;
        case SND_MIXER_PRI_GAIN:
                snd_es1869_write(codec, 0xb4, (left << 4) | right);
                break;
        case SND_MIXER_PRI_3D_SPACE:
                snd_es1869_mixer_write(codec, 0x52, left);
                break;
        default:
                snd_spin_unlock(codec, mixer, &flags);
                snd_printd("Invalid channel priority: %x\n", channel->hw.priority);
                return;
        }
        snd_spin_unlock(codec, mixer, &flags);
}

static void snd_es1869_mute(snd_kmixer_t * mixer, snd_kmixer_channel_t * channel, unsigned int mute)
{
        unsigned long flags;
        es1869_t *codec;
        codec = (es1869_t *) mixer->private_data;
        snd_spin_lock(codec, mixer, &flags);
        switch (channel->hw.priority) {
        case SND_MIXER_PRI_MASTER:
                snd_es1869_mixer_bits(codec, 0x60, 
                                      (mute & SND_MIX_MUTE_LEFT) ? 0x40 : 0x00, 0x40);
                snd_es1869_mixer_bits(codec, 0x62, 
                                      (mute & SND_MIX_MUTE_RIGHT) ? 0x40 : 0x00, 0x40);
                break;
        case SND_MIXER_PRI_3D_SPACE:
                snd_es1869_mixer_bits(codec, 0x50, mute ? 0x00 : 0x08, 0x08);
                break;
        default:
                snd_spin_unlock(codec, mixer, &flags);
                snd_printd("Invalid channel priority: %x\n", channel->hw.priority);
                return;
        }
        snd_spin_unlock(codec, mixer, &flags);
}


static struct snd_stru_mixer_channel_hw snd_es1869_mixs[] =
{
        {
                SND_MIXER_PRI_PCM,	/* priority */
                SND_MIXER_PRI_MASTER,	/* parent priority */
                SND_MIXER_ID_PCM,	/* device name */
                SND_MIXER_OSS_PCM,	/* OSS device # */
                SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_RECORDVOLUME,
                0, 15,			/* min, max value */
                -4600, 0, 288,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                snd_es1869_record_source, /* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_PCM1,	/* priority */
                SND_MIXER_PRI_MASTER,	/* parent priority */
                SND_MIXER_ID_PCM1,	/* device name */
                SND_MIXER_OSS_ALTPCM,	/* OSS device # */
                SND_MIXER_CINFO_CAP_STEREO,
                0, 15,			/* min, max value */
                -4600, 0, 288,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                NULL,			/* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_MIC,	/* priority */
                SND_MIXER_PRI_MASTER,	/* parent priority */
                SND_MIXER_ID_MIC,	/* device name */
                SND_MIXER_OSS_MIC,	/* OSS device # */
                SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_RECORDVOLUME,
                0, 15,			/* min, max value */
                -4600, 0, 288,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                snd_es1869_record_source, /* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_MASTER,	/* priority */
                SND_MIXER_PRI_PARENT,	/* parent priority */
                SND_MIXER_ID_MASTER,	/* device name */
                SND_MIXER_OSS_VOLUME,	/* OSS device # */
#if 0
                SND_MIXER_CINFO_CAP_JOINRECORD |
#endif
                SND_MIXER_CINFO_CAP_STEREO,
                0, 63,			/* min, max value */
                -4600, 0, 288,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
#if 0
                /* Only possible if we add a SND_MIXER_CINFO_CAP_EXCL_RECORD
                   capability (reset all other record sources when we select
                   this source and reset this record source when we select
                   another source) */
                snd_es1869_record_source, /* record source */
#else
                NULL,			/* record source */
#endif
                snd_es1869_mute,	/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_FM,	/* priority */
                SND_MIXER_PRI_MASTER,	/* parent priority */
                SND_MIXER_ID_FM,	/* device name */
                SND_MIXER_OSS_SYNTH,	/* OSS device # */
                SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_RECORDVOLUME,
                0, 15,			/* min, max value */
                -4600, 0, 288,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                snd_es1869_record_source, /* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_CD,	/* priority */
                SND_MIXER_PRI_MASTER,	/* parent priority */
                SND_MIXER_ID_CD,	/* device name */
                SND_MIXER_OSS_CD,	/* OSS device # */
                SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_INPUT | SND_MIXER_CINFO_CAP_RECORDVOLUME,
                0, 15,			/* min, max value */
                -4600, 0, 288,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                snd_es1869_record_source, /* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_AUXA,	/* priority */
                SND_MIXER_PRI_MASTER,	/* parent priority */
                SND_MIXER_ID_AUXA,	/* device name */
                SND_MIXER_OSS_LINE1,	/* OSS device # */
                SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_INPUT | SND_MIXER_CINFO_CAP_RECORDVOLUME,
                0, 15,			/* min, max value */
                -4600, 0, 288,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                snd_es1869_record_source, /* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_SPEAKER,	/* priority */
                SND_MIXER_PRI_MASTER,	/* parent priority */
                SND_MIXER_ID_SPEAKER,	/* device name */
                SND_MIXER_OSS_SPEAKER,	/* OSS device # */
                SND_MIXER_CINFO_CAP_INPUT,
                0, 7,			/* max value */
                -4600, 0, 575,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                NULL,			/* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_LINE,	/* priority */
                SND_MIXER_PRI_MASTER,	/* parent priority */
                SND_MIXER_ID_LINE,	/* device name */
                SND_MIXER_OSS_LINE,	/* OSS device # */
                SND_MIXER_CINFO_CAP_STEREO | SND_MIXER_CINFO_CAP_INPUT | SND_MIXER_CINFO_CAP_RECORDVOLUME,
                0, 15,			/* min, max value */
                -4600, 0, 288,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                snd_es1869_record_source, /* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_GAIN,	/* priority */
                SND_MIXER_PRI_PARENT,	/* parent priority */
                SND_MIXER_ID_GAIN,	/* device name */
                SND_MIXER_OSS_IMIX,	/* OSS device # */
                SND_MIXER_CINFO_CAP_STEREO,
                0, 15,			/* min, max value */
                -600, 1650, 150,	/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                NULL,			/* record source */
                NULL,			/* set mute */
                snd_es1869_volume_level,/* set volume level */
                NULL,			/* set route */
        },
        {
                SND_MIXER_PRI_3D_SPACE,	/* priority */
                SND_MIXER_PRI_PARENT,	/* parent priority */
                SND_MIXER_ID_3D_SPACE,	/* device name */
                SND_MIXER_OSS_LINE3,	/* OSS device # */
                0,
                0, 63,			/* min, max value */
                0, 0, 0,		/* min, max, step - dB */
                0,
                NULL,			/* compute dB -> linear */
                NULL,			/* compute linear -> dB */
                NULL,			/* record source */
                snd_es1869_mute,	/* set mute */
                snd_es1869_volume_level,	/* set volume level */
                NULL,			/* set route */
        },
};

snd_kmixer_t *snd_es1869_mixer(es1869_t *codec)
{
        int idx;
        snd_kmixer_t *mixer;
        snd_kmixer_channel_t *channel;

        if (!codec || !codec->card)
                return NULL;
        mixer = snd_mixer_new(codec->card, "ES1869");
        if (!mixer)
                return NULL;
        strcpy(mixer->name, "ESS AudioDrive ES1869");
        for(idx = 0; idx < sizeof(snd_es1869_mixs)/sizeof(struct snd_stru_mixer_channel_hw); idx++) {
                channel = snd_mixer_new_channel(mixer, &snd_es1869_mixs[idx]);
                if (!channel) {
                        snd_mixer_free(mixer);
                        return NULL;
                }
        }
        mixer->private_data = codec;
        return mixer;
}

/*
 *  INIT part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_es1869_export;
#endif

int init_module(void)
{
#ifndef LINUX_2_1
        if (register_symtab(&snd_symbol_table_es1869_export) < 0)
                return -ENOMEM;
#endif
        return 0;
}

void cleanup_module(void)
{
}

