/*
 *   Generic MIDI synth driver for ALSA sequencer
 *   Copyright (c) 1998 by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *                         Jaroslav Kysela <perex@jcu.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
 
/* 
Possible options for midisynth module:
	- automatic opening of midi ports on first received event or subscription
	  (close will be performed when client leaves)
*/


#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "midi.h"
#include "seq_kernel.h"
#include "seq_midi.h"


/* data for this midi synth driver */
typedef struct {
	snd_card_t *card;
	int device;
	snd_rawmidi_t *input_rmidi;
	snd_rawmidi_t *output_rmidi;
	int seq_client;
	int seq_port;
	unsigned long jiffies;

	unsigned char running_state;	/* for keeping the last status byte */
} seq_midisynth_t;

typedef struct {
	int seq_client;
	seq_midisynth_t *ports[SND_RAWMIDI_DEVICES];
} seq_midisynth_client_t;

static seq_midisynth_client_t *synths[SND_CARDS];
snd_mutex_define_static(register);

/* fill standard header data, source port & channel are filled in */
static void snd_seq_midi_setheader(snd_seq_event_t * ev, int port, int channel)
{
	memset(ev, 0, sizeof(snd_seq_event_t));

	ev->flags = SND_SEQ_EVENT_LENGTH_FIXED;

	ev->source.queue = SND_SEQ_ADDRESS_UNKNOWN;
	ev->source.port = port;
	ev->source.channel = channel;

	ev->dest.queue = SND_SEQ_ADDRESS_SUBSCRIBERS;
	ev->dest.channel = channel;

	ev->flags = SND_SEQ_TIME_STAMP_TICK | SND_SEQ_TIME_MODE_REL;
}


/*
 * the ALSA low-level midi routines appear to return 'whole' midi events and 
 * have already handled midi running state. I don't know if will remain in 
 * the future. If not, some more elaborate MIDI parser is needed.
 */
static void snd_midi_command(snd_rawmidi_t * rmidi, void *cmd_private_data, unsigned char *command, int count)
{
	seq_midisynth_t *msynth = (seq_midisynth_t *) cmd_private_data;
	int channel;
	int port;
	int client;
	snd_seq_event_t ev;

	if (msynth == NULL) {
		snd_printd("msynth == NULL\n");
		return;
	}
	port = msynth->seq_port;
	client = msynth->seq_client;
	channel = command[0] & 0x0f;

	switch (command[0] & 0xf0) {

		case 0x80:	// note off

			if (count == 3) {
				snd_seq_midi_setheader(&ev, port, channel);

				ev.type = SND_SEQ_EVENT_NOTEOFF;
				ev.data.note.note = command[1];
				ev.data.note.velocity = command[2];

				snd_seq_kernel_client_enqueue(client, &ev, 1);
				return;
			}
			break;

		case 0x90:	// note on

			if (count == 3) {
				snd_seq_midi_setheader(&ev, port, channel);

				ev.type = SND_SEQ_EVENT_NOTEON;
				ev.data.note.note = command[1];
				ev.data.note.velocity = command[2];

				snd_seq_kernel_client_enqueue(client, &ev, 1);
				return;
			}
			break;

		case 0xa0:	// poly key pressure

			if (count == 3) {
				snd_seq_midi_setheader(&ev, port, channel);

				ev.type = SND_SEQ_EVENT_KEYPRESS;
				ev.data.control.param = command[1];
				ev.data.control.value = command[2];

				snd_seq_kernel_client_enqueue(client, &ev, 1);
				return;
			}
			break;

		case 0xb0:	// control change

			if (count == 3) {
				snd_seq_midi_setheader(&ev, port, channel);

				ev.type = SND_SEQ_EVENT_CONTROLLER;
				ev.data.control.param = command[1];
				ev.data.control.value = command[2];

				snd_seq_kernel_client_enqueue(client, &ev, 1);
				return;
			}
			break;

		case 0xc0:	// program change

			if (count == 2) {
				snd_seq_midi_setheader(&ev, port, channel);

				ev.type = SND_SEQ_EVENT_PGMCHANGE;
				ev.data.control.value = command[1];

				snd_seq_kernel_client_enqueue(client, &ev, 1);
				return;
			}
			break;

		case 0xd0:	// channel pressure

			if (count == 2) {
				snd_seq_midi_setheader(&ev, port, channel);

				ev.type = SND_SEQ_EVENT_CHANPRESS;
				ev.data.control.value = command[1];

				snd_seq_kernel_client_enqueue(client, &ev, 1);
				return;
			}
			break;

		case 0xe0:	// pitch bender

			if (count == 3) {
				snd_seq_midi_setheader(&ev, port, channel);

				ev.type = SND_SEQ_EVENT_PITCHBEND;
				ev.data.control.value = (command[1] & 0x7f) + ((command[2] & 0x7f) << 7) - 8192;

				snd_seq_kernel_client_enqueue(client, &ev, 1);
				return;
			}
			break;

		case 0xf0:
			switch (command[0]) {
				case 0xf0:	/* sysex */
					snd_seq_midi_setheader(&ev, port, 0);
					ev.flags = (ev.flags & ~SND_SEQ_EVENT_LENGTH_MASK) | SND_SEQ_EVENT_LENGTH_VARIABLE;
					ev.type = SND_SEQ_EVENT_SYSEX;
					ev.data.ext.ptr = snd_seq_ext_malloc(count, 1);
					ev.data.ext.len = count;
					if (ev.data.ext.ptr) {
						memcpy(ev.data.ext.ptr, command, count);
						snd_seq_kernel_client_enqueue(client, &ev, 1);
					} else {
						snd_printd("failed to get %d bytes for sysex\n", count);
					}
					return;

				case 0xf1:	/* MTC quarter frame */
					if (count == 2) {
						snd_seq_midi_setheader(&ev, port, 0);

						ev.type = SND_SEQ_EVENT_QFRAME;
						ev.data.control.value = command[1];

						snd_seq_kernel_client_enqueue(client, &ev, 1);
						return;
					}
					break;

				case 0xf2:	/* song position */
					if (count == 3) {
						snd_seq_midi_setheader(&ev, port, 0);

						ev.type = SND_SEQ_EVENT_SONGPOS;
						ev.data.control.value = (command[1] & 0x7f) + ((command[2] & 0x7f) << 7);

						snd_seq_kernel_client_enqueue(client, &ev, 1);
						return;
					}
					break;

				case 0xf3:	/* song select */
					if (count == 2) {
						snd_seq_midi_setheader(&ev, port, 0);

						ev.type = SND_SEQ_EVENT_SONGSEL;
						ev.data.control.value = command[1];

						snd_seq_kernel_client_enqueue(client, &ev, 1);
						return;
					}
					break;

				case 0xf4:	/* undefined */
					return;

				case 0xf5:	/* undefined */
					return;

				case 0xf6:	/* tune request */
					snd_printd("Rx: tune request\n");
					return;

				case 0xf7:	/* end of sysex */
					return;

					// system real-time messages

				case 0xf8:	/* timing clock */
					if (count == 1) {
						snd_seq_midi_setheader(&ev, port, 0);

						ev.type = SND_SEQ_EVENT_CLOCK;
						snd_seq_kernel_client_enqueue(client, &ev, 1);
						return;
					}
				case 0xfa:	/* start */
					if (count == 1) {
						snd_seq_midi_setheader(&ev, port, 0);

						ev.type = SND_SEQ_EVENT_START;
						snd_seq_kernel_client_enqueue(client, &ev, 1);
						return;
					}
				case 0xfb:	// continue

					if (count == 1) {
						snd_seq_midi_setheader(&ev, port, 0);

						ev.type = SND_SEQ_EVENT_CONTINUE;
						snd_seq_kernel_client_enqueue(client, &ev, 1);
						return;
					}
				case 0xfc:	// stop

					if (count == 1) {
						snd_seq_midi_setheader(&ev, port, 0);

						ev.type = SND_SEQ_EVENT_STOP;
						snd_seq_kernel_client_enqueue(client, &ev, 1);
						return;
					}
				case 0xfd:	/* undefined */
					return;

				case 0xfe:	// active sensing

					if (count == 1) {
						snd_seq_midi_setheader(&ev, port, channel);

						ev.type = SND_SEQ_EVENT_HEARTBEAT;
						snd_seq_kernel_client_enqueue(client, &ev, 1);
						return;
					}
				case 0xff:	// system reset

					snd_printd("Rx: system reset\n");
					return;
			}
	}

#ifdef SNDCFG_DEBUG
	/* not a legal MIDI sequence.... */
	snd_printk("rx command '%s': ", rmidi->name);
	while (count-- > 0)
		printk("%02x:", *command++);
	printk("\n");
#endif
}



/* send data to specified midi device */
static void dump_midi(snd_rawmidi_t * rmidi, unsigned char *buf, int count)
{
	int done = snd_midi_transmit(rmidi, buf, count);

#if 0
	{
		int i;
	
		printk("dump: ");
		for (i = 0; i < count; i++)
			printk("%02x:", *buf++);
		printk("\n");
	}
#endif

	if (done != count) {
		snd_printd("only wrote %d instead of %d bytes to midi device\n", done, count);
	}
}


static int event_process_midi(snd_seq_event_t * ev, void *private_data)
{
	seq_midisynth_t *msynth = (seq_midisynth_t *) private_data;
	int channel = ev->dest.channel, client;
	unsigned char msg[10];	/* buffer for constructing midi messages */
	snd_rawmidi_t *rmidi;
	snd_seq_event_t event;

	if (msynth == NULL) {
		snd_printd("msynth == NULL\n");
		return -EINVAL;
	}
	client = msynth->seq_client;
	rmidi = msynth->output_rmidi;
	if (!rmidi)
		return -EINVAL;

#if 0
	if ((signed)(jiffies - msynth->jiffies) > HZ/50)
		msynth->running_state = 0;
	msynth->jiffies = jiffies;
#endif

#if 0
	printk("ev source: %i.%i.%i.%i, dest = %i.%i.%i.%i\n",
			ev->source.queue,
			ev->source.client,
			ev->source.port,
			ev->source.channel,
			ev->dest.queue,
			ev->dest.client,
			ev->dest.port,
			ev->dest.channel);
#endif
			
	/* decode actual event data... */
	switch (ev->type) {
		case SND_SEQ_EVENT_NOTE:
			/* note event with specified length, first trigger note on */
			msg[0] = (channel & 0x0f) | 0x90;	/* note on */
			msg[1] = ev->data.note.note & 0x7f;
			msg[2] = ev->data.note.velocity & 0x7f;
			if (msynth->running_state == msg[0]) {
				dump_midi(rmidi, msg + 1, 2);
			} else {
				msynth->running_state = msg[0];
				dump_midi(rmidi, msg, 3);
			}

			/* enqueue note off event */
			memcpy(&event, ev, sizeof(event));
			switch (event.flags & SND_SEQ_TIME_STAMP_MASK) {
				case SND_SEQ_TIME_STAMP_TICK:
					event.time.tick = event.data.note.duration;
					break;
				case SND_SEQ_TIME_STAMP_REAL:
					event.time.real.tv_sec = event.data.note.duration / 1000;	/* unit for duration is ms */
					event.time.real.tv_nsec = 1E6 * (event.data.note.duration % 1000);
					break;
			}
			event.flags = (event.flags & ~SND_SEQ_TIME_MODE_MASK) | SND_SEQ_TIME_MODE_REL;
			event.type = SND_SEQ_EVENT_NOTEOFF;
			snd_seq_kernel_client_enqueue(client, &event, 1);
			break;


		case SND_SEQ_EVENT_NOTEOFF:
			msg[0] = (channel & 0x0f) | 0x80;	/* note off */
			msg[1] = ev->data.note.note & 0x7f;
			msg[2] = ev->data.note.velocity & 0x7f;
			if (msynth->running_state == msg[0]) {
				dump_midi(rmidi, msg + 1, 2);
			} else {
				msynth->running_state = msg[0];
				dump_midi(rmidi, msg, 3);
			}
			break;

		case SND_SEQ_EVENT_NOTEON:
			msg[0] = (channel & 0x0f) | 0x90;	/* note on */
			msg[1] = ev->data.note.note & 0x7f;
			msg[2] = ev->data.note.velocity & 0x7f;
			if (msynth->running_state == msg[0]) {
				dump_midi(rmidi, msg + 1, 2);
			} else {
				msynth->running_state = msg[0];
				dump_midi(rmidi, msg, 3);
			}
			break;

		case SND_SEQ_EVENT_KEYPRESS:
			msg[0] = (channel & 0x0f) | 0xa0;	/* polyphonic key pressure */
			msg[1] = ev->data.control.param & 0x7f;
			msg[2] = ev->data.control.value & 0x7f;
			if (msynth->running_state == msg[0]) {
				dump_midi(rmidi, msg + 1, 2);
			} else {
				msynth->running_state = msg[0];
				dump_midi(rmidi, msg, 3);
			}
			break;

		case SND_SEQ_EVENT_CONTROLLER:
			msg[0] = (channel & 0x0f) | 0xb0;	/* control change */
			msg[1] = ev->data.control.param & 0x7f;
			msg[2] = ev->data.control.value & 0x7f;
			if (msynth->running_state == msg[0]) {
				dump_midi(rmidi, msg + 1, 2);
			} else {
				msynth->running_state = msg[0];
				dump_midi(rmidi, msg, 3);
			}
			break;

		case SND_SEQ_EVENT_PGMCHANGE:
			msg[0] = (channel & 0x0f) | 0xc0;	/* program change */
			msg[1] = ev->data.control.value & 0x7f;
			if (msynth->running_state == msg[0]) {
				dump_midi(rmidi, msg + 1, 1);
			} else {
				msynth->running_state = msg[0];
				dump_midi(rmidi, msg, 2);
			}
			break;

		case SND_SEQ_EVENT_CHANPRESS:
			msg[0] = (channel & 0x0f) | 0xd0;	/* channel pressure */
			msg[1] = ev->data.control.value & 0x7f;
			if (msynth->running_state == msg[0]) {
				dump_midi(rmidi, msg + 1, 1);
			} else {
				msynth->running_state = msg[0];
				dump_midi(rmidi, msg, 2);
			}
			break;

		case SND_SEQ_EVENT_PITCHBEND:
			msg[0] = (channel & 0x0f) | 0xe0;	/* pitch bender */
			msg[1] = (ev->data.control.value + 8192) & 0x7f;	/* lsb */
			msg[2] = ((ev->data.control.value + 8192) >> 7) & 0x7f;		/* msb */
			if (msynth->running_state == msg[0]) {
				dump_midi(rmidi, msg + 1, 2);
			} else {
				msynth->running_state = msg[0];
				dump_midi(rmidi, msg, 3);
			}
			break;



		case SND_SEQ_EVENT_SYSEX:{
				unsigned char *sysex = (unsigned char *) ev->data.ext.ptr;

				//printf("Event  = System Exclusive len=%d\n", ev->data.ext.len);
				dump_midi(rmidi, sysex, ev->data.ext.len);
				msynth->running_state = 0;
			}
			break;


		case SND_SEQ_EVENT_QFRAME:
			msg[0] = 0xf1;	/* MTC quarter frame */
			msg[1] = ev->data.control.value & 0x7f;
			dump_midi(rmidi, msg, 2);
			msynth->running_state = 0;
			break;

		case SND_SEQ_EVENT_CLOCK:
			msg[0] = 0xf8;
			dump_midi(rmidi, msg, 1);
			msynth->running_state = 0;
			break;

		case SND_SEQ_EVENT_START:
			msg[0] = 0xfa;
			dump_midi(rmidi, msg, 1);
			msynth->running_state = 0;
			break;

		case SND_SEQ_EVENT_CONTINUE:
			msg[0] = 0xfb;
			dump_midi(rmidi, msg, 1);
			msynth->running_state = 0;
			break;

		case SND_SEQ_EVENT_STOP:
			msg[0] = 0xfc;
			dump_midi(rmidi, msg, 1);
			msynth->running_state = 0;
			break;

		case SND_SEQ_EVENT_HEARTBEAT:
			msg[0] = 0xfe;	/* active sensing */
			dump_midi(rmidi, msg, 1);
			msynth->running_state = 0;
			break;


			/* messages which we are sending over the MIDI buss */
		case SND_SEQ_EVENT_TEMPO:
		case SND_SEQ_EVENT_TIMESIGN:
		case SND_SEQ_EVENT_KEYSIGN:
			break;


		default:
			snd_printd("Event  = Decoding for type %d is not implemented\n", ev->type);
	}
	return 0;
}


/* call-back function for event input */
static int event_input(snd_seq_event_t * ev, void *private_data)
{
	seq_midisynth_client_t *mclient = (seq_midisynth_client_t *) private_data;
	seq_midisynth_t *synth;

	if (!mclient || !ev)
		return -EINVAL;
	if (ev->dest.port >= SND_RAWMIDI_DEVICES)
		return -EINVAL;
	synth = mclient->ports[ev->dest.port];
	if (!synth)
		return -ENOENT;
	event_process_midi(ev, synth);
	return 0;
}


static seq_midisynth_t *snd_seq_midisynth_new(snd_card_t *card, int device)
{
	seq_midisynth_t *msynth;
	
	msynth = snd_calloc(sizeof(seq_midisynth_t));
	if (msynth == NULL) {
		snd_printd("malloc failed for snd_seq_midisynth_new()\n");
		return NULL;
	}
	
	msynth->card = card;
	msynth->device = device;
	
	return msynth;
}

/* open associated midi device for input */
static int midisynth_subscribe(void *private_data)
{
	int err;
	seq_midisynth_t *msynth = (seq_midisynth_t *)private_data;

	MOD_INC_USE_COUNT;
	/* open midi port */
	if ((err = snd_midi_open(msynth->card->number, msynth->device, SND_RAWMIDI_LFLG_INPUT, &msynth->input_rmidi)) < 0) {
		snd_printd("midi input open failed!!!\n");
		MOD_DEC_USE_COUNT;
		return err;
	}
	msynth->input_rmidi->input.u.p.command = snd_midi_command;
	msynth->input_rmidi->input.u.p.cmd_private_data = msynth;
	snd_midi_start_input(msynth->input_rmidi);
	return 0;
}

/* close associated midi device for input */
static int midisynth_unsubscribe(void *private_data)
{
	int err;
	seq_midisynth_t *msynth = (seq_midisynth_t *)private_data;

	if (!msynth->input_rmidi)
		return -EINVAL;
	msynth->input_rmidi = NULL;
	err = snd_midi_close(msynth->card->number, msynth->device, SND_RAWMIDI_LFLG_INPUT);
	MOD_DEC_USE_COUNT;
	return err;
}

/* open associated midi device for output */
static int midisynth_use(void *private_data)
{
	int err;
	seq_midisynth_t *msynth = (seq_midisynth_t *)private_data;

	MOD_INC_USE_COUNT;
	/* open midi port */
	if ((err = snd_midi_open(msynth->card->number, msynth->device, SND_RAWMIDI_LFLG_OUTPUT, &msynth->output_rmidi)) < 0) {
		snd_printd("midi output open failed!!!\n");
		MOD_DEC_USE_COUNT;
		return err;
	}
	return 0;
}

/* close associated midi device for output */
static int midisynth_unuse(void *private_data)
{
	int err;
	seq_midisynth_t *msynth = (seq_midisynth_t *)private_data;

	if (!msynth->output_rmidi)
		return -EINVAL;
	snd_midi_flush_output(msynth->output_rmidi);
	msynth->output_rmidi = NULL;
	err = snd_midi_close(msynth->card->number, msynth->device, SND_RAWMIDI_LFLG_OUTPUT);
	MOD_DEC_USE_COUNT;
	return err;
}


static void snd_seq_midisynth_delete(seq_midisynth_t **ms)
{
	seq_midisynth_t *msynth;
	snd_seq_port_info_t port;
	
	if (ms == NULL) {
		snd_printd("snd_seq_midisynth_delete called with NULL\n");
		return;
	}
	msynth = *ms;
	*ms = NULL;

	if (msynth == NULL)
		return;

	/* delete port */
	memset(&port, 0, sizeof(port));
	port.client = msynth->seq_client;
	port.port = msynth->seq_port;
	snd_seq_kernel_client_ctl(port.client, SND_SEQ_IOCTL_DELETE_PORT, &port);

	snd_free(msynth,sizeof(seq_midisynth_t));		
}


/* register new midi synth port */
/* if portname = NULL, the name will be fetched from the MIDI device name */
int snd_seq_midisynth_register_port(snd_card_t *card, int device, char *portname)
{
	seq_midisynth_client_t *client;
	seq_midisynth_t *msynth;
	snd_seq_port_info_t port;
	snd_rawmidi_info_t info;
	int newclient = 0;
	snd_seq_client_callback_t callbacks;
	snd_seq_port_callback_t pcallbacks;
	snd_seq_client_info_t inf;

	if (!card || device < 0 || device >= SND_RAWMIDI_DEVICES)
		return -EINVAL;
	snd_mutex_down_static(register);
	client = synths[card->number];
	if (!client) {
		newclient = 1;
		client = snd_calloc(sizeof(seq_midisynth_client_t));
		if (!client) {
			snd_mutex_up_static(register);
			return -ENOMEM;
		}
		memset(&callbacks, 0, sizeof(callbacks));
		callbacks.private_data = client;
		callbacks.input = event_input;
		callbacks.allow_input = callbacks.allow_output = 1;
		client->seq_client = snd_seq_create_kernel_client(card, 0, &callbacks);
		if (client->seq_client < 0) {
			snd_free(client, sizeof(seq_midisynth_client_t));
			snd_mutex_up_static(register);
			return -ENOMEM;
		}
		/* set our client name */
		memset(&inf,0,sizeof(snd_seq_client_info_t));
		sprintf(inf.name, "%i: MIDI Synth", card->number);
		snd_seq_kernel_client_ctl(client->seq_client, SND_SEQ_IOCTL_SET_CLIENT_INFO, &inf);
	}
	client->ports[device] = msynth = snd_seq_midisynth_new(card, device);
	if (!client->ports[device] || snd_midi_info(card->number, device, &info)<0) {
		if (newclient) {
			snd_seq_delete_kernel_client(client->seq_client);
			snd_free(client, sizeof(seq_midisynth_client_t));
		}
		snd_mutex_up_static(register);
		return -EBUSY;
	}
	msynth->seq_client = client->seq_client;
	
	/* declare port */
	memset(&port, 0, sizeof(port));
	port.client = client->seq_client;
	if (portname) {
		strcpy(port.name, portname);
	} else {
		sprintf(port.name, "%i: %s", card->number, info.name);
	}
	port.capability = SND_SEQ_PORT_CAP_SUBSCRIPTION;
	if (info.flags & SND_RAWMIDI_INFO_OUTPUT)
		port.capability |= SND_SEQ_PORT_CAP_OUT | SND_SEQ_PORT_CAP_SYNC_OUT;
	if (info.flags & SND_RAWMIDI_INFO_INPUT)
		port.capability |= SND_SEQ_PORT_CAP_OUT | SND_SEQ_PORT_CAP_SYNC_OUT;
	if (info.flags & SND_RAWMIDI_INFO_DUPLEX)
		port.capability |= SND_SEQ_PORT_CAP_DUPLEX;
	port.type = SND_SEQ_PORT_TYPE_MIDI_GENERIC;
	port.midi_channels = 16;
	memset(&pcallbacks, 0, sizeof(pcallbacks));
	pcallbacks.private_data = msynth;
	pcallbacks.subscribe = midisynth_subscribe;
	pcallbacks.unsubscribe = midisynth_unsubscribe;
	pcallbacks.use = midisynth_use;
	pcallbacks.unuse = midisynth_unuse;
	port.kernel = &pcallbacks;
	if (snd_seq_kernel_client_ctl(client->seq_client, SND_SEQ_IOCTL_CREATE_PORT, &port)<0) {
		if (newclient) {
			snd_seq_delete_kernel_client(client->seq_client);
			snd_free(client, sizeof(seq_midisynth_client_t));
		}
		snd_mutex_up_static(register);
		return -ENOMEM;
	}

	msynth->seq_port = port.port;
	if (newclient)
		synths[card->number] = client;
	snd_mutex_up_static(register);
	return 0;	/* success */
}

/* release midi synth port */
int snd_seq_midisynth_unregister_port(snd_card_t *card, int device)
{
	seq_midisynth_client_t *client;
	int c;
	
	snd_mutex_down_static(register);
	client = synths[card->number];
	if (!client) {
		snd_mutex_up_static(register);
		return -ENODEV;
	}
	snd_seq_midisynth_delete(&client->ports[device]);
	for (c = 0; c < SND_RAWMIDI_DEVICES; c++)
		if (client->ports[device])
			break;
	if (c >= SND_RAWMIDI_DEVICES) {
		snd_seq_delete_kernel_client(client->seq_client);
		synths[card->number] = NULL;
		snd_free(client, sizeof(seq_midisynth_client_t));
	}
	snd_mutex_up_static(register);
	return 0;
}

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_seq_midi_export;
#endif


int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_seq_midi_export) < 0)
		return -ENOMEM;
#endif
	memset(&synths, 0, sizeof(synths));
	return 0;
}


void cleanup_module(void)
{
}


#ifdef MODULE_PARM		/* hey - we have new 2.1.18+ kernel... */
MODULE_AUTHOR("Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>");
MODULE_DESCRIPTION("Advanced Linux Sound Architecture sequencer MIDI synth.");
MODULE_SUPPORTED_DEVICE("sound");
#endif
