/*
 *  Digital Audio (PCM) abstract layer
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "minors.h"
#include "pcm.h"
#include "control.h"
#include "info.h"
#include "ulaw.h"
#include "sndpersist.h"

static int snd_pcm_switch_read(snd_pcm_t * pcm,
			       snd_pcm_channel_t * pchn,
			       snd_pcm_switch_t * _sw);
static int snd_pcm_switch_write(snd_pcm_t * pcm,
                                snd_pcm_channel_t * pchn,
                                snd_pcm_switch_t * _sw);

snd_pcm_t *snd_pcm_devices[SND_CARDS * SND_PCM_DEVICES];
static struct snd_stru_pcm_notify *snd_pcm_notify_first = NULL;
snd_mutex_define_static(register);

void snd_pcm_lock(void)
{
	snd_mutex_down_static(register);
}

void snd_pcm_unlock(void)
{
	snd_mutex_up_static(register);
}

static int snd_pcm_control_ioctl(snd_card_t * card,
				 snd_control_t * control,
				 unsigned int cmd, unsigned long arg)
{
	unsigned int tmp;
	int idx;
	snd_pcm_t *pcm;

	tmp = card->number << 2;
	switch (cmd) {
	case SND_CTL_IOCTL_HW_INFO:
		{
			struct snd_ctl_hw_info *ptr = (struct snd_ctl_hw_info *) arg;

			ptr->pcmdevs = 0;
			for (idx = SND_PCM_DEVICES - 1; idx >= 0; idx--) {
				if (snd_pcm_devices[tmp + idx]) {
					ptr->pcmdevs = idx + 1;
					break;
				}
			}
			return 0;
		}
	case SND_CTL_IOCTL_PCM_DEVICE:
		{
			int val = snd_ioctl_in((long *) arg);
			if (val < 0 || val >= SND_PCM_DEVICES)
				return -EINVAL;
			if (!snd_pcm_devices[tmp + val])
				return -EINVAL;
			control->pcm_device = val;
			return 0;
		}
	case SND_CTL_IOCTL_PCM_PSWITCHES:
	case SND_CTL_IOCTL_PCM_PSWITCH_READ:
	case SND_CTL_IOCTL_PCM_PSWITCH_WRITE:
	case SND_CTL_IOCTL_PCM_RSWITCHES:
	case SND_CTL_IOCTL_PCM_RSWITCH_READ:
	case SND_CTL_IOCTL_PCM_RSWITCH_WRITE:
		pcm = snd_pcm_devices[tmp + control->pcm_device];
		if (!pcm)
			return -ENOENT;
		switch (cmd) {
		case SND_CTL_IOCTL_PCM_PSWITCHES:
			return snd_ioctl_out((long *) arg, pcm->playback.switches_count);
		case SND_CTL_IOCTL_PCM_PSWITCH_READ:
			return snd_pcm_switch_read(pcm, &pcm->playback, (snd_pcm_switch_t *) arg);
		case SND_CTL_IOCTL_PCM_PSWITCH_WRITE:
			return snd_pcm_switch_write(pcm, &pcm->playback, (snd_pcm_switch_t *) arg);
		case SND_CTL_IOCTL_PCM_RSWITCHES:
			return snd_ioctl_out((long *) arg, pcm->record.switches_count);
		case SND_CTL_IOCTL_PCM_RSWITCH_READ:
			return snd_pcm_switch_read(pcm, &pcm->record, (snd_pcm_switch_t *) arg);
		case SND_CTL_IOCTL_PCM_RSWITCH_WRITE:
			return snd_pcm_switch_write(pcm, &pcm->record, (snd_pcm_switch_t *) arg);
		}
		break;
	}
	return -EAGAIN;
}

snd_pcm_t *snd_pcm_new_device(snd_card_t * card, char *id, snd_minor_t * reg)
{
	snd_pcm_t *pcm;

	if (!card)
		return NULL;
	pcm = (snd_pcm_t *) snd_calloc(sizeof(snd_pcm_t));
	if (!pcm)
		return NULL;
	pcm->card = card;
	if (id) {
		strncpy(pcm->id, id, sizeof(pcm->id) - 1);
	}
	pcm->reg = reg;
	snd_mutex_prepare(&pcm->playback, switches);
	snd_mutex_prepare(&pcm->record, switches);
	return pcm;
}

int snd_pcm_free(snd_pcm_t * pcm)
{
	int idx;

	if (!pcm)
		return -EINVAL;
	if (pcm->playback.switches_count > 0) {
		for (idx = 0; idx < pcm->playback.switches_count; idx++)
			snd_free(pcm->playback.switches[idx],
				 sizeof(snd_pcm_kswitch_t));
		snd_free(pcm->playback.switches,
			 pcm->playback.switches_count * sizeof(void *));
	}
	if (pcm->record.switches_count > 0) {
		for (idx = 0; idx < pcm->record.switches_count; idx++)
			snd_free(pcm->record.switches[idx],
				 sizeof(snd_pcm_kswitch_t));
		snd_free(pcm->record.switches,
			 pcm->record.switches_count * sizeof(void *));
	}
	if (pcm->private_free)
		pcm->private_free(pcm->private_data);
	snd_free(pcm, sizeof(snd_pcm_t));
	return 0;
}

static void snd_pcm_store(snd_pcm_t * pcm)
{
	int idx, size;
	unsigned int pos;
	char *buffer;
	snd_pcm_switch_t uswitch;
	mm_segment_t fs;
	char key[32];
	
	size = sizeof(unsigned)*2 +
	       sizeof(pcm->id) + sizeof(pcm->name) +
	       pcm->playback.switches_count*sizeof(snd_pcm_switch_t) +
	       pcm->record.switches_count*sizeof(snd_pcm_switch_t);
	buffer = (char *) vmalloc(size);
	if (!buffer) {
		snd_printk("snd: cannot store pcm settings - no enough memory\n" );
		return;
	}
	*(((unsigned int *)buffer) + 0) = pcm->playback.switches_count;
	*(((unsigned int *)buffer) + 1) = pcm->record.switches_count;
	pos = sizeof(unsigned)*2;
	memcpy(buffer + pos, pcm->id, sizeof(pcm->id));
	pos += sizeof(pcm->id);
	memcpy(buffer + pos, pcm->name, sizeof(pcm->name));
	pos += sizeof(pcm->name);
	for (idx = 0; idx < pcm->playback.switches_count; idx++) {
		memset(&uswitch, 0, sizeof(uswitch));
		uswitch.switchn = idx;
		fs = snd_enter_user();
		snd_pcm_switch_read(pcm, &pcm->playback, &uswitch);
		snd_leave_user(fs);
		memcpy(buffer + pos, &uswitch, sizeof(uswitch));
		pos += sizeof(snd_pcm_switch_t);
	}
	for (idx = 0; idx < pcm->record.switches_count; idx++) {
		memset(&uswitch, 0, sizeof(uswitch));
		uswitch.switchn = idx;
		fs = snd_enter_user();
		snd_pcm_switch_read(pcm, &pcm->record, &uswitch);
		snd_leave_user(fs);
		memcpy(buffer + pos, &uswitch, sizeof(uswitch));
		pos += sizeof(snd_pcm_switch_t);
	}
	sprintf(key, "snd:%i/pcm", pcm->card->number);
	snd_persist_store(key, buffer, size);
	vfree(buffer);
}

static void snd_pcm_restore(snd_pcm_t * pcm)
{
	int idx, size, swcountp, swcountr;
	unsigned int pos;
	char *buffer;
	snd_pcm_switch_t uswitch;
	mm_segment_t fs;
	char key[32];
	
	size = sizeof(unsigned)*2 +
	       sizeof(pcm->id) + sizeof(pcm->name) +
	       pcm->playback.switches_count*sizeof(snd_pcm_switch_t) +
	       pcm->record.switches_count*sizeof(snd_pcm_switch_t);
	buffer = (char *) vmalloc(size);
	if (!buffer) {
		snd_printk("snd: cannot restore pcm settings - no enough memory\n" );
		return;
	}
	sprintf(key, "snd:%i/pcm", pcm->card->number);
	size = snd_persist_restore(key, buffer, size);
	/* ok. I'm very paranoid in bellow code */
	if (size < sizeof(unsigned)*2 + sizeof(pcm->id) + sizeof(pcm->name)) {
		vfree(buffer);
		return;
	}
	swcountp = *(((unsigned int *)buffer) + 0);
	swcountr = *(((unsigned int *)buffer) + 0);
	if (swcountp != pcm->playback.switches_count ||
	    swcountr != pcm->record.switches_count) {
	    	vfree(buffer);
	    	return;
	}
	if (sizeof(unsigned)*2 +
	    sizeof(pcm->id) + sizeof(pcm->name) +
	    swcountp*sizeof(snd_pcm_switch_t) +
	    swcountr*sizeof(snd_pcm_switch_t) != size) {
	    	vfree(buffer);
	    	return;
	}
	pos = sizeof(unsigned)*2;
	if (strcmp(buffer + pos, pcm->id)) {
		vfree(buffer);
		return;
	}
	pos += sizeof(pcm->id);
	if (strcmp(buffer + pos, pcm->name)) {
		vfree(buffer);
		return;
	}
	pos += sizeof(pcm->name);
	for (idx = 0; idx < swcountp; idx++) {
		memcpy(&uswitch, buffer + pos, sizeof(uswitch));
		fs = snd_enter_user();
		snd_pcm_switch_write(pcm, &pcm->playback, &uswitch);
		snd_leave_user(fs);
		pos += sizeof(snd_pcm_switch_t);
	}
	for (idx = 0; idx < swcountr; idx++) {
		memcpy(&uswitch, buffer + pos, sizeof(uswitch));
		fs = snd_enter_user();
		snd_pcm_switch_write(pcm, &pcm->record, &uswitch);
		snd_leave_user(fs);
		pos += sizeof(snd_pcm_switch_t);
	}
	vfree(buffer);
}

int snd_pcm_register(snd_pcm_t * pcm, int pcm_device)
{
	int idx, err;
	unsigned short minor;
	struct snd_stru_pcm_notify *notify;
	char str[16];

	snd_pcm_lock();
	if (pcm_device < 0 || pcm_device >= SND_PCM_DEVICES) {
		snd_pcm_unlock();
		return -EINVAL;
	}
	idx = (pcm->card->number * SND_PCM_DEVICES) + pcm_device;
	if (snd_pcm_devices[idx]) {
		snd_pcm_unlock();
		return -EBUSY;
	}
	pcm->device = pcm_device;
	snd_pcm_devices[idx] = pcm;
	minor = SND_MINOR_PCM + idx;
	if ((err = snd_register_minor(minor, pcm->reg)) < 0) {
		snd_pcm_devices[idx] = NULL;
		snd_pcm_unlock();
		return err;
	}
	sprintf(str, "pcm%i%i", pcm->card->number, pcm->device);
	if ((pcm->dev = snd_info_create_device(str, minor, 0)) == NULL) {
		snd_unregister_minor(minor);
		snd_pcm_devices[idx] = NULL;
		snd_pcm_unlock();
		return -ENOMEM;
	}
	for (notify = snd_pcm_notify_first; notify; notify = notify->next) {
		if (notify->n_register)
			notify->n_register(minor, pcm);
	}
	snd_pcm_proc_init(pcm);
	snd_pcm_unlock();
	snd_pcm_restore(pcm);
	return 0;
}

int snd_pcm_unregister(snd_pcm_t * pcm)
{
	int idx;
	struct snd_stru_pcm_notify *notify;

	if (!pcm)
		return -EINVAL;
	snd_pcm_lock();
	idx = (pcm->card->number * SND_PCM_DEVICES) + pcm->device;
	if (snd_pcm_devices[idx] != pcm) {
		snd_pcm_unlock();
		return -EINVAL;
	}
	snd_pcm_store(pcm);
	snd_pcm_proc_done(pcm);
	snd_info_free_device(pcm->dev);
	snd_unregister_minor(SND_MINOR_PCM + idx);
	for (notify = snd_pcm_notify_first; notify; notify = notify->next) {
		if (notify->n_unregister)
			notify->n_unregister(SND_MINOR_PCM + idx, pcm);
	}
	snd_pcm_devices[idx] = NULL;
	snd_pcm_unlock();
	return snd_pcm_free(pcm);
}

int snd_pcm_notify(struct snd_stru_pcm_notify *notify, int nfree)
{
	int idx;
	struct snd_stru_pcm_notify *tnotify;

	if (!notify || !notify->n_register || !notify->n_unregister)
		return -EINVAL;
	snd_pcm_lock();
	if (nfree) {
		tnotify = snd_pcm_notify_first;
		if (tnotify == notify) {
			snd_pcm_notify_first = notify->next;
		} else {
			while (tnotify && tnotify->next != notify)
				tnotify = tnotify->next;
			if (!tnotify) {
				snd_pcm_unlock();
				return -ENOENT;
			}
			tnotify->next = tnotify->next->next;
		}
		for (idx = 0; idx < SND_CARDS * SND_PCM_DEVICES; idx++) {
			if (snd_pcm_devices[idx] == NULL)
				continue;
			notify->n_unregister(idx + SND_MINOR_PCM,
				             snd_pcm_devices[idx]);
		}
	} else {
		notify->next = NULL;
		if (!snd_pcm_notify_first) {
			snd_pcm_notify_first = notify;
		} else {
			for (tnotify = snd_pcm_notify_first;
			     tnotify->next;
			     tnotify = tnotify->next);
			tnotify->next = notify;
		}
		for (idx = 0; idx < SND_CARDS * SND_PCM_DEVICES; idx++) {
			if (snd_pcm_devices[idx] == NULL)
				continue;
			notify->n_register(idx + SND_MINOR_PCM,
				           snd_pcm_devices[idx]);
		}
	}
	snd_pcm_unlock();
	return 0;
}

static int snd_pcm_switch_read(snd_pcm_t * pcm,
			       snd_pcm_channel_t * pchn,
			       snd_pcm_switch_t * _sw)
{
	struct snd_pcm_switch sw;
	snd_pcm_kswitch_t *ksw;
	int err = 0;
	unsigned int switchn;

	if (verify_area(VERIFY_WRITE, (void *) _sw, sizeof(snd_pcm_switch_t)))
		return -EFAULT;
	copy_from_user(&sw, _sw, sizeof(sw));
	switchn = sw.switchn;
	snd_mutex_down(pchn, switches);
	if (switchn >= pchn->switches_count) {
		snd_mutex_up(pchn, switches);
		return -EINVAL;
	}
	ksw = pchn->switches[switchn];
	memset(&sw, 0, sizeof(sw));
	strncpy(sw.name, ksw->name, sizeof(sw.name) - 1);
	err = ksw->get_switch(pcm, ksw, &sw);
	snd_mutex_up(pchn, switches);
	if (!err)
		copy_to_user(_sw, &sw, sizeof(sw));
	return err;
}

static int snd_pcm_switch_write(snd_pcm_t * pcm,
			        snd_pcm_channel_t * pchn,
			        snd_pcm_switch_t * _sw)
{
	struct snd_pcm_switch sw;
	snd_pcm_kswitch_t *ksw;
	int err = 0;
	unsigned int switchn;

	memset(&sw, 0, sizeof(sw));
	if (verify_area(VERIFY_READ, (void *) _sw, sizeof(snd_pcm_switch_t)))
		return -EFAULT;
	copy_from_user(&sw, _sw, sizeof(sw));
	switchn = sw.switchn;
	snd_mutex_down(pchn, switches);
	if (switchn >= pchn->switches_count) {
		snd_mutex_up(pchn, switches);
		return -EINVAL;
	}
	ksw = pchn->switches[switchn];
	memset(&sw, 0, sizeof(sw));
	err = ksw->set_switch(pcm, ksw, &sw);
	snd_mutex_up(pchn, switches);
	return err;
}

int snd_pcm_ioctl(snd_pcm_t * pcm, unsigned int cmd, unsigned long arg)
{
	switch (cmd) {
	case SND_PCM_IOCTL_PSWITCHES:
		return snd_ioctl_out((long *) arg, pcm->playback.switches_count);
	case SND_PCM_IOCTL_PSWITCH_READ:
		return snd_pcm_switch_read(pcm, &pcm->playback, (snd_pcm_switch_t *) arg);
	case SND_PCM_IOCTL_PSWITCH_WRITE:
		return snd_pcm_switch_write(pcm, &pcm->playback, (snd_pcm_switch_t *) arg);
	case SND_PCM_IOCTL_RSWITCHES:
		return snd_ioctl_out((long *) arg, pcm->record.switches_count);
	case SND_PCM_IOCTL_RSWITCH_READ:
		return snd_pcm_switch_read(pcm, &pcm->record, (snd_pcm_switch_t *) arg);
	case SND_PCM_IOCTL_RSWITCH_WRITE:
		return snd_pcm_switch_write(pcm, &pcm->record, (snd_pcm_switch_t *) arg);
	}
	return -ENXIO;
}

snd_pcm_kswitch_t *snd_pcm_new_switch(snd_pcm_t * pcm, snd_pcm_channel_t * pchn, snd_pcm_kswitch_t * ksw)
{
	snd_pcm_kswitch_t **x;
	snd_pcm_kswitch_t *nsw;

	snd_mutex_down(pchn, switches);
	x = (snd_pcm_kswitch_t **)
			snd_malloc((pchn->switches_count + 1) * sizeof(void *));
	if (!x) {
		snd_mutex_up(pchn, switches);
		return NULL;
	}
	nsw = (snd_pcm_kswitch_t *) snd_malloc(sizeof(snd_pcm_kswitch_t));
	if (!nsw) {
		snd_free(x, (pchn->switches_count + 1) * sizeof(void *));
		snd_mutex_up(pchn, switches);
		return NULL;
	}
	memcpy(nsw, ksw, sizeof(snd_pcm_kswitch_t));
	if (pchn->switches) {
		memcpy(x, pchn->switches,
				(pchn->switches_count + 1) * sizeof(void *));
		snd_free(pchn->switches,
				pchn->switches_count * sizeof(void *));
	}
	pchn->switches = x;
	pchn->switches[pchn->switches_count++] = nsw;
	snd_mutex_up(pchn, switches);
	return 0;
}

/*
 *  Info interface
 */

static void snd_pcm_proc_read(snd_info_buffer_t * buffer, void *private_data)
{
	int idx;
	snd_pcm_t *pcm;

	snd_mutex_down_static(register);
	for (idx = 0; idx < SND_CARDS * SND_PCM_DEVICES; idx++) {
		pcm = snd_pcm_devices[idx];
		if (pcm == NULL)
			continue;
		snd_iprintf(buffer, "%02i-%02i: %s : %s\n", idx >> 2, idx & 3, pcm->id, pcm->name);
	}
	snd_mutex_up_static(register);
}

/*
 *  ENTRY functions
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_pcm_export;
#endif

static snd_info_entry_t *snd_pcm_proc_entry = NULL;

int init_module(void)
{
	snd_info_entry_t *entry;

#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_pcm_export) < 0)
		return -ENOMEM;
#endif
	snd_control_register_ioctl(snd_pcm_control_ioctl);
	if ((entry = snd_info_create_entry(NULL, "pcm")) != NULL) {
		entry->t.text.read_size = SND_CARDS * SND_PCM_DEVICES * 128;
		entry->t.text.read = snd_pcm_proc_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	snd_pcm_proc_entry = entry;
	return 0;
}

void cleanup_module(void)
{
	snd_control_unregister_ioctl(snd_pcm_control_ioctl);
	if (snd_pcm_proc_entry) {
		snd_info_unregister(snd_pcm_proc_entry);
		snd_pcm_proc_entry = NULL;
	}
}
