/*
 *  Abstract routines for MIXER control
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "minors.h"
#include "mixer.h"
#include "info.h"
#include "control.h"
#include "sndpersist.h"

#define SND_MIXERS		(SND_CARDS * SND_MINOR_MIXERS)

static snd_kmixer_t *snd_mixers[SND_MIXERS] = {[0 ... (SND_MIXERS - 1)] = NULL};

snd_mutex_define_static(register);

snd_kmixer_channel_t *snd_mixer_oss_channel(snd_kmixer_t * mixer, int device)
{
	int idx;
	snd_kmixer_channel_t *channel;

	for (idx = 0; idx < mixer->channels_count; idx++) {
		channel = mixer->channels[idx];
		if (channel->hw.ossdev == device)
			return channel;
	}
	return NULL;
}

static int snd_mixer_open(unsigned short minor, int cardnum, int device,
                          struct file *file)
{
	snd_kmixer_file_t *mfile, *mfile1;
	snd_kmixer_t *mixer;

	if (!(mixer = snd_mixers[(cardnum * SND_MINOR_MIXERS) + device]))
		return -ENODEV;
	mfile = (snd_kmixer_file_t *) snd_calloc(sizeof(snd_kmixer_file_t));
	if (!mfile)
		return -ENOMEM;
	if (minor < SND_MINOR_BEGIN)
		mfile->osscompat = 1;
	snd_sleep_prepare(mfile, change);
	mfile->mixer = mixer;
	if (!mfile->mixer) {
		snd_free(mfile, sizeof(snd_kmixer_file_t));
		return -ENOMEM;
	}
	file->private_data = mfile;
	MOD_INC_USE_COUNT;
	mixer->card->use_inc(mixer->card);
	snd_mutex_down(mixer, ffile);
	if (mixer->ffile) {
		for (mfile1 = mixer->ffile;
		     mfile1->next;
		     mfile1 = mfile1->next);
		mfile1->next = mfile;
	} else {
		mixer->ffile = mfile;
	}
	snd_mutex_up(mixer, ffile);
	return 0;
}

static int snd_mixer_release(unsigned short minor, int cardnum, int device,
                             struct file *file)
{
	snd_kmixer_file_t *mfile, *mfile1;
	snd_kmixer_t *mixer;

	if (file->private_data) {
		mfile = (snd_kmixer_file_t *) file->private_data;
		mixer = mfile->mixer;
		snd_mutex_down(mixer, ffile);
		if (mixer->ffile == mfile) {
			mixer->ffile = mfile->next;
		} else {
			for (mfile1 = mixer->ffile;
			     mfile1->next != mfile;
			     mfile1 = mfile1->next);
			mfile1->next = mfile->next;
		}
		snd_mutex_up(mixer, ffile);
		snd_free(file->private_data, sizeof(snd_kmixer_file_t));
		file->private_data = NULL;
		mixer->card->use_dec(mixer->card);
	}
	MOD_DEC_USE_COUNT;
	return 0;
}

void snd_mixer_set_kernel_mute(snd_kmixer_t * mixer,
			       unsigned int priority,
                               unsigned short mute,
                               int recordvolume)
{
	int idx, mask, left, right, volumemask;
	snd_kmixer_channel_t *channel;
	struct snd_stru_mixer_volume *volume;

	for (idx = 0; idx < mixer->channels_count; idx++) {
		channel = mixer->channels[idx];
		if (channel->hw.priority != priority) continue;
		if (recordvolume && !(channel->hw.caps & SND_MIXER_CINFO_CAP_RECORDVOLUME))
			return;	/* wrong input */
		volume = !recordvolume ? &channel->output : &channel->input;
		if (!(channel->hw.caps & SND_MIXER_CINFO_CAP_STEREO) ||
		    (channel->hw.caps & SND_MIXER_CINFO_CAP_JOINMUTE))
			mute = mute & SND_MIX_MUTE ? SND_MIX_MUTE : 0;
		if (channel->kmute == mute) return;	/* no change */
		mask = channel->kmute = mute;
		mask |= channel->umute;
		if (mask == channel->mute) return;	/* no change */
		channel->mute = mask;
		if (channel->hw.set_mute) {
			channel->hw.set_mute(mixer, channel, mask);
		} else {
			left = mask & SND_MIX_MUTE_LEFT ?
					channel->hw.min : volume->uleft;
			right = mask & SND_MIX_MUTE_RIGHT ?
					channel->hw.min : volume->uright;
			volumemask = recordvolume ? SND_MIX_RECORD_VOLUME : 0;
			if (volume->left != left || volume->right != right) {
				if (channel->hw.set_volume_level)
					channel->hw.set_volume_level(mixer,
								     channel,
								     left | volumemask,
								     right | volumemask);
				volume->left = left;
				volume->right = right;
			}
		}
		return;
	}
}

static void snd_mixer_notify_change(snd_kmixer_file_t * mfile,
				    unsigned int channel, int force)
{
	unsigned long flags;
	snd_kmixer_file_t *mfile1;
	snd_kmixer_t *mixer;

	mixer = mfile->mixer;
#if 0
	printk("notify change: 0x%x (%i)\n", (int) (1 << channel), (int) channel);
#endif
	channel = 1 << channel;
	for (mfile1 = mixer->ffile; mfile1; mfile1 = mfile1->next) {
		if (force || mfile1 != mfile) {
			snd_spin_lock(mixer, lock, &flags);
			if (mfile1->changes & channel) {
				snd_spin_unlock(mixer, lock, &flags);
				continue;
			}
			mfile1->changes |= channel;
			if (snd_getlock(mfile1, change) & SND_WK_SLEEP) {
				snd_getlock(mfile1, change) &= ~SND_WK_SLEEP;
				snd_wakeup(mfile1, change);
			}
			snd_spin_unlock(mixer, lock, &flags);
		}
	}
}

static void snd_mixer_notify_switch_change(snd_kmixer_file_t * mfile,
				           unsigned int switchidx, int force)
{
	unsigned long flags;
	snd_kmixer_file_t *mfile1;
	snd_kmixer_t *mixer;

	mixer = mfile->mixer;
#if 0
	printk("switch notify change: 0x%x (%i)\n", (int) (1 << switchidx), (int) switchidx);
#endif
	switchidx = 1 << switchidx;
	for (mfile1 = mixer->ffile; mfile1; mfile1 = mfile1->next) {
		if (force || mfile1 != mfile) {
			snd_spin_lock(mixer, lock, &flags);
			if (mfile1->schanges & switchidx) {
				snd_spin_unlock(mixer, lock, &flags);
				continue;
			}
			mfile1->schanges |= switchidx;
			if (snd_getlock(mfile1, change) & SND_WK_SLEEP) {
				snd_getlock(mfile1, change) &= ~SND_WK_SLEEP;
				snd_wakeup(mfile1, change);
			}
			snd_spin_unlock(mixer, lock, &flags);
		}
	}
}

static int snd_mixer_set_record_source(snd_kmixer_file_t * mfile,
				       snd_kmixer_channel_t * channel,
				       unsigned int rec, int force, int notify)
{
	snd_kmixer_t *mixer;
	snd_kmixer_channel_t *channel1;
	int idx;

	if (!channel->hw.set_record_source)
		return 0;
	if (!(channel->hw.caps & SND_MIXER_CINFO_CAP_STEREO) ||
	    (channel->hw.caps & SND_MIXER_CINFO_CAP_JOINMUTE))
		rec = rec & SND_MIX_REC ? SND_MIX_REC : 0;
	if (!force && channel->record == rec)
		return 0;
	mixer = mfile->mixer;
	if (mixer->hw.caps & SND_MIXER_INFO_CAP_EXCL_RECORD) {
		for (idx = mixer->channels_count - 1; idx >= 0; idx--) {
			channel1 = mixer->channels[idx];
			if (channel1 == channel)
				continue;
			if ((force || channel1->record) &&
			    channel1->hw.set_record_source) {
				channel1->record = 0;
				channel1->hw.set_record_source(mixer,
							       channel1,
							       0);
				snd_mixer_notify_change(mfile,
							channel1->channel,
							1);
			}
		}
	}
	channel->record = rec;
	channel->hw.set_record_source(mixer, channel, rec);
	if (notify)
		snd_mixer_notify_change(mfile, channel->channel, 0);
	return 1;
}

static int snd_mixer_compute_dB(snd_kmixer_channel_t *channel, int vol)
{
	int result, volrange, dbrange;

	volrange = channel->hw.max - channel->hw.min;
	dbrange = channel->hw.max_dB - channel->hw.min_dB;
	result = (dbrange * (vol - channel->hw.min)) + (volrange / 2);
	result /= volrange;
        result += channel->hw.min_dB;
	return result - (result % channel->hw.step_dB);
}

static int snd_mixer_compute_vol(snd_kmixer_channel_t *channel, int db)
{
	int result, volrange, dbrange;

	volrange = channel->hw.max - channel->hw.min;
	dbrange = channel->hw.max_dB - channel->hw.min_dB;
	db %= channel->hw.step_dB;
	result = (volrange * (db - channel->hw.min_dB)) + (dbrange / 2);
	return (result / dbrange) + channel->hw.min;
}

static int snd_mixer_device_read(snd_kmixer_file_t * mfile,
				 snd_mixer_channel_t * _device,
				 int recordvolume)
{
	unsigned long flags;
	snd_kmixer_t *mixer;
	snd_mixer_channel_t device;
	snd_kmixer_channel_t *channel;
	struct snd_stru_mixer_volume *volume;
	int tmp;

	mixer = mfile->mixer;
	if (verify_area(VERIFY_READ, _device, sizeof(device)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _device, sizeof(device)))
		return -EFAULT;
	copy_from_user(&device, _device, sizeof(device));
	tmp = device.channel;
	memset(&device, 0, sizeof(snd_mixer_channel_t));
	device.channel = tmp;
	if (!mfile->osscompat) {
		if (tmp >= mixer->channels_visible)
			return -ENODEV;
	} else {
		if (tmp >= mixer->channels_count)
			return -ENODEV;
	}
	channel = mixer->channels[tmp];
	if (recordvolume && !(channel->hw.caps & SND_MIXER_CINFO_CAP_RECORDVOLUME))
		return -ENOENT;
	volume = !recordvolume ? &channel->input : &channel->output;

	snd_spin_lock(mixer, lock, &flags);
	device.flags = 0;
	if (channel->record & SND_MIX_REC_LEFT)
		device.flags |= SND_MIXER_FLG_RECORD_LEFT;
	if (channel->record & SND_MIX_REC_RIGHT)
		device.flags |= SND_MIXER_FLG_RECORD_RIGHT;
	if (channel->umute & SND_MIX_MUTE_LEFT)
		device.flags |= SND_MIXER_FLG_MUTE_LEFT;
	if (channel->umute & SND_MIX_MUTE_RIGHT)
		device.flags |= SND_MIXER_FLG_MUTE_RIGHT;
	if (channel->route & SND_MIX_ROUTE_LTOR_OUT)
		device.flags |= SND_MIXER_FLG_LTOR_OUT;
	if (channel->route & SND_MIX_ROUTE_RTOL_OUT)
		device.flags |= SND_MIXER_FLG_RTOL_OUT;
	if (channel->route & SND_MIX_ROUTE_LTOR_IN)
		device.flags |= SND_MIXER_FLG_LTOR_IN;
	if (channel->route & SND_MIX_ROUTE_RTOL_IN)
		device.flags |= SND_MIXER_FLG_RTOL_IN;
	if (mfile->exact) {
		device.left = volume->uleft;
		device.right = volume->uright;
	} else {
		device.left = volume->aleft;
		device.right = volume->aright;
	}
	if (channel->hw.compute_dB) {
		device.left_dB = channel->hw.compute_dB(mixer,
							channel,
							volume->uleft);
		device.right_dB = channel->hw.compute_dB(mixer,
							 channel,
							 volume->uright);
	} else {
		if (channel->hw.step_dB > 0) {
			device.left_dB = snd_mixer_compute_dB(channel,
							      volume->uleft);
			device.right_dB = snd_mixer_compute_dB(channel,
							       volume->uright);
		} else {
			device.left_dB = device.right_dB = 0;
		}
	}
	snd_spin_unlock(mixer, lock, &flags);
	copy_to_user(_device, &device, sizeof(device));
	return 0;
}

static int snd_mixer_device_write(snd_kmixer_file_t * mfile,
				  snd_mixer_channel_t * _device,
				  int recordvolume)
{
	unsigned int mask, left, right;
	snd_mixer_channel_t device;
	snd_kmixer_t *mixer;
	snd_kmixer_channel_t *channel;
	struct snd_stru_mixer_volume *volume;
	int force = 0, change = 0, tmp, aleft, aright;

	mixer = mfile->mixer;
	if (verify_area(VERIFY_READ, _device, sizeof(device)))
		return -EFAULT;
	copy_from_user(&device, _device, sizeof(device));
	if (!mfile->osscompat) {
		if (device.channel >= mixer->channels_visible)
			return -ENODEV;
	} else {
		if (device.channel >= mixer->channels_count)
			return -ENODEV;
	}
	channel = mixer->channels[device.channel];
	if (recordvolume && !(channel->hw.caps & SND_MIXER_CINFO_CAP_RECORDVOLUME))
		return -ENOENT;
	volume = !recordvolume ? &channel->input : &channel->output;

	force = (device.flags & SND_MIXER_FLG_FORCE) != 0;

	/* record source */

	mask = 0;
	if (channel->hw.set_record_source) {
		if (device.flags & SND_MIXER_FLG_RECORD_LEFT)
			mask |= SND_MIX_REC_LEFT;
		if (device.flags & SND_MIXER_FLG_RECORD_RIGHT)
			mask |= SND_MIX_REC_RIGHT;
	}
	if (snd_mixer_set_record_source(mfile, channel, mask, force, 0))
		change = 1;

	/* mute */

	mask = 0;
	if (!force || channel->hw.set_mute) {
		if ((channel->hw.caps & SND_MIXER_CINFO_CAP_STEREO) &&
		    !(channel->hw.caps & SND_MIXER_CINFO_CAP_JOINMUTE)) {
			if (device.flags & SND_MIXER_FLG_MUTE_LEFT)
				mask |= SND_MIX_MUTE_LEFT;
			if (device.flags & SND_MIXER_FLG_MUTE_RIGHT)
				mask |= SND_MIX_MUTE_RIGHT;
		} else {
			if (device.flags & SND_MIXER_FLG_MUTE)
				mask = SND_MIX_MUTE;
		}
	}
	if (channel->umute != mask)
		change = 1;
	channel->umute = mask;

	/* route */

	mask = 0;
	if (channel->hw.set_route) {
		if (device.flags & SND_MIXER_FLG_LTOR_OUT)
			mask |= SND_MIX_ROUTE_LTOR_OUT;
		if (device.flags & SND_MIXER_FLG_RTOL_OUT)
			mask |= SND_MIX_ROUTE_RTOL_OUT;
		if (channel->hw.caps & SND_MIXER_CINFO_CAP_SWITCH_OUT)
			mask = (mask & SND_MIX_ROUTE_OUT) ?
					SND_MIX_ROUTE_OUT : 0;
		if (device.flags & SND_MIXER_FLG_LTOR_IN)
			mask |= SND_MIX_ROUTE_LTOR_IN;
		if (device.flags & SND_MIXER_FLG_RTOL_IN)
			mask |= SND_MIX_ROUTE_RTOL_IN;
		if (channel->hw.caps & SND_MIXER_CINFO_CAP_SWITCH_IN)
			mask = (mask & SND_MIX_ROUTE_IN) ?
					(mask | SND_MIX_ROUTE_IN) : mask;
	}
	if (force || channel->route != mask) {
		if (channel->route != mask)
			change = 1;
		channel->route = mask;
		if (channel->hw.set_route)
			channel->hw.set_route(mixer, channel, mask);
	}

	/* volume */

	if (!(device.flags & SND_MIXER_FLG_DECIBEL)) {
		left = device.left;
		right = device.right;
	} else {
		if (device.left_dB < channel->hw.min_dB)
			device.left_dB = channel->hw.min_dB;
		if (device.right_dB < channel->hw.min_dB)
			device.right_dB = channel->hw.min_dB;
		if (device.left_dB > channel->hw.max_dB)
			device.left_dB = channel->hw.max_dB;
		if (device.right_dB > channel->hw.max_dB)
			device.right_dB = channel->hw.max_dB;
		if (channel->hw.compute_linear) {
			left = channel->hw.compute_linear(mixer, channel,
							  device.left_dB);
			right = channel->hw.compute_linear(mixer, channel,
							   device.right_dB);
		} else {
			if (channel->hw.step_dB > 0) {
				left = snd_mixer_compute_vol(channel,
				                             device.left_dB);
				right = snd_mixer_compute_vol(channel,
							      device.right_dB);
			} else {
				left = right = 0;
			}
		}
	}
	if (!(channel->hw.caps & SND_MIXER_CINFO_CAP_STEREO)) {
		right = left = (left + right) / 2;
	}
	if (mfile->exact || (device.flags & SND_MIXER_FLG_DECIBEL)) {
		if (left < channel->hw.min)
			left = channel->hw.min;
		if (right < channel->hw.min)
			right = channel->hw.min;
		if (left > channel->hw.max)
			left = channel->hw.max;
		if (right > channel->hw.max)
			right = channel->hw.max;
		tmp = channel->hw.max - channel->hw.min;
		aleft = ((100 * (left - channel->hw.min)) + (tmp >> 1)) / tmp;
		aright = ((100 * (right - channel->hw.min)) + (tmp >> 1)) / tmp;
	} else {
		if (left < 0)
			left = 0;
		if (right < 0)
			right = 0;
		if (left > 100)
			left = 100;
		if (right > 100)
			right = 100;
		left = ((channel->hw.max * (aleft = left)) + 50) / 100;
		right = ((channel->hw.max * (aright = right)) + 50) / 100;
	}
	if (volume->aleft != aleft || volume->aright != aright)
		change = 1;
	volume->aleft = aleft;
	volume->aright = aright;
	if (volume->uleft != left || volume->uright != right)
		change = 1;
	volume->uleft = left;
	volume->uright = right;

	/* make changes */

	left = volume->uleft;
	right = volume->uright;
	mask = channel->umute | channel->kmute;
	if (force || channel->mute != mask) {
		channel->mute = mask;
		if (channel->hw.set_mute)
			channel->hw.set_mute(mixer, channel, mask);
		change = 1;
	}
	if (!channel->hw.set_mute) {
		if (channel->mute & SND_MIX_MUTE_LEFT)
			left = channel->hw.min;
		if (channel->mute & SND_MIX_MUTE_RIGHT)
			right = channel->hw.min;
	}
#if 0
	snd_printk("hw.mute = 0x%x, mute = 0x%x, left = %i, right = %i\n", channel->hw.mute, channel->mute, left, right);
#endif
	if (force || volume->left != left || volume->right != right) {
		volume->left = left;
		volume->right = right;
		if (recordvolume) {
			left |= SND_MIX_RECORD_VOLUME;
			right |= SND_MIX_RECORD_VOLUME;
		}
		if (channel->hw.set_volume_level) 
			channel->hw.set_volume_level(mixer, channel,
						     left, right);
		change = 1;
	}
	/* let other mixers know that something were changed */

	if (change)
		snd_mixer_notify_change(mfile, channel->channel, 0);

	return 0;
}

static int snd_mixer_switch_read(snd_kmixer_file_t * mfile,
				 snd_mixer_switch_t * _uswitch)
{
	snd_kmixer_t *mixer;
	snd_kmixer_switch_t *kswitch;
	snd_mixer_switch_t uswitch;
	int err;
	unsigned int switchn;

	if (verify_area(VERIFY_READ, _uswitch, sizeof(uswitch)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _uswitch, sizeof(uswitch)))
		return -EFAULT;
	copy_from_user(&uswitch, _uswitch, sizeof(uswitch));
	mixer = mfile->mixer;
	if (uswitch.switchn >= mixer->switches_count)
		return -EINVAL;
	kswitch = mixer->switches[switchn = uswitch.switchn];
	if (!kswitch || !kswitch->get_switch)
		return -ENXIO;
	memset(&uswitch, 0, sizeof(uswitch));
	uswitch.switchn = switchn;
	strncpy(uswitch.name, kswitch->name, sizeof(uswitch.name) - 1);
	if ((err = kswitch->get_switch(mixer, kswitch, &uswitch)) < 0)
		return err;
	copy_to_user(_uswitch, &uswitch, sizeof(uswitch));
	return 0;
}

static int snd_mixer_switch_write(snd_kmixer_file_t * mfile,
				  snd_mixer_switch_t * _uswitch)
{
	snd_kmixer_t *mixer;
	snd_kmixer_switch_t *kswitch;
	snd_mixer_switch_t uswitch;
	int err;

	if (verify_area(VERIFY_READ, _uswitch, sizeof(uswitch)))
		return -EFAULT;
	copy_from_user(&uswitch, _uswitch, sizeof(uswitch));
	mixer = mfile->mixer;
	if (uswitch.switchn >= mixer->switches_count)
		return -EINVAL;
	kswitch = mixer->switches[uswitch.switchn];
	if (!kswitch || !kswitch->set_switch)
		return -ENXIO;
	if ((err = kswitch->set_switch(mixer, kswitch, &uswitch)) < 0)
		return err;
	snd_mixer_notify_switch_change(mfile, uswitch.switchn, 0);
	return 0;
}

static int snd_mixer_info(snd_kmixer_t * mixer, snd_mixer_info_t * _info)
{
	snd_mixer_info_t info;

	if (verify_area(VERIFY_WRITE, _info, sizeof(info)))
		return -EFAULT;
	memset(&info, 0, sizeof(snd_mixer_info_t));
	info.type = mixer->card->type;
	info.channels = mixer->channels_visible;
	info.caps = mixer->hw.caps;
	strncpy(info.id, mixer->id, sizeof(info.id));
	strncpy(info.name, mixer->name, sizeof(info.name));
	info.switches = mixer->switches_count;
	copy_to_user(_info, &info, sizeof(info));
	return 0;
}

static int snd_mixer_device_info(snd_kmixer_file_t * mfile,
				 snd_mixer_channel_info_t * _info)
{
	snd_kmixer_t *mixer;
	snd_mixer_channel_info_t info;
	snd_kmixer_channel_t *channel;
	int channel_idx, i;

	mixer = mfile->mixer;
	if (verify_area(VERIFY_READ, _info, sizeof(info)))
		return -EFAULT;
	if (verify_area(VERIFY_WRITE, _info, sizeof(info)))
		return -EFAULT;
	copy_from_user(&info, _info, sizeof(info));
	channel_idx = info.channel;
	memset(&info, 0, sizeof(snd_mixer_channel_info_t));
	info.channel = channel_idx;
	if (channel_idx >= mixer->channels_visible)
		return -ENODEV;
	channel = mixer->channels[channel_idx];
	strncpy(info.name, channel->hw.name, sizeof(info.name) - 1);
	if (channel->hw.parent_priority != SND_MIXER_PRI_PARENT) {
		for (i = 0; i < mixer->channels_visible; i++) {
			if (mixer->channels[i]->hw.priority == channel->hw.parent_priority) {
				info.parent = i;
				break;
			}
		}
		if (i >= mixer->channels_visible) {
			snd_printd("Oops... Parent 0x%x not found!!!\n", channel->hw.parent_priority);
			info.parent = SND_MIXER_PRI_PARENT;
		}
	} else {
		info.parent = SND_MIXER_PRI_PARENT;
	}
	info.caps = channel->hw.caps;
	if (channel->hw.set_record_source)
		info.caps |= SND_MIXER_CINFO_CAP_RECORD;
	info.caps |= SND_MIXER_CINFO_CAP_MUTE;	/* mute is always emulated */
	if (channel->hw.set_mute)
		info.caps |= SND_MIXER_CINFO_CAP_HWMUTE;
	if (!(channel->hw.caps & SND_MIXER_CINFO_CAP_STEREO)) {
		info.caps |= SND_MIXER_CINFO_CAP_JOINMUTE;
		if (channel->hw.caps & SND_MIXER_CINFO_CAP_RECORD) {
			info.caps |= SND_MIXER_CINFO_CAP_JOINRECORD;
		}
	}
	if (mfile->exact) {
		info.min = channel->hw.min;
		info.max = channel->hw.max;
	} else {
		info.min = 0;
		info.max = 100;
	}
	info.min_dB = channel->hw.min_dB;
	info.max_dB = channel->hw.max_dB;
	info.step_dB = channel->hw.step_dB;
	copy_to_user(_info, &info, sizeof(info));
	return 0;
}

static int snd_mixer_set_recsrc(snd_kmixer_file_t * mfile, int recsrc)
{
	int idx, value;
	snd_kmixer_t *mixer;
	snd_kmixer_channel_t *channel;

	mixer = mfile->mixer;
	for (idx = SND_MIXER_OSS_DEVS - 1; idx >= 0; idx--) {
		channel = snd_mixer_oss_channel(mixer, idx);
		if (channel == NULL)
			continue;
		value = (recsrc & (1 << idx)) ? 1 : 0;
		if (channel->record != value) {
			mixer->modify_counter++;
			snd_mixer_set_record_source(mfile, channel,
						    value, 0, 1);
		}
	}
	return 0;
}

static int snd_mixer_devmask(snd_kmixer_t * mixer)
{
	int result, idx, ossdev;

	result = 0;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		ossdev = mixer->channels[idx]->hw.ossdev;
		if (ossdev == SND_MIXER_OSS_UNKNOWN)
			continue;
		result |= (1 << ossdev);
	}
	return result;
}

static int snd_mixer_stereodevs(snd_kmixer_t * mixer)
{
	int result, idx;
	snd_kmixer_channel_t *channel;

	result = 0;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		channel = mixer->channels[idx];
		if (channel->hw.ossdev == SND_MIXER_OSS_UNKNOWN)
			continue;
		if (channel->hw.caps & SND_MIXER_CINFO_CAP_STEREO)
			result |= (1 << channel->hw.ossdev);
	}
	return result;
}

static int snd_mixer_recmask(snd_kmixer_t * mixer)
{
	int result, idx;
	snd_kmixer_channel_t *channel;

	result = 0;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		channel = mixer->channels[idx];
		if (channel->hw.ossdev == SND_MIXER_OSS_UNKNOWN)
			continue;
		if (channel->hw.set_record_source)
			result |= (1 << channel->hw.ossdev);
	}
	return result;
}

static int snd_mixer_get_recsrc(snd_kmixer_t * mixer)
{
	int result, idx;
	snd_kmixer_channel_t *channel;

	result = 0;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		channel = mixer->channels[idx];
		if (channel->hw.ossdev == SND_MIXER_OSS_UNKNOWN)
			continue;
		if (channel->record)
			result |= (1 << channel->hw.ossdev);
	}
	return result;
}

static int snd_mixer_osscaps(snd_kmixer_t * mixer)
{
	int result;

	result = 0;
	if (mixer->hw.caps & SND_MIXER_INFO_CAP_EXCL_RECORD)
		result |= SND_MIXER_OSS_CAP_EXCL_INPUT;
	return result;
}

static int snd_mixer_set_volume(snd_kmixer_file_t * mfile,
                                int oss_device, int volume)
{
	mm_segment_t fs;
	int err;
	snd_kmixer_t *mixer;
	snd_kmixer_channel_t *channel;
	snd_mixer_channel_t device;

	mixer = mfile->mixer;
	channel = snd_mixer_oss_channel(mixer, oss_device);
	if (!channel)
		return -ENXIO;
	device.channel = channel->channel;
	fs = snd_enter_user();
	if ((err = snd_mixer_device_read(mfile, &device, 0)) < 0) {
		snd_leave_user(fs);
		return err;
	}
	device.left = volume & 0xff;
	device.right = (volume >> 8) & 0xff;
	if ((err = snd_mixer_device_write(mfile, &device, 0)) < 0) {
		snd_leave_user(fs);
		return err;
	}
	if (channel->hw.caps & SND_MIXER_CINFO_CAP_RECORDVOLUME) {
		if ((err = snd_mixer_device_write(mfile, &device, 1)) < 0) {
			snd_leave_user(fs);
			return err;
		}
	}
	snd_leave_user(fs);
	mixer->modify_counter++;
	return (unsigned short) device.left |
	    ((unsigned short) device.right << 8);
}

static int snd_mixer_get_volume(snd_kmixer_file_t * mfile, int oss_device)
{
	mm_segment_t fs;
	int err;
	snd_kmixer_t *mixer;
	snd_kmixer_channel_t *channel;
	snd_mixer_channel_t device;

	mixer = mfile->mixer;
	channel = snd_mixer_oss_channel(mixer, oss_device);
	if (!channel)
		return -ENXIO;
	device.channel = channel->channel;
	fs = snd_enter_user();
	if ((err = snd_mixer_device_read(mfile, &device, 0)) < 0) {
		snd_leave_user(fs);
		return err;
	}
	snd_leave_user(fs);
	return (unsigned short) device.left |
	    ((unsigned short) device.right << 8);
}

int snd_mixer_ioctl_card(snd_card_t * card, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	snd_kmixer_file_t *mfile, sfile;
	snd_kmixer_t *mixer;
	int tmp;

	if (card) {
		/* note: this finds only first mixer for card */
		for (tmp = 0; tmp < SND_MIXERS; tmp++) {
			mixer = snd_mixers[tmp];
			if (mixer && mixer->card == card)
				break;
		}
		if (!mixer)
			return -ENXIO;
		memset(&sfile, 0, sizeof(sfile));
		sfile.mixer = mixer;
		mfile = &sfile;
	} else {
		mfile = (snd_kmixer_file_t *) file->private_data;
		if (!mfile)
			return -EIO;
		mixer = mfile->mixer;
	}
	if (((cmd >> 8) & 0xff) == 'R') {
		switch (cmd) {
		case SND_MIXER_IOCTL_PVERSION:
			return snd_ioctl_out((long *) arg, SND_MIXER_VERSION);
		case SND_MIXER_IOCTL_CHANNELS:
			return snd_ioctl_out((long *) arg, mixer->channels_visible);
		case SND_MIXER_IOCTL_INFO:
			return snd_mixer_info(mixer, (snd_mixer_info_t *) arg);
		case SND_MIXER_IOCTL_EXACT:
			tmp = snd_ioctl_in((long *) arg);
			mfile->exact = tmp != 0;
			snd_ioctl_out((long *) arg, mfile->exact);
			return 0;
		case SND_MIXER_IOCTL_CHANNEL_INFO:
			return snd_mixer_device_info(mfile, (snd_mixer_channel_info_t *) arg);
		case SND_MIXER_IOCTL_CHANNEL_READ:
			return snd_mixer_device_read(mfile, (snd_mixer_channel_t *) arg, 0);
		case SND_MIXER_IOCTL_CHANNEL_WRITE:
			return snd_mixer_device_write(mfile, (snd_mixer_channel_t *) arg, 0);
		case SND_MIXER_IOCTL_SWITCHES:
			return snd_ioctl_out((long *) arg, mixer->switches_count);
		case SND_MIXER_IOCTL_SWITCH_READ:
			return snd_mixer_switch_read(mfile, (snd_mixer_switch_t *) arg);
		case SND_MIXER_IOCTL_SWITCH_WRITE:
			return snd_mixer_switch_write(mfile, (snd_mixer_switch_t *) arg);
		case SND_MIXER_IOCTL_CHANNEL_RREAD:
			return snd_mixer_device_read(mfile, (snd_mixer_channel_t *) arg, 1);
		case SND_MIXER_IOCTL_CHANNEL_RWRITE:
			return snd_mixer_device_write(mfile, (snd_mixer_channel_t *) arg, 1);
		}
		return -ENXIO;
	}
	if (((cmd >> 8) & 0xff) != 'M')
		return -ENXIO;
#if 0
	snd_printk(">> mixer ioctl - cmd = 0x%x\n", cmd);
#endif
	if (cmd == SND_MIXER_OSS_INFO) {
		struct snd_oss_mixer_info info;

		memset(&info, 0, sizeof(info));
		strncpy(info.id, mixer->id, sizeof(info.id) - 1);
		strncpy(info.name, mixer->name, sizeof(info.name) - 1);
		info.modify_counter = mixer->modify_counter;
		if (verify_area(VERIFY_WRITE, (void *) arg, sizeof(snd_mixer_info)))
			return -EFAULT;
		copy_to_user((void *) arg, &snd_mixer_info, sizeof(snd_mixer_info));
		return 0;
	}
	if (cmd == SND_MIXER_OSS_OLD_INFO) {
		struct snd_oss_mixer_info_obsolete info;

		memset(&info, 0, sizeof(info));
		strncpy(info.id, mixer->id, sizeof(info.id) - 1);
		strncpy(info.name, mixer->name, sizeof(info.name) - 1);
		if (verify_area(VERIFY_WRITE, (void *) arg, sizeof(snd_mixer_info)))
			return -EFAULT;
		copy_to_user((void *) arg, &snd_mixer_info, sizeof(snd_mixer_info));
		return 0;
	}
	if (cmd & IOC_IN)
		switch (cmd) {
		case SND_MIXER_OSS_SET_RECSRC:
			return snd_ioctl_out((long *) arg, snd_mixer_set_recsrc(mfile, snd_ioctl_in((long *) arg)));
		default:
			tmp = snd_mixer_set_volume(mfile, cmd & 0xff, snd_ioctl_in((long *) arg));
			return tmp >= 0 ? snd_ioctl_out((long *) arg, tmp) : tmp;
	} else if (cmd & IOC_OUT)
		switch (cmd) {
		case SND_OSS_GETVERSION:
			return snd_ioctl_out((long *) arg, SND_OSS_VERSION);
		case SND_MIXER_OSS_DEVMASK:
			return snd_ioctl_out((long *) arg, snd_mixer_devmask(mixer));
		case SND_MIXER_OSS_STEREODEVS:
			return snd_ioctl_out((long *) arg, snd_mixer_stereodevs(mixer));
		case SND_MIXER_OSS_RECMASK:
			return snd_ioctl_out((long *) arg, snd_mixer_recmask(mixer));
		case SND_MIXER_OSS_CAPS:
			return snd_ioctl_out((long *) arg, snd_mixer_osscaps(mixer));
		case SND_MIXER_OSS_RECSRC:
			return snd_ioctl_out((long *) arg, snd_mixer_get_recsrc(mixer));
		default:
			tmp = snd_mixer_get_volume(mfile, cmd & 0xff);
			return tmp >= 0 ? snd_ioctl_out((long *) arg, tmp) : tmp;
		}
#ifdef SNDCFG_DEBUG
	snd_printk("MIXER ERR\n");
	snd_printk("  MIXER REQUEST: 0x%x, 0x%x\n", cmd, (int) snd_ioctl_in((long *) arg));
#endif
	return -ENXIO;
}

static int snd_mixer_control_ioctl(snd_card_t * card, snd_control_t * control, unsigned int cmd, unsigned long arg)
{
	snd_kmixer_t *mixer;
	snd_kmixer_file_t sfile;

	switch (cmd) {
	case SND_CTL_IOCTL_HW_INFO:
		{
			struct snd_ctl_hw_info *ptr = (struct snd_ctl_hw_info *) arg;
			if (snd_mixers[(card->number << 1) + 1])
				ptr->mixerdevs = 2;
			else if (snd_mixers[(card->number << 1) + 0])
				ptr->mixerdevs = 1;
			else
				ptr->mixerdevs = 0;
			return 0;
		}
	case SND_CTL_IOCTL_MIXER_DEVICE:
		{
			int val = snd_ioctl_in((long *) arg);
			if (val < 0 || val > 1)
				return -EINVAL;
			if (!snd_mixers[(card->number << 1) + 1] && val > 1)
				return -EINVAL;
			if (!snd_mixers[(card->number << 1) + 0])
				return -EINVAL;
			control->mixer_device = val;
			return 0;
		}
	case SND_CTL_IOCTL_MIXER_INFO:
	case SND_CTL_IOCTL_MIXER_SWITCHES:
	case SND_CTL_IOCTL_MIXER_SWITCH_READ:
	case SND_CTL_IOCTL_MIXER_SWITCH_WRITE:
		mixer = snd_mixers[(card->number << 1) + control->mixer_device];
		if (!mixer)
			return -ENOENT;
		switch (cmd) {
		case SND_CTL_IOCTL_MIXER_INFO:
			return snd_mixer_info(mixer, (snd_mixer_info_t *) arg);
		case SND_CTL_IOCTL_MIXER_SWITCHES:
			return snd_ioctl_out((long *) arg, mixer->switches_count);
		case SND_CTL_IOCTL_MIXER_SWITCH_READ:
			memset(&sfile, 0, sizeof(sfile));
			sfile.mixer = mixer;
			return snd_mixer_switch_read(&sfile, (snd_mixer_switch_t *) arg);
		case SND_CTL_IOCTL_MIXER_SWITCH_WRITE:
			memset(&sfile, 0, sizeof(sfile));
			sfile.mixer = mixer;
			return snd_mixer_switch_write(&sfile, (snd_mixer_switch_t *) arg);
		}
		break;
	}
	return -EAGAIN;
}

static int snd_mixer_ioctl(struct file *file,
			   unsigned int cmd,
			   unsigned long arg)
{
	return snd_mixer_ioctl_card(NULL, file, cmd, arg);
}

static long snd_mixer_read(struct file *file, char *buffer, long count)
{
	unsigned long flags;
	unsigned int device;
	unsigned char buf[8];
	snd_kmixer_file_t *mfile;
	snd_kmixer_t *mixer;
	long size = 0;

	mfile = (snd_kmixer_file_t *) file->private_data;
	if (!mfile)
		return -ENOENT;
	mixer = mfile->mixer;
	if (!mixer)
		return -ENOENT;
	if (count < 8)
		return 0;
	memset(buf, 0, sizeof(buf));
	for (device = 0; device < mfile->mixer->channels_visible; device++) {
		snd_spin_lock(mixer, lock, &flags);
		if (mfile->changes & (1 << device)) {
			mfile->changes &= ~(1 << device);
			snd_spin_unlock(mixer, lock, &flags);
			*(unsigned int *) &buf[4] = device;
			copy_to_user(buffer, buf, 8);
			buffer += 8;
			size += 8;
			count -= 8;
			if (count < 8)
				return size;
		} else {
			snd_spin_unlock(mixer, lock, &flags);
		}
	}
	*(unsigned int *) &buf[0] = 1;	/* switch ID */
	for (device = 0; device < mfile->mixer->switches_count; device++) {
		snd_spin_lock(mixer, lock, &flags);
		if (mfile->schanges & (1 << device)) {
			mfile->schanges &= ~(1 << device);
			snd_spin_unlock(mixer, lock, &flags);
			*(unsigned int *) &buf[4] = device;
			copy_to_user(buffer, buf, 8);
			buffer += 8;
			size += 8;
			count -= 8;
			if (count < 8)
				return size;
		} else {
			snd_spin_unlock(mixer, lock, &flags);
		}
	}
	return size;
}

#ifdef SND_POLL
static unsigned int snd_mixer_poll(struct file *file, poll_table * wait)
{
	unsigned long flags;
	unsigned int mask;
	snd_kmixer_file_t *mfile;
	snd_kmixer_t *mixer;

	mfile = (snd_kmixer_file_t *) file->private_data;
	if (!mfile)
		return -EIO;
	mixer = mfile->mixer;

	snd_spin_lock(mixer, lock, &flags);
	snd_getlock(mfile, change) |= SND_WK_SLEEP;
	snd_spin_unlock(mixer, lock, &flags);
	snd_poll_wait(file, mfile, change, wait);

	mask = 0;
	if (mfile->changes || mfile->schanges)
		mask |= POLLIN | POLLRDNORM;

	return mask;
}
#else
static int snd_mixer_select(struct file *file,
                            int sel_type,
                            select_table * wait)
{
	unsigned long flags;
	snd_kmixer_file_t *mfile;
	snd_kmixer_t *mixer;

	mfile = (snd_kmixer_file_t *) file->private_data;
	if (!mfile)
		return -EIO;
	mixer = mfile->mixer;

	switch (sel_type) {
	case SEL_IN:
		snd_spin_lock(mixer, lock, &flags);
		if (!mfile->changes && !mfile->schanges) {
			snd_getlock(mfile, change) |= SND_WK_SLEEP;
			snd_spin_unlock(mixer, lock, &flags);
			snd_select_wait(mfile, change, wait);
			return 0;
		}
		snd_getlock(mfile, change) &= ~SND_WK_SLEEP;
		snd_spin_unlock(mixer, lock, &flags);
		return 1;
	case SEL_OUT:
		break;
	case SEL_EX:
		break;
	}
	return 0;
}
#endif

/*
 *  /proc interface
 */

static void snd_mixer_proc_read(snd_info_buffer_t * buffer, void *private_data)
{
	snd_kmixer_t *mixer;
	snd_kmixer_channel_t *channel;
	int idx;

	mixer = (snd_kmixer_t *) private_data;
	snd_iprintf(buffer, "Mixer '%s' '%s'\n", mixer->id, mixer->name);
	for (idx = 0; idx < mixer->channels_visible; idx++) {
		channel = mixer->channels[idx];
		if (!channel)
			continue;
		snd_iprintf(buffer, "Channel '%s' %d %d %d %d",
			    channel->hw.name,
			    channel->hw.min,
			    channel->hw.max,
			    channel->output.uleft,
			    channel->output.uright);
		if (channel->hw.caps & SND_MIXER_CINFO_CAP_RECORDVOLUME) {
			snd_iprintf(buffer, " [%d %d]",
				    channel->input.uleft,
				    channel->input.uright);
		}
		snd_iprintf(buffer, "%s%s%s%s%s%s%s%s\n",
			    channel->umute & SND_MIX_MUTE_LEFT ? " lmute" : "",
			    channel->umute & SND_MIX_MUTE_RIGHT ? " rmute" : "",
			    channel->record & SND_MIX_REC_LEFT ? " lrec" : "",
			    channel->record & SND_MIX_REC_RIGHT ? " rrec" : "",
			    channel->route & SND_MIX_ROUTE_LTOR_OUT ? "ltor_out" : "",
			    channel->route & SND_MIX_ROUTE_RTOL_OUT ? "rtol_out" : "",
			    channel->route & SND_MIX_ROUTE_LTOR_IN ? "ltor_in" : "",
			    channel->route & SND_MIX_ROUTE_RTOL_IN ? "rtol_in" : "");
	}
}

/*
 *  REGISTRATION PART
 */

static snd_minor_t snd_mixer_reg =
{
	"mixer",

	NULL,			/* unregister */

	NULL,			/* lseek */
	snd_mixer_read,		/* read */
	NULL,			/* write */
	snd_mixer_open,		/* open */
	snd_mixer_release,	/* release */
#ifdef SND_POLL
	snd_mixer_poll,		/* poll */
#else
	snd_mixer_select,	/* select */
#endif
	snd_mixer_ioctl,	/* ioctl */
	NULL			/* mmap */
};

snd_kmixer_t *snd_mixer_new(snd_card_t * card, char *id)
{
	snd_kmixer_t *mixer;

	mixer = (snd_kmixer_t *) snd_calloc(sizeof(snd_kmixer_t));
	if (!mixer)
		return NULL;
	mixer->card = card;
	if (id) {
		strncpy(mixer->id, id, sizeof(mixer->id) - 1);
	}
	snd_mutex_prepare(mixer, ffile);
	snd_spin_prepare(mixer, lock);
	return mixer;
}

int snd_mixer_free(snd_kmixer_t * mixer)
{
	int idx;
	snd_kmixer_channel_t *channel;
	snd_kmixer_switch_t *kswitch;

	if (!mixer)
		return -EINVAL;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		channel = mixer->channels[idx];
		snd_free(channel, sizeof(snd_kmixer_channel_t));
	}
	for (idx = 0; idx < mixer->switches_count; idx++) {
		kswitch = mixer->switches[idx];
		snd_free(kswitch, sizeof(snd_kmixer_switch_t));
	}
	if (mixer->private_data && mixer->private_free)
		mixer->private_free(mixer->private_data);
	snd_free(mixer, sizeof(snd_kmixer_t));
	return 0;
}

snd_kmixer_channel_t *snd_mixer_new_channel(snd_kmixer_t * mixer,
					    struct snd_stru_mixer_channel_hw * hw)
{
	int idx, idx1, priority;
	snd_kmixer_channel_t *channel;

	snd_mutex_down(mixer, ffile);
	if (!mixer || mixer->channels_count >= SND_MIXER_CHANNELS) {
		snd_mutex_up(mixer, ffile);
		return NULL;
	}
	channel = (snd_kmixer_channel_t *)
				snd_calloc(sizeof(snd_kmixer_channel_t));
	if (!channel) {
		snd_mutex_up(mixer, ffile);
		return NULL;
	}
	memcpy(&channel->hw, hw, sizeof(struct snd_stru_mixer_channel_hw));
	priority = channel->hw.priority;
	for (idx = 0; idx < mixer->channels_count; idx++)
		if (mixer->channels[idx]->hw.priority > priority)
			break;
	for (idx1 = mixer->channels_count; idx1 > idx; idx1--) {
		mixer->channels[idx1] = mixer->channels[idx1 - 1];
	}
	mixer->channels_count++;
	if (channel->hw.priority != SND_MIXER_PRI_HIDDEN)
		mixer->channels_visible++;
	mixer->channels[idx] = channel;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		mixer->channels[idx]->channel = idx;
	}
	snd_mutex_up(mixer, ffile);
	return channel;
}

void snd_mixer_reorder_channel(snd_kmixer_t * mixer,
			       snd_kmixer_channel_t * channel)
{
	int idx, idx1;

	if (!mixer || !channel)
		return;
	snd_mutex_down(mixer, ffile);
	for (idx = 0; idx < mixer->channels_count; idx++) {
		if (mixer->channels[idx] == channel) {
			for (idx1 = idx; idx1 + 1 < mixer->channels_count; idx1++)
				mixer->channels[idx1] = mixer->channels[idx1 + 1];
			mixer->channels_count--;
			break;
		}
	}
	for (idx = 0; idx < mixer->channels_count; idx++)
		if (mixer->channels[idx]->hw.priority > channel->hw.priority)
			break;
	for (idx1 = mixer->channels_count; idx1 > idx; idx1--) {
		mixer->channels[idx1] = mixer->channels[idx1 - 1];
	}
	mixer->channels[idx] = channel;
	mixer->channels_count++;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		mixer->channels[idx]->channel = idx;
	}
	snd_mutex_up(mixer, ffile);
}

snd_kmixer_switch_t *snd_mixer_new_switch(snd_kmixer_t * mixer,
                                          snd_kmixer_switch_t * kswitch)
{
	snd_kmixer_switch_t *rswitch;

	snd_mutex_down(mixer, ffile);
	if (!mixer || mixer->switches_count >= SND_MIXER_SWITCHES) {
		snd_mutex_up(mixer, ffile);
		return NULL;
	}
	rswitch = (snd_kmixer_switch_t *)
				snd_malloc(sizeof(snd_kmixer_switch_t));
	if (!rswitch) {
		snd_mutex_up(mixer, ffile);
		return NULL;
	}
	memcpy(rswitch, kswitch, sizeof(snd_kmixer_switch_t));
	mixer->switches[mixer->switches_count++] = rswitch;
	snd_mutex_up(mixer, ffile);
	return rswitch;
}

static void snd_mixer_store(snd_kmixer_t * mixer)
{
	int idx, size;
	unsigned int pos;
	char *buffer;
	snd_mixer_channel_t channel;
	snd_mixer_switch_t uswitch;
	mm_segment_t fs;
	snd_kmixer_file_t sfile;
	char key[32];
	
	size = sizeof(unsigned)*2 +
	       sizeof(mixer->id) + sizeof(mixer->name) +
	       mixer->channels_count*sizeof(snd_mixer_channel_t)*2 +
	       mixer->switches_count*sizeof(snd_mixer_switch_t);
	buffer = (char *) vmalloc(size);
	if (!buffer) {
		snd_printk("snd: cannot store mixer settings - no enough memory\n" );
		return;
	}
	*(((unsigned int *)buffer) + 0) = mixer->channels_count;
	*(((unsigned int *)buffer) + 1) = mixer->switches_count;
	pos = 2*sizeof(unsigned);
	memcpy(buffer + pos, mixer->id, sizeof(mixer->id));
	pos += sizeof(mixer->id);
	memcpy(buffer + pos, mixer->name, sizeof(mixer->name));
	pos += sizeof(mixer->name);
	memset(&sfile, 0, sizeof(sfile));
	sfile.mixer = mixer;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		/* output volume */
		memset(&channel, 0, sizeof(channel));
		channel.channel = idx;
		fs = snd_enter_user();
		snd_mixer_device_read(&sfile, &channel, 0);
		snd_leave_user(fs);
		memcpy(buffer + pos, &channel, sizeof(channel));
		pos += sizeof(snd_mixer_channel_t);
		/* input volume */
		fs = snd_enter_user();
		snd_mixer_device_read(&sfile, &channel, 1);
		snd_leave_user(fs);
		memcpy(buffer + pos, &channel, sizeof(channel));
		pos += sizeof(snd_mixer_channel_t);
	}
	for (idx = 0; idx < mixer->switches_count; idx++) {
		memset(&uswitch, 0, sizeof(uswitch));
		uswitch.switchn = idx;
		fs = snd_enter_user();
		snd_mixer_switch_read(&sfile, &uswitch);
		snd_leave_user(fs);
		memcpy(buffer + pos, &uswitch, sizeof(uswitch));
		pos += sizeof(snd_mixer_switch_t);
	}
	sprintf(key, "snd:%i/mixer", mixer->card->number);
	snd_persist_store(key, buffer, size);
	vfree(buffer);
}

static void snd_mixer_restore(snd_kmixer_t * mixer)
{
	int idx, size, swcount, chncount;
	unsigned int pos;
	char *buffer;
	snd_mixer_channel_t channel;
	snd_mixer_switch_t uswitch;
	mm_segment_t fs;
	snd_kmixer_file_t sfile;
	char key[32];
	
	size = sizeof(unsigned)*2 +
	       sizeof(mixer->id) + sizeof(mixer->name) +
	       mixer->channels_count*sizeof(snd_mixer_channel_t)*2 +
	       mixer->switches_count*sizeof(snd_mixer_switch_t);
	buffer = (char *) vmalloc(size);
	if (!buffer) {
		snd_printk("snd: cannot restore mixer settings - no enough memory\n" );
		return;
	}
	sprintf(key, "snd:%i/mixer", mixer->card->number);
	size = snd_persist_restore(key, buffer, size);
	/* ok. I'm very paranoid in bellow code */
	if (size < sizeof(unsigned)*2 + sizeof(mixer->id) + sizeof(mixer->name)) {
		vfree(buffer);
		return;
	}
	swcount = *(((unsigned int *)buffer) + 0);
	chncount = *(((unsigned int *)buffer) + 1);
	if (chncount != mixer->channels_count ||
	    swcount != mixer->switches_count) {
	    	vfree(buffer);
	    	return;
	}
	if (sizeof(unsigned)*2 +
	    sizeof(mixer->id) + sizeof(mixer->name) +
	    chncount*sizeof(snd_mixer_channel_t)*2 +
	    swcount*sizeof(snd_mixer_switch_t) != size) {
	    	vfree(buffer);
	    	return;
	}
	pos = 2*sizeof(unsigned);
	if (strcmp(buffer + pos, mixer->id)) {
		vfree(buffer);
		return;
	}
	pos += sizeof(mixer->id);
	if (strcmp(buffer + pos, mixer->name)) {
		vfree(buffer);
		return;
	}
	pos += sizeof(mixer->name);
	memset(&sfile, 0, sizeof(sfile));
	sfile.mixer = mixer;
	for (idx = 0; idx < chncount; idx++) {
		/* output volume */
		memcpy(&channel, buffer + pos, sizeof(channel));
		fs = snd_enter_user();
		snd_mixer_device_write(&sfile, &channel, 0);
		snd_leave_user(fs);
		pos += sizeof(snd_mixer_channel_t);
		/* input volume */
		memcpy(&channel, buffer + pos, sizeof(channel));
		fs = snd_enter_user();
		snd_mixer_device_write(&sfile, &channel, 1);
		snd_leave_user(fs);
		pos += sizeof(snd_mixer_channel_t);
	}
	for (idx = 0; idx < swcount; idx++) {
		memcpy(&uswitch, buffer + pos, sizeof(uswitch));
		fs = snd_enter_user();
		snd_mixer_switch_write(&sfile, &uswitch);
		snd_leave_user(fs);
		pos += sizeof(snd_mixer_switch_t);
	}
	vfree(buffer);
}

int snd_mixer_register(snd_kmixer_t * mixer, int device)
{
	mm_segment_t fs;
	char name[16];
	int idx, err, cardnum, minor;
	snd_info_entry_t *entry;
	snd_kmixer_channel_t *channel;
	snd_kmixer_file_t sfile;
	snd_mixer_channel_t mdevice;

	if (device < 0 || device > 1)
		return -EINVAL;
	if (!mixer->card)
		return -EINVAL;
	idx = ((cardnum = mixer->card->number) * SND_MINOR_MIXERS) + device;
	snd_mutex_down_static(register);
	if (snd_mixers[idx]) {
		snd_mutex_up_static(register);
		return -EBUSY;
	}
	mixer->device = device;
	snd_mixers[idx] = mixer;
	minor = SND_MINOR_MIXER + (mixer->card->number * SND_MINOR_MIXERS) + device;
	if ((err = snd_register_minor(minor, &snd_mixer_reg)) < 0) {
		snd_mutex_up_static(register);
		snd_mixers[idx] = NULL;
		return err;
	}
	sprintf(name, "mixer%i%i", mixer->card->number, device);
	if ((mixer->dev = snd_info_create_device(name, minor, 0)) == NULL) {
		snd_unregister_minor(minor);
		snd_mutex_up_static(register);
		snd_mixers[idx] = NULL;
		return -ENOMEM;
	}
	if ((idx & (SND_MINOR_MIXERS - 1)) == 0)
		snd_register_minor((cardnum << 4) + SND_MINOR_OSS_MIXER, &snd_mixer_reg);
	if ((idx & (SND_MINOR_MIXERS - 1)) == 1)
		snd_register_minor((cardnum << 4) + SND_MINOR_OSS_MIXER1, &snd_mixer_reg);
	snd_mutex_up_static(register);
	sprintf(name, "mixer%i", device);
	if ((entry = snd_info_create_entry(mixer->card, name)) != NULL) {
		entry->private_data = mixer;
		entry->mode = S_IFREG | S_IRUGO | S_IWUSR;
		entry->t.text.read_size = 256 + (mixer->channels_count * 128);
		entry->t.text.read = snd_mixer_proc_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	mixer->proc_entry = entry;
	if (!mixer->device)
		snd_oss_info_register(SND_OSS_INFO_DEV_MIXERS, mixer->card->number, mixer->name);
	for (idx = 0; idx < mixer->channels_visible; idx++) {
		channel = mixer->channels[idx];
		if (!channel)
			continue;

		memset(&sfile, 0, sizeof(sfile));
		sfile.mixer = mixer;
		sfile.exact = 1;

		memset(&mdevice, 0, sizeof(mdevice));
		mdevice.channel = channel->channel;
		mdevice.flags = SND_MIXER_FLG_MUTE | SND_MIXER_FLG_FORCE;
		mdevice.left = channel->hw.min;
		mdevice.right = channel->hw.min;
		fs = snd_enter_user();
		if ((err = snd_mixer_device_write(&sfile, &mdevice, 0)) < 0) {
			snd_printk("snd_mixer_register: oops, can't setup channel '%s' (%i) - output!!!\n", channel->hw.name, err);
		}
		if (channel->hw.caps & SND_MIXER_CINFO_CAP_RECORDVOLUME) {
			if ((err = snd_mixer_device_write(&sfile, &mdevice, 1)) < 0) {
				snd_printk("snd_mixer_register: oops, can't setup channel '%s' (%i) - input!!!\n", channel->hw.name, err);
			}
		}
		snd_leave_user(fs);
	}
	snd_mixer_restore(mixer);
	return 0;
}

int snd_mixer_unregister(snd_kmixer_t * mixer)
{
	int idx, cardnum;

	if (!mixer)
		return -EINVAL;
	if (!mixer->device)
		snd_oss_info_unregister(SND_OSS_INFO_DEV_MIXERS,
		                        mixer->card->number);
	if (mixer->proc_entry) {
		snd_info_unregister(mixer->proc_entry);
		mixer->proc_entry = NULL;
	}
	snd_mixer_store(mixer);
	snd_mutex_down_static(register);
	idx = (mixer->card->number * SND_MINOR_MIXERS) + mixer->device;
	if (snd_mixers[idx] != mixer) {
		snd_mutex_up_static(register);
		return -EINVAL;
	}
	snd_info_free_device(mixer->dev);
	snd_unregister_minor(SND_MINOR_MIXER + idx);
	if ((idx & (SND_MINOR_MIXERS - 1)) == 0) {
		cardnum = idx >> 1;
		snd_unregister_minor((cardnum << 4) + SND_MINOR_OSS_MIXER);
	}
	if ((idx & (SND_MINOR_MIXERS - 1)) == 1) {
		cardnum = idx >> 1;
		snd_unregister_minor((cardnum << 4) + SND_MINOR_OSS_MIXER1);
	}
	snd_mixers[idx] = NULL;
	snd_mutex_up_static(register);
	return snd_mixer_free(mixer);
}

/*
 *  INIT PART
 */

snd_kmixer_channel_t *snd_mixer_find_channel(snd_kmixer_t * mixer,
					     unsigned int priority)
{
	int idx;
	snd_kmixer_channel_t *channel;

	if (!mixer)
		return NULL;
	for (idx = 0; idx < mixer->channels_count; idx++) {
		channel = mixer->channels[idx];
		if (channel->hw.priority == priority)
			return channel;
	}
	return NULL;
}

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_mixer_export;
#endif

int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_mixer_export) < 0)
		return -ENOMEM;
#endif
	snd_control_register_ioctl(snd_mixer_control_ioctl);
	return 0;
}

void cleanup_module(void)
{
	snd_control_unregister_ioctl(snd_mixer_control_ioctl);
}
