/*
 *  Routines for driver control
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "minors.h"
#include "info.h"
#include "control.h"
#include "sndpersist.h"

struct snd_stru_control_ioctl {
	snd_control_ioctl_t fioctl;
	struct snd_stru_control_ioctl *next;
};

static struct snd_stru_control_ioctl *snd_first_ioctl = NULL;
static int snd_first_ioctl_use = 0;
static snd_info_entry_t *snd_control_dev[SND_CARDS] =
				{[0 ... (SND_CARDS - 1)] = NULL};

snd_mutex_define_static(register);

static int snd_control_open(unsigned short minor, int cardnum, int device,
			    struct file *file)
{
	snd_card_t *card;
	snd_control_t *control;

	if (!(card = snd_cards[cardnum]))
		return -ENXIO;
	control = (snd_control_t *) snd_malloc(sizeof(snd_control_t));
	if (!control)
		return -ENOMEM;
	control->card = card;
	file->private_data = control;
	MOD_INC_USE_COUNT;
	card->use_inc(card);
	return 0;
}

static int snd_control_release(unsigned short minor, int cardnum, int device,
			       struct file *file)
{
	snd_control_t *control;

	if (file->private_data) {
		control = (snd_control_t *) file->private_data;
		file->private_data = NULL;
		control->card->use_dec(control->card);
		snd_free(control, sizeof(snd_control_t));
	}
	MOD_DEC_USE_COUNT;
	return 0;
}

static int snd_control_hw_info(snd_card_t * card, snd_control_t * control,
			       unsigned int cmd, unsigned long arg)
{
	struct snd_stru_control_ioctl *p;
	struct snd_ctl_hw_info info;

	memset(&info, 0, sizeof(info));
	if (verify_area(VERIFY_WRITE, (void *) arg,
			sizeof(struct snd_ctl_hw_info)))
		 return -EFAULT;
	snd_mutex_down(card, control);
	info.type = card->type;
	strncpy(info.id, card->id, 8);
	strncpy(info.abbreviation, card->abbreviation, sizeof(info.abbreviation) - 1);
	strncpy(info.name, card->shortname, sizeof(info.name) - 1);
	strncpy(info.longname, card->longname, sizeof(info.longname) - 1);
	info.switches = card->switches_count;
	for (p = snd_first_ioctl; p; p = p->next)
		p->fioctl(card, control, cmd, (unsigned long) &info);
	snd_mutex_up(card, control);
	copy_to_user((void *) arg, &info, sizeof(struct snd_ctl_hw_info));
	return 0;
}

static int snd_control_switch_read(snd_card_t * card, struct snd_ctl_switch *_sw)
{
	snd_ctl_switch_t sw;
	snd_ctl_kswitch_t *ksw;
	int err = 0;
	unsigned int switchn;

	if (verify_area(VERIFY_READ, (void *) _sw, sizeof(struct snd_ctl_switch)))
		 return -EFAULT;
	if (verify_area(VERIFY_WRITE, (void *) _sw, sizeof(struct snd_ctl_switch)))
		 return -EFAULT;
	copy_from_user(&sw, _sw, sizeof(sw));
	switchn = sw.switchn;
	snd_mutex_down(card, control);
	if (switchn >= card->switches_count) {
		snd_mutex_up(card, control);
		return -EINVAL;
	}
	ksw = card->switches[switchn];
	memset(&sw, 0, sizeof(sw));
	strncpy(sw.name, ksw->name, sizeof(sw.name) - 1);
	err = ksw->get_switch(card, ksw, &sw);
	snd_mutex_up(card, control);
	if (!err)
		copy_to_user(_sw, &sw, sizeof(sw));
	return err;
}

static int snd_control_switch_write(snd_card_t * card, struct snd_ctl_switch *_sw)
{
	static char *root_only[] =
	{
		SND_CTL_SW_JOYSTICK,
		SND_CTL_SW_JOYSTICK_ADDRESS,
		NULL
	};
	snd_ctl_switch_t sw;
	snd_ctl_kswitch_t *ksw;
	int err = 0;
	unsigned int switchn;
	char **root_str = root_only;

	memset(&sw, 0, sizeof(sw));
	if (verify_area(VERIFY_READ, (void *) _sw, sizeof(struct snd_ctl_switch)))
		 return -EFAULT;
	copy_from_user(&sw, _sw, sizeof(sw));
	switchn = sw.switchn;
	snd_mutex_down(card, control);
	if (switchn >= card->switches_count) {
		snd_mutex_up(card, control);
		return -EINVAL;
	}
	ksw = card->switches[switchn];
	while (*root_str) {
		if (!strcmp(ksw->name, *root_str)) {
			if (current->euid != 0 && current->egid != 0) {
				snd_mutex_up(card, control);
				return -EPERM;
			}
		}
		root_str++;
	}
	err = ksw->set_switch(card, ksw, &sw);
	snd_mutex_up(card, control);
	return err;
}

static int snd_control_ioctl(struct file *file,
			     unsigned int cmd,
			     unsigned long arg)
{
	snd_control_t *control;
	snd_card_t *card;
	struct snd_stru_control_ioctl *p;
	int err;

	control = (snd_control_t *) file->private_data;
	if (!control)
		return -EINVAL;
	card = control->card;
	if (!card)
		return -EINVAL;
	switch (cmd) {
	case SND_CTL_IOCTL_PVERSION:
		return snd_ioctl_out((long *) arg, SND_CTL_VERSION);
	case SND_CTL_IOCTL_HW_INFO:
		return snd_control_hw_info(card, control, cmd, arg);
	case SND_CTL_IOCTL_SWITCHES:
		return snd_ioctl_out((long *) arg, card->switches_count);
	case SND_CTL_IOCTL_SWITCH_READ:
		return snd_control_switch_read(card, (struct snd_ctl_switch *) arg);
	case SND_CTL_IOCTL_SWITCH_WRITE:
		return snd_control_switch_write(card, (struct snd_ctl_switch *) arg);
	}
	snd_mutex_down(card, control);
	for (p = snd_first_ioctl; p; p = p->next) {
		err = p->fioctl(card, control, cmd, arg);
		if (err != -EAGAIN) {
			snd_mutex_up(card, control);
			return err;
		}
	}
	snd_mutex_up(card, control);
	return -ENXIO;
}

int snd_control_register_ioctl(snd_control_ioctl_t fcn)
{
	struct snd_stru_control_ioctl *p, *pn;

	pn = (struct snd_stru_control_ioctl *)
			snd_calloc(sizeof(struct snd_stru_control_ioctl));
	if (!pn)
		return -ENOMEM;
	pn->fioctl = fcn;
	snd_mutex_down_static(register);
	if (!snd_first_ioctl) {
		snd_first_ioctl = pn;
	} else {
		for (p = snd_first_ioctl; p->next; p = p->next);
		p->next = pn;
	}
	snd_mutex_up_static(register);
	return 0;
}

int snd_control_unregister_ioctl(snd_control_ioctl_t fcn)
{
	struct snd_stru_control_ioctl *p, *pn, *po;

	snd_mutex_down_static(register);
	if (!snd_first_ioctl) {
		snd_mutex_up_static(register);
		return -EINVAL;
	}
	p = snd_first_ioctl;
	if (p->fioctl == fcn) {
		snd_first_ioctl = p->next;
		snd_mutex_up_static(register);
		snd_free(p, sizeof(struct snd_stru_control_ioctl));
		return 0;
	}
	for (; p; p = pn) {
		pn = p->next;
		if (pn && pn->fioctl == fcn) {
			po = pn;
			p->next = pn = pn->next;
			snd_mutex_up_static(register);
			snd_free(po, sizeof(struct snd_stru_control_ioctl));
			return 0;
		}
	}
	snd_mutex_up_static(register);
	return -EINVAL;
}

snd_ctl_kswitch_t *snd_control_new_switch(snd_card_t * card, snd_ctl_kswitch_t * ksw)
{
	snd_ctl_kswitch_t **x;
	snd_ctl_kswitch_t *nsw;

	snd_mutex_down(card, control);
	x = (snd_ctl_kswitch_t **)
			snd_malloc((card->switches_count + 1) * sizeof(void *));
	if (!x) {
		snd_mutex_up(card, control);
		return NULL;
	}
	nsw = (snd_ctl_kswitch_t *) snd_malloc(sizeof(snd_ctl_kswitch_t));
	if (!nsw) {
		snd_free(x, (card->switches_count + 1) * sizeof(void *));
		snd_mutex_up(card, control);
		return NULL;
	}
	memcpy(nsw, ksw, sizeof(snd_ctl_kswitch_t));
	if (card->switches) {
		memcpy(x, card->switches,
		       (card->switches_count + 1) * sizeof(void *));
		snd_free(card->switches,
		         card->switches_count * sizeof(void *));
	}
	card->switches = x;
	card->switches[card->switches_count++] = nsw;
	snd_mutex_up(card, control);
	return nsw;
}

/*
 *  INIT PART
 */

static snd_minor_t snd_control_reg =
{
	"control",

	NULL,			/* unregister */

	NULL,			/* lseek */
	NULL,			/* read */
	NULL,			/* write */
	snd_control_open,	/* open */
	snd_control_release,	/* release */
#ifdef SND_POLL
	NULL,			/* poll */
#else
	NULL,			/* select */
#endif
	snd_control_ioctl,	/* ioctl */
	NULL			/* mmap */
};

static void snd_control_store(snd_card_t * card)
{
	int idx, size;
	unsigned int pos;
	char *buffer;
	snd_ctl_switch_t uswitch;
	mm_segment_t fs;
	char key[32];
	
	size = sizeof(unsigned) +
	       sizeof(card->id) + sizeof(card->longname) +
	       card->switches_count*sizeof(snd_ctl_switch_t);
	buffer = (char *) vmalloc(size);
	if (!buffer) {
		snd_printk("snd: cannot store control settings - no enough memory\n" );
		return;
	}
	*(((unsigned int *)buffer) + 0) = card->switches_count;
	pos = sizeof(unsigned);
	memcpy(buffer + pos, card->id, sizeof(card->id));
	pos += sizeof(card->id);
	memcpy(buffer + pos, card->longname, sizeof(card->longname));
	pos += sizeof(card->longname);
	for (idx = 0; idx < card->switches_count; idx++) {
		memset(&uswitch, 0, sizeof(uswitch));
		uswitch.switchn = idx;
		fs = snd_enter_user();
		snd_control_switch_read(card, &uswitch);
		snd_leave_user(fs);
		memcpy(buffer + pos, &uswitch, sizeof(uswitch));
		pos += sizeof(snd_ctl_switch_t);
	}
	sprintf(key, "snd:%i/control", card->number);
	snd_persist_store(key, buffer, size);
	vfree(buffer);
}

static void snd_control_restore(snd_card_t * card)
{
	int idx, size, swcount;
	unsigned int pos;
	char *buffer;
	snd_ctl_switch_t uswitch;
	mm_segment_t fs;
	char key[32];
	
	size = sizeof(unsigned) +
	       sizeof(card->id) + sizeof(card->longname) +
	       card->switches_count*sizeof(snd_ctl_switch_t);
	buffer = (char *) vmalloc(size);
	if (!buffer) {
		snd_printk("snd: cannot restore control settings - no enough memory\n" );
		return;
	}
	sprintf(key, "snd:%i/control", card->number);
	size = snd_persist_restore(key, buffer, size);
	/* ok. I'm very paranoid in bellow code */
	if (size < sizeof(unsigned) + sizeof(card->id) + sizeof(card->longname)) {
		vfree(buffer);
		return;
	}
	swcount = *(((unsigned int *)buffer) + 0);
	if (swcount != card->switches_count) {
	    	vfree(buffer);
	    	return;
	}
	if (sizeof(unsigned) +
	    sizeof(card->id) + sizeof(card->longname) +
	    swcount*sizeof(snd_ctl_switch_t) != size) {
	    	vfree(buffer);
	    	return;
	}
	pos = sizeof(unsigned);
	if (strcmp(buffer + pos, card->id)) {
		vfree(buffer);
		return;
	}
	pos += sizeof(card->id);
	if (strcmp(buffer + pos, card->longname)) {
		vfree(buffer);
		return;
	}
	pos += sizeof(card->longname);
	for (idx = 0; idx < swcount; idx++) {
		memcpy(&uswitch, buffer + pos, sizeof(uswitch));
		fs = snd_enter_user();
		snd_control_switch_write(card, &uswitch);
		snd_leave_user(fs);
		pos += sizeof(snd_ctl_switch_t);
	}
	vfree(buffer);
}


int snd_control_register(snd_card_t *card)
{
	int err, cardnum;
	char name[16];

	if (!card)
		return -EINVAL;
	cardnum = card->number;
	if (cardnum < 0 || cardnum >= SND_CARDS)
		return -EINVAL;
	if ((err = snd_register_minor(SND_MINOR_CONTROL + cardnum,
				      &snd_control_reg)) < 0)
		return err;
	sprintf(name, "control%i", cardnum);
	if ((snd_control_dev[cardnum] = snd_info_create_device(name,
				SND_MINOR_CONTROL + cardnum, 0)) == NULL) {
		snd_unregister_minor(SND_MINOR_CONTROL + cardnum);
		return -ENOMEM;
	}
	snd_first_ioctl_use++;
	snd_control_restore(card);
	return 0;
}

int snd_control_unregister(snd_card_t *card)
{
	int err, cardnum;

	if (!card)
		return -EINVAL;
	cardnum = card->number;
	if (cardnum < 0 || cardnum >= SND_CARDS)
		return -EINVAL;
	snd_control_store(card);
	snd_info_free_device(snd_control_dev[cardnum]);
	if ((err = snd_unregister_minor(SND_MINOR_CONTROL + cardnum)) < 0)
		return err;
	snd_first_ioctl_use--;
	return 0;
}
