/*
 *   ALSA sequencer Timer
 *   Copyright (c) 1998 by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "seq_timer.h"
#include "seq_queue.h"
#include "seq_info.h"

/* static data */
static snd_seq_real_time_t timer_period;

static void snd_seq_timer_reset(timer_t * tmr);


/* create new timer (constructor) */
timer_t *snd_seq_timer_new(void)
{
	timer_t *tmr;
	
	tmr = snd_malloc(sizeof(timer_t));
	if (tmr == NULL) {
		snd_printk("malloc failed for snd_seq_timer_new() \n");
		return NULL;
	}

	/* clear unused variables */
	memset(tmr, 0, sizeof(timer_t));

	/* setup defaults */
	tmr->ppq = 96;		/* 96 PPQ */
	tmr->tempo = 500000;	/* 120 BPM */	
	tmr->running = 0;

	/* reset time */
	snd_seq_timer_reset(tmr);
	
	return tmr;
}

/* delete timer (destructor) */
void snd_seq_timer_delete(timer_t **tmr)
{
	timer_t *t = *tmr;
	*tmr = NULL;

	if (t == NULL) {
		snd_printk("oops: snd_seq_timer_delete() called with NULL timer\n");
		return;
	}
	t->running = 0;

	/* reset time */
	snd_seq_timer_reset(t);

	snd_free(t,sizeof(timer_t));
}


static void snd_seq_timer_reset(timer_t * tmr)
{
	/* reset time & songposition */
	tmr->cur_tick = 0;
	tmr->cur_time.tv_sec = 0;
	tmr->cur_time.tv_nsec = 0;

	tmr->tempo_tick = 0;
	tmr->tempo_time.tv_sec = 0;
	tmr->tempo_time.tv_nsec = 0;

	tmr->sync_tmp = 0;
}


/* called by timer interrupt routine. the period time since previous invocation is passed */
int snd_seq_timer_interrupt(snd_seq_real_time_t * period)
{
	volatile int timers_running = 0;	/* flag if we did increment times */
	int c;
	queue_t *q;
	timer_t *tmr;


	/* FIXME: temporary hack to get timer resolution, should be set by registration call (together with name) */
	memcpy(&timer_period, period, sizeof(snd_seq_real_time_t));


	/* process all timers... */

	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);
		if (q == NULL)
			continue;

		tmr = q->timer;

		if (tmr != NULL) {
			if (tmr->running) {
				double delta_time;

				/* update timer */
				snd_seq_inc_real_time(&tmr->cur_time, period);
				timers_running = 1;

				/* calculate current tick */
				/* FIXME: use of floating point stuff, illegal in kernel (??) */
				if (tmr->tempo > 0) {
					delta_time = tmr->cur_time.tv_sec - tmr->tempo_time.tv_sec + (1.0E-9 * (double) (tmr->cur_time.tv_nsec - tmr->tempo_time.tv_nsec));
					tmr->cur_tick = tmr->tempo_tick + (tmr->ppq * delta_time * 1.0E6 / tmr->tempo);
				}
			}
		}
	}

	if (timers_running) {

#ifdef 0
		/* sync timer has expired *//* FIXME: hack to send midi sync info */
		if ((tmr->cur_tick > tmr->sync_tmp) && (snd_seq_unused_cells() > 10)) {
			snd_seq_event_t ev;

			ev.type = SND_SEQ_EVENT_CLOCK;
			ev.flags = SND_SEQ_TIME_STAMP_TICK | SND_SEQ_TIME_MODE_ABS;
			ev.time.tick = 0;
			ev.source.queue = 0;
			ev.source.client = 0;
			ev.source.port = 0;
			ev.dest.queue = 1;
			ev.dest.client = 255;
			ev.dest.port = 255;
			if (snd_seq_kernel_client_dispatch(0, &ev)) {
				tmr->sync_tmp += tmr->ppq / 24;
			}
		}
#endif
		/* check queues and dispatch events */
		snd_seq_check_queues();
	}
	return 0;		/* FIXME ??? perhaps return the timers_running flag, so the timer can decrease frequency if there are no active queues */
}


void snd_seq_timer_set_tempo(timer_t * tmr, int tempo)
{
	if (tmr) {
		/* store location of tempo chamge */
		tmr->tempo_tick = tmr->cur_tick;
		tmr->tempo_time.tv_sec = tmr->cur_time.tv_sec;
		tmr->tempo_time.tv_nsec = tmr->cur_time.tv_nsec;

		tmr->tempo = tempo;
	}
}


void snd_seq_timer_set_ppq(timer_t * tmr, int ppq)
{
	if (tmr) {
		if (tmr->running & (ppq != tmr->ppq)) {
			/* refuse to change ppq on running timers, because it wil upset the song position (ticks) */
			snd_printk("seq: cannot change ppq of a running timer\n");
		} else {
			tmr->ppq = ppq;
		}
	}
}


void snd_seq_timer_stop(timer_t * tmr)
{
	tmr->running = 0;
}


void snd_seq_timer_start(timer_t * tmr)
{
	tmr->running = 1;

	/* reset time & songposition */
	snd_seq_timer_reset(tmr);
}


void snd_seq_timer_continue(timer_t * tmr)
{
	tmr->running = 1;
}


/* exported to seq_info.c */
void snd_seq_info_timer_read(snd_info_buffer_t * buffer, void *private_data)
{
	snd_iprintf(buffer, "Timer       : %s\n", "Some unknown Timer...");
	snd_iprintf(buffer, "Period time : %d.%09d\n", timer_period.tv_sec, timer_period.tv_nsec);
}
