/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *     and (c) 1999 Steve Ratcliffe <steve@parabola.demon.co.uk>
 *  Copyright (C) 1999-2000 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *  Routines for control of EMU8000 chip
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define SND_MAIN_OBJECT_FILE
#include "emu8000_local.h"

MODULE_AUTHOR("Takashi Iwai, Steve Ratcliffe");
MODULE_PARM(snd_seq_ports, "i");
MODULE_PARM_DESC(snd_seq_ports, "number of sequencer ports to be created");

int snd_seq_ports = 4;


/*
 * create a new hardware dependent device for Emu8000
 */
int snd_emu8000_new_device(snd_seq_device_t *dev)
{
	emu8000_t *hw;
	emu8000_arg_t *arg;
	snd_emux_t *emu;

	hw = snd_magic_kcalloc(emu8000_t, 0, GFP_KERNEL);
	if (hw == NULL)
		return -ENOMEM;
	hw->reg_lock = SPIN_LOCK_UNLOCKED;
	arg = SND_SEQ_DEVICE_ARGPTR(dev);
	hw->port1 = arg->port;
	hw->port2 = arg->port + 0x400;
	hw->port3 = arg->port + 0x800;
	hw->mem_size = 0;

	if (snd_emu8000_detect(hw) < 0) {
		snd_magic_kfree(hw);
		return -ENODEV;
	}

	if (snd_emux_new(&emu) < 0) {
		snd_magic_kfree(hw);
		return -ENOMEM;
	}
	hw->emu = emu;
	snd_emu8000_ops_setup(hw);

	emu->hw = hw;
	emu->max_voices = EMU8000_DRAM_VOICES;
	emu->num_ports = snd_seq_ports;

	snd_emu8000_init_hw(hw);
	hw->memhdr = snd_emux_memhdr_new(hw->mem_size);
	if (hw->memhdr == NULL) {
		snd_emux_free(emu);
		snd_magic_kfree(hw);
		return -ENOMEM;
	}

	emu->memhdr = hw->memhdr;
	if (snd_emux_register(emu, dev->card, arg->index, "Emu8000 WaveTable") < 0) {
		snd_emux_free(emu);
		snd_emux_memhdr_free(hw->memhdr);
		snd_magic_kfree(hw);
		return -ENOMEM;
	}

	snd_emu8000_create_mixer(hw, arg->mixer, arg->mixer_dest, arg->mixer_index);
	
	hw->bass_level = 5;
	hw->treble_level = 9;
	hw->chorus_mode = 2;
	hw->reverb_mode = 4;

	snd_emu8000_start(hw);

	dev->driver_data = hw;

	return 0;
}


/*
 * free all resources
 */
int snd_emu8000_delete_device(snd_seq_device_t *dev)
{
	emu8000_t *hw;

	hw = snd_magic_cast(emu8000_t, dev->driver_data, -EINVAL);

	snd_emu8000_free_mixer(hw);
	if (hw->emu)
		snd_emux_free(hw->emu);
	if (hw->memhdr)
		snd_emux_memhdr_free(hw->memhdr);
	snd_magic_kfree(hw);
	return 0;
}

/*
 *  INIT part
 */

#ifdef MODULE
int __init init_module(void)
#else
int __init alsa_emu8000_init(void)
#endif
{
	
	static snd_seq_dev_ops_t ops = {
		snd_emu8000_new_device,
		snd_emu8000_delete_device,
	};
	return snd_seq_device_register_driver(SND_SEQ_DEV_ID_EMU8000, &ops, sizeof(emu8000_arg_t));
}

#ifdef MODULE

void __exit cleanup_module(void)
{
	snd_seq_device_unregister_driver(SND_SEQ_DEV_ID_EMU8000);
}

#endif
