/*
 *  Midi synth routines for the Emu8k/Emu10k1
 *
 *  Copyright (C) 1999 Steve Ratcliffe
 *  Copyright (c) 1999-2000 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *  Contains code based on awe_wave.c by Takashi Iwai
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "emux_voice.h"

#ifdef SND_EMUX_USE_RAW_EFFECT
/*
 * effects table
 */

#define xoffsetof(type,tag)	((long)(&((type)NULL)->tag) - (long)(NULL))

#define parm_offset(tag)	xoffsetof(soundfont_voice_parm_block_t*, tag)

#define PARM_BYTE	0
#define PARM_WORD	1
#define PARM_SIGN	2

static struct emux_parm_defs {
	int type;	/* byte or word */
	int low, high;	/* value range */
	long offset;	/* offset in parm_block (-1 = not written) */
	int update;	/* flgas for real-time update */
} parm_defs[EMUX_NUM_EFFECTS] = {
	{PARM_WORD, 0, 0x8000, parm_offset(moddelay), 0},	/* env1 delay */
	{PARM_BYTE, 1, 0x80, parm_offset(modatk), 0},	/* env1 attack */
	{PARM_BYTE, 0, 0x7e, parm_offset(modhld), 0},	/* env1 hold */
	{PARM_BYTE, 1, 0x7f, parm_offset(moddcy), 0},	/* env1 decay */
	{PARM_BYTE, 1, 0x7f, parm_offset(modrel), 0},	/* env1 release */
	{PARM_BYTE, 0, 0x7f, parm_offset(modsus), 0},	/* env1 sustain */
	{PARM_BYTE, 0, 0xff, parm_offset(env1pit), 0},	/* env1 pitch */
	{PARM_BYTE, 0, 0xff, parm_offset(env1fc), 0},	/* env1 fc */

	{PARM_WORD, 0, 0x8000, parm_offset(voldelay), 0},	/* env2 delay */
	{PARM_BYTE, 1, 0x80, parm_offset(volatk), 0},	/* env2 attack */
	{PARM_BYTE, 0, 0x7e, parm_offset(volhld), 0},	/* env2 hold */
	{PARM_BYTE, 1, 0x7f, parm_offset(voldcy), 0},	/* env2 decay */
	{PARM_BYTE, 1, 0x7f, parm_offset(volrel), 0},	/* env2 release */
	{PARM_BYTE, 0, 0x7f, parm_offset(volsus), 0},	/* env2 sustain */

	{PARM_WORD, 0, 0x8000, parm_offset(lfo1delay), 0},	/* lfo1 delay */
	{PARM_BYTE, 0, 0xff, parm_offset(lfo1freq), SND_EMUX_UPDATE_TREMFREQ},	/* lfo1 freq */
	{PARM_SIGN, -128, 127, parm_offset(lfo1vol), SND_EMUX_UPDATE_TREMFREQ},	/* lfo1 vol */
	{PARM_SIGN, -128, 127, parm_offset(lfo1pit), SND_EMUX_UPDATE_FMMOD},	/* lfo1 pitch */
	{PARM_BYTE, 0, 0xff, parm_offset(lfo1fc), SND_EMUX_UPDATE_FMMOD},	/* lfo1 cutoff */

	{PARM_WORD, 0, 0x8000, parm_offset(lfo2delay), 0},	/* lfo2 delay */
	{PARM_BYTE, 0, 0xff, parm_offset(lfo2freq), SND_EMUX_UPDATE_FM2FRQ2},	/* lfo2 freq */
	{PARM_SIGN, -128, 127, parm_offset(lfo2pit), SND_EMUX_UPDATE_FM2FRQ2},	/* lfo2 pitch */

	{PARM_WORD, 0, 0xffff, -1, SND_EMUX_UPDATE_PITCH},	/* initial pitch */
	{PARM_BYTE, 0, 0xff, parm_offset(chorus), 0},	/* chorus */
	{PARM_BYTE, 0, 0xff, parm_offset(reverb), 0},	/* reverb */
	{PARM_BYTE, 0, 0xff, parm_offset(cutoff), SND_EMUX_UPDATE_VOLUME},	/* cutoff */
	{PARM_BYTE, 0, 15, parm_offset(filterQ), SND_EMUX_UPDATE_Q},	/* resonance */

	{PARM_WORD, 0, 0xffff, -1, 0},	/* sample start */
	{PARM_WORD, 0, 0xffff, -1, 0},	/* loop start */
	{PARM_WORD, 0, 0xffff, -1, 0},	/* loop end */
	{PARM_WORD, 0, 0xffff, -1, 0},	/* coarse sample start */
	{PARM_WORD, 0, 0xffff, -1, 0},	/* coarse loop start */
	{PARM_WORD, 0, 0xffff, -1, 0},	/* coarse loop end */
	{PARM_BYTE, 0, 0xff, -1, SND_EMUX_UPDATE_VOLUME},	/* initial attenuation */
};

/* set byte effect value */
static void
effect_set_byte(unsigned char *valp, snd_midi_channel_t *chan, int type)
{
	short effect;
	snd_emux_effect_table_t *fx = chan->private;

	effect = fx->val[type];
	if (fx->flag[type] == EMUX_FX_FLAG_ADD) {
		if (parm_defs[type].type == PARM_SIGN)
			effect += *(char*)valp;
		else
			effect += *valp;
	}
	if (effect < parm_defs[type].low)
		effect = parm_defs[type].low;
	else if (effect > parm_defs[type].high)
		effect = parm_defs[type].high;
	*valp = (unsigned char)effect;
}

/* set word effect value */
static void
effect_set_word(unsigned short *valp, snd_midi_channel_t *chan, int type)
{
	int effect;
	snd_emux_effect_table_t *fx = chan->private;

	effect = *(unsigned short*)&fx->val[type];
	if (fx->flag[type] == EMUX_FX_FLAG_ADD)
		effect += *valp;
	if (effect < parm_defs[type].low)
		effect = parm_defs[type].low;
	else if (effect > parm_defs[type].high)
		effect = parm_defs[type].high;
	*valp = (unsigned short)effect;
}

/* address offset */
static int
effect_get_offset(snd_midi_channel_t *chan, int lo, int hi, int mode)
{
	int addr = 0;
	snd_emux_effect_table_t *fx = chan->private;

	if (fx->flag[hi])
		addr = (short)fx->val[hi];
	addr = addr << 15;
	if (fx->flag[lo])
		addr += (short)fx->val[lo];
	if (!(mode & SND_SFNT_SAMPLE_8BITS))
		addr /= 2;
	return addr;
}

#ifdef CONFIG_SND_OSSEMUL
/* change effects - for OSS sequencer compatibility */
void
snd_emux_send_effect_oss(snd_emux_port_t *port, snd_midi_channel_t *chan, int type, int val)
{
	int mode;

	if (type & 0x40)
		mode = EMUX_FX_FLAG_OFF;
	else if (type & 0x80)
		mode = EMUX_FX_FLAG_ADD;
	else
		mode = EMUX_FX_FLAG_SET;
	type &= 0x3f;

	snd_emux_send_effect(port, chan, type, val, mode);
}
#endif

/* Modify the effect value.
 * if update is necessary, call emu8000_control
 */
void
snd_emux_send_effect(snd_emux_port_t *port, snd_midi_channel_t *chan, int type, int val, int mode)
{
	int i;
	int offset;
	unsigned char *srcp, *origp;
	snd_emux_t *emu;
	snd_emux_effect_table_t *fx;
	unsigned long flags;

	emu = port->emu;
	fx = chan->private;
	if (emu == NULL || fx == NULL)
		return;
	if (type < 0 || type >= EMUX_NUM_EFFECTS)
		return;

	fx->val[type] = val;
	fx->flag[type] = mode;

	/* do we need to modify the register in realtime ? */
	if (! parm_defs[type].update || (offset = parm_defs[type].offset) < 0)
		return;

	/* modify the register values */
	spin_lock_irqsave(&emu->voice_lock, flags);
	for (i = 0; i < emu->max_voices; i++) {
		snd_emux_voice_t *vp = &emu->voices[i];
		if (!STATE_IS_PLAYING(vp->state) || vp->chan != chan)
			continue;
		srcp = (unsigned char*)&vp->reg.parm + offset;
		origp = (unsigned char*)&vp->zone->v.parm + offset;
		switch (parm_defs[type].type) {
		case PARM_BYTE:
		case PARM_SIGN:
			*srcp = *origp;
			effect_set_byte(srcp, chan, type);
			break;
		case PARM_WORD:
			*(unsigned short*)srcp = *(unsigned short*)origp;
			effect_set_word((unsigned short*)srcp, chan, type);
			break;
		}
	}
	spin_unlock_irqrestore(&emu->voice_lock, flags);

	/* activate them */
	snd_emux_update_channel(port, chan, parm_defs[type].update);
}


/* copy wavetable registers to voice table */
void
snd_emux_setup_effect(snd_emux_voice_t *vp)
{
	snd_midi_channel_t *chan = vp->chan;
	snd_emux_effect_table_t *fx;
	unsigned char *srcp;
	int i;

	if (! (fx = chan->private))
		return;

	/* modify the register values via effect table */
	for (i = 0; i < EMUX_FX_END; i++) {
		int offset;
		if (! fx->flag[i] || (offset = parm_defs[i].offset) < 0)
			continue;
		srcp = (unsigned char*)&vp->reg.parm + offset;
		switch (parm_defs[i].type) {
		case PARM_BYTE:
		case PARM_SIGN:
			effect_set_byte(srcp, chan, i);
			break;
		case PARM_WORD:
			effect_set_word((unsigned short*)srcp, chan, i);
			break;
		}
	}

	/* correct sample and loop points */
	vp->reg.start += effect_get_offset(chan, EMUX_FX_SAMPLE_START,
					   EMUX_FX_COARSE_SAMPLE_START,
					   vp->reg.sample_mode);

	vp->reg.loopstart += effect_get_offset(chan, EMUX_FX_LOOP_START,
					       EMUX_FX_COARSE_LOOP_START,
					       vp->reg.sample_mode);

	vp->reg.loopend += effect_get_offset(chan, EMUX_FX_LOOP_END,
					     EMUX_FX_COARSE_LOOP_END,
					     vp->reg.sample_mode);
}

/*
 * effect table
 */
void
snd_emux_create_effect(snd_emux_port_t *p)
{
	int i;
	p->effect = snd_kcalloc(sizeof(snd_emux_effect_table_t) * p->chset.max_channels, GFP_KERNEL);
	if (p->effect) {
		for (i = 0; i < p->chset.max_channels; i++)
			p->chset.channels[i].private = p->effect + i;
	} else {
		for (i = 0; i < p->chset.max_channels; i++)
			p->chset.channels[i].private = NULL;
	}
}

void
snd_emux_delete_effect(snd_emux_port_t *p)
{
	if (p->effect) {
		snd_kfree(p->effect);
		p->effect = NULL;
	}
}

void
snd_emux_clear_effect(snd_emux_port_t *p)
{
	if (p->effect) {
		memset(p->effect, 0, sizeof(snd_emux_effect_table_t) * p->chset.max_channels);
	}
}

#endif /* SND_EMUX_USE_RAW_EFFECT */
