/*
 *  Copyright (C) 2000 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *  Routines for control of EMU WaveTable chip
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define SND_MAIN_OBJECT_FILE
#include "../../include/emux_synth.h"
#include "emux_voice.h"

MODULE_AUTHOR("Takashi Iwai");

/*
 * create a new hardware dependent device for Emu8000
 */
int snd_emux_new(snd_emux_t **remu)
{
	snd_emux_t *emu;

	*remu = NULL;
	emu = snd_magic_kcalloc(snd_emux_t, 0, GFP_KERNEL);
	if (emu == NULL)
		return -ENOMEM;

	emu->voice_lock = SPIN_LOCK_UNLOCKED;
	init_MUTEX(&emu->register_mutex);

	emu->client = -1;
#ifdef CONFIG_SND_OSSEMUL
	emu->oss_synth = NULL;
#endif
	emu->max_voices = 0;
	emu->use_time = 0;

	*remu = emu;
	return 0;
}


/*
 */
int snd_emux_register(snd_emux_t *emu, snd_card_t *card, int index, char *name)
{
	snd_sf_callback_t sf_cb;

	snd_debug_check(emu->hw == NULL, -EINVAL);
	snd_debug_check(emu->max_voices <= 0, -EINVAL);
	snd_debug_check(card == NULL, -EINVAL);
	snd_debug_check(name == NULL, -EINVAL);

	emu->card = card;
	emu->name = snd_kmalloc_strdup(name, GFP_KERNEL);
	emu->voices = snd_kcalloc(sizeof(snd_emux_voice_t) * emu->max_voices, GFP_KERNEL);
	if (emu->voices == NULL)
		return -ENOMEM;

	/* create soundfont list */
	memset(&sf_cb, 0, sizeof(sf_cb));
	sf_cb.private_data = emu;
	sf_cb.sample_new = emu->ops.sample_new;
	sf_cb.sample_free = emu->ops.sample_free;
	sf_cb.sample_reset = emu->ops.sample_reset;
	emu->sflist = snd_sf_new(&sf_cb, emu->memhdr);
	if (emu->sflist == NULL)
		return -ENOMEM;

	snd_emux_init_voices(emu);

	snd_emux_init_seq(emu, card, index, name);
#ifdef CONFIG_SND_OSSEMUL
	snd_emux_init_seq_oss(emu, name);
#endif

#ifdef CONFIG_PROC_FS
	snd_emux_proc_init(emu, card, index);
#endif
	return 0;
}


/*
 */
int snd_emux_free(snd_emux_t *emu)
{
	if (! emu)
		return -EINVAL;

#ifdef CONFIG_PROC_FS
	snd_emux_proc_free(emu);
#endif
#ifdef CONFIG_SND_OSSEMUL
	snd_emux_detach_seq_oss(emu);
#endif
	snd_emux_detach_seq(emu);

	if (emu->sflist)
		snd_sf_free(emu->sflist);

	if (emu->voices)
		snd_kfree(emu->voices);

	if (emu->name)
		snd_kfree(emu->name);

	snd_magic_kfree(emu);
	return 0;
}


/*
 *  INIT part
 */

#ifdef MODULE

int __init init_module(void)
{
	return 0;
}

void __exit cleanup_module(void)
{
}

#endif
