/*
 *   Generic MIDI synth driver for ALSA sequencer
 *   Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *                         Jaroslav Kysela <perex@suse.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
 
/* 
Possible options for midisynth module:
	- automatic opening of midi ports on first received event or subscription
	  (close will be performed when client leaves)
*/


#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/rawmidi.h"
#include "../../include/seq_kernel.h"
#include "../../include/seq_device.h"
#include "../../include/seq_midi_event.h"

MODULE_AUTHOR("Frank van de Pol <frank@vande-pol.demon.nl>");
MODULE_DESCRIPTION("Advanced Linux Sound Architecture sequencer MIDI synth.");
MODULE_SUPPORTED_DEVICE("sound");
int output_buffer_size = PAGE_SIZE;
MODULE_PARM(output_buffer_size, "i");
MODULE_PARM_DESC(output_buffer_size, "Output buffer size in bytes.");
int input_buffer_size = PAGE_SIZE;
MODULE_PARM(input_buffer_size, "i");
MODULE_PARM_DESC(input_buffer_size, "Input buffer size in bytes.");

/* data for this midi synth driver */
typedef struct {
	snd_card_t *card;
	int device;
	snd_rawmidi_t *input_rmidi;
	snd_rawmidi_t *output_rmidi;
	int seq_client;
	int seq_port;
	snd_midi_event_t *parser;
} seq_midisynth_t;

typedef struct {
	int seq_client;
	seq_midisynth_t *ports[SND_RAWMIDI_DEVICES];
} seq_midisynth_client_t;

static seq_midisynth_client_t *synths[SND_CARDS];
static DECLARE_MUTEX(register_mutex);

/* fill standard header data, source port are filled in */
static void snd_seq_midi_setheader(snd_seq_event_t * ev, int port)
{
	ev->flags = SND_SEQ_EVENT_LENGTH_FIXED;
	ev->source.port = port;
	ev->dest.client = SND_SEQ_ADDRESS_SUBSCRIBERS;
}

/* handle rawmidi input event (MIDI v1.0 stream) */
static void snd_midi_input_event(snd_rawmidi_t * rmidi)
{
	seq_midisynth_t *msynth;
	snd_seq_event_t ev;
	char buf[16], *pbuf;
	long res, count;

	if (rmidi == NULL)
		return;
	msynth = (seq_midisynth_t *) rmidi->chn[SND_RAWMIDI_CHANNEL_INPUT].private_data;
	if (msynth == NULL)
		return;
	memset(&ev, 0, sizeof(ev));
	snd_seq_midi_setheader(&ev, msynth->seq_port);
	while (rmidi->chn[SND_RAWMIDI_CHANNEL_INPUT].used > 0) {
		res = snd_rawmidi_kernel_read(rmidi, buf, sizeof(buf));
		if (res <= 0)
			continue;
		if (msynth->parser == NULL)
			continue;
		pbuf = buf;
		while (res > 0) {
			count = snd_midi_event_encode(msynth->parser, pbuf, res, &ev);
			if (count < 0)
				break;
			pbuf += count;
			res -= count;
			if (ev.type != SND_SEQ_EVENT_NONE) {
				snd_seq_kernel_client_dispatch(msynth->seq_client, &ev, 1, 0);
				/* release sysex data */
				if (snd_seq_ev_is_variable(&ev) && ev.data.ext.ptr)
					snd_seq_ext_free(ev.data.ext.ptr, ev.data.ext.len);
				/* clear event and reset header */
				memset(&ev, 0, sizeof(ev));
				snd_seq_midi_setheader(&ev, msynth->seq_port);
			}
		}
	}
}

static int dump_midi(snd_rawmidi_t *rmidi, const char *buf, int count)
{
	snd_debug_check(rmidi == NULL || buf == NULL, -EINVAL);
	if (rmidi->chn[SND_RAWMIDI_CHANNEL_OUTPUT].size - rmidi->chn[SND_RAWMIDI_CHANNEL_OUTPUT].used < count)
		return -ENOMEM;
	if (snd_rawmidi_kernel_write(rmidi, buf, count) < count)
		return -EINVAL;
	return 0;
}

static int event_process_midi(snd_seq_event_t * ev, int direct,
			      void *private_data, int atomic, int hop)
{
	seq_midisynth_t *msynth = (seq_midisynth_t *) private_data;
	unsigned char msg[10];	/* buffer for constructing midi messages */
	snd_rawmidi_t *rmidi;
	int res;

	snd_debug_check(msynth == NULL, -EINVAL);
	rmidi = msynth->output_rmidi;
	snd_debug_check(rmidi == NULL, -EINVAL);
	if (ev->type == SND_SEQ_EVENT_SYSEX) {	/* special case, to save space */
		if ((ev->flags & SND_SEQ_EVENT_LENGTH_MASK) != SND_SEQ_EVENT_LENGTH_VARIABLE) {
			/* invalid event */
			snd_printd("seq_midi: invalid sysex event flags = 0x%x\n", ev->flags);
			return 0;
		}
		if ((res = dump_midi(rmidi, (char *)ev->data.ext.ptr, ev->data.ext.len)) < 0) {
			snd_midi_event_reset_decode(msynth->parser);
			return res;
		}
	} else {
		if (msynth->parser == NULL)
			return -EIO;
		res = snd_midi_event_decode(msynth->parser, msg, sizeof(msg), ev);
		if (res < 0)
			return res;
		if ((res = dump_midi(rmidi, msg, res)) < 0) {
			snd_midi_event_reset_decode(msynth->parser);
			return res;
		}
	}
	return 0;
}


static seq_midisynth_t *snd_seq_midisynth_new(snd_card_t *card, int device)
{
	seq_midisynth_t *msynth;
	
	msynth = snd_kcalloc(sizeof(seq_midisynth_t), GFP_KERNEL);
	if (msynth == NULL) {
		snd_printd("malloc failed for snd_seq_midisynth_new()\n");
		return NULL;
	}
	
	if (snd_midi_event_new(&msynth->parser) < 0) {
		snd_kfree(msynth);
		return NULL;
	}
	
	msynth->card = card;
	msynth->device = device;

	return msynth;
}

/* open associated midi device for input */
static int midisynth_subscribe(void *private_data, snd_seq_port_subscribe_t *info)
{
	int err;
	seq_midisynth_t *msynth = (seq_midisynth_t *)private_data;
	snd_rawmidi_params_t params;

	MOD_INC_USE_COUNT;

	if ((info->midi_channels && info->midi_channels != 16) ||
	    info->midi_voices || info->synth_voices) {
	    	MOD_DEC_USE_COUNT;
	    	return -EINVAL;
	}

	/* open midi port */
	if ((err = snd_rawmidi_kernel_open(msynth->card->number, msynth->device, SND_RAWMIDI_LFLG_INPUT, &msynth->input_rmidi)) < 0) {
		snd_printd("midi input open failed!!!\n");
		MOD_DEC_USE_COUNT;
		return err;
	}
	memset(&params, 0, sizeof(params));
	params.min = 1;
	params.size = input_buffer_size;
	if ((err = snd_rawmidi_input_params(msynth->input_rmidi, &params)) < 0) {
		snd_rawmidi_kernel_release(msynth->input_rmidi, SND_RAWMIDI_LFLG_INPUT);
		MOD_DEC_USE_COUNT;
		return err;
	}
	msynth->input_rmidi->chn[SND_RAWMIDI_CHANNEL_INPUT].event = snd_midi_input_event;
	msynth->input_rmidi->chn[SND_RAWMIDI_CHANNEL_INPUT].private_data = msynth;
	snd_rawmidi_kernel_read(msynth->input_rmidi, NULL, 0);
	return 0;
}

/* close associated midi device for input */
static int midisynth_unsubscribe(void *private_data, snd_seq_port_subscribe_t *info)
{
	int err;
	seq_midisynth_t *msynth = (seq_midisynth_t *)private_data;

	snd_debug_check(msynth->input_rmidi == NULL, -EINVAL);
	err = snd_rawmidi_kernel_release(msynth->input_rmidi, SND_RAWMIDI_LFLG_INPUT);
	msynth->input_rmidi = NULL;
	MOD_DEC_USE_COUNT;
	return err;
}

/* open associated midi device for output */
static int midisynth_use(void *private_data, snd_seq_port_subscribe_t *info)
{
	int err;
	seq_midisynth_t *msynth = (seq_midisynth_t *)private_data;
	snd_rawmidi_params_t params;

	MOD_INC_USE_COUNT;

	if ((info->midi_channels && info->midi_channels != 16) ||
	    info->midi_voices || info->synth_voices) {
	    	MOD_DEC_USE_COUNT;
	    	return -EINVAL;
	}

	/* open midi port */
	if ((err = snd_rawmidi_kernel_open(msynth->card->number, msynth->device, SND_RAWMIDI_LFLG_OUTPUT, &msynth->output_rmidi)) < 0) {
		snd_printd("midi output open failed!!!\n");
		MOD_DEC_USE_COUNT;
		return err;
	}
	memset(&params, 0, sizeof(params));
	params.room = 1;
	params.max = output_buffer_size - 1;
	params.size = output_buffer_size;
	if ((err = snd_rawmidi_output_params(msynth->output_rmidi, &params)) < 0) {
		snd_rawmidi_kernel_release(msynth->output_rmidi, SND_RAWMIDI_LFLG_OUTPUT);
		MOD_DEC_USE_COUNT;
		return err;
	}
	return 0;
}

/* close associated midi device for output */
static int midisynth_unuse(void *private_data, snd_seq_port_subscribe_t *info)
{
	int err;
	seq_midisynth_t *msynth = (seq_midisynth_t *)private_data;
	unsigned char buf = 0xfe;/*active sensing midi event*/

	snd_debug_check(msynth->output_rmidi == NULL, -EINVAL);
	/* sending single active sensing message to shut the device up */
	snd_rawmidi_kernel_write(msynth->output_rmidi, &buf, 1);
	snd_rawmidi_flush_output(msynth->output_rmidi);
	err = snd_rawmidi_kernel_release(msynth->output_rmidi, SND_RAWMIDI_LFLG_OUTPUT);
	msynth->output_rmidi = NULL;
	MOD_DEC_USE_COUNT;
	return err;
}


static void snd_seq_midisynth_delete(seq_midisynth_t **ms)
{
	seq_midisynth_t *msynth;
	snd_seq_port_info_t port;
	
	if (ms == NULL) {
		snd_printd("snd_seq_midisynth_delete called with NULL\n");
		return;
	}
	msynth = *ms;
	*ms = NULL;

	if (msynth == NULL)
		return;

	/* delete port */
	memset(&port, 0, sizeof(port));
	port.client = msynth->seq_client;
	port.port = msynth->seq_port;
	snd_seq_kernel_client_ctl(port.client, SND_SEQ_IOCTL_DELETE_PORT, &port);

	if (msynth->parser)
		snd_midi_event_free(msynth->parser);
	snd_kfree(msynth);
}


/* register new midi synth port */
int
snd_seq_midisynth_register_port(snd_seq_device_t *dev)
{
	seq_midisynth_client_t *client;
	seq_midisynth_t *msynth;
	snd_seq_port_info_t port;
	snd_rawmidi_info_t info;
	int newclient = 0;
	snd_seq_client_callback_t callbacks;
	snd_seq_port_callback_t pcallbacks;
	snd_seq_client_info_t inf;
	snd_card_t *card = dev->card;
	int device = dev->device;

	snd_debug_check(card == NULL || device < 0 || device >= SND_RAWMIDI_DEVICES, -EINVAL);
	down(&register_mutex);
	client = synths[card->number];
	if (client == NULL) {
		newclient = 1;
		client = snd_kcalloc(sizeof(seq_midisynth_client_t), GFP_KERNEL);
		if (client == NULL) {
			up(&register_mutex);
			return -ENOMEM;
		}
		memset(&callbacks, 0, sizeof(callbacks));
		callbacks.private_data = client;
		callbacks.allow_input = callbacks.allow_output = 1;
		client->seq_client = snd_seq_create_kernel_client(card, 0, &callbacks);
		if (client->seq_client < 0) {
			snd_kfree(client);
			up(&register_mutex);
			return -ENOMEM;
		}
		/* set our client name */
		memset(&inf,0,sizeof(snd_seq_client_info_t));
		inf.client = client->seq_client;
		inf.type = KERNEL_CLIENT;
		sprintf(inf.name, "External MIDI %i", card->number);
		strcpy(inf.group, SND_SEQ_GROUP_DEVICE);
		snd_seq_kernel_client_ctl(client->seq_client, SND_SEQ_IOCTL_SET_CLIENT_INFO, &inf);
	}
	client->ports[device] = msynth = snd_seq_midisynth_new(card, device);
	if (client->ports[device] == NULL || snd_rawmidi_kernel_info(card->number, device, &info)<0) {
		if (newclient) {
			snd_seq_delete_kernel_client(client->seq_client);
			snd_kfree(client);
		}
		up(&register_mutex);
		return -EBUSY;
	}
	msynth->seq_client = client->seq_client;
	
	/* declare port */
	memset(&port, 0, sizeof(port));
	port.client = client->seq_client;
	if (dev->name)
		strcpy(port.name, dev->name);
	strcpy(port.group, SND_SEQ_GROUP_DEVICE);
	if (info.flags & SND_RAWMIDI_INFO_OUTPUT)
		port.capability |= SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SYNC_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE;
	if (info.flags & SND_RAWMIDI_INFO_INPUT)
		port.capability |= SND_SEQ_PORT_CAP_READ | SND_SEQ_PORT_CAP_SYNC_READ | SND_SEQ_PORT_CAP_SUBS_READ;
	if (info.flags & SND_RAWMIDI_INFO_DUPLEX)
		port.capability |= SND_SEQ_PORT_CAP_DUPLEX;
	port.type = SND_SEQ_PORT_TYPE_MIDI_GENERIC;
	port.midi_channels = 16;
	memset(&pcallbacks, 0, sizeof(pcallbacks));
	pcallbacks.private_data = msynth;
	pcallbacks.subscribe = midisynth_subscribe;
	pcallbacks.unsubscribe = midisynth_unsubscribe;
	pcallbacks.use = midisynth_use;
	pcallbacks.unuse = midisynth_unuse;
	pcallbacks.event_input = event_process_midi;
	port.kernel = &pcallbacks;
	if (snd_seq_kernel_client_ctl(client->seq_client, SND_SEQ_IOCTL_CREATE_PORT, &port)<0) {
		snd_seq_midisynth_delete(&client->ports[device]);
		if (newclient) {
			snd_seq_delete_kernel_client(client->seq_client);
			snd_kfree(client);
		}
		up(&register_mutex);
		return -ENOMEM;
	}

	msynth->seq_port = port.port;
	if (newclient)
		synths[card->number] = client;
	up(&register_mutex);
	return 0;	/* success */
}

/* release midi synth port */
int
snd_seq_midisynth_unregister_port(snd_seq_device_t *dev)
{
	seq_midisynth_client_t *client;
	snd_card_t *card = dev->card;
	int device = dev->device;
	int c;
	
	down(&register_mutex);
	client = synths[card->number];
	if (client == NULL) {
		up(&register_mutex);
		return -ENODEV;
	}
	snd_seq_event_port_detach(client->seq_client, client->ports[device]->seq_port);
	snd_seq_midisynth_delete(&client->ports[device]);
	for (c = 0; c < SND_RAWMIDI_DEVICES; c++)
		if (client->ports[device])
			break;
	if (c >= SND_RAWMIDI_DEVICES) {
		snd_seq_delete_kernel_client(client->seq_client);
		synths[card->number] = NULL;
		snd_kfree(client);
	}
	up(&register_mutex);
	return 0;
}


#ifdef MODULE
int __init init_module(void)
#else
int __init alsa_seq_midi_init(void)
#endif
{
	static snd_seq_dev_ops_t ops = {
		snd_seq_midisynth_register_port,
		snd_seq_midisynth_unregister_port,
	};
	memset(&synths, 0, sizeof(synths));
	snd_seq_device_register_driver(SND_SEQ_DEV_ID_MIDISYNTH, &ops, 0);
	return 0;
}

#ifdef MODULE

void __exit cleanup_module(void)
{
	snd_seq_device_unregister_driver(SND_SEQ_DEV_ID_MIDISYNTH);
}

#endif
