/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Copyright (c) by Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *  EMU10K1 memory page allocation (PTB area)
 *  EMU10K1 chip can handle only 32MByte of the memory at the same time.
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../include/driver.h"
#include "../../include/emu10k1.h"

/* page arguments of these two macros are Emu page (4096 bytes), not like
 * aligned pages in others
 */
#define __set_ptb_phys(emu,page,ptr)	((emu)->ptb_pages[page] = (ptr << 1) | (page))
#define __ptb_to_ptr(emu,pg)	(void*)phys_to_virt(((emu)->ptb_pages[pg] >> 1) & ~(EMUPAGESIZE - 1))

#if PAGE_SIZE == 4096
/* page size == EMUPAGESIZE */
#define ALIGN_PAGE_SIZE		PAGE_SIZE	/* minimum page size for allocation */
#define MAX_ALIGN_PAGES		MAXPAGES	/* maxmium aligned pages */
/* fill PTB entrie(s) corresponding to page with ptr */
#define set_ptb_phys(emu,page,ptr)	__set_ptb_phys(emu,page,ptr)
/* fill PTB entrie(s) corresponding to page with silence pointer */
#define set_silent_ptb(emu,page)	set_ptb_virt(emu, page, emu->silent_page)
/* get aligned page from offset address */
#define get_aligned_page(offset)	((offset) >> 12)
/* get offset address from aligned page */
#define aligned_page_offset(page)	((page) << 12)
/* get buffer address from aligned page */
#define page_to_ptr(emu,page)	__ptb_to_ptr(emu, page)

#elif PAGE_SIZE == 8192
/* page size == EMUPAGESIZE x 2*/
#define ALIGN_PAGE_SIZE		PAGE_SIZE
#define MAX_ALIGN_PAGES		(MAXPAGES / 2)
#define get_aligned_page(offset)	((offset) >> 13)
#define aligned_page_offset(page)	((page) << 13)
#define page_to_ptr(emu,page)	__ptb_to_ptr(emu, (page) << 1)

/* fill PTB entries -- we need to fill two entries */
static inline void set_ptb_phys(emu10k1_t *emu, int page, unsigned long ptr)
{
	page <<= 1;
	__set_ptb_phys(emu, page, ptr);
	page++;
	ptr += EMUPAGESIZE;
	__set_ptb_phys(emu, page, ptr);
}
static inline void set_silent_ptb(emu10k1_t *emu, int page)
{
	unsigned long ptr = (unsigned long)virt_to_phys(emu->silent_page);
	page <<= 1;
	__set_ptb_phys(emu, page, ptr);
	page++;
	/* do not increment ptr */
	__set_ptb_phys(emu, page, ptr);
}

#else
/* arbitrary size -- so far, this doesn't exist on Linux, though.. */
#define UNIT_PAGES		(PAGE_SIZE / EMUPAGESIZE)
#define ALIGN_PAGE_SIZE		(EMUPAGESIZE * UNIT_PAGES)
#define MAX_ALIGN_PAGES		(MAXPAGES / UNIT_PAGES)
/* Note: if alignment doesn't match to the maximum size, the last few blocks
 * become unusable.  To use such blocks, you'll need to check the validity
 * of accessing page in set_ptb_phys and set_silent_ptb.  search_empty()
 * should also check it, too.
 */
#define get_aligned_page(offset)	((offset) / ALIGN_PAGE_SIZE)
#define aligned_page_offset(page)	((page) * ALIGN_PAGE_SIZE)
#define page_to_ptr(emu,page)	__ptb_to_ptr(emu, (page) * UNIT_PAGES)

/* fill PTB entries -- UNIT_PAGES entries must be filled */
static inline void set_ptb_phys(emu10k1_t *emu, int page, unsigned long ptr)
{
	int i;
	page *= UNIT_PAGES;
	for (i = 0; i < UNIT_PAGES; i++, page++) {
		__set_ptb_phys(emu, page, ptr);
		ptr += EMUPAGESIZE;
	}
}
static inline void set_silent_ptb(emu10k1_t *emu, int page)
{
	unsigned long ptr = (unsigned long)virt_to_phys(emu->silent_page);
	int i;
	page *= UNIT_PAGES;
	for (i = 0; i < UNIT_PAGES; i++, page++)
		__set_ptb_phys(emu, page, ptr);
}

#endif /* PAGE_SIZE */

/* calculate buffer pointer from offset address */
inline static void *offset_ptr(emu10k1_t *emu, int offset)
{
	char *ptr;
	ptr = page_to_ptr(emu, get_aligned_page(offset));
	ptr += offset % ALIGN_PAGE_SIZE;
	return (void*)ptr;
}

/* fill PTB entries with virtual address ptr */
#define set_ptb_virt(emu,page,ptr)	set_ptb_phys(emu, page, (unsigned int)virt_to_phys(ptr))

/* first and last (aligned) pages of memory block */
#define firstpg(blk)	(((snd_emu10k1_memblk_arg_t*)snd_emux_memblk_argptr(blk))->first_page)
#define lastpg(blk)	(((snd_emu10k1_memblk_arg_t*)snd_emux_memblk_argptr(blk))->last_page)

/*
 * search empty pages which may contain given size
 */
static snd_emux_memblk_t *
search_empty(snd_emux_memhdr_t *hdr, int size)
{
	snd_emux_memblk_t *blk, *prev;
	int page, psize;

	psize = get_aligned_page(size + ALIGN_PAGE_SIZE -1);
	prev = NULL;
	page = 0;
	for (blk = hdr->block; blk; prev = blk, blk = blk->next) {
		if (page + psize <= firstpg(blk))
			goto __found_pages;
		page = lastpg(blk) + 1;
	}
	if (page + psize > MAX_ALIGN_PAGES)
		return NULL;

__found_pages:
	/* create a new memory block */
	blk = __snd_emux_memblk_new(hdr, psize * ALIGN_PAGE_SIZE, prev);
	if (blk == NULL)
		return NULL;
	blk->offset = aligned_page_offset(page); /* set aligned offset */
	firstpg(blk) = page;
	lastpg(blk) = page + psize - 1;
	return blk;
}


/*
 * check if the given pointer is valid for pages
 */
static int is_valid_page(void *pages)
{
	unsigned long ptr = (unsigned long)virt_to_phys(pages);
	if (ptr & ~0x7fffffffUL) {
		snd_printk("emu10k1: max memory size is 2GB!!\n");
		return 0;
	}
	if (ptr & (EMUPAGESIZE-1)) {
		snd_printk("emu10k1: page is not aligned\n");
		return 0;
	}
	return 1;
}

/*
 * page allocation for DMA
 */
snd_emux_memblk_t *
snd_emu10k1_alloc_pages(emu10k1_t *emu, void *pages, unsigned long size)
{
	unsigned long ptr;
	snd_emux_memhdr_t *hdr;
	snd_emux_memblk_t *blk;
	int page;

	snd_debug_check(emu == NULL, NULL);
	snd_debug_check(size <= 0 || size >= MAXPAGES * EMUPAGESIZE, NULL);
	hdr = emu->memhdr;
	snd_debug_check(hdr == NULL, NULL);

	if (! is_valid_page(pages))
		return NULL;

	down(&hdr->block_mutex);
	blk = search_empty(hdr, size);
	if (blk == NULL) {
		up(&hdr->block_mutex);
		return NULL;
	}
	/* set PTB entries */
	ptr = (unsigned long)virt_to_phys(pages);
	for (page = firstpg(blk); page <= lastpg(blk); page++) {
		set_ptb_phys(emu, page, ptr);
		ptr += ALIGN_PAGE_SIZE;
	}
	up(&hdr->block_mutex);
	return blk;
}


/*
 * release DMA buffer from page table
 */
int snd_emu10k1_free_pages(emu10k1_t *emu, snd_emux_memblk_t *blk)
{
	snd_emux_memhdr_t *hdr;
	snd_emux_memblk_t *prev;
	int page;

	snd_debug_check(emu == NULL, -EINVAL);
	snd_debug_check(blk == NULL, -EINVAL);

	hdr = emu->memhdr;
	down(&hdr->block_mutex);
	if (__snd_emux_mem_find_prev(hdr, blk, &prev)) {
		up(&hdr->block_mutex);
		snd_printk("emu10k1: can't find memory block (addr=0x%x, size=0x%x)\n", blk->offset, blk->size);
		return -EINVAL;
	}
	/* reset PTB entries */
	for (page = firstpg(blk); page <= lastpg(blk); page++)
		set_silent_ptb(emu, page);
	/* free memory block */
	__snd_emux_mem_free(hdr, blk, prev);
	up(&hdr->block_mutex);
	return 0;
}


/*----------------------------------------------------------------
 * memory allocation using multiple pages (for synth)
 *----------------------------------------------------------------
 * Unlike the DMA allocation above, non-contiguous pages are
 * assigned to PTB.
 *----------------------------------------------------------------*/

/*
 */
static int synth_alloc_pages(emu10k1_t *hw, snd_emux_memblk_t *blk, snd_emux_memblk_t *prev);
static int synth_free_pages(emu10k1_t *hw, snd_emux_memblk_t *blk, snd_emux_memblk_t *prev);

/*
 * allocate a synth sample area
 */
snd_emux_memblk_t *
snd_emu10k1_synth_alloc(emu10k1_t *hw, unsigned int size)
{
	snd_emux_memblk_t *blk, *prev;
	snd_emux_memhdr_t *hdr = hw->memhdr; 

	down(&hdr->block_mutex);
	blk = __snd_emux_mem_alloc(hdr, size, &prev);
	if (blk == NULL) {
		up(&hdr->block_mutex);
		return NULL;
	}
	if (synth_alloc_pages(hw, blk, prev)) {
		__snd_emux_mem_free(hdr, blk, prev);
		up(&hdr->block_mutex);
		return NULL;
	}
	up(&hdr->block_mutex);
	return blk;
}


/*
 * free a synth sample area
 */
int
snd_emu10k1_synth_free(emu10k1_t *hw, snd_emux_memblk_t *blk)
{
	snd_emux_memblk_t *prev;
	snd_emux_memhdr_t *hdr = hw->memhdr; 

	down(&hdr->block_mutex);
	if (__snd_emux_mem_find_prev(hdr, blk, &prev)) {
		up(&hdr->block_mutex);
		return -EINVAL;
	}
	synth_free_pages(hw, blk, prev);
	 __snd_emux_mem_free(hdr, blk, prev);
	up(&hdr->block_mutex);
	return 0;
}


/*
 * reset PTB entry and free kernel page
 */
static void clear_ptb(emu10k1_t *emu, int page)
{
	void *ptr = page_to_ptr(emu, page);
	set_silent_ptb(emu, page);
	snd_free_pages(ptr, ALIGN_PAGE_SIZE);
}

/*
 * allocate kernel pages and assign them to PTB
 */
static int synth_alloc_pages(emu10k1_t *hw, snd_emux_memblk_t *blk, snd_emux_memblk_t *prev)
{
	int page, first_page, last_page;
	void *ptr;

	first_page = get_aligned_page(blk->offset);
	firstpg(blk) = first_page;
	last_page = get_aligned_page(blk->offset + blk->size - 1);
	lastpg(blk) = last_page;
	/* check new allocation range */
	if (prev && lastpg(prev) == first_page)
		first_page++;  /* first page was already allocated */
	if (blk->next && firstpg(blk->next) == last_page)
		last_page--; /* last page was already allocated */
	/* allocate a kernel page for each Emu page -
	 * fortunately Emu page size and kernel PAGE_SIZE is identical!
	 */
	for (page = first_page; page <= last_page; page++) {
		ptr = snd_malloc_pages(ALIGN_PAGE_SIZE, NULL, 0);
		if (ptr == NULL || ! is_valid_page(ptr))
			goto __fail;
		set_ptb_virt(hw, page, ptr);
	}
	return 0;

__fail:
	/* release allocated pages */
	last_page = page - 1;
	for (page = first_page; page <= last_page; page++)
		clear_ptb(hw, page);

	return -ENOMEM;
}

/*
 * free pages
 */
static int synth_free_pages(emu10k1_t *emu, snd_emux_memblk_t *blk, snd_emux_memblk_t *prev)
{
	int page, first_page, last_page;

	first_page = firstpg(blk);
	if (prev && lastpg(prev) == first_page)
		first_page++; /* first_page is still used */
	last_page = lastpg(blk);
	if (blk->next && firstpg(blk->next) == last_page)
		last_page--; /* last_page is still used */
	for (page = first_page; page <= last_page; page++)
		clear_ptb(emu, page);

	return 0;
}

/*
 * bzero(blk + offset, size)
 */
int snd_emu10k1_synth_bzero(emu10k1_t *emu, snd_emux_memblk_t *blk, int offset, int size)
{
	int page, nextofs, end_offset, temp, temp1;

	offset += blk->offset;
	end_offset = offset + size;
	page = get_aligned_page(offset) + 1;
	do {
		nextofs = aligned_page_offset(page);
		temp = nextofs - offset;
		temp1 = end_offset - offset;
		if (temp1 < temp)
			temp = temp1;
		memset(offset_ptr(emu, offset), 0, temp);
		offset = nextofs;
		page++;
	} while (offset < end_offset);
	return 0;
}

/*
 * copy_from_user(blk + offset, data, size)
 */
int snd_emu10k1_synth_copy_from_user(emu10k1_t *emu, snd_emux_memblk_t *blk, int offset, const char *data, int size)
{
	int page, nextofs, end_offset, temp, temp1;

	offset += blk->offset;
	end_offset = offset + size;
	page = get_aligned_page(offset) + 1;
	do {
		nextofs = aligned_page_offset(page);
		temp = nextofs - offset;
		temp1 = end_offset - offset;
		if (temp1 < temp)
			temp = temp1;
		if (copy_from_user(offset_ptr(emu, offset), data, temp))
			return -EFAULT;
		offset = nextofs;
		data += temp;
		page++;
	} while (offset < end_offset);
	return 0;
}

EXPORT_SYMBOL(snd_emu10k1_synth_alloc);
EXPORT_SYMBOL(snd_emu10k1_synth_free);
EXPORT_SYMBOL(snd_emu10k1_synth_bzero);
EXPORT_SYMBOL(snd_emu10k1_synth_copy_from_user);
