/*
 *  Copyright (C) 2000 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *  Routines for control of EMU10K1 WaveTable synth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define SND_MAIN_OBJECT_FILE
#include "emu10k1_synth_local.h"

MODULE_AUTHOR("Takashi Iwai");
MODULE_PARM(snd_seq_ports, "i");
MODULE_PARM_DESC(snd_seq_ports, "number of sequencer ports to be created");
MODULE_PARM(max_voices, "i");
MODULE_PARM_DESC(max_voices, "maximum number of voices for WaveTable");

int snd_seq_ports = 4;
int max_voices = 62;	/* two voices are kept for PCM */

/*
 * create a new hardware dependent device for Emu8000
 */
int snd_emu10k1_synth_new_device(snd_seq_device_t *dev)
{
	snd_emux_t *emu;
	snd_emu10k1_synth_arg_t *arg;

	if (snd_emux_new(&emu) < 0)
		return -ENOMEM;

	snd_emu10k1_ops_setup(emu);
	arg = SND_SEQ_DEVICE_ARGPTR(dev);
	emu->hw = arg->hwptr;
	if (max_voices < 1)
		max_voices = 1;
	else if (max_voices > 64)
		max_voices = 64;
	emu->max_voices = max_voices;
	emu->num_ports = snd_seq_ports;
	emu->pitch_shift = -501;
	emu->memhdr = arg->hwptr->memhdr;

	if (snd_emux_register(emu, dev->card, arg->index, "Emu10k1 WaveTable") < 0) {
		snd_emux_free(emu);
		return -ENOMEM;
	}

	dev->driver_data = emu;

	return 0;
}

int snd_emu10k1_synth_delete_device(snd_seq_device_t *dev)
{
	snd_emux_t *emu;
	emu = snd_magic_cast(snd_emux_t, dev->driver_data, -EINVAL);
	snd_emux_free(emu);
	return 0;
}


/*
 *  INIT part
 */

#ifdef MODULE
int __init init_module(void)
#else
int __init alsa_emu10k1_synth_init(void)
#endif
{
	
	static snd_seq_dev_ops_t ops = {
		snd_emu10k1_synth_new_device,
		snd_emu10k1_synth_delete_device,
	};
	return snd_seq_device_register_driver(SND_SEQ_DEV_ID_EMU10K1_SYNTH, &ops, sizeof(snd_emu10k1_synth_arg_t));
}

#ifdef MODULE

void __exit cleanup_module(void)
{
	snd_seq_device_unregister_driver(SND_SEQ_DEV_ID_EMU10K1_SYNTH);
}

#endif
