/*
 *  Midi Sequencer interface routines.
 *
 *  Copyright (C) 1999 Steve Ratcliffe
 *  Copyright (c) 1999 by Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../../include/driver.h"
#include "../../include/emu8000.h"
#include "../../include/seq_midi_event.h"
#include "emu8000_port.h"
#include "emu8000_equalizer.h"


/*
 * module parameters
 */
int seqports = 4;
MODULE_PARM(seqports, "i");
MODULE_PARM_DESC(seqports, "number of sequencer ports to be opened");


/* Prototypes for static functions */
static void free_port(void *private);
static void snd_emu8000_init_port(emu8000_port_t *p);
static int emu8000_use(void *private, snd_seq_port_subscribe_t *info);
static int emu8000_unuse(void *private, snd_seq_port_subscribe_t *info);
static int get_client(snd_card_t *card, char *name);
static void create_effect_table(emu8000_port_t *p);
static void delete_effect_table(emu8000_port_t *p);
static void clear_effect_table(emu8000_port_t *p);

/*
 * MIDI emulation operators
 */
static snd_midi_op_t emu8000_ops = {
	snd_emu8000_note_on,
	snd_emu8000_note_off,
	snd_emu8000_key_press,
	snd_emu8000_note_terminate,
	snd_emu8000_control,
	snd_emu8000_nrpn,
	snd_emu8000_sysex,
};


/*
 * number of MIDI channels
 */
#define MIDI_CHANNELS		16

/*
 * type flags for MIDI sequencer port
 */
#define DEFAULT_MIDI_TYPE	(SND_SEQ_PORT_TYPE_MIDI_GENERIC |\
				 SND_SEQ_PORT_TYPE_MIDI_GM |\
				 SND_SEQ_PORT_TYPE_MIDI_GS |\
				 SND_SEQ_PORT_TYPE_MIDI_XG |\
				 SND_SEQ_PORT_TYPE_DIRECT_SAMPLE)

/*
 * Initialise the EMU8000 Synth by creating a client and registering
 * a series of ports.
 * Each of the ports will contain the 16 midi channels.  Applications
 * can connect to these ports to play midi data.
 */
int
snd_emu8000_init_seq(emu8000_t *emu)
{
	int  i;
	snd_seq_port_callback_t pinfo;

	if (EMU8000_CHECK(emu) < 0)
		return -ENODEV;

	emu->client = get_client(emu->card, "AWE Wave Table Synth");
	if (emu->client < 0) {
		snd_printk("emu8000: can't create client\n");
		return -ENODEV;
	}

	if (seqports < 0) {
		snd_printk("emu8000: seqports must be greater than zero\n");
		seqports = 1;
	} else if (seqports >= EMU8000_MAX_PORTS) {
		snd_printk("emu8000: too many ports."
			   "limited max. ports %d\n", EMU8000_MAX_PORTS);
		seqports = EMU8000_MAX_PORTS;
	}

	emu->used = 0;

	memset(&pinfo, 0, sizeof(pinfo));
	pinfo.use = emu8000_use;
	pinfo.unuse = emu8000_unuse;
	pinfo.event_input = snd_emu8000_event_input;

	for (i = 0; i < seqports; i++) {
		emu8000_port_t *p;
		char name[30];

		sprintf(name, "Emu8000 port %d", i);
		p = snd_emu8000_create_port(emu, name, MIDI_CHANNELS,
					    0, &pinfo);
		if (p == NULL) {
			snd_printk("emu8000: can't create port\n");
			return -ENOMEM;
		}

		p->port_mode =  PORT_MODE_MIDI;
		snd_emu8000_init_port(p);
		emu->ports[i] = p->chset.port;
	}

	return 0;
}


/*
 * Detach from the ports that were set up for this synthesizer and
 * destroy the kernel client.
 */
void
snd_emu8000_detach_seq(emu8000_t *emu)
{
	if (emu == NULL)
		return;

	if (emu->voices)
		snd_emu8000_terminate_all(emu);
		
	down(&emu->register_mutex);

	if (emu->client >= 0) {
		snd_seq_delete_kernel_client(emu->client);
		emu->client = -1;
	}

	up(&emu->register_mutex);
}


/*
 * create a sequencer port and channel_set
 */

emu8000_port_t *
snd_emu8000_create_port(emu8000_t *emu, char *name,
			int max_channels, int oss_port,
			snd_seq_port_callback_t *callback)
{
	emu8000_port_t *p;
	int i, type, cap;

	/* Allocate structures for this channel */
	if ((p = snd_kcalloc(sizeof(*p), GFP_KERNEL)) == NULL) {
		snd_printk("emu8000: no memory\n");
		return NULL;
	}
	p->chset.channels = snd_kcalloc(max_channels * sizeof(snd_midi_channel_t), GFP_KERNEL);
	if (p->chset.channels == NULL) {
		snd_printk("emu8000: no memory\n");
		snd_kfree(p);
		return NULL;
	}
	for (i = 0; i < max_channels; i++)
		p->chset.channels[i].number = i;
	p->chset.private_data = p;
	p->chset.max_channels = max_channels;
	p->emu = emu;
	p->chset.client = emu->client;
	create_effect_table(p);

	callback->private_free = free_port;
	callback->private_data = p;

	cap = SND_SEQ_PORT_CAP_WRITE;
	if (oss_port) {
		type = SND_SEQ_PORT_TYPE_SPECIFIC;
	} else {
		type = DEFAULT_MIDI_TYPE;
		cap |= SND_SEQ_PORT_CAP_SUBS_WRITE;
	}

	p->chset.port = snd_seq_event_port_attach(emu->client, callback,
						  cap, type, name);

	return p;
}


/*
 * release memory block for port
 */
static void
free_port(void *private)
{
	emu8000_port_t *p = private;
	if (p) {
		delete_effect_table(p);
		if (p->chset.channels)
			snd_kfree(p->chset.channels);
		snd_kfree(p);
	}
}


#define DEFAULT_DRUM_FLAGS	(1<<9)

/*
 * initialize the port specific parameters
 */
static void
snd_emu8000_init_port(emu8000_port_t *p)
{
	p->drum_flags = DEFAULT_DRUM_FLAGS;
	p->volume_atten = 0;

	snd_emu8000_reset_port(p);
}


/*
 * reset port
 */

void
snd_emu8000_reset_port(emu8000_port_t *port)
{
	int i;
	emu8000_t *emu;

	emu = port->emu;
	if (EMU8000_CHECK(emu) < 0)
		return;

	/* stop all sounds */
	snd_emu8000_sounds_off_all(emu, port);

	snd_midi_channel_set_clear(&port->chset);

	clear_effect_table(port);

	/* set port specific control parameters */
	port->ctrls[EMU8000_MD_DEF_BANK] = 0;
	port->ctrls[EMU8000_MD_DEF_DRUM] = 0;
	port->ctrls[EMU8000_MD_REALTIME_PAN] = 1;

	for (i = 0; i < port->chset.max_channels; i++) {
		snd_midi_channel_t *chan = port->chset.channels + i;
		chan->drum_channel = ((port->drum_flags >> i) & 1) ? 1 : 0;
	}
}


/*
 * input sequencer event
 */
int
snd_emu8000_event_input(snd_seq_event_t *ev, int direct, void *private, int atomic, int hop)
{
	emu8000_port_t *port = private;
	snd_debug_check(port == NULL || ev == NULL, -EINVAL);

	snd_midi_process_event(&emu8000_ops, ev, &port->chset);

	return 0;
}


/*
 * increment usage count
 */
void
snd_emu8000_inc_count(emu8000_t *emu)
{
	emu->used++;
	MOD_INC_USE_COUNT;
	if (emu->card->use_inc)
		emu->card->use_inc(emu->card);
}


/*
 * decrease usage count
 */
void
snd_emu8000_dec_count(emu8000_t *emu)
{
	MOD_DEC_USE_COUNT;
	if (emu->card->use_dec)
		emu->card->use_dec(emu->card);
	emu->used--;
	if (emu->used <= 0)
		snd_emu8000_terminate_all(emu);
}


/*
 * Routine that is called upon a first use of a particular port
 */
static int
emu8000_use(void *private, snd_seq_port_subscribe_t *info)
{
	emu8000_port_t *p = private;
	emu8000_t *emu;

	if (p == NULL || (emu = p->emu) == NULL) {
		snd_printk("emu8000: illegal port use callback\n");
		return -EIO;
	}

	down(&emu->register_mutex);
	snd_emu8000_init_port(p);
	snd_emu8000_inc_count(emu);
	up(&emu->register_mutex);
	return 0;
}

/*
 * Routine that is called upon the last unuse() of a particular port.
 */
static int
emu8000_unuse(void *private, snd_seq_port_subscribe_t *info)
{
	emu8000_port_t *p = private;
	emu8000_t *emu;

	if (p == NULL || (emu = p->emu) == NULL) {
		snd_printk("emu8000: illegal port use callback\n");
		return -EIO;
	}

	down(&emu->register_mutex);
	snd_emu8000_sounds_off_all(emu, p);
	snd_emu8000_dec_count(emu);
	up(&emu->register_mutex);
	return 0;
}


/*
 * Create a sequencer client
 */
static int
get_client(snd_card_t *card, char *name)
{
	snd_seq_client_callback_t callbacks;
	snd_seq_client_info_t cinfo;
	int  client;
	int  n;

	memset(&callbacks, 0, sizeof(callbacks));
	callbacks.private_data = NULL;
	callbacks.allow_input = 1;
	callbacks.allow_output = 1;

	/* Find a free client, start from 1 as the MPU expects to use 0 */
	for (n = 1; n < 7; n++) {
		client = snd_seq_create_kernel_client(card, n, &callbacks);
		if (client >= 0)
			break;
	}
	if (client < 0)
		client = snd_seq_create_kernel_client(NULL, -1, &callbacks);
	if (client < 0)
		return client;

	memset(&cinfo, 0, sizeof(cinfo));
	cinfo.client = client;
	cinfo.type = KERNEL_CLIENT;
	sprintf(cinfo.name, "%s : %d", name, card->number);
	strcpy(cinfo.group, SND_SEQ_GROUP_DEVICE);
	snd_seq_kernel_client_ctl(client, SND_SEQ_IOCTL_SET_CLIENT_INFO, &cinfo);

	return client;
}


/*
 * effect table
 */
static void
create_effect_table(emu8000_port_t *p)
{
	int i;
	p->effect = snd_kcalloc(sizeof(emu8000_effect_table_t) * p->chset.max_channels, GFP_KERNEL);
	if (p->effect) {
		for (i = 0; i < p->chset.max_channels; i++)
			p->chset.channels[i].private = p->effect + i;
	} else {
		for (i = 0; i < p->chset.max_channels; i++)
			p->chset.channels[i].private = NULL;
	}
}

static void
delete_effect_table(emu8000_port_t *p)
{
	if (p->effect) {
		snd_kfree(p->effect);
		p->effect = NULL;
	}
}

static void
clear_effect_table(emu8000_port_t *p)
{
	if (p->effect) {
		memset(p->effect, 0, sizeof(emu8000_effect_table_t) * p->chset.max_channels);
	}
}

