/*
 *   ALSA sequencer FIFO
 *   Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../include/driver.h"
#include "seq_fifo.h"
#include "seq_lock.h"


/* FIFO */

/* create new fifo */
fifo_t *snd_seq_fifo_new(int poolsize)
{
	fifo_t *f;

	f = snd_kcalloc(sizeof(fifo_t), GFP_KERNEL);
	if (f == NULL) {
		snd_printd("malloc failed for snd_seq_fifo_new() \n");
		return NULL;
	}

	f->pool = snd_seq_pool_new(poolsize);
	if (f->pool == NULL) {
		snd_kfree(f);
		return NULL;
	}
	if (snd_seq_pool_init(f->pool) < 0) {
		snd_seq_pool_delete(&f->pool);
		snd_kfree(f);
		return NULL;
	}

	f->lock = SPIN_LOCK_UNLOCKED;
	init_MUTEX(&f->use_mutex);
	init_waitqueue_head(&f->input_sleep);

	f->head = NULL;
	f->tail = NULL;
	f->cells = 0;
	
	return f;
}

void snd_seq_fifo_delete(fifo_t **fifo)
{
	fifo_t *f;

	if (fifo == NULL) {
		snd_printd("oops: snd_seq_fifo_delete() called with NULL pointer to fifo\n");
		return;
	}
	f = *fifo;
	if (fifo == NULL) {
		snd_printd("oops: snd_seq_fifo_delete() called with NULL fifo\n");
		return;
	}
	*fifo = NULL;

	snd_seq_fifo_clear(f);

	/* wake up clients if any */
	if (waitqueue_active(&f->input_sleep))
		wake_up(&f->input_sleep);

	/* release resources...*/
	/*....................*/

	if (f->pool) {
		snd_seq_pool_done(f->pool);
		snd_seq_pool_delete(&f->pool);
	}
	
	snd_kfree(f);
}


/* clear queue */
void snd_seq_fifo_clear(fifo_t *f)
{
	snd_seq_event_cell_t *cell;

	/* clear overflow flag */
	f->overflow = 0;

	down(&f->use_mutex); /* down semaphore */
	/* drain the fifo */
	while ((cell = snd_seq_fifo_cell_out(f)) != NULL) {
		snd_seq_cell_free(cell);
	}
	up(&f->use_mutex); /* down semaphore */
}


/* enqueue event to fifo */
int snd_seq_fifo_event_in(fifo_t *f, snd_seq_event_t *event, int atomic)
{
	snd_seq_event_cell_t *cell;
	unsigned long flags;
	int err;

	if (f == NULL) {
		snd_printd("oops: snd_seq_fifo_cell_in() called with NULL fifo\n");
		return -EINVAL;
	}

	spin_lock_irqsave(&f->lock, flags);

	err = snd_seq_event_dup(f->pool, event, &cell, atomic);
	if (cell == NULL) {
		if (err == -EAGAIN)
			f->overflow++; /* set overflow flag */
		spin_unlock_irqrestore(&f->lock, flags);
		return err;
	}

	/* add new cell to tail of the fifo */
	if (f->tail != NULL)
		f->tail->ptr_l = cell;
	f->tail = cell;
	cell->ptr_l = NULL;

	/* register if this is the first element */
	if (f->head == NULL)
		f->head = cell;

	f->cells++;

	/* wakeup client */
	if (waitqueue_active(&f->input_sleep))
		wake_up(&f->input_sleep);

	spin_unlock_irqrestore(&f->lock, flags);

	return 0; /* success */
}

/* dequeue cell from fifo */
static snd_seq_event_cell_t *fifo_cell_out(fifo_t *f)
{
	snd_seq_event_cell_t *cell;

	if ((cell = f->head) != NULL) {
		f->head = cell->ptr_l;

		/* reset tail if this was the last element */
		if (f->tail == cell)
			f->tail = NULL;

		cell->ptr_l = NULL;
		f->cells--;
	}

	return cell;
}

/* dequeue cell from fifo on non-blocking mode */
snd_seq_event_cell_t *snd_seq_fifo_cell_out(fifo_t *f)
{
	snd_seq_event_cell_t *cell;
	unsigned long flags;

	spin_lock_irqsave(&f->lock, flags);
	cell = fifo_cell_out(f);
	spin_unlock_irqrestore(&f->lock, flags);

	return cell;
}

/* dequeue cell from fifo and lock fifo */
snd_seq_event_cell_t *snd_seq_fifo_locked_cell_out(fifo_t *f, int *err, int nonsleep)
{
	snd_seq_event_cell_t *cell;
	unsigned long flags;

	*err = 0;
	if (f == NULL) {
		snd_printd("oops: snd_seq_fifo_cell_out() called with NULL fifo\n");
		*err = -EINVAL;
		return NULL;
	}

	spin_lock_irqsave(&f->lock, flags);
	while (f->cells == 0) {
		if (nonsleep) {
			/* non-blocking - return immediately */
			spin_unlock_irqrestore(&f->lock, flags);
			return NULL;
		}

		snd_seq_sleep_in_lock(&f->input_sleep, &f->lock);

		if (signal_pending(current)) {
			*err = -EINTR;
			spin_unlock_irqrestore(&f->lock, flags);
			return NULL;
		}
	}

	cell = fifo_cell_out(f);

	spin_unlock_irqrestore(&f->lock, flags);

	down(&f->use_mutex); /* down semaphore */

	return cell;
}

/*
 * Put a cell back at the beginning of the queue
 */
void snd_seq_fifo_cell_putback(fifo_t *f, snd_seq_event_cell_t *cell)
{
	unsigned long flags;

	spin_lock_irqsave(&f->lock, flags);
	cell->ptr_l = f->head;
	f->head = cell;
	if (f->tail == NULL)
		f->tail = cell;
	f->cells++;
	spin_unlock_irqrestore(&f->lock, flags);
}

/* unlock FIFO */
void snd_seq_fifo_unlock(fifo_t *f)
{
	up(&f->use_mutex); /* up semaphore */
}


/* polling; return non-zero if queue is available */
int snd_seq_fifo_poll_wait(fifo_t *f, struct file *file, poll_table *wait)
{
	poll_wait(file, &f->input_sleep, wait);
	return (f->cells > 0);
}

/* change the size of pool; all old events are removed */
int snd_seq_fifo_resize(fifo_t *f, int poolsize)
{
	unsigned long flags;
	pool_t *newpool, *oldpool;
	snd_seq_event_cell_t *cell, *next, *oldhead;

	snd_debug_check(f == NULL || f->pool == NULL, -EINVAL);

	/* allocate new pool */
	newpool = snd_seq_pool_new(poolsize);
	if (newpool == NULL)
		return -ENOMEM;
	if (snd_seq_pool_init(newpool) < 0) {
		snd_seq_pool_delete(&newpool);
		return -ENOMEM;
	}

	down(&f->use_mutex);
	spin_lock_irqsave(&f->lock, flags);
	/* remember old pool */
	oldpool = f->pool;
	oldhead = f->head;
	/* exchange pools */
	f->pool = newpool;
	f->head = NULL;
	f->tail = NULL;
	f->cells = 0;
	/* NOTE: overflow flag is not cleared */
	spin_unlock_irqrestore(&f->lock, flags);
	up(&f->use_mutex);

	/* release cells in old pool */
	for (cell = oldhead; cell; cell = next) {
		next = cell->ptr_l;
		snd_seq_cell_free(cell);
	}
	snd_seq_pool_delete(&oldpool);

	return 0;
}
