/*
 *  ALSA sequencer device management
 *  Copyright (c) 1999 by Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/seq_device.h"
#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif

MODULE_AUTHOR("Takashi Iwai <iwai@ww.uni-erlangen.de>");
MODULE_DESCRIPTION("ALSA sequencer device management");
MODULE_SUPPORTED_DEVICE("sound");

/*
 * driver list
 */
typedef struct ops_list ops_list_t;

/* driver state */
#define DRIVER_EMPTY		0
#define DRIVER_LOADED		(1<<0)
#define DRIVER_REQUESTED	(1<<1)
#define DRIVER_LOCKED		(1<<2)

struct ops_list {
	char id[ID_LEN];	/* driver id */
	int driver;		/* driver state */
	int used;		/* reference counter */
	int argsize;		/* argument size */

	/* operators */
	snd_seq_dev_ops_t ops;

	/* registred devices */
	snd_seq_device_t *devhead, *devtail;	/* list of devices */
	int num_devices;	/* number of associated devices */
	int num_init_devices;	/* number of initialized devices */
	struct semaphore reg_mutex;

	ops_list_t *next;	/* next driver */
};


static ops_list_t *opslist = NULL;
static int num_ops = 0;
static DECLARE_MUTEX(ops_mutex);

/*
 * prototypes
 */
static int snd_seq_device_free(snd_seq_device_t *dev);
static int snd_seq_device_register(snd_seq_device_t *dev, snd_device_t *devptr);
static int snd_seq_device_unregister(snd_seq_device_t *dev);

static int init_device(snd_seq_device_t *dev, ops_list_t *ops);
static int free_device(snd_seq_device_t *dev, ops_list_t *ops);
static ops_list_t *find_driver(char *id, int create_if_empty);
static ops_list_t *create_driver(char *id);
static void unlock_driver(ops_list_t *ops);
static void remove_drivers(void);

/*
 * load all registered drivers (called from seq_clientmgr.c)
 */

void snd_seq_device_load_drivers(void)
{
#ifdef CONFIG_KMOD
	ops_list_t *ops;
	char modname[64];

	down(&ops_mutex);
	for (ops = opslist; ops; ops = ops->next) {
		if (! (ops->driver & DRIVER_LOADED) &&
		    ! (ops->driver & DRIVER_REQUESTED)) {
			ops->used++;
			up(&ops_mutex);
			ops->driver |= DRIVER_REQUESTED;
			sprintf(modname, "snd-%s", ops->id);
			request_module(modname);
			down(&ops_mutex);
			ops->used--;
		}
	}
	up(&ops_mutex);
#endif
}

/*
 * register a sequencer device
 * card = card info (NULL allowed)
 * device = device number (if any)
 * id = id of driver
 * result = return pointer (NULL allowed if unnecessary)
 */
int snd_seq_device_new(snd_card_t *card, int device, char *id, int argsize,
		       snd_seq_device_t **result)
{
	snd_seq_device_t *dev;
	ops_list_t *ops;
	int err;
	static snd_device_ops_t dops = {
		(snd_dev_free_t *)snd_seq_device_free,
		(snd_dev_register_t *)snd_seq_device_register,
		(snd_dev_unregister_t *)snd_seq_device_unregister
	};

	if (result)
		*result = NULL;

	snd_debug_check(id == NULL, -EINVAL);

	ops = find_driver(id, 1);
	if (ops == NULL)
		return -ENOMEM;

	dev = snd_kcalloc(sizeof(*dev) + argsize, GFP_KERNEL);
	if (dev == NULL) {
		unlock_driver(ops);
		return -ENOMEM;
	}

	/* set up device info */
	dev->card = card;
	dev->device = device;
	strncpy(dev->id, id, sizeof(dev->id) - 1);
	dev->id[sizeof(dev->id) - 1] = 0;
	dev->argsize = argsize;
	dev->status = SND_SEQ_DEVICE_FREE;
	dev->next = NULL;

	/* add this device to the list */
	down(&ops->reg_mutex);
	if (ops->devtail)
		ops->devtail->next = dev;
	else
		ops->devhead = dev;
	ops->devtail = dev;
	ops->num_devices++;
	up(&ops->reg_mutex);

	unlock_driver(ops);
	
	if ((err = snd_device_new(card, SND_DEV_SEQUENCER, dev, 0, &dops, NULL)) < 0) {
		snd_seq_device_free(dev);
		return err;
	}
	
	if (result)
		*result = dev;

	return 0;
}

/*
 * free the existing device
 */
static int snd_seq_device_free(snd_seq_device_t *dev)
{
	ops_list_t *ops;
	snd_seq_device_t *prev, *p;

	snd_debug_check(dev == NULL, -EINVAL);

	ops = find_driver(dev->id, 0);
	if (ops == NULL)
		return -ENXIO;

	/* remove the device from the list */
	down(&ops->reg_mutex);
	prev = NULL;
	for (p = ops->devhead; p; prev = p, p = p->next) {
		if (p == dev) {
			if (prev)
				prev->next = p->next;
			else
				ops->devhead = p->next;
			if (ops->devtail == p)
				ops->devtail = prev;
			ops->num_devices--;
			break;
		}
	}
	up(&ops->reg_mutex);
	if (p == NULL) {
		unlock_driver(ops);
		return -ENOENT;
	}

	free_device(dev, ops);
	snd_kfree(dev);

	unlock_driver(ops);

	return 0;
}

/*
 * find the matching device
 */
snd_seq_device_t * snd_seq_device_find(snd_card_t *card, int device, char *id)
{
	ops_list_t *ops;
	snd_seq_device_t *dev;

	ops = find_driver(id, 0);
	if (ops == NULL)
		return NULL;
	down(&ops->reg_mutex);
	for (dev = ops->devhead; dev; dev = dev->next) {
		if (dev->card == card && dev->device == device)
			break;
	}
	up(&ops->reg_mutex);
	unlock_driver(ops);

	return dev;
}


/*
 * register the device
 */
static int snd_seq_device_register(snd_seq_device_t *dev, snd_device_t *devptr)
{
	ops_list_t *ops;

	ops = find_driver(dev->id, 0);
	if (ops == NULL)
		return -ENOENT;

	/* initialize this device if the corresponding driver was
	 * already loaded
	 */
	if (ops->driver & DRIVER_LOADED)
		init_device(dev, ops);

	unlock_driver(ops);
	return 0;
}

/*
 * unregister the existing device
 */
static int snd_seq_device_unregister(snd_seq_device_t *dev)
{
	return snd_seq_device_free(dev);
}

/*
 * register device driver
 * id = driver id
 * entry = driver operators - duplicated to each instance
 */
int snd_seq_device_register_driver(char *id, snd_seq_dev_ops_t *entry, int argsize)
{
	snd_seq_device_t *dev;
	ops_list_t *ops;

	if (id == NULL || entry == NULL ||
	    entry->init_device == NULL || entry->free_device == NULL)
		return -EINVAL;

	ops = find_driver(id, 1);
	if (ops == NULL)
		return -ENOMEM;
	if (ops->driver & DRIVER_LOADED) {
		snd_printk("driver_register: driver '%s' already exists\n", id);
		unlock_driver(ops);
		return -EBUSY;
	}

	down(&ops->reg_mutex);
	/* copy driver operators */
	memcpy(&ops->ops, entry, sizeof(*entry));
	ops->driver |= DRIVER_LOADED;
	ops->argsize = argsize;

	/* initialize existing devices if necessary */
	for (dev = ops->devhead; dev; dev = dev->next) {
		init_device(dev, ops);
	}
	up(&ops->reg_mutex);

	unlock_driver(ops);

	return 0;
}


/*
 * create driver record
 */
static ops_list_t * create_driver(char *id)
{
	ops_list_t *ops;

	ops = snd_kcalloc(sizeof(*ops), GFP_KERNEL);
	if (ops == NULL)
		return ops;

	/* set up driver entry */
	strncpy(ops->id, id, sizeof(ops->id) - 1);
	ops->id[sizeof(ops->id) - 1] = 0;
	init_MUTEX(&ops->reg_mutex);
	ops->driver = DRIVER_EMPTY;
	/* lock this instance */
	ops->used = 1;

	/* register driver entry */
	down(&ops_mutex);
	ops->next = opslist;
	opslist = ops;
	num_ops++;
	up(&ops_mutex);

	return ops;
}


/*
 * unregister the specified driver
 */
int snd_seq_device_unregister_driver(char *id)
{
	ops_list_t *ops;
	snd_seq_device_t *dev;

	ops = find_driver(id, 0);
	if (ops == NULL)
		return -ENXIO;
	if (! (ops->driver & DRIVER_LOADED) ||
	    (ops->driver & DRIVER_LOCKED)) {
		snd_printk("driver_unregister: cannot unload driver '%s': status=%x\n", id, ops->driver);
		unlock_driver(ops);
		return -EBUSY;
	}

	/* close and release all devices associated with this driver */
	down(&ops->reg_mutex);
	ops->driver |= DRIVER_LOCKED; /* do not remove this driver recursively */
	for (dev = ops->devhead; dev; dev = dev->next)
		free_device(dev, ops);

	ops->driver = 0;
	if (ops->num_init_devices > 0)
		snd_printk("free_driver: init_devices > 0!! (%d)\n", ops->num_init_devices);
	up(&ops->reg_mutex);

	unlock_driver(ops);

	/* remove empty driver entries */
	remove_drivers();

	return 0;
}


/*
 * remove empty driver entries
 */
static void remove_drivers(void)
{
	ops_list_t *ops, *prev, *next;

	down(&ops_mutex);
	prev = NULL;
	ops = opslist;
	while (ops) {
		next = ops->next;
		if (! (ops->driver & DRIVER_LOADED) &&
		    ops->used == 0 && ops->num_devices == 0) {
			if (prev)
				prev->next = ops->next;
			else
				opslist = ops->next;
			snd_kfree(ops);
			num_ops--;
		} else {
			prev = ops;
		}
		ops = next;
	}
	up(&ops_mutex);
}

/*
 * initialize the device - call init_device operator
 */
static int init_device(snd_seq_device_t *dev, ops_list_t *ops)
{
	if (! (ops->driver & DRIVER_LOADED))
		return 0; /* driver is not loaded yet */
	if (dev->status != SND_SEQ_DEVICE_FREE)
		return 0; /* already initialized */
	if (ops->argsize != dev->argsize) {
		snd_printk("snd_seq_device: incompatible device '%s' for plug-in '%s'\n", dev->name, ops->id);
printk(" %d %d\n", ops->argsize, dev->argsize);
		return -EINVAL;
	}
	if (ops->ops.init_device(dev) >= 0) {
		dev->status = SND_SEQ_DEVICE_REGISTERED;
		ops->num_init_devices++;
	} else {
		snd_printk("init_device failed: %s: %s\n", dev->name, dev->id);
	}

	return 0;
}

/*
 * release the device - call free_device operator
 */
static int free_device(snd_seq_device_t *dev, ops_list_t *ops)
{
	int result;

	if (! (ops->driver & DRIVER_LOADED))
		return 0; /* driver is not loaded yet */
	if (dev->status != SND_SEQ_DEVICE_REGISTERED)
		return 0; /* not registered */
	if (ops->argsize != dev->argsize) {
		snd_printk("snd_seq_device: incompatible device '%s' for plug-in '%s'\n", dev->name, ops->id);
		return -EINVAL;
	}
	if ((result = ops->ops.free_device(dev)) >= 0 || result == -ENOENT) {
		dev->status = SND_SEQ_DEVICE_FREE;
		dev->driver_data = NULL;
		ops->num_init_devices--;
	} else {
		snd_printk("free_device failed: %s: %s\n", dev->name, dev->id);
	}

	return 0;
}

/*
 * find the matching driver with given id
 */
static ops_list_t * find_driver(char *id, int create_if_empty)
{
	ops_list_t *ops;

	down(&ops_mutex);
	for (ops = opslist; ops; ops = ops->next) {
		if (strcmp(ops->id, id) == 0) {
			ops->used++;
			break;
		}
	}
	up(&ops_mutex);
	if (ops)
		return ops;
	if (create_if_empty)
		return create_driver(id);
	return NULL;
}

static void unlock_driver(ops_list_t *ops)
{
	down(&ops_mutex);
	ops->used--;
	up(&ops_mutex);
}


/*
 * module part
 */

#ifdef MODULE

int __init init_module(void)
{
	return 0;
}

void __exit cleanup_module(void)
{
	remove_drivers();
	if (num_ops)
		snd_printk("seq_device: drivers not released (%d)\n", num_ops);
}

#endif

EXPORT_SYMBOL(snd_seq_device_load_drivers);
EXPORT_SYMBOL(snd_seq_device_new);
EXPORT_SYMBOL(snd_seq_device_find);
EXPORT_SYMBOL(snd_seq_device_register_driver);
EXPORT_SYMBOL(snd_seq_device_unregister_driver);
