/* 
    ALSA memory allocation module for the RME Digi9652
  
 	Copyright(c) 1999 IEM - Winfried Ritsch
        Copyright (C) 1999 Paul Barton-Davis 

    This module is only needed if you compiled the rme9652 driver with
    the PREALLOCATE_MEMORY option. It allocates the memory need to
    run the board and holds it until the module is unloaded. Because
    we need 2 contiguous 1.6MB regions for the board, it can be
    a problem getting them once the system memory has become fairly
    fragmented. 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    rme9652_mem.c,v 1.1 2000/01/16 16:48:58 perex Exp
*/

#define SND_MAIN_OBJECT_FILE

#include "../../include/driver.h"
#include "../../include/rme9652.h"

/* export */

void *rme9652_rec_busmaster_memory;
void *rme9652_play_busmaster_memory;

EXPORT_SYMBOL(rme9652_rec_busmaster_memory);
EXPORT_SYMBOL(rme9652_play_busmaster_memory);

MODULE_AUTHOR("Winfried Ritsch, Paul Barton-Davis <pbd@op.net>");
MODULE_DESCRIPTION("Memory allocator for RME Digi9652");

/* These are here so that we have absolutely no dependencies on any
   other modules. Dependencies can (1) cause us to lose in the rush
   for 2x 1.6MB chunks of contiguous memory and (2) make driver
   debugging difficult because unloading and reloading the snd module
   causes us to have to do the same for this one. We inevitably fail
   to allocate contiguous memory after the first load, so this is bad.  
*/

static char *rme9652_malloc_pages(unsigned long size, int *res_pg, int dma)
{
	int pg;
	char *buf;

	for (pg = 0; PAGE_SIZE * (1 << pg) < size; pg++);
	buf = (char *) __get_free_pages(GFP_KERNEL | (dma ? GFP_DMA : 0), pg);
	if (buf) {
		int page, last_page;

		page = MAP_NR(buf);
		last_page = page + (1 << pg);
		while (page < last_page)
			set_bit(PG_reserved, &mem_map[page++].flags);
	}
	*res_pg = pg;
	return buf;
}


static void rme9652_free_pages(char *ptr, unsigned long size)
{
	int pg, page, last_page;

	if (ptr == NULL)
		return;
	for (pg = 0; PAGE_SIZE * (1 << pg) < size; pg++);
	page = MAP_NR(ptr);
	last_page = page + (1 << pg);
	while (page < last_page)
		clear_bit(PG_reserved, &mem_map[page++].flags);
	free_pages((unsigned long) ptr, pg);
}

int init_module(void)
{
  
  int pg;

  rme9652_rec_busmaster_memory = NULL;
  rme9652_play_busmaster_memory = NULL;

  /* Since we don't know if its a Hammerfall or a Hammerfall/Light, 
     assume the worst, and go for 26 channels (the Hammerfall).
     
     See note in rme9652.h about why we allocate for an extra
     channel.
  */

#define TOTAL_SIZE (26+1)*(RME9652_CHANNEL_BUFFER_BYTES)

  rme9652_rec_busmaster_memory = rme9652_malloc_pages (TOTAL_SIZE, &pg, 1);
  rme9652_play_busmaster_memory = rme9652_malloc_pages (TOTAL_SIZE, &pg, 1);

  if (rme9652_play_busmaster_memory == 0 || 
      rme9652_rec_busmaster_memory == 0) {
	  snd_printk (CARD_NAME ": no memory available for buffers\n");
	  return -ENOMEM;
  }

  return 0;
}

void cleanup_module(void)
{

	if(rme9652_rec_busmaster_memory)
		rme9652_free_pages(rme9652_rec_busmaster_memory,TOTAL_SIZE);
	if(rme9652_play_busmaster_memory)
		rme9652_free_pages(rme9652_play_busmaster_memory,TOTAL_SIZE);
	
}

