/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of ForteMedia FM801 chips
 *
 *  BUGS:
 *    --
 *
 *  TODO:
 *    --
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/control.h"
#include "../../include/info.h"
#include "../../include/fm801.h"

/*
 *  constants
 */

/*
 *  common I/O routines
 */

static void snd_fm801_codec_write(void *private_data,
				  unsigned short reg,
				  unsigned short val)
{
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, );
	int idx;

	/*
	 *  Wait until the codec interface is not ready..
	 */
	for (idx = 0; idx < 100; idx++) {
		if (!(inw(FM801_REG(codec, AC97_CMD)) & (1<<9)))
			break;
		udelay(10);
	}
	if (inw(FM801_REG(codec, AC97_CMD)) & (1<<9)) {
		snd_printk("fm801: AC'97 interface is busy (1)\n");
		return;
	}
	/* write data and address */
	outw(val, FM801_REG(codec, AC97_DATA));
	outw(reg | FM801_AC97_ADDR, FM801_REG(codec, AC97_CMD));
	/*
	 *  Wait until the write command is not completed..
         */
	for (idx = 0; idx < 1000; idx++) {
		if (!(inw(FM801_REG(codec, AC97_CMD)) & (1<<9)))
			break;
		udelay(10);
	}
	if (inw(FM801_REG(codec, AC97_CMD)) & (1<<9))
		snd_printk("fm801: AC'97 interface is busy (2)\n");
}

static unsigned short snd_fm801_codec_read(void *private_data,
					    unsigned short reg)
{
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);
	int idx;

	/*
	 *  Wait until the codec interface is not ready..
	 */
	for (idx = 0; idx < 100; idx++) {
		if (!(inw(FM801_REG(codec, AC97_CMD)) & (1<<9)))
			break;
		udelay(10);
	}
	if (inw(FM801_REG(codec, AC97_CMD)) & (1<<9)) {
		snd_printk("fm801: AC'97 interface is busy (1)\n");
		return 0;
	}
	/* read command */
	outw(reg | FM801_AC97_ADDR | (1<<7), FM801_REG(codec, AC97_CMD));
	for (idx = 0; idx < 100; idx++) {
		if (!(inw(FM801_REG(codec, AC97_CMD)) & (1<<9)))
			break;
		udelay(10);
	}
	/*
	 *  Wait until the codec interface is not ready..
	 */
	if (inw(FM801_REG(codec, AC97_CMD)) & (1<<9)) {
		snd_printk("fm801: AC'97 interface is busy (2)\n");
		return 0;
	}
	for (idx = 0; idx < 1000; idx++) {
		if (inw(FM801_REG(codec, AC97_CMD)) & (1<<8))
			break;
		udelay(10);
	}
	if (!(inw(FM801_REG(codec, AC97_CMD)) & (1<<8))) {
		snd_printk("fm801: AC'97 interface is not valid (2)\n");
		return 0;
	}
	return inw(FM801_REG(codec, AC97_DATA));
}

/*
 *  Sample rate routines
 */

static unsigned int snd_fm801_real_rate(int rate)
{
	static unsigned int rates[] = {
		 5500,  8000,  9600, 11025,
		16000, 19200, 22050, 32000,
		38400, 44100, 48000 };
	int idx;

	for (idx = 0; idx < 11; idx++)
		if (rates[idx] >= rate)
			return rates[idx];
	return rates[10];
}

static unsigned short snd_fm801_rate_bits(int rate)
{
	static unsigned int rates[] = {
		 5500,  8000,  9600, 11025,
		16000, 19200, 22050, 32000,
		38400, 44100, 48000 };
	int idx;

	for (idx = 0; idx < 11; idx++)
		if (rates[idx] >= rate)
			return idx;
	return 10;
}

/*
 *  PCM part
 */

static int snd_fm801_playback_ioctl(void *private_data,
				    snd_pcm_subchn_t * subchn,
				    unsigned int cmd,
				    unsigned long *arg)
{
	int result;
	
	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	if (cmd == SND_PCM_IOCTL1_PARAMS)
		subchn->runtime->format.rate = snd_fm801_real_rate(subchn->runtime->format.rate);
	return 0;
}

static int snd_fm801_capture_ioctl(void *private_data,
				   snd_pcm_subchn_t * subchn,
				   unsigned int cmd,
				   unsigned long *arg)
{
	int result;
	
	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	if (cmd == SND_PCM_IOCTL1_PARAMS)
		subchn->runtime->format.rate = snd_fm801_real_rate(subchn->runtime->format.rate);
	return 0;
}

static int snd_fm801_playback_trigger(void *private_data,
				      snd_pcm_subchn_t * subchn,
				      int cmd)
{
	unsigned long flags;
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);
	int result = 0;

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (cmd == SND_PCM_TRIGGER_GO) {
		codec->ply_ctrl &= ~(FM801_BUF1_LAST |
				     FM801_BUF2_LAST |
				     FM801_PAUSE);
		codec->ply_ctrl |= FM801_START |
				   FM801_IMMED_STOP;
		outw(codec->ply_ctrl, FM801_REG(codec, PLY_CTRL));
	} else if (cmd == SND_PCM_TRIGGER_STOP) {
		codec->ply_ctrl &= ~FM801_START;
		outw(codec->ply_ctrl, FM801_REG(codec, PLY_CTRL));
	} else {
		result = -EINVAL;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

static int snd_fm801_capture_trigger(void *private_data,
				     snd_pcm_subchn_t * subchn,
				     int cmd)
{
	unsigned long flags;
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);
	int result = 0;

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (cmd == SND_PCM_TRIGGER_GO) {
		codec->cap_ctrl &= ~(FM801_BUF1_LAST |
				     FM801_BUF2_LAST |
				     FM801_PAUSE);
		codec->cap_ctrl |= FM801_START |
				   FM801_IMMED_STOP;
		outw(codec->cap_ctrl, FM801_REG(codec, CAP_CTRL));
	} else if (cmd == SND_PCM_TRIGGER_STOP) {
		codec->cap_ctrl &= ~FM801_START;
		outw(codec->cap_ctrl, FM801_REG(codec, CAP_CTRL));
	} else {
		result = -EINVAL;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

static int snd_fm801_playback_prepare(void *private_data,
				      snd_pcm_subchn_t * subchn)
{
	unsigned long flags;
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;

	codec->ply_size = snd_pcm_lib_transfer_size(subchn);
	codec->ply_count = snd_pcm_lib_transfer_fragment(subchn);
	spin_lock_irqsave(&codec->reg_lock, flags);
	codec->ply_ctrl &= ~(FM801_START | FM801_16BIT |
			     FM801_STEREO | FM801_RATE_MASK);
	if (snd_pcm_format_width(runtime->format.format) == 16)
		codec->ply_ctrl |= FM801_16BIT;
	if (runtime->format.voices > 1)
		codec->ply_ctrl |= FM801_STEREO;
	codec->ply_ctrl |= snd_fm801_rate_bits(runtime->format.rate) << FM801_RATE_SHIFT;
	codec->ply_buf = 0;
	outw(codec->ply_ctrl, FM801_REG(codec, PLY_CTRL));
	outw(codec->ply_count - 1, FM801_REG(codec, PLY_COUNT));
	codec->ply_buffer = virt_to_bus(runtime->dma_area->buf);
	codec->ply_pos = 0;
	outl(codec->ply_buffer, FM801_REG(codec, PLY_BUF1));
	outl(codec->ply_buffer + (codec->ply_count % codec->ply_size), FM801_REG(codec, PLY_BUF2));
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return 0;
}

static int snd_fm801_capture_prepare(void *private_data,
				     snd_pcm_subchn_t * subchn)
{
	unsigned long flags;
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;

	codec->cap_size = snd_pcm_lib_transfer_size(subchn);
	codec->cap_count = snd_pcm_lib_transfer_fragment(subchn);
	spin_lock_irqsave(&codec->reg_lock, flags);
	codec->cap_ctrl &= ~(FM801_START | FM801_16BIT |
			     FM801_STEREO | FM801_RATE_MASK);
	if (snd_pcm_format_width(runtime->format.format) == 16)
		codec->cap_ctrl |= FM801_16BIT;
	if (runtime->format.voices > 1)
		codec->cap_ctrl |= FM801_STEREO;
	codec->cap_ctrl |= snd_fm801_rate_bits(runtime->format.rate) << FM801_RATE_SHIFT;
	codec->cap_buf = 0;
	outw(codec->cap_ctrl, FM801_REG(codec, CAP_CTRL));
	outw(codec->cap_count - 1, FM801_REG(codec, CAP_COUNT));
	codec->cap_buffer = virt_to_bus(runtime->dma_area->buf);
	codec->cap_pos = 0;
	outl(codec->cap_buffer, FM801_REG(codec, CAP_BUF1));
	outl(codec->cap_buffer + (codec->cap_count % codec->cap_size), FM801_REG(codec, CAP_BUF2));
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return 0;
}

static unsigned int snd_fm801_playback_pointer(void *private_data,
					       snd_pcm_subchn_t * subchn)
{
	unsigned long flags;
	unsigned int result;
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (codec->ply_ctrl & FM801_START) {
		result = codec->ply_pos + (codec->ply_count - (inw(FM801_REG(codec, PLY_COUNT)) + 1));
	} else {
		result = 0;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

static unsigned int snd_fm801_capture_pointer(void *private_data,
					      snd_pcm_subchn_t * subchn)
{
	unsigned long flags;
	unsigned int result;
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (codec->cap_ctrl & FM801_START) {
		result = codec->cap_pos + (codec->cap_count - (inw(FM801_REG(codec, CAP_COUNT)) + 1));
	} else {
		result = 0;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

void snd_fm801_interrupt(fm801_t *codec, unsigned short status)
{
	unsigned int tmp;

	if ((status & (FM801_IRQ_PLAYBACK | FM801_IRQ_CAPTURE)) == 0)
		return;
	if (codec->pcm && (status & FM801_IRQ_PLAYBACK)) {
		spin_lock_irq(&codec->reg_lock);
		codec->ply_buf++;
		codec->ply_pos += codec->ply_count;
		codec->ply_pos %= codec->ply_size;
		tmp = codec->ply_pos + codec->ply_count;
		tmp %= codec->ply_size;
		outl(codec->ply_buffer + tmp,
				(codec->ply_buf & 1) ?
					FM801_REG(codec, PLY_BUF1) :
					FM801_REG(codec, PLY_BUF2));
		spin_unlock_irq(&codec->reg_lock);
		snd_pcm_transfer_done(codec->playback_subchn);
	}
	if (codec->pcm && (status & FM801_IRQ_CAPTURE)) {
		spin_lock_irq(&codec->reg_lock);
		codec->cap_buf++;
		codec->cap_pos += codec->cap_count;
		codec->cap_pos %= codec->cap_size;
		tmp = codec->cap_pos + codec->cap_count;
		tmp %= codec->cap_size;
		outl(codec->cap_buffer + tmp,
				(codec->cap_buf & 1) ?
					FM801_REG(codec, PLY_BUF1) :
					FM801_REG(codec, PLY_BUF2));
		spin_unlock_irq(&codec->reg_lock);
		snd_pcm_transfer_done(codec->capture_subchn);
	}
	/* clear interrupt */
	outw((status & (FM801_IRQ_PLAYBACK|FM801_IRQ_CAPTURE)), FM801_REG(codec, IRQ_STATUS));
}

static snd_pcm_hardware_t snd_fm801_playback =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_BLOCK_TRANSFER |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* formats */
	SND_PCM_RATE_KNOT | SND_PCM_RATE_8000_48000,
	5500,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	64,			/* min. fragment size */
	(128*1024),		/* max. fragment size */
	3,			/* fragment align */
	0,			/* FIFO size (unknown) */
	4,			/* transfer block size */
	snd_fm801_playback_ioctl,
	snd_fm801_playback_prepare,
	snd_fm801_playback_trigger,
	snd_pcm_playback_write,
	snd_fm801_playback_pointer
};

static snd_pcm_hardware_t snd_fm801_capture =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_BLOCK_TRANSFER |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* formats */
	SND_PCM_RATE_KNOT | SND_PCM_RATE_8000_48000,
	5500,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	64,			/* min. fragment size */
	(128*1024),		/* max. fragment size */
	3,			/* fragment align */
	0,			/* FIFO size (unknown) */
	4,			/* transfer block size */
	snd_fm801_capture_ioctl,
	snd_fm801_capture_prepare,
	snd_fm801_capture_trigger,
	snd_pcm_capture_read,
	snd_fm801_capture_pointer
};

static int snd_fm801_playback_open(void *private_data,
				   snd_pcm_subchn_t * subchn)
{
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);
	int err;

	if ((err = snd_pcm_dma_alloc(subchn, codec->dma1ptr, "FM801 - DAC")) < 0)
		return err;
	codec->playback_subchn = subchn;
	subchn->runtime->hw = &snd_fm801_playback;
	snd_pcm_set_mixer(subchn, codec->mixer->device, codec->ac97->me_playback);
	return 0;
}

static int snd_fm801_capture_open(void *private_data,
				  snd_pcm_subchn_t * subchn)
{
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, -ENXIO);
	int err;

	if ((err = snd_pcm_dma_alloc(subchn, codec->dma2ptr, "FM801 - ADC")) < 0)
		return err;
	codec->capture_subchn = subchn;
	subchn->runtime->hw = &snd_fm801_capture;
	snd_pcm_set_mixer(subchn, codec->mixer->device, codec->ac97->me_playback);
	return 0;
}

static int snd_fm801_playback_close(void *private_data,
				    snd_pcm_subchn_t * subchn)
{
	snd_pcm_dma_free(subchn);
	return 0;
}

static int snd_fm801_capture_close(void *private_data,
				   snd_pcm_subchn_t * subchn)
{
	snd_pcm_dma_free(subchn);
	return 0;
}

static void snd_fm801_pcm_free(void *private_data)
{
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, );
	codec->pcm = NULL;
}

int snd_fm801_pcm(fm801_t * codec, int device, snd_pcm_t ** rpcm)
{
	snd_pcm_t *pcm;
	int err;

	*rpcm = NULL;
	if ((err = snd_pcm_new(codec->card, "FM801", device, 1, 1, &pcm)) < 0)
		return err;

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_fm801_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_fm801_playback_close;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_fm801_capture_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_fm801_capture_close;

	pcm->private_data = codec;
	pcm->private_free = snd_fm801_pcm_free;
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE |
			  SND_PCM_INFO_DUPLEX;
	strcpy(pcm->name, "FM801");
	*rpcm = codec->pcm = pcm;
	return 0;
}

/*
 *  Mixer routines
 */

static void snd_fm801_mixer_free_ac97(void *private_data)
{
	fm801_t *codec = snd_magic_cast(fm801_t, private_data, );
	codec->mixer = NULL;
}

int snd_fm801_mixer(fm801_t * codec, int device, snd_pcm_t * pcm, snd_kmixer_t ** rmixer)
{
	ac97_t ac97;
	snd_kmixer_t *mixer;
	int err;

	*rmixer = NULL;
	memset(&ac97, 0, sizeof(ac97));
	ac97.write = snd_fm801_codec_write;
	ac97.read = snd_fm801_codec_read;
	ac97.private_data = codec;
	ac97.private_free = snd_fm801_mixer_free_ac97;
	if ((err = snd_ac97_mixer(codec->card, device, &ac97, 1, &pcm->device, &mixer)) < 0)
		return err;
	codec->ac97 = snd_magic_cast(ac97_t, mixer->private_data, -ENXIO);
	*rmixer = codec->mixer = mixer;
	return 0;
}

/*
 *  initialization routines
 */

int snd_fm801_create(snd_card_t * card,
		     struct pci_dev * pci,
		     snd_dma_t * dma1ptr,
		     snd_dma_t * dma2ptr,
		     snd_irq_t * irqptr,
		     fm801_t ** rcodec)
{
	fm801_t *codec;
	unsigned short cmdw;
	unsigned char cmdb;
	int err;
	static snd_device_ops_t ops = {
		(snd_dev_free_t *)snd_fm801_free,
		NULL,
		NULL
	};
	
	*rcodec = NULL;
	codec = snd_magic_kcalloc(fm801_t, 0, GFP_KERNEL);
	if (codec == NULL)
		return -ENOMEM;
	codec->reg_lock = SPIN_LOCK_UNLOCKED;
	codec->card = card;
	codec->pci = pci;
	codec->dma1ptr = dma1ptr;
	codec->dma2ptr = dma2ptr;
	codec->irqptr = irqptr;
#ifdef NEW_PCI
	codec->port = pci->resource[0].start;
#else
	codec->port = pci->base_address[0] & ~PCI_BASE_ADDRESS_SPACE;
#endif
	pci_set_master(pci);
	pci_read_config_word(pci, PCI_COMMAND, &cmdw);
	if ((cmdw & PCI_COMMAND_IO) != PCI_COMMAND_IO) {
		cmdw |= PCI_COMMAND_IO;
		pci_write_config_word(pci, PCI_COMMAND, cmdw);
	}
	pci_set_master(pci);
	pci_read_config_byte(pci, PCI_LATENCY_TIMER, &cmdb);
	if (cmdb < 32)
		cmdb = 32;
	pci_write_config_byte(pci, PCI_LATENCY_TIMER, cmdb);

	/* codec cold reset + AC'97 warm reset */
	outw((1<<5)|(1<<6), FM801_REG(codec, CODEC_CTRL));
	udelay(100);
	outw(0, FM801_REG(codec, CODEC_CTRL));
	mdelay(1);

	/* init volume */
	outw(0x0808, FM801_REG(codec, PCM_VOL));
	outw(0x0808, FM801_REG(codec, FM_VOL));
	outw(0x0808, FM801_REG(codec, I2S_VOL));

	/* interrupt setup - unmask MPU, PLAYBACK & CAPTURE */
	cmdw = inw(FM801_REG(codec, IRQ_MASK));
	cmdw &= ~0x0083;
	outw(cmdw, FM801_REG(codec, IRQ_MASK));

	/* interrupt clear */
	outw(FM801_IRQ_PLAYBACK|FM801_IRQ_CAPTURE|FM801_IRQ_MPU, FM801_REG(codec, IRQ_STATUS));

	if ((err = snd_device_new(card, SND_DEV_LOWLEVEL, codec, 0, &ops, NULL)) < 0) {
		snd_fm801_free(codec);
		return err;
	}

	*rcodec = codec;
	return 0;
}

int snd_fm801_free(fm801_t * codec)
{
	unsigned short cmdw;

	/* interrupt setup - mask everything */
	cmdw = inw(FM801_REG(codec, IRQ_MASK));
	cmdw |= 0x00c3;
	outw(cmdw, FM801_REG(codec, IRQ_MASK));

	snd_magic_kfree(codec);
	return 0;
}

EXPORT_SYMBOL(snd_fm801_create);
EXPORT_SYMBOL(snd_fm801_free);
EXPORT_SYMBOL(snd_fm801_interrupt);
EXPORT_SYMBOL(snd_fm801_pcm);
EXPORT_SYMBOL(snd_fm801_mixer);

/*
 *  INIT part
 */

#ifdef MODULE

int __init init_module(void)
{
	return 0;
}

void __exit cleanup_module(void)
{
}

#endif
