/*
 *   ALSA driver for RME Digi9652 audio interfaces 
 *
 *	Copyright (c) 1999 IEM - Winfried Ritsch
 *      Copyright (c) 1999 Paul Barton-Davis
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */      

#define SND_MAIN_OBJECT_FILE

#include "../include/driver.h"
#include "../include/pcm.h"
#include "../include/hwdep.h"
#include "../include/rme9652.h"
#include "../include/info.h"
#include "../include/initval.h"

static rme9652_t *snd_rme9652_cards[SND_CARDS] = SND_DEFAULT_PTR;

int   snd_offset[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */

MODULE_PARM(snd_offset, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_offset, "Index value for RME Digi9652 (Hammerfall) soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for RME Digi9652 (Hammerfall) soundcard.");
MODULE_AUTHOR("Paul Barton-Davis <pbd@op.net>, Winfried Ritsch");
MODULE_DESCRIPTION("\
Driver: RME Digi9652 (HammerFall + Hammerfall-Light)\n\
");

static void snd_rme9652_use_inc (snd_card_t *);
static void snd_rme9652_use_dec (snd_card_t *);
static int  snd_rme9652_detect (snd_card_t *, rme9652_t *s, unsigned int);
static int  snd_rme9652_probe (int dev, rme9652_t *);

static void snd_rme9652_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_rme9652_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static int snd_rme9652_detect(snd_card_t * card, 
			      rme9652_t *s,
			      unsigned int device_id)
{
	if ((s->pci = pci_find_device (PCI_VENDOR_ID_RME,
				       device_id,
				       s->pci)) == NULL) {
		return -ENODEV;
	}

#ifdef NEW_PCI
	if ((s->port = s->pci->resource[0].start) == 0) {
		return -ENODEV;
	}
#else
	if (s->pci->base_address[0] == 0 ||
	    (s->pci->base_address[0] & PCI_BASE_ADDRESS_SPACE ) == 
	    PCI_BASE_ADDRESS_SPACE_IO) {
		return -ENODEV;
	}
	s->port = s->pci->base_address[0] & ~PCI_BASE_ADDRESS_SPACE;
#endif

	if (s->pci->irq == 0) {
		return -ENODEV;
	}
	
	if (snd_register_ioport (card, s->port, RME9652_IO_EXTENT, 
				 CARD_NAME, NULL) < 0) {
		return -ENODEV;
	}

	return 0;
}

static int snd_rme9652_probe(int dev, rme9652_t *rme9652)
{
	snd_card_t *card;
	int err;

	card = snd_card_new (snd_offset[dev], snd_id[dev],
			     snd_rme9652_use_inc, 
			     snd_rme9652_use_dec);

	if (!card) {
		return -ENOMEM;
	}

	card->static_data = rme9652;
	rme9652->pci = NULL;

	if (snd_rme9652_detect (card, rme9652, PCI_DEVICE_ID_HAMMERFALL) < 0) {
		goto __nodev;
	}

	if ((err = snd_register_interrupt (card,
					   CARD_NAME,
					   rme9652->pci->irq,
					   SND_IRQ_TYPE_PCI, 
					   snd_rme9652_interrupt,
					   rme9652, 
					   NULL,
					   &rme9652->irqptr)) < 0) {
		goto __nodev;
	}

	if (snd_rme9652_create (card, dev, rme9652) < 0) {
		goto __nodev;
	}

	strcpy(card->abbreviation, "Hammerfall");
	strcpy(card->shortname, "RME Digi9652");

	if (card->type == SND_CARD_TYPE_HAMMERFALL) {
		sprintf(card->longname, "%s at 0x%lx, irq %i",
			card->shortname,
			rme9652->port,
			rme9652->irqptr->irq);
	} else {
		sprintf(card->longname, "%s/Light at 0x%lx, irq %i",
			card->shortname,
			rme9652->port,
			rme9652->irqptr->irq);
	}

	if (!snd_card_register(card)) {
		return 0;
	}

  __nodev:
	snd_card_free (card);
	return -ENXIO;
}

#ifdef MODULE
int __init init_module(void)
#else
int __init alsa_card_hammerfall_init(void)
#endif

{
	int dev, cards;
	rme9652_t *rme9652;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		rme9652 = (rme9652_t *)
			snd_kcalloc(sizeof(rme9652_t), GFP_KERNEL);

		rme9652->last_spdif_sample_rate = -1;
		rme9652->last_adat_sample_rate = -1;

		if (!rme9652) {
			return -ENOMEM;
		}

		if (snd_rme9652_probe (dev, rme9652) < 0) {
			snd_kfree (rme9652);
			break;
		}

		snd_rme9652_cards[dev] = rme9652;
		cards++;
	}

	if (!cards) {
		snd_printk (CARD_NAME ": no cards found\n");
		return -ENODEV;
	}

	return 0;
}

#ifdef MODULE

void __exit 
cleanup_module(void)

{
	int dev;
	rme9652_t *rme9652;

	for (dev = 0; dev < SND_CARDS; dev++) {
		rme9652 = snd_rme9652_cards[dev];
		snd_rme9652_cards[dev] = NULL;

		if (rme9652) {
			snd_rme9652_free (rme9652);
		}
	}

	return;
}

#endif
