///////////////////////////////////////////////////////////////////////////
//	MOTU Midi Timepiece ALSA Main routines
//	Copyright by Michael T. Mayers (c) Jan 09, 2000
//	Thanks to John Galbraith
///////////////////////////////////////////////////////////////////////////
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
///////////////////////////////////////////////////////////////////////////
//	This driver is for the 'Mark Of The Unicorn' (MOTU) MidiTimePiece AV multiport MIDI interface 
//
//	IOPORTS
//	///////////
//	8 MIDI Ins and 8 MIDI outs
//	Video Sync In (BNC), Word Sync Out (BNC), 
//	ADAT Sync Out (DB9)
//	SMPTE in/out (1/4")
//	2 programmable pedal/footswitch inputs and 4 programmable MIDI controller knobs.
//	Macintosh RS422 serial port
//	RS422 "network" port for ganging multiple MTP's
//	PC Parallel Port ( which this driver currently uses )
//
//	MISC FEATURES
//	///////////////
//	Hardware MIDI routing, merging, and filtering	
//	MIDI Synchronization to Video, ADAT, SMPTE and other Clock sources
//	128 'scene' memories, recallable from MIDI program change
///////////////////////////////////////////////////////////////////////////
//	remember to set SND_RAWMIDI_PORTS in rawmidi.h to >= 18
///////////////////////////////////////////////////////////////////////////

//
#define SND_MAIN_OBJECT_FILE

/////////////////////////////////////////////////////////////////
//	includes

#include "../include/mtpav.h"

/////////////////////////////////////////////////////////////////
//	globals

MODULE_DESCRIPTION("\
Driver: MOTU MidiTimePiece AV multiport MIDI\n\
");

int snd_index = SND_DEFAULT_IDX1;
char *snd_id = SND_DEFAULT_STR1;
int snd_port = 0x378;			/* 0x378, 0x278 */
int snd_irq = 7;			/* 7, 5 */
MODULE_PARM(snd_index, "i");
MODULE_PARM_DESC(snd_index, "Index value for MotuMTPAV MIDI." );
MODULE_PARM(snd_id, "s");
MODULE_PARM_DESC(snd_id, "ID string for MotuMTPAV MIDI." );
MODULE_PARM(snd_port, "i");
MODULE_PARM_DESC(snd_port, "Parallel port # for MotuMTPAV MIDI." );
MODULE_PARM(snd_irq, "i");
MODULE_PARM_DESC(snd_irq, "Parallel IRQ # for MotuMTPAV MIDI." );

TSmtp *mtp_card;
U8 outmsg_lastport;
U8 inmsg_lastport;
U8 inmsg_port;

//////////////////////////////////////////////////////////////////

static void snd_mtpav_use_inc(snd_card_t * card);
static void snd_mtpav_use_dec(snd_card_t * card);
extern void snd_mtpav_output_write(snd_rawmidi_t * rmidi);
extern int snd_mtpav_input_open(snd_rawmidi_t * rmidi);
extern int snd_mtpav_input_close(snd_rawmidi_t * rmidi);
extern void snd_mtpav_input_trigger(snd_rawmidi_t * rmidi, int up);
extern int snd_mtpav_output_open(snd_rawmidi_t * rmidi);
extern int snd_mtpav_output_close(snd_rawmidi_t * rmidi);
void note_on_port( U8 port, U8 chan, U8 note, U8 vel );
void note_off_port( U8 port, U8 chan, U8 note );
void mputreg( U16 reg, U8 val );
U8 mgetreg( U16 reg );
void read_bus_handler( void );
extern U32 mtp_inmidistate;

void mtpav_inmidi_h( U8 inbyte );
void mtpav_outmidi_h( U32 port, U8 outbyte );
void write_byte( U8 byte );

//////////////////////////////////////////////////////////////////////////

/*static*/ void snd_mtpav_output_write(snd_rawmidi_t * rmidi)
{	U8 outbyte;
	U32 port;
	
	int ifsent;
	
	port =  (U32)  rmidi->private_data;
	
	//ifsent = rmidi->output.data( rmidi, &outbyte, 1 );
	ifsent = snd_rawmidi_transmit(rmidi, &outbyte, 1);
	
	if( ifsent != 1 )
	{
	}
	else
	{	mtpav_outmidi_h( port, outbyte );
	}
}

//////////////////////////////////////////////////////////////////////////

/*static*/
int snd_mtpav_input_open(snd_rawmidi_t * rmidi)
{	U32 flags;
	U32 port = (U32) rmidi->private_data;

	spin_lock_irqsave( & mtp_card->spinlock, flags);
	mtp_card->mode[port] |= MTPAV_MODE_INPUT_OPENED;
	spin_unlock_irqrestore( & mtp_card->spinlock, flags);
	return 0;
}

//////////////////////////////////////////////////////////////////

/*static*/
int snd_mtpav_input_close(snd_rawmidi_t * rmidi)
{	U32 flags;
	U32 port = (U32) rmidi->private_data;

	spin_lock_irqsave( & mtp_card->spinlock, flags );
	mtp_card->mode[port] &= (~MTPAV_MODE_INPUT_OPENED);
	spin_unlock_irqrestore( & mtp_card->spinlock, flags);
	return 0;
}

//////////////////////////////////////////////////////////////////

/*static*/
void snd_mtpav_input_trigger(snd_rawmidi_t * rmidi, int up)
{	U32 flags;
	U32 port = (U32) rmidi->private_data;
	
	spin_lock_irqsave( & mtp_card->spinlock, flags);
	if (up)
		mtp_card->mode[port] |= MTPAV_MODE_INPUT_TRIGGERED;
	else
		mtp_card->mode[port] &= ~MTPAV_MODE_INPUT_TRIGGERED;

	spin_unlock_irqrestore( & mtp_card->spinlock, flags);

}

///////////////////////////////////////////////////////////////////////////////

/*static*/
int snd_mtpav_output_open(snd_rawmidi_t * rmidi)
{	U32 flags;
	U32 port = (U32) rmidi->private_data;
	
	spin_lock_irqsave( & mtp_card->spinlock, flags);
	mtp_card->mode[port] |= MTPAV_MODE_OUTPUT_OPENED;
	spin_unlock_irqrestore( & mtp_card->spinlock, flags);
	return 0;
};

///////////////////////////////////////////////////////////////////////////////

/*static*/
int snd_mtpav_output_close(snd_rawmidi_t * rmidi)
{	U32 flags;
	U32 port = (U32) rmidi->private_data;
	
	spin_lock_irqsave( & mtp_card->spinlock, flags );
	mtp_card->mode[port] &= (~MTPAV_MODE_INPUT_OPENED);
	spin_unlock_irqrestore( & mtp_card->spinlock, flags);
	return 0;
};

///////////////////////////////////////////////////////////////////////////////

U8 mgetreg( U16 reg )
{
	U8 rval = 0;

	#ifndef	USE_FAKE_MTP

	if( reg == SREG )
	{	rval = inb( MTPAV_IOBASE+SREG );
		rval = (rval & 0xf8);
	}
	else if( reg == CREG )
	{	rval = inb( MTPAV_IOBASE+CREG );
		rval = (rval & 0x1c);
	}

	#endif
	
	return rval;
}

///////////////////////////////////////////////////////////////////////////////

void mputreg( U16 reg, U8 val )
{
	#ifndef	USE_FAKE_MTP
	
	if( reg == DREG )
	{	outb( val, (MTPAV_IOBASE+DREG) );
	}
	else if( reg == CREG )
	{	outb( val, (MTPAV_IOBASE+CREG) );
	}
	
	#endif
}

//////////////////////////////////////////////////////////////////

void read_bus_handler( void )
{	
	#ifndef	USE_FAKE_MTP

	U8 clrread, setread;
	U8 mtp_read_byte;
	U8 read_sr0, read_sr1, read_sr2, read_sr3;
   
	U8 sbyt = mgetreg( SREG );

	if( sbyt & SIGS_BYTE )
	{	U8 cbyt = mgetreg( CREG );
		clrread = cbyt & ( SIGC_READ ^ 0xff );
		setread = cbyt | SIGC_READ;
						
		mputreg( CREG, setread );
		read_sr0 = mgetreg( SREG );
		mputreg( CREG, clrread );

   		mputreg( CREG, setread );
		read_sr1 = mgetreg( SREG );
		mputreg( CREG, clrread );

		mputreg( CREG, setread );
		read_sr2 = mgetreg( SREG );
		mputreg( CREG, clrread );

   		mputreg( CREG, setread );
   		read_sr3 = mgetreg( SREG );
   		mputreg( CREG, clrread );

   		mtp_read_byte = ((read_sr0 & (SIGS_IN0|SIGS_IN1) ) /16 );
	   	mtp_read_byte |= (((read_sr1 & (SIGS_IN0|SIGS_IN1) ) /16)<<2);
	   	mtp_read_byte |= (((read_sr2 & (SIGS_IN0|SIGS_IN1) ) /16)<<4);
	   	mtp_read_byte |= (((read_sr3 & (SIGS_IN0|SIGS_IN1) ) /16)<<6);

		mtpav_inmidi_h( mtp_read_byte );
	}

	#endif
}

///////////////////////////////////////////////////////////////////////////////

U32 mtp_lastoutport = 0xffffffff;

void mtpav_outmidi_h( U32 port, U8 outbyte )
{	
	U8 u8port = (U8) port;
	
	if( port != mtp_lastoutport )
	{	mtp_lastoutport = port;
		write_byte( 0xf5 );
		write_byte( u8port );
		snd_printk( "new outport: 0x%x\n", (unsigned int) port );
	}
	write_byte( outbyte );
	
}

void write_byte( U8 outbyte )
{
	U8 tcbyt;
	U8 clrwrite;
	U8 setwrite;

	U8 sbyt = mgetreg( SREG );

	while( ! (sbyt & SIGS_RFD) ) // wait for RFD hi
	{	sbyt = mgetreg( SREG );
	}

	/////////////////
	
	tcbyt = mgetreg( CREG );
	clrwrite = tcbyt & ( SIGC_WRITE ^ 0xff );
	setwrite = tcbyt | SIGC_WRITE;

	/////////////////
	// this stage is loopable!
   
	{	mputreg( DREG, outbyte );
		mputreg( CREG, clrwrite );	// clear write bit
		mputreg( CREG, setwrite );	// set write bit
	}

}

//////////////////////////////////////////////////////////////////

void note_on_port( U8 port, U8 chan, U8 note, U8 vel )
{	write_byte( 0xf5 );
	write_byte( port );
	write_byte( 0x90 | chan );
	write_byte( note );
	write_byte( vel );
}

//////////////////////////////////////////////////////////////////

void note_off_port( U8 port, U8 chan, U8 note )
{	write_byte( 0xf5 );
	write_byte( port );
	write_byte( 0x90 | chan );
	write_byte( note );
	write_byte( 0 );
}

/////////////////////////////////////////////////////////////////////

void send_inbyte_to_rawmidi( U32 port, U8 inbyte )
{ 	
	if( mtp_card->mode[port] & MTPAV_MODE_INPUT_TRIGGERED )
	{}	//CVS mtp_card->rmidi[port]->input.data( mtp_card->rmidi[port], & inbyte, 1);
}

/////////////////////////////////////////////////////////////////////

U32 mtp_inmidistate;
U8 mtp_incable;

void mtpav_inmidi_h( U8 inbyte )
{	snd_rawmidi_t *rmidi;
	//char ibyte = inbyte;
	
	if( mtp_inmidistate == 0 ) // awaiting command
	{	if( inbyte == 0xf5 ) // MTP port #
		{	mtp_inmidistate = 1;
		}
		else
		{	rmidi = mtp_card->rmidi[ mtp_incable ];
			if ( (mtp_card->mode[ mtp_incable ] & MTPAV_MODE_INPUT_OPENED ) != 0 )
				snd_rawmidi_receive( rmidi, &inbyte, 1 );
		}
	}
	else if( mtp_inmidistate == 1 )
	{	mtp_incable = inbyte;
		mtp_inmidistate = 0;
		snd_printk( "cable %d\n", mtp_incable );
	}
 }

//////////////////////////////////////////////////////////////////

static void snd_mtpav_irqh(int irq, void *dev_id, struct pt_regs *regs)
{	U32 flags = 0;
   	
   	spin_lock_irqsave( & mtp_card->spinlock, flags );
   	read_bus_handler();
	spin_unlock_irqrestore( & mtp_card->spinlock, flags );
}

/////////////////////////////////////////////////////////////////

static struct snd_stru_rawmidi_channel_hw snd_mtpav_output =
{
	SND_RAWMIDI_HW_POLL,	// flags
	NULL,			/// private_data
	NULL,			/// private_free
	snd_mtpav_output_open,	/// open
	snd_mtpav_output_close,	/// close
	NULL,			/// trigger
	{snd_mtpav_output_write},	// io.write
	NULL,			// abort/

};

/////////////////////////////////////////////////////////////////

static struct snd_stru_rawmidi_channel_hw snd_mtpav_input =
{
	0,			// flags
	NULL,			// private_data
	NULL,			// private_free
	snd_mtpav_input_open,	// open
	snd_mtpav_input_close,	// close
	snd_mtpav_input_trigger,	// trigger
	{NULL},			// io.read
	NULL,			// abort

};

/////////////////////////////////////////////////////////////////
// get ISA resources

int mtp_get_isa_resources( void )
{	U32 flags;
	int possible_irqs[2];
	UBOOL irqtst;
	spin_lock_irqsave(  & mtp_card->spinlock, flags );
	if( snd_register_ioport( mtp_card->card, snd_irq, 3, "MotuMTPAV MIDI", & mtp_card->rioport ) < 0)
	{	snd_printk("snd_uart16550_register_resources snd_register_ioport failed, port\n");
		spin_unlock_irqrestore( &mtp_card->spinlock, flags );
		return -EBUSY;
	}
	possible_irqs[0] = snd_irq;
	possible_irqs[1] = -1;
	snd_printk("snd_mtpav_register_resources requesting irq=%d\n", MTPAV_IRQ );
	//irqtst = snd_register_interrupt( mtp_card->card, "MOTU MTPAV", MTPAV_IRQ, SND_IRQ_TYPE_ISA, snd_mtpav_irqh, (void *) mtp_card->rmidi[0], possible_irqs, & mtp_card->irq );
	irqtst = snd_register_interrupt( mtp_card->card, "MOTU MTPAV", MTPAV_IRQ, SND_IRQ_TYPE_ISA, snd_mtpav_irqh, (void *) 0, possible_irqs, & mtp_card->irq );
	snd_printk("mtpav irqtst: %d\n", irqtst );
	spin_unlock_irqrestore( & mtp_card->spinlock, flags );
	return 0;
}

/////////////////////////////////////////////////////////////////
// get RAWMIDI resources

char rmidinames1[80];
char rmidinames2[80];

int mtp_get_rawmidi_ports( void )
{	U32 i;
	int rval = 0;
	snd_rawmidi_t *rawmidi;
	
	for( i=0; i<NUMPORTS; i++ )
	{	sprintf( rmidinames1, "MTPAV Midiport %d", i );
		
		rawmidi = mtp_card->rmidi[i];
		
		if ((rval = snd_rawmidi_new( mtp_card->card, rmidinames1, i, & rawmidi )) < 0)
			return rval;
		
		if ( rawmidi == NULL) return 0;
		mtp_card->spinlock = SPIN_LOCK_UNLOCKED;
		memcpy(& rawmidi->chn[SND_RAWMIDI_CHANNEL_OUTPUT].hw, & snd_mtpav_output, sizeof(snd_mtpav_output) );
		memcpy(& rawmidi->chn[SND_RAWMIDI_CHANNEL_INPUT].hw, & snd_mtpav_input, sizeof(snd_mtpav_input) );
		rawmidi->private_data = (void *) i;
		rawmidi->info_flags |= SND_RAWMIDI_INFO_OUTPUT | SND_RAWMIDI_INFO_INPUT | SND_RAWMIDI_INFO_DUPLEX;
		sprintf( rawmidi->name, "MotuMIDI %d", (int) i );
	}
	return 0;
}
/////////////////////////////////////////////////////////////////
TSmtp *new_TSmtp( void )
{	U32 i;
	TSmtp *ncrd = (TSmtp *) snd_kcalloc( sizeof( TSmtp ), GFP_KERNEL );
	if( ncrd != NULL )
	{	ncrd->card = NULL;
		ncrd->rioport = NULL;
		ncrd->irq = NULL;
		
		for( i=0; i<NUMPORTS; i++ )
		{	ncrd->rmidi[i] = NULL;
			ncrd->mode[i] = 0;
		}
	}
	return ncrd;
}
/////////////////////////////////////////////////////////////////
void free_TSmtp( TSmtp *crd )
{	snd_kfree( crd );
}
/////////////////////////////////////////////////////////////////
static void snd_mtpav_use_inc(snd_card_t * card)
{	MOD_INC_USE_COUNT;
}
/////////////////////////////////////////////////////////////////
static void snd_mtpav_use_dec(snd_card_t * card)
{	MOD_DEC_USE_COUNT;
}
/////////////////////////////////////////////////////////////////
int init_module( void )
{
	int err = 0;
	char longname_buffer[80];
	
	mtp_card = new_TSmtp();
	if( mtp_card == NULL )
		return -ENOMEM;
	
	mtp_card->card = snd_card_new(snd_index, snd_id, snd_mtpav_use_inc, snd_mtpav_use_dec);

	mtp_card->card->type = SND_CARD_TYPE_MTPAV;
	strcpy(mtp_card->card->abbreviation, "mtpav");
	strcpy(mtp_card->card->shortname, "mtpav on parallel port" );
	memset(longname_buffer, 0, sizeof(longname_buffer));
	sprintf(longname_buffer, "mtpav on parallel port at" );

	err = mtp_get_isa_resources();
	if( err == 0 )
	{	err = mtp_get_rawmidi_ports();
		if( err == 0 )
		{
		}
		else
			return err;
	}
	else
		return err;		

	if ( ! snd_card_register( mtp_card->card ) )  // dont snd_card_register until AFTER all cards reources done!
	{	err = 0;
	}
	else
	{	snd_card_free( mtp_card->card );
		err =  -ENOMEM;
		return err;
	}

   	mputreg( CREG, SIGC_INTEN ); // enable pport interrupts

	mtp_inmidistate = 0;
	inmsg_lastport = 0xff;
	outmsg_lastport = 0xff;
   
	// mtp port scan (put mtp into smart mode)
	// this will hang your system if you do not have an mtp connected
	// this will be fixed by adding timeout code
	//for( tport = 1; tport<=8; tport++ )
	//{	for( tchan=0; tchan<16; tchan++ )
	//	{    note_on_port( tport, tchan, 0x20, 0x7f );
	//	}
	//	for( tchan=0; tchan<16; tchan++ )
	//	{    note_off_port( tport, tchan, 0x20 );
	//	}
	//}
	
	return err;
}

/////////////////////////////////////////////////////////////////

void __exit cleanup_module( void )
{	if ( mtp_card == NULL)
		return;
	snd_card_unregister( mtp_card->card );
	free_TSmtp( mtp_card );
}
