/*
 *  Generic driver for CS4231 chips
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Originally the CS4232/CS4232A driver, modified for use on CS4231 by
 *  Tugrul Galatali <galatalt@stuy.edu>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/cs4231.h"
#include "../include/mpu401.h"
#include "../include/initval.h"

MODULE_DESCRIPTION("\
Driver: Generic CS4231\n\
");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_port[SND_CARDS] = SND_DEFAULT_PORT;	/* PnP setup */
int snd_mpu_port[SND_CARDS] = SND_DEFAULT_PORT;	/* PnP setup */
int snd_irq[SND_CARDS] = SND_DEFAULT_IRQ;	/* 5,7,9,11,12,15 */
int snd_mpu_irq[SND_CARDS] = SND_DEFAULT_IRQ;	/* 9,11,12,15 */
int snd_dma1[SND_CARDS] = SND_DEFAULT_DMA;	/* 0,1,3,5,6,7 */
int snd_dma1_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
int snd_dma2[SND_CARDS] = SND_DEFAULT_DMA;	/* 0,1,3,5,6,7 */
int snd_dma2_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for CS4231 soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for CS4231 soundcard.");
MODULE_PARM(snd_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_port, "Port # for CS4231 driver. [PORT12]");
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port # for CS4231 driver. [PORT12]");
MODULE_PARM(snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_irq, "IRQ # for CS4231 driver. [IRQ]");
MODULE_PARM(snd_mpu_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_irq, "MPU-401 IRQ # for CS4231 driver. [IRQ]");
MODULE_PARM(snd_dma1, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1, "DMA1 # for CS4231 driver. [DMA]");
MODULE_PARM(snd_dma1_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1_size, "Size of DMA1 # for CS4231 driver. [DMA_SIZE]");
MODULE_PARM(snd_dma2, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma2, "DMA2 # for CS4231 driver. [DMA]");
MODULE_PARM(snd_dma2_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma2_size, "Size of DMA2 # for CS4231 driver. [DMA_SIZE]");

static struct snd_card_cs4231 {
	snd_irq_t *irqptr;
	snd_irq_t *mpuirqptr;
	snd_dma_t *dma1ptr;
	snd_dma_t *dma2ptr;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	cs4231_t *codec;
	unsigned short pcm_status_reg;
} *snd_card_cs4231_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_card_cs4231_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_cs4231_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static snd_pcm_t * __init snd_card_cs4231_detect(snd_card_t * card,
						 unsigned short port,
						 unsigned short mpu_port,
						 snd_irq_t *irqptr,
						 snd_dma_t *dma1ptr,
						 snd_dma_t *dma2ptr)
{
	snd_pcm_t *pcm;

	if (snd_register_ioport(card, port, 4, "CS4231", NULL) < 0)
		return NULL;
	if (mpu_port != SND_AUTO_PORT) {
		if (snd_register_ioport(card, mpu_port, 2, "CS4231 - MPU-401", NULL) < 0) {
			snd_unregister_ioports(card);
			snd_printk("ports for CS4231 MPU-401 are already used, try another\n");
			return NULL;
		}
	}
	if (snd_cs4231_new_pcm(card, 0, port, irqptr,
			       dma1ptr, dma2ptr, CS4231_HW_DETECT, 0, &pcm) < 0) {
		snd_unregister_ioports(card);
		return NULL;
	}
	return pcm;
}

static void snd_card_cs4231_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_card_cs4231 *acard;
	unsigned char status;

	acard = (struct snd_card_cs4231 *) dev_id;
	if (acard == NULL || acard->pcm == NULL)
		return;
	status = inb(acard->pcm_status_reg);
	if (status & 0x01)
		snd_cs4231_interrupt(acard->pcm, status);
}

static void snd_card_cs4231_midi_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_card_cs4231 *acard;

	acard = (struct snd_card_cs4231 *) dev_id;
	if (acard == NULL || acard->rmidi == NULL)
		return;
	snd_mpu401_uart_interrupt(acard->rmidi);
}

static int __init snd_card_cs4231_resources(int dev,
					    struct snd_card_cs4231 *acard,
					    snd_card_t * card)
{
	static int possible_irqs[] = {-1};
	static int possible_dmas[] = {-1};
	int err;

	if ((err = snd_register_interrupt(card, "CS4231",
			snd_irq[dev], SND_IRQ_TYPE_ISA,
			snd_card_cs4231_interrupt, acard,
			possible_irqs, &acard->irqptr)) < 0)
		return err;
	acard->mpuirqptr = NULL;
	if (snd_mpu_port[dev] >= 0 && snd_mpu_port[dev] != SND_AUTO_PORT && snd_mpu_irq[dev] >= 0) {
		if (snd_register_interrupt(card,
				"CS4231 - MPU401", snd_mpu_irq[dev],
				SND_IRQ_TYPE_ISA,
				snd_card_cs4231_midi_interrupt,
				acard, possible_irqs, &acard->mpuirqptr) < 0) {
			snd_mpu_port[dev] = SND_AUTO_PORT;
			snd_printk("warning: can't allocate IRQ for MPU401, midi port is disabled\n");
		}
	} else {
		snd_mpu_port[dev] = SND_AUTO_PORT;
		acard->mpuirqptr = NULL;
	}
	if ((err = snd_register_dma_channel(card,
			"CS4231 - DMA1", snd_dma1[dev],
			SND_DMA_TYPE_ISA, snd_dma1_size[dev],
			possible_dmas, &acard->dma1ptr)) < 0)
		return err;
	if (snd_dma2[dev] >= 0 && snd_dma2[dev] < 8) {
		if ((err = snd_register_dma_channel(card,
				"CS4231 - DMA2", snd_dma2[dev],
				SND_DMA_TYPE_ISA, snd_dma2_size[dev],
				possible_dmas, &acard->dma2ptr)) < 0)
			return err;
	} else {
		acard->dma2ptr = NULL;
	}
	return 0;
}

static int __init snd_card_cs4231_probe(int dev, struct snd_card_cs4231 *acard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	snd_rawmidi_t *rmidi = NULL;
	cs4231_t *codec;

	card = snd_card_new(snd_index[dev], snd_id[dev],
		       snd_card_cs4231_use_inc, snd_card_cs4231_use_dec);
	if (card == NULL)
		return -ENOMEM;
	card->type = SND_CARD_TYPE_CS4231;
	if (snd_card_cs4231_resources(dev, acard, card) < 0) {
		snd_card_free(card);
		return -EBUSY;
	}
	if (snd_mpu_port[dev] < 0)
		snd_mpu_port[dev] = SND_AUTO_PORT;
	pcm = snd_card_cs4231_detect(card, snd_port[dev],
				     snd_mpu_port[dev],
				     acard->irqptr, acard->dma1ptr,
				     acard->dma2ptr == NULL ?
				     	acard->dma1ptr : acard->dma2ptr);
	if (pcm == NULL) {
		snd_card_free(card);
		return -ENODEV;
	}
	codec = snd_magic_cast(cs4231_t, pcm->private_data, -ENXIO);
	acard->codec = codec;
	acard->pcm_status_reg = codec->port + 2;

	if (snd_cs4231_new_mixer(pcm, 0, &mixer) < 0)
		goto __nodev;

	if (acard->mpuirqptr) {
		if (snd_mpu401_uart_new(card, 0, MPU401_HW_CS4232,
					snd_mpu_port[dev],
					acard->mpuirqptr->irq,
					&rmidi) < 0)
			snd_printk("CS4231: MPU401 not detected\n");
	}
	strcpy(card->abbreviation, "CS4231");
	strcpy(card->shortname, pcm->name);
	sprintf(card->longname, "%s at 0x%x, irq %i, dma %i",
		pcm->name,
		codec->port,
		acard->irqptr->irq,
		acard->dma1ptr->dma);
	if (acard->dma2ptr)
		sprintf(card->longname + strlen(card->longname), "&%i",
			acard->dma2ptr->dma);
	if (!snd_card_register(card)) {
		acard->card = card;
		acard->pcm = pcm;
		acard->mixer = mixer;
		acard->rmidi = rmidi;
		return 0;
	}

      __nodev:
	snd_card_free(card);
	return -ENXIO;
}

#ifdef MODULE
int __init init_module(void)
#else
int __init alsa_card_cs4231_init(void)
#endif
{
	int dev, cards;
	struct snd_card_cs4231 *acard;

	for (dev = cards = 0; dev < SND_CARDS && snd_port[dev] > 0; dev++) {
		acard = (struct snd_card_cs4231 *)
				snd_kcalloc(sizeof(struct snd_card_cs4231), GFP_KERNEL);
		if (acard == NULL)
			continue;
		if (snd_card_cs4231_probe(dev, acard) < 0) {
			snd_kfree(acard);
			if (snd_port[dev] == SND_AUTO_PORT)
				break;
#ifdef MODULE
			snd_printk("CS4231 soundcard #%i not found at 0x%x or device busy\n", dev + 1, snd_port[dev]);
#endif
			continue;
		}
		snd_card_cs4231_cards[dev] = acard;
		cards++;
	}
	if (!cards) {
#ifdef MODULE
		snd_printk("CS4231 soundcard #%i not found or device busy\n", dev + 1);
#endif
		return -ENODEV;
	}
	return 0;
}

#ifdef MODULE

void __exit cleanup_module(void)
{
	int idx;
	struct snd_card_cs4231 *acard;

	for (idx = 0; idx < SND_CARDS; idx++) {
		acard = snd_card_cs4231_cards[idx];
		if (acard) {
			snd_card_unregister(acard->card);
			snd_kfree(acard);
		}
	}
}

#endif
