/*
 * Copyright (C) by Paul Barton-Davis 1998-1999
 *
 * This file is distributed under the GNU GENERAL PUBLIC LICENSE (GPL)
 * Version 2 (June 1991). See the "COPYING" file distributed with this
 * software for more info.  
 */

/* The low level driver for the WaveFront ICS2115 MIDI interface(s)
 *
 * Note that there is also an MPU-401 emulation (actually, a UART-401
 * emulation) on the CS4232 on the Tropez and Tropez Plus. This code
 * has nothing to do with that interface at all.
 *
 * The interface is essentially just a UART-401, but is has the
 * interesting property of supporting what Turtle Beach called
 * "Virtual MIDI" mode. In this mode, there are effectively *two*
 * MIDI buses accessible via the interface, one that is routed
 * solely to/from the external WaveFront synthesizer and the other
 * corresponding to the pin/socket connector used to link external
 * MIDI devices to the board.
 *
 * This driver fully supports this mode, allowing two distinct MIDI
 * busses to be used completely independently, giving 32 channels of
 * MIDI routing, 16 to the WaveFront synth and 16 to the external MIDI
 * bus. The devices are named /dev/snd/midiCnD0 and /dev/snd/midiCnD1,
 * where `n' is the card number. Note that the device numbers may be
 * something other than 0 and 1 if the CS4232 UART/MPU-401 interface
 * is enabled.
 *
 * Switching between the two is accomplished externally by the driver
 * using the two otherwise unused MIDI bytes. See the code for more details.
 *
 * NOTE: VIRTUAL MIDI MODE IS ON BY DEFAULT (see lowlevel/isa/wavefront.c)
 *
 * The main reason to turn off Virtual MIDI mode is when you want to
 * tightly couple the WaveFront synth with an external MIDI
 * device. You won't be able to distinguish the source of any MIDI
 * data except via SysEx ID, but thats probably OK, since for the most
 * part, the WaveFront won't be sending any MIDI data at all.
 *  
 * The main reason to turn on Virtual MIDI Mode is to provide two
 * completely independent 16-channel MIDI buses, one to the
 * WaveFront and one to any external MIDI devices. Given the 32
 * voice nature of the WaveFront, its pretty easy to find a use
 * for all 16 channels driving just that synth.
 *  
 */

#include "../../include/driver.h"
#include "../../include/snd_wavefront.h"

static inline int 
wf_mpu_status (snd_wavefront_midi_t *midi)

{
	return inb (midi->mpu_status_port);
}

static inline int 
input_avail (snd_wavefront_midi_t *midi)

{
	return !(wf_mpu_status(midi) & INPUT_AVAIL);
}

static inline int
output_ready (snd_wavefront_midi_t *midi)

{
	return !(wf_mpu_status(midi) & OUTPUT_READY);
}

static inline int 
read_data (snd_wavefront_midi_t *midi)

{
	return inb (midi->mpu_data_port);
}

static inline void 
write_data (snd_wavefront_midi_t *midi, unsigned char byte)

{
	outb (byte, midi->mpu_data_port);
}

static snd_wavefront_midi_t *
get_wavefront_midi (snd_rawmidi_t *rmidi)

{
	snd_card_t *card;
	snd_wavefront_card_t *acard;

	if (rmidi == NULL) 
	        return NULL;

	card = rmidi->card;

	if (card == NULL) 
	        return NULL;

	if (card->private_data == NULL) 
 	        return NULL;

	acard = card->private_data;

	return &acard->wavefront.midi;
}

static void
snd_wavefront_midi_output_write (snd_rawmidi_t *rmidi)
{
	snd_wavefront_midi_t *midi;
	snd_wavefront_mpu_id  mpu;
	unsigned char switchch;
	unsigned long flags;
	unsigned char byte;
	int max = 256;
	int timeout;
	unsigned int mode;

	if (rmidi->private_data == NULL)
		return;

	mpu = *((snd_wavefront_mpu_id *) rmidi->private_data);

	if ((midi = get_wavefront_midi (rmidi)) == NULL)
	        return;

	if (mpu == internal_mpu) {
		mode = midi->internal_mode;
	} else {
		mode = midi->external_mode;
	}
	if (!(mode & MPU401_MODE_OUTPUT)) {
		return;
	}

	/* Its not OK to try to change the status of "virtuality" of
	   the MIDI interface while we're outputting stuff.  See
	   snd_wavefront_midi_{enable,disable}_virtual () for the
	   other half of this.  
	*/

	spin_lock_irqsave (&midi->virtual, flags);

	if (midi->isvirtual && mpu != midi->output_mpu) {
		
		if (mpu == internal_mpu) { 
			switchch = WF_INTERNAL_SWITCH;
		} else {
			switchch = WF_EXTERNAL_SWITCH;
		}

		/* XXX fix me - no hard timing loops allowed! */
	  again:
		for (timeout = 30000; timeout > 0; timeout--) {
			if (output_ready (midi))
			        break;
		}

		if (output_ready (midi)) {
			write_data(midi, switchch);
		} else {
			goto again;
		}
	} 

	midi->output_mpu = mpu;

	while (max > 0) {

		/* XXX fix me - no hard timing loops allowed! */

		for (timeout = 30000; timeout > 0; timeout--) {
			if (output_ready (midi))
				break;
		}

		if (output_ready (midi)) {
			if (mpu == internal_mpu) {
				mode = midi->internal_mode;
			} else {
				mode = midi->external_mode;
			}

			if (!(mode & MPU401_MODE_OUTPUT) ||
			    rmidi->output.data (rmidi, &byte, 1) != 1) {
				spin_unlock_irqrestore (&midi->virtual, flags);
				return;
			}

			write_data(midi, byte);
			max--;
		} else {
			break;
		}
	}
}

static int 
snd_wavefront_midi_input_open(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	snd_wavefront_midi_t *midi;
	snd_wavefront_mpu_id mpu;

	if (rmidi == NULL) 
	        return -EIO;

	if (rmidi->private_data == NULL)
	        return -EIO;

	mpu = *((snd_wavefront_mpu_id *) rmidi->private_data);

	if ((midi = get_wavefront_midi (rmidi)) == NULL)
	        return -EIO;

	spin_lock_irqsave (&midi->open, flags);
	if (mpu == internal_mpu) {
		midi->internal_mode |= MPU401_MODE_INPUT;
	} else {
		midi->external_mode |= MPU401_MODE_INPUT;
	}
	spin_unlock_irqrestore (&midi->open, flags);

	return 0;
}

static int 
snd_wavefront_midi_output_open(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	snd_wavefront_midi_t *midi;
	snd_wavefront_mpu_id mpu;

	if (rmidi == NULL) 
	        return -EIO;

	if (rmidi->private_data == NULL)
	        return -EIO;

	mpu = *((snd_wavefront_mpu_id *) rmidi->private_data);

	if ((midi = get_wavefront_midi (rmidi)) == NULL)
	        return -EIO;

	spin_lock_irqsave (&midi->open, flags);

	if (mpu == internal_mpu) {
		midi->internal_mode |= MPU401_MODE_OUTPUT;
	} else {
		midi->external_mode |= MPU401_MODE_OUTPUT;
	}
	spin_unlock_irqrestore (&midi->open, flags);

	return 0;
}

static int snd_wavefront_midi_input_close(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	snd_wavefront_midi_t *midi;
	snd_wavefront_mpu_id mpu;

	if (rmidi == NULL) 
	        return -EIO;

	if (rmidi->private_data == NULL)
	        return -EIO;

	mpu = *((snd_wavefront_mpu_id *) rmidi->private_data);

	if ((midi = get_wavefront_midi (rmidi)) == NULL)
	        return -EIO;

	spin_lock_irqsave (&midi->open, flags);
	if (mpu == internal_mpu) {
		midi->internal_mode &= ~MPU401_MODE_INPUT;
	} else {
		midi->external_mode &= ~MPU401_MODE_INPUT;
	}
	spin_unlock_irqrestore (&midi->open, flags);

	return 0;
}

static int snd_wavefront_midi_output_close(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	snd_wavefront_midi_t *midi;
	snd_wavefront_mpu_id mpu;

	if (rmidi == NULL) 
	        return -EIO;

	if (rmidi->private_data == NULL)
	        return -EIO;

	mpu = *((snd_wavefront_mpu_id *) rmidi->private_data);

	if ((midi = get_wavefront_midi (rmidi)) == NULL)
	        return -EIO;

	spin_lock_irqsave (&midi->open, flags);

	if (mpu == internal_mpu) {
		midi->internal_mode &= ~MPU401_MODE_OUTPUT;
	} else {
		midi->external_mode &= ~MPU401_MODE_OUTPUT;
	}

	spin_unlock_irqrestore (&midi->open, flags);
	return 0;
}

static void snd_wavefront_midi_input_trigger(snd_rawmidi_t * rmidi, int up)
{
	unsigned long flags;
	snd_wavefront_midi_t *midi;
	snd_wavefront_mpu_id mpu;

	if (rmidi == NULL) 
	        return;

	if (rmidi->private_data == NULL)
	        return;

	mpu = *((snd_wavefront_mpu_id *) rmidi->private_data);

	if ((midi = get_wavefront_midi (rmidi)) == NULL) {
		return;
	}

	spin_lock_irqsave (&midi->input, flags);
	if (up) {
		if (mpu == internal_mpu) {
			midi->internal_mode |= MPU401_MODE_INPUT_TRIGGER;
		} else {
			midi->external_mode |= MPU401_MODE_INPUT_TRIGGER;
		}
	} else {
		if (mpu == internal_mpu) {
			midi->internal_mode &= ~MPU401_MODE_INPUT_TRIGGER;
		} else {
			midi->external_mode &= ~MPU401_MODE_INPUT_TRIGGER;
		}
	}
	spin_unlock_irqrestore (&midi->input, flags);
}

void
snd_wavefront_midi_interrupt (snd_wavefront_card_t *card)

{
	unsigned long flags;
	snd_wavefront_midi_t *midi;
	static snd_rawmidi_t *rmidi = NULL;
	unsigned int mode;
	int max = 128;
	unsigned char byte;

	midi = &card->wavefront.midi;

	if (!input_avail (midi)) { /* not for us */
		return;
	}

	while (--max) {
		spin_lock_irqsave (&midi->input, flags);

		if (input_avail (midi)) {
			byte = read_data (midi);

			if (midi->isvirtual) {
				
				if (byte == WF_EXTERNAL_SWITCH) {
					rmidi = card->ics2115_external_rmidi;
				} else if (byte == WF_INTERNAL_SWITCH) { 
					rmidi = card->ics2115_internal_rmidi;
				} /* else just leave it as it is */
				
			} else {
				rmidi = card->ics2115_internal_rmidi;
			}

			if (rmidi == card->ics2115_internal_rmidi) {
				mode = card->wavefront.midi.internal_mode;
			} else if (rmidi == card->ics2115_external_rmidi) {
				mode = card->wavefront.midi.external_mode;
			} else {
				return;
			}

			spin_unlock_irqrestore (&midi->input, flags);
			if (mode & MPU401_MODE_INPUT_TRIGGER) {
				rmidi->input.data (rmidi, &byte, 1);
			} 
		} else {
			spin_unlock_irqrestore (&midi->input, flags);
			break;
		}
	} 
}

void
snd_wavefront_midi_enable_virtual (snd_wavefront_card_t *card)

{
	unsigned long flags;

	spin_lock_irqsave (&card->wavefront.midi.virtual, flags);
	card->wavefront.midi.isvirtual = 1;
	card->wavefront.midi.output_mpu = internal_mpu;
	card->wavefront.midi.input_mpu = internal_mpu;
	spin_unlock_irqrestore (&card->wavefront.midi.virtual, flags);
}

void
snd_wavefront_midi_disable_virtual (snd_wavefront_card_t *card)

{
	unsigned long flags;

	spin_lock_irqsave (&card->wavefront.midi.virtual, flags);
	snd_wavefront_midi_input_close (card->ics2115_external_rmidi);
	snd_wavefront_midi_output_close (card->ics2115_external_rmidi);
	card->wavefront.midi.isvirtual = 0;
	spin_unlock_irqrestore (&card->wavefront.midi.virtual, flags);
}

int
snd_wavefront_midi_start (snd_wavefront_card_t *card)

{
	int ok, i;
	unsigned char rbuf[4], wbuf[4];
	snd_wavefront_t *dev;
	snd_wavefront_midi_t *midi;

	dev = &card->wavefront;
	midi = &dev->midi;

	/* The ICS2115 MPU-401 interface doesn't do anything
	   until its set into UART mode.
	*/

	/* XXX fix me - no hard timing loops allowed! */

	for (i = 0; i < 30000 && !output_ready (midi); i++);

	if (!output_ready (midi)) {
		snd_printk (LOGNAME "MIDI interface not ready for command\n");
		return -1;
	}

	/* Any interrupts received from now on
	   are owned by the MIDI side of things.
	*/

	dev->interrupts_are_midi = 1;
	
	outb (UART_MODE_ON, midi->mpu_command_port);

	for (ok = 0, i = 50000; i > 0 && !ok; i--) {
		if (input_avail (midi)) {
			if (read_data (midi) == MPU_ACK) {
				ok = 1;
				break;
			}
		}
	}

	if (!ok) {
		snd_printk (LOGNAME "cannot set UART mode for MIDI interface");
		dev->interrupts_are_midi = 0;
		return -1;
	}

	/* Route external MIDI to WaveFront synth (by default) */
    
	if (snd_wavefront_cmd (dev, WFC_MISYNTH_ON, rbuf, wbuf)) {
		snd_printk (LOGNAME "can't enable MIDI-IN-2-synth routing.\n");
		/* XXX error ? */
	}

	/* Turn on Virtual MIDI, but first *always* turn it off,
	   since otherwise consectutive reloads of the driver will
	   never cause the hardware to generate the initial "internal" or 
	   "external" source bytes in the MIDI data stream. This
	   is pretty important, since the internal hardware generally will
	   be used to generate none or very little MIDI output, and
	   thus the only source of MIDI data is actually external. Without
	   the switch bytes, the driver will think it all comes from
	   the internal interface. Duh.
	*/

	if (snd_wavefront_cmd (dev, WFC_VMIDI_OFF, rbuf, wbuf)) { 
		snd_printk (LOGNAME "virtual MIDI mode not disabled\n");
		return 0; /* We're OK, but missing the external MIDI dev */
	}

	snd_wavefront_midi_enable_virtual (card);

	if (snd_wavefront_cmd (dev, WFC_VMIDI_ON, rbuf, wbuf)) {
		snd_printk (LOGNAME "cannot enable virtual MIDI mode.\n");
		snd_wavefront_midi_disable_virtual (card);
	} 
	return 0;
}

struct snd_stru_rawmidi_direction_hw snd_wavefront_midi_output =
{
	SND_RAWMIDI_HW_POLL,		     /* flags */
	NULL,				     /* private_data */
	NULL,				     /* private_free */
	snd_wavefront_midi_output_open,	     /* open */
	snd_wavefront_midi_output_close,     /* close */
	NULL,                                /* io.trigger */
	{ snd_wavefront_midi_output_write }, /* io.write */
	NULL,				     /* abort */
};

struct snd_stru_rawmidi_direction_hw snd_wavefront_midi_input =
{
	0,				 /* flags */
	NULL,				 /* private_data */
	NULL,				 /* private_free */
	snd_wavefront_midi_input_open,	 /* open */
	snd_wavefront_midi_input_close,	 /* close */
	snd_wavefront_midi_input_trigger,/* trigger */
	{NULL},				 /* io.read */
	NULL,				 /* abort */
};

