/*
 *  ALSA sequencer Client Manager
 *  Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *                        Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../include/driver.h"
#include "../../include/minors.h"
#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif

#include "../../include/seq_kernel.h"
#include "seq_clientmgr.h"
#include "seq_memory.h"
#include "seq_queue.h"
#include "seq_timer.h"
#include "seq_info.h"
#include "seq_system.h"
#include "seq_lock.h"

#undef TRACK_SUBSCRIBE
#define SHOW_CAPABILITY

/* Client Manager

 * this module handles the connections of userland and kernel clients
 * 
 */

#define SND_SEQ_LFLG_INPUT	0x0001
#define SND_SEQ_LFLG_OUTPUT	0x0002
#define SND_SEQ_LFLG_OPEN	(SND_SEQ_LFLG_INPUT|SND_SEQ_LFLG_OUTPUT)

#define semaphore_of(fp)	((fp)->f_dentry->d_inode->i_sem)

static spinlock_t clients_lock = SPIN_LOCK_UNLOCKED;
static DECLARE_MUTEX(register_mutex);

static char clienttablock[SND_SEQ_MAX_CLIENTS];
static client_t *clienttab[SND_SEQ_MAX_CLIENTS];
static usage_t client_usage = {0, 0};

static inline unsigned short snd_seq_file_flags(struct file *file)
{
        switch (file->f_mode & (FMODE_READ | FMODE_WRITE)) {
        case FMODE_WRITE:
                return SND_SEQ_LFLG_OUTPUT;
        case FMODE_READ:
                return SND_SEQ_LFLG_INPUT;
        default:
                return SND_SEQ_LFLG_OPEN;
        }
}

static inline int snd_seq_write_pool_allocated(client_t *client)
{
	return snd_seq_total_cells(client->pool) > 0;
}

static inline int snd_seq_output_ok(client_t *client)
{
	return snd_seq_unused_cells(client->pool) >= client->pool_room;
}

/* return pointer to client structure for specified id */
static client_t *clientptr(int clientid)
{
	if (clientid < 0 || clientid >= SND_SEQ_MAX_CLIENTS) {
		snd_printd("Seq: oops. Trying to get pointer to client %d\n", clientid);
		return NULL;
	}
	return clienttab[clientid];
}

extern int snd_seq_client_load[];

client_t *snd_seq_client_use_ptr(int clientid)
{
	unsigned long flags;
	client_t *client;

	spin_lock_irqsave(&clients_lock, flags);
	client = clientptr(clientid);
	if (client)
		goto __lock;
	spin_unlock_irqrestore(&clients_lock, flags);
#ifdef CONFIG_KMOD
	if (!in_interrupt()) {
		if (clientid < 64) {
			int idx;
			char name[32];
			
			for (idx = 0; idx < 64; idx++) {
				if (snd_seq_client_load[idx] < 0)
					break;
				if (snd_seq_client_load[idx] == clientid) {
					sprintf(name, "snd-seq-client-%i", clientid);
					request_module(name);
					break;
				}
			}
		} else if (clientid >= 64 && clientid < 128) {
			snd_request_card((clientid - 64) / 8);
		}
		spin_lock_irqsave(&clients_lock, flags);
		client = clientptr(clientid);
		if (client)
			goto __lock;
		spin_unlock_irqrestore(&clients_lock, flags);
	}
#endif
	return NULL;

      __lock:
	spin_lock(&client->use_lock);
	client->use++;
	spin_unlock(&client->use_lock);
	spin_unlock_irqrestore(&clients_lock, flags);
	return client;
}

void snd_seq_client_unlock(client_t *client)
{
	unsigned long flags;

	if (!client)
		return;
	spin_lock_irqsave(&client->use_lock, flags);
	if (!(--client->use) && waitqueue_active(&client->use_sleep))
	    	wake_up(&client->use_sleep);
	spin_unlock_irqrestore(&client->use_lock, flags);
}

static void usage_alloc(usage_t * res, int num)
{
	res->cur += num;
	if (res->cur > res->peak)
		res->peak = res->cur;
}

static void usage_free(usage_t * res, int num)
{
	res->cur -= num;
}

/* initialise data structures */
void client_init_data(void)
{
	/* zap out the client table */
	memset(&clienttablock, 0, sizeof(clienttablock));
	memset(&clienttab, 0, sizeof(clienttab));
}


static client_t *seq_create_client1(int client_index, int poolsize)
{
	unsigned long flags;
	int c;
	client_t *client;

	/* init client data */
	client = snd_kcalloc(sizeof(client_t), GFP_KERNEL);
	if (!client)
		return NULL;
	client->pool = snd_seq_pool_new();
	if (! client->pool) {
		snd_kfree(client);
		return NULL;
	}
	client->pool_size = poolsize;
	client->pool_room = (poolsize + 1) / 2;
	client->type = NO_CLIENT;
	client->use_lock = SPIN_LOCK_UNLOCKED;
	init_waitqueue_head(&client->use_sleep);
	client->ports_lock = SPIN_LOCK_UNLOCKED;
	init_waitqueue_head(&client->ports_sleep);

	client->output_sleep_lock = SPIN_LOCK_UNLOCKED;
	init_waitqueue_head(&client->output_sleep);
 
	/* find free slot in the client table */
	spin_lock_irqsave(&clients_lock, flags);
	if (client_index < 0) {
		for (c = 128; c < SND_SEQ_MAX_CLIENTS; c++) {
			if (clienttab[c] || clienttablock[c])
				continue;
			clienttab[client->number = c] = client;
			spin_unlock_irqrestore(&clients_lock, flags);
			return client;
		}
	} else {
		if (!clienttab[client_index] && !clienttablock[client_index]) {
			clienttab[client->number = client_index] = client;
			spin_unlock_irqrestore(&clients_lock, flags);
			return client;
		}
	}
	spin_unlock_irqrestore(&clients_lock, flags);
	snd_kfree(client);
	return NULL;	/* no free slot found or busy, return failure code */
}


static int seq_free_client1(client_t *client)
{
	unsigned long flags;

	if (!client)
		return -EINVAL;
	spin_lock_irqsave(&clients_lock, flags);
	clienttablock[client->number] = 1;
	clienttab[client->number] = NULL;
	spin_unlock_irqrestore(&clients_lock, flags);
	spin_lock_irqsave(&client->use_lock, flags);
	while (client->use) {
		snd_seq_sleep_in_lock(&client->use_sleep, &client->use_lock);
	}
	spin_unlock_irqrestore(&client->use_lock, flags);
	snd_seq_queue_client_termination(client->number);
	snd_seq_delete_ports(client);
	snd_seq_queue_client_leave(client->number);
	if (client->pool) {
		snd_seq_pool_done(client->pool);
		snd_seq_pool_delete(&client->pool);
	}
	clienttablock[client->number] = 0;
	return 0;
}


static void seq_free_client(client_t * client)
{
	if (!client)
		return;

	down(&register_mutex);
	switch (client->type) {
	case NO_CLIENT:
		snd_printk("Seq: Trying to free unused client %d\n", client->number);
		break;
	case USER_CLIENT:
	case KERNEL_CLIENT:
		seq_free_client1(client);
		usage_free(&client_usage, 1);
		break;

	default:
		snd_printk("Seq: Trying to free client %d with undefined type = %d\n", client->number, client->type);
	}
	up(&register_mutex);

	snd_seq_system_client_ev_client_exit(client->number);
}



/* -------------------------------------------------------- */

/* create a user client */
static int snd_seq_open(unsigned short minor, int cardnum, int device, struct file *file)
{
	int c, mode;			/* client id */
	client_t *client;
	user_client_t *user;

	down(&register_mutex);
	client = seq_create_client1(-1, SND_SEQ_DEFAULT_EVENTS);
	if (!client) {
		up(&register_mutex);
		return -ENOMEM;	/* failure code */
	}

	mode = snd_seq_file_flags(file);
	if (mode & SND_SEQ_LFLG_INPUT)
		client->accept_input = 1;
	if (mode & SND_SEQ_LFLG_OUTPUT)
		client->accept_output = 1;

	user = &client->data.user;
	user->fifo = NULL;
	user->fifo_pool_size = 0;

	if (mode & SND_SEQ_LFLG_INPUT) {
		user->fifo_pool_size = SND_SEQ_DEFAULT_CLIENT_EVENTS;
		user->fifo = snd_seq_fifo_new(user->fifo_pool_size);
		if (! user->fifo) {
			seq_free_client1(client);
			snd_kfree(client);
			up(&register_mutex);
			return -ENOMEM;
		}
	}

	usage_alloc(&client_usage, 1);
	client->type = USER_CLIENT;
	up(&register_mutex);

	c = client->number;
	(user_client_t *) file->private_data = client;

	/* fill client data */
	user->file = file;
	sprintf(client->name, "Client-%d", c);
	strcpy(client->group, SND_SEQ_GROUP_APPLICATION);

	/* make others aware this new client */
	snd_seq_system_client_ev_client_start(c);

	MOD_INC_USE_COUNT;

	return 0;
}

static int snd_seq_release(unsigned short minor, int cardnum, int device, struct file *file)
{
	client_t *client = (client_t *) file->private_data;

	if (client) {
		seq_free_client(client);
		if (client->data.user.fifo)
			snd_seq_fifo_delete(&client->data.user.fifo);
		snd_kfree(client);
	}

	MOD_DEC_USE_COUNT;
	return 0;
}


/* handle client read() */
static long snd_seq_read(struct file *file, char *buf, long count)
{
	client_t *client = (client_t *) file->private_data;
	fifo_t *fifo;
	int err;
	long result = 0;

	if (!(snd_seq_file_flags(file) & SND_SEQ_LFLG_INPUT))
		return -ENXIO;

	if (verify_area(VERIFY_WRITE, buf, count))
		return -EFAULT;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;

	if (!client->accept_input || !(fifo = client->data.user.fifo))
		return -EIO;

	if (fifo->overflow) {
		/* buffer overflow is detected */
		snd_seq_fifo_clear(fifo);
		/* return error code */
		return -ERANGE;
	}

	/* while data available in queue */
	while (count >= sizeof(snd_seq_event_t)) {
		snd_seq_event_cell_t *cell;
		int len;
		int atomic;

		atomic = ((file->f_flags & O_NONBLOCK) || result > 1)? 1: 0;
		cell = snd_seq_fifo_cell_out(fifo, &err, atomic);
		if (cell == NULL)
			break;

		/* check if buffer is big enough for the new event */
		switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
		case SND_SEQ_EVENT_LENGTH_VARIABLE:
			/* handle variable length data */
			len = sizeof(snd_seq_event_t) + cell->event.data.ext.len;
			break;
		case SND_SEQ_EVENT_LENGTH_FIXED:
		default:
			/* normal event, no special action needed */
			len = sizeof(snd_seq_event_t);
			break;
		}
		/* check if buffer is big enough for the new event */
		if (count < len) {
			if (result == 0)
				err = -EINVAL; /* This will never work */

			/* Put the cell back on the fifo */
			snd_seq_fifo_cell_putback(fifo, cell);
			break; /* not enough room for this new event */
		}

		/* Place the cell into the users read buffer */
		switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
		case SND_SEQ_EVENT_LENGTH_VARIABLE:
			/* handle variable length data */
			len = sizeof(snd_seq_event_t) + cell->event.data.ext.len;
			copy_to_user(buf, &cell->event, sizeof(snd_seq_event_t));
			if (cell->event.data.ext.ptr != NULL) {
				copy_to_user(buf + sizeof(snd_seq_event_t), cell->event.data.ext.ptr, cell->event.data.ext.len);
			}
			break;

		case SND_SEQ_EVENT_LENGTH_FIXED:
		default:
			/* normal event, no special action needed */
			len = sizeof(snd_seq_event_t);
			copy_to_user(buf, &cell->event, sizeof(snd_seq_event_t));
			break;
		}
		/* free cell - DO NOT use snd_seq_cell_free here! */
		snd_seq_fifo_cell_free(fifo, cell);

		result += len;
		count -= len;
		buf += len;
	}

	return err ? err : result;
}


/*
 * check access permission to the port
 * if capability doesn't match, check the group capability, too.
 */
static int check_port_perm(client_port_t *port, char *group, int flags)
{
	if ((port->capability & flags) != flags) {
		if (strncmp(port->group, group, sizeof(port->group)) != 0 ||
		    (port->cap_group & flags) != flags)
			return 0;
	}
	return flags;
}

/*
 * check if the destination client is available, and return the pointer
 * if filter is non-zero, client filter bitmap is tested.
 */
static client_t *get_event_dest_client(snd_seq_event_t *event, int filter)
{
	client_t *dest;

	dest = snd_seq_client_use_ptr(event->dest.client);
	if (! dest)
		return NULL;
	if (! dest->accept_input)
		goto __not_avail;
	if ((dest->filter & SND_SEQ_FILTER_USE_EVENT) &&
	    ! test_bit(event->type, &dest->event_filter))
		goto __not_avail;
	if (filter && !(dest->filter & filter))
		goto __not_avail;

	return dest; /* ok - accessible */
__not_avail:
	snd_seq_client_unlock(dest);
	return NULL;
}


/*
 * deliver an event to the specified destination.
 * if fill_time is true, the current queue time is copied (for direct passing).
 * if filter is non-zero, client filter bitmap is tested.
 *
 *  RETURN VALUE: 1 : if succeeded
 *                0 : event was not passed to the port
 *		 <0 : error
 */
static int snd_seq_deliver_single_event(client_port_t *src_port,
					snd_seq_event_t *event,
					int fill_time, int filter,
					int atomic, int hop)
{
	client_t *dest;
	client_port_t *dest_port;
	int result = 0; /* default return value is zero - event was ignored */
	int direct;

	dest = get_event_dest_client(event, filter);
	if (! dest)
		return 0;

	dest_port = snd_seq_port_use_ptr(dest, event->dest.port);
	if (! dest_port)
		goto __skip;

	/* check permission */
	if (! check_port_perm(dest_port, src_port->group, SND_SEQ_PORT_CAP_WRITE))
		goto __skip;
		
	direct = (event->flags & SND_SEQ_DEST_MASK);
	if (direct)
		fill_time = 1;

	/* fill the current time if required */
	if (fill_time && event->dest.queue < SND_SEQ_MAX_QUEUES) {
		queue_t *q;
		q = queueptr(event->dest.queue);
		if (q && test_bit(dest->number, &q->clients_bitmap)) {
			switch (event->flags & SND_SEQ_TIME_STAMP_MASK) {
			case SND_SEQ_TIME_STAMP_TICK:
				event->time.tick = snd_seq_timer_get_cur_tick(q->timer);
				break;
			case SND_SEQ_TIME_STAMP_REAL:
				event->time.real = snd_seq_timer_get_cur_time(q->timer);
				break;
  			}
		}
	}

	switch (dest->type) {
	case USER_CLIENT:
		if (dest->data.user.fifo) {
			/* LENGTH_VARUSR or LENGTH_VARIPC event will be
			 * converted to LENGTH_VARIABLE in cell_alloc(),
			 * so we don't have to worry about it.
			 */
			if (snd_seq_fifo_event_in(dest->data.user.fifo, event, atomic) == 0)
				result = 1;
			else
				result = 0;
		}
		break;

	case KERNEL_CLIENT:
		if (dest_port->event_input == NULL)
			goto __skip;  /* no input? - just ignored */
		if ((event->flags & SND_SEQ_EVENT_LENGTH_MASK) == SND_SEQ_EVENT_LENGTH_VARUSR &&
		    event->type < SND_SEQ_EVENT_NORMAL_CONTROLS) {
			/* copy VARUSR event to VARIABLE event */
			snd_seq_event_t newev;
			char *ptr;

			memcpy(&newev, event, sizeof(newev));
			ptr = snd_seq_ext_malloc(event->data.ext.len, atomic);
			newev.flags &= ~SND_SEQ_EVENT_LENGTH_MASK;
			newev.flags |= SND_SEQ_EVENT_LENGTH_VARIABLE;
			/* copy data to temporary kernel space */
			if (ptr == NULL) {
				result = -ENOMEM;
				goto __skip;
			}
			if (copy_from_user(ptr, event->data.ext.ptr, event->data.ext.len)) {
				snd_seq_ext_free(ptr, event->data.ext.len);
				result = -EFAULT;
				goto __skip;
			}
			newev.data.ext.ptr = ptr;

			if (dest_port->event_input(&newev, direct, dest_port->private_data, atomic, hop) >= 0)
				result = 1;
			/* free temporary buffer */
			snd_seq_ext_free(ptr, event->data.ext.len);
		} else {
			/* send the event as it is */
			if (dest_port->event_input(event, direct, dest_port->private_data, atomic, hop) >= 0)
				result = 1;
		}
		break;
	default:
		break;
	}

  __skip:
	if (dest_port)
		snd_seq_ports_unlock(dest);
	snd_seq_client_unlock(dest);
	return result;
}


/*
 * send the event to all subscribers:
 */
static int deliver_to_subscribers(client_port_t *src_port,
				  snd_seq_event_t *event,
				  int atomic, int hop)
{
	subscribers_t *subs;
	int err = 0, num_ev = 0;

	if (! src_port->input.list)
		return 0; /* no subscription - skip */

	snd_seq_subscribers_lock(&src_port->input);
	for (subs = src_port->input.list; subs; subs = subs->next) {
		event->dest.client = subs->addr.client;
		event->dest.port = subs->addr.port;
		event->dest.queue = subs->addr.queue;
		/* convert time according to flag with subscription */
		event->flags &= ~SND_SEQ_TIME_STAMP_MASK;
		event->flags |=
			subs->realtime ? SND_SEQ_TIME_STAMP_REAL : SND_SEQ_TIME_STAMP_TICK;
		err = snd_seq_deliver_single_event(src_port, event, 1, 0, atomic, hop);
		if (err < 0)
			break;
		num_ev += err;
	}
	snd_seq_subscribers_unlock(&src_port->input);
	return err ? err : num_ev;
}

/*
 * broadcast to all ports:
 */
static int port_broadcast_event(client_port_t *src_port,
				snd_seq_event_t *event,
				int atomic, int hop)
{
	int num_ev = 0, err = 0;
	client_t *dest_client;
	client_port_t *p;

	dest_client = get_event_dest_client(event, SND_SEQ_FILTER_BROADCAST);
	if (! dest_client)
		return 0;

	snd_seq_ports_lock(dest_client);
	for (p = dest_client->ports; p; p = p->next) {
		event->dest.port = p->port;
		err = snd_seq_deliver_single_event(src_port, event, 0,
						   SND_SEQ_FILTER_BROADCAST,
						   atomic, hop);
		if (err < 0)
			break;
		num_ev++;
	}
	snd_seq_ports_unlock(dest_client);
	snd_seq_client_unlock(dest_client);
	return err ? err : num_ev;
}

/*
 * send the event to all clients:
 * if destination port is also ADDRESS_BROADCAST, deliver to all ports.
 */
static int broadcast_event(client_t *client, client_port_t *src_port,
			   snd_seq_event_t *event, int atomic, int hop)
{
	int err = 0, num_ev = 0;
	int dest, dest_port;

	dest_port = event->dest.port;
	if (event->dest.queue == SND_SEQ_ADDRESS_BROADCAST)
		event->dest.queue = SND_SEQ_ADDRESS_UNKNOWN; /* to be sure */

	for (dest = 0; dest < SND_SEQ_MAX_CLIENTS; dest++) {
		/* don't send to itself */
		if (dest == client->number)
			continue;
		event->dest.client = dest;
		event->dest.port = dest_port;
		if (dest_port == SND_SEQ_ADDRESS_BROADCAST)
			err = port_broadcast_event(src_port, event, atomic, hop);
		else
			err = snd_seq_deliver_single_event(src_port, event, 0,
							   SND_SEQ_FILTER_BROADCAST,
							   atomic, hop);
		if (err < 0)
			break;
		num_ev += err;
	}
	return err ? err : num_ev;
}


/* multicast - not supported yet */
static int multicast_event(client_port_t *src_port, snd_seq_event_t *event,
			   int atomic, int hop)
{
	snd_printd("seq: multicast not supported yet.\n");
	return 0; /* ignored */
}


/* deliver an event to the destination port(s).
 * if the event is to subscribers or broadcast, the event is dispatched
 * to multiple targets.
 * The event record might be overwritten and broken.
 *
 * RETURN VALUE: n > 0  : the number of delivered events.
 *               n == 0 : the event was not passed to any client.
 *               n < 0  : error - event was not processed.
 */
static int snd_seq_deliver_event(client_t *client, snd_seq_event_t *event,
				 int atomic, int hop)
{
	int result;
	client_port_t *src_port;

	hop++;
	if (hop >= SND_SEQ_MAX_HOPS) {
		snd_printd("too long delivery path (%d:%d->%d:%d)\n",
			   event->source.client, event->source.port,
			   event->dest.client, event->dest.port);
		return -EMLINK;
	}

	/* get port pointer - don't forget to release port before return! */
	src_port = snd_seq_port_use_ptr(client, event->source.port);
	if (! src_port)
		return -EINVAL; /* skip */

	if (event->dest.queue == SND_SEQ_ADDRESS_SUBSCRIBERS ||
	    event->dest.client == SND_SEQ_ADDRESS_SUBSCRIBERS)
		result = deliver_to_subscribers(src_port, event, atomic, hop);

	else if (event->dest.queue == SND_SEQ_ADDRESS_BROADCAST ||
		 event->dest.client == SND_SEQ_ADDRESS_BROADCAST)
		result = broadcast_event(client, src_port, event, atomic, hop);
	
	else if (event->dest.client >= SND_SEQ_MAX_CLIENTS)
		result = multicast_event(src_port, event, atomic, hop);

	else if (event->dest.port == SND_SEQ_ADDRESS_BROADCAST)
		result = port_broadcast_event(src_port, event, atomic, hop);

	else
		result = snd_seq_deliver_single_event(src_port, event, 0, 0, atomic, hop);

	snd_seq_ports_unlock(client);
	return result;
}

/*
 * dispatch an event cell:
 * This function is called only from queue check routines in timer
 * interrupts or after enqueued.
 *
 * RETURN VALUE: n > 0  : the number of delivered events.
 *		 n == 0 : the event was not passed to any client.
 *		 n < 0  : error - event was not processed.
 */
int snd_seq_dispatch_event(snd_seq_event_cell_t *cell, int atomic, int hop)
{
	client_t *client;
	unsigned long flags;
	int result, note_converted = 0;

	client = snd_seq_client_use_ptr(cell->event.source.client);
	if (! client)
		return -EINVAL;

	if (cell->event.type == SND_SEQ_EVENT_NOTE) {
		note_converted = 1;
		cell->event.type = SND_SEQ_EVENT_NOTEON;
	}

	result = snd_seq_deliver_event(client, &cell->event, atomic, hop);

	if (note_converted) {
		snd_seq_event_t *ev;

		/*
		 * This was originally a note event.  We now re-use the
		 * cell for the note-off event.
		 */

		ev = &cell->event;
		ev->type = SND_SEQ_EVENT_NOTEOFF;

		/* add the duration time */
		switch (ev->flags & SND_SEQ_TIME_STAMP_MASK) {
		case SND_SEQ_TIME_STAMP_TICK:
			ev->time.tick += ev->data.note.duration;
			break;
		case SND_SEQ_TIME_STAMP_REAL:
			/* unit for duration is ms */
			ev->time.real.tv_nsec +=
			1000000 * (ev->data.note.duration % 1000);
			ev->time.real.tv_sec += ev->data.note.duration / 1000 +
				ev->time.real.tv_nsec % 1000000000;
			ev->time.real.tv_nsec %= 1000000000;
			break;
		}

		/* Now queue this cell as the note off event */
		snd_seq_enqueue_event(cell, atomic, hop);

	} else {

		/* event was processed, or duplicated on fifo.
		   now release the cell */
		snd_seq_cell_free(cell);

		/* if the client is sleeping, wake it up */
		spin_lock_irqsave(&client->output_sleep_lock, flags);
		if (waitqueue_active(&client->output_sleep)) {
			/* has enough space now? */
			if (snd_seq_output_ok(client))
				wake_up(&client->output_sleep);
		}
		spin_unlock_irqrestore(&client->output_sleep_lock, flags);
	}

	snd_seq_client_unlock(client);
	return result;
}


/* Allocate a cell from client pool and enqueue it to queue:
 * if pool is empty and blocking is TRUE, sleep until a new cell is
 * available.
 *
 * possible error values:
 *	-EINVAL	invalid queue / destination
 *	-EAGAIN	no cell (for non-blocking mode)
 *	-EINTR	interrupted while sleeping
 *	-ENOMEM	no memory
 *	-EMLINK too long delivery path (only in direct dispatching)
 *
 * NOTE: VARUSR event is converted to VARIABLE event in seq_event_dup().
 *       Event data is copied from user space to kernel malloced buffer.
 */
static int snd_seq_client_enqueue_event(client_t *client,
					snd_seq_event_t *event,
					struct file *file, int blocking,
					int atomic, int hop)
{
	unsigned long flags;
	snd_seq_event_cell_t *cell;
	queue_t *q;

	/* special queue values - force direct passing */
	if (event->dest.queue == SND_SEQ_ADDRESS_SUBSCRIBERS ||
	    event->dest.queue == SND_SEQ_ADDRESS_BROADCAST)
		event->flags |= SND_SEQ_DEST_DIRECT;

	/* direct event processing without enqueued */
	if ((event->flags & SND_SEQ_DEST_MASK) == SND_SEQ_DEST_DIRECT) {
		if (event->type == SND_SEQ_EVENT_NOTE)
			return -EINVAL; /* this event must be enqueued! */
		return snd_seq_deliver_event(client, event, atomic, hop);
	}

	/* Not direct, normal queuing */
	if (event->dest.queue >= SND_SEQ_MAX_QUEUES)
		return -EINVAL; /* invalid queue */
	q = queueptr(event->dest.queue);
	if (!q || !test_bit(client->number, &q->clients_bitmap))
		return -EINVAL; /* invalid queue */
	if (! snd_seq_write_pool_allocated(client))
		return -EINVAL; /* queue is not allocated */

	/* allocate an event cell */
	cell = NULL;

	spin_lock_irqsave(&client->output_sleep_lock, flags);
	if (snd_seq_unused_cells(client->pool) > 0)
		cell = snd_seq_event_dup(client->pool, event, atomic);

	/* sleep until pool has enough room */
	while (cell == NULL) {
		if (! blocking) {
			spin_unlock_irqrestore(&client->output_sleep_lock, flags);
			return -EAGAIN;
		}
		/* change semaphore to allow other clients
		   to access device file */
		if (file)
			up(&semaphore_of(file));

		snd_seq_sleep_in_lock(&client->output_sleep, &client->output_sleep_lock);

		/* restore semaphore again */
		if (file)
			down(&semaphore_of(file));
		if (signal_pending(current)) {
			spin_unlock_irqrestore(&client->output_sleep_lock, flags);
			return -EINTR;
		}
		/* try again.. */
		cell = snd_seq_event_dup(client->pool, event, atomic);
	}
	spin_unlock_irqrestore(&client->output_sleep_lock, flags);

	/* we got a cell. enqueue it. */
	if (snd_seq_enqueue_event(cell, atomic, hop) < 0) {
		snd_seq_cell_free(cell);
		return -ENOMEM;
	}

	return 0;
}



/* handle write() */
static long snd_seq_write(struct file *file, const char *buf, long count)
{
	client_t *client = (client_t *) file->private_data;
	int written = 0, len;
	int err = 0;
	snd_seq_event_t event;

	if (!(snd_seq_file_flags(file) & SND_SEQ_LFLG_OUTPUT))
		return -ENXIO;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;
		
	if (!client->accept_output || client->pool == NULL)
		return -EIO;

	/* allocate the pool now if the pool is not allocated yet */ 
	if (client->pool_size > 0 && !snd_seq_write_pool_allocated(client)) {
		if (snd_seq_pool_init(client->pool, client->pool_size) < 0)
			return -ENOMEM;
	}

	/* only process whole events */
	while (count >= sizeof(snd_seq_event_t)) {
		int extlen;
		
		/* Read in the event header from the user */
		len = sizeof(event);
		if (copy_from_user(&event, buf, len)) {
			err = -EFAULT;
			break;
		}
		event.source.queue = SND_SEQ_ADDRESS_UNKNOWN;
		event.source.client = client->number;	/* fill in client number */

		/* Check for extension data length */
		switch (event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
		case SND_SEQ_EVENT_LENGTH_VARIABLE:
		case SND_SEQ_EVENT_LENGTH_VARUSR:
			/* Check for invalid extension data length */
			extlen = event.data.ext.len;
			if (extlen < 0) {
				/* back out, will get an error this time or next */
				err = -EINVAL;
				goto error_exit;
			}
			if ((event.flags & SND_SEQ_EVENT_LENGTH_MASK) == SND_SEQ_EVENT_LENGTH_VARIABLE) {
				if (extlen + len > count) {
					/* back out, will get an error this time or next */
					err = -EINVAL;
					goto error_exit;
				}
				/* use VARUSR event */
				event.flags &= ~SND_SEQ_EVENT_LENGTH_MASK;
				event.flags |= SND_SEQ_EVENT_LENGTH_VARUSR;
				/* set user space pointer */
				event.data.ext.ptr = (char*)buf + sizeof(snd_seq_event_t);
				len += extlen;  /* Inc in case of skip */
			}				
			break;
		case SND_SEQ_EVENT_LENGTH_VARIPC:
			err = -EINVAL; /* FIXME: not supported yet.. */
			goto error_exit;
		}

		/* ok, enqueue it */
		err = snd_seq_client_enqueue_event(client, &event, file,
						   !(file->f_flags & O_NONBLOCK),
						   0, 0);
		if (err < 0) {
			if (err != -EINVAL) /* fatal error? */
				break;
		}

		/* Update pointers and counts */
		count -= len;
		buf += len;
		written += len;
	}

  error_exit:
	return written ? written : err;
}


/*
 * handle polling
 */
static unsigned int snd_seq_poll(struct file *file, poll_table * wait)
{
	client_t *client = (client_t *) file->private_data;
	unsigned int mask = 0;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;

	if ((snd_seq_file_flags(file) & SND_SEQ_LFLG_INPUT) &&
	    client->data.user.fifo) {

		/* check if data is available in the outqueue */
		if (snd_seq_fifo_poll_wait(client->data.user.fifo, file, wait))
			mask |= POLLIN | POLLRDNORM;
	}

	if (snd_seq_file_flags(file) & SND_SEQ_LFLG_OUTPUT) {

		poll_wait(file, &client->output_sleep, wait);

		/* check if data is available in the pool */
		if (!snd_seq_write_pool_allocated(client) || snd_seq_output_ok(client))
			mask |= POLLOUT | POLLWRNORM;
	}

	return mask;
}


/*-----------------------------------------------------*/


/* SYSTEM_INFO ioctl() */
static int snd_seq_ioctl_system_info(client_t *client, snd_seq_system_info_t * _info)
{
	snd_seq_system_info_t info;

	/* fill the info fields */
	info.queues = SND_SEQ_MAX_QUEUES;
	info.clients = SND_SEQ_MAX_CLIENTS;
	info.ports = 256;	/* fixed limit */
	info.channels = 256;	/* fixed limit */

	if (copy_to_user(_info, &info, sizeof(info)))
		return -EFAULT;
	return 0;
}


/* CLIENT_INFO ioctl() */
static void get_client_info(client_t *cptr, snd_seq_client_info_t *info)
{
	memset(info, 0, sizeof(*info));
	info->client = cptr->number;

	/* fill the info fields */
	info->type = cptr->type;
	strcpy(info->name, cptr->name);
	strcpy(info->group, cptr->group);
	info->filter = cptr->filter;
	memcpy(info->multicast_filter, cptr->multicast_filter, sizeof(info->multicast_filter));
	memcpy(info->event_filter, cptr->event_filter, sizeof(info->event_filter));
	info->num_ports = cptr->num_ports;
}

static int snd_seq_ioctl_get_client_info(client_t * client, snd_seq_client_info_t * _client_info)
{
	client_t *cptr;
	snd_seq_client_info_t client_info;

	if (copy_from_user(&client_info, _client_info, sizeof(client_info)))
		return -EFAULT;

	/* requested client number */
	cptr = snd_seq_client_use_ptr(client_info.client);
	if (!cptr)
		return -ENOENT;		/* don't change !!! */

	get_client_info(cptr, &client_info);
	snd_seq_client_unlock(cptr);

	if (copy_to_user(_client_info, &client_info, sizeof(client_info)))
		return -EFAULT;
	return 0;
}


/* CLIENT_INFO ioctl() */
static int snd_seq_ioctl_set_client_info(client_t * client, snd_seq_client_info_t * _client_info)
{
	snd_seq_client_info_t client_info;

	if (copy_from_user(&client_info, _client_info, sizeof(client_info)))
		return -EFAULT;

	/* it is not allowed to set the info fields for an another client */
	if (client->number != client_info.client)
		return -EPERM;
	/* also client type must be set now */
	if (client->type != client_info.type)
		return -EINVAL;

	/* fill the info fields */
	if (client_info.name[0]) {
		strncpy(client->name, client_info.name, sizeof(client->name)-1);
		client->name[sizeof(client->name)-1] = '\0';
	}
	if (client_info.group[0]) {
		strncpy(client->group, client_info.group, sizeof(client->group)-1);
		client->group[sizeof(client->group)-1] = '\0';
	}
	client->filter = client_info.filter;
	memcpy(client->multicast_filter, client_info.multicast_filter, sizeof(client->multicast_filter));
	memcpy(client->event_filter, client_info.event_filter, sizeof(client->event_filter));

	return 0;
}


/* 
 * CREATE PORT ioctl() 
 */
static int snd_seq_ioctl_create_port(client_t * client, snd_seq_port_info_t * _info)
{
	client_port_t *port;
	snd_seq_port_info_t info;
	snd_seq_port_callback_t *callback;

	if (in_interrupt())
		return -EBUSY;

	if (!client || !_info)
		return -EINVAL;
	port = snd_seq_create_port(client);
	if (!port)
		return -ENOMEM;

	if (copy_from_user(&info, _info, sizeof(snd_seq_port_info_t))) {
		snd_seq_delete_port(client, port->port);
		return -EFAULT;
	}

	/* it is not allowed to create the port for an another client */
	if (info.client != client->number)
		return -EPERM;
	if (client->type == USER_CLIENT && info.kernel) {
		snd_seq_delete_port(client, port->port);
		return -EINVAL;
	}
	if (client->type == KERNEL_CLIENT) {
		if ((callback = info.kernel) != NULL) {
			port->private_data = callback->private_data;
			port->subscribe = callback->subscribe;
			port->unsubscribe = callback->unsubscribe;
			port->use = callback->use;
			port->unuse = callback->unuse;
			port->event_input = callback->event_input;
			port->private_free = callback->private_free;
		} else {
			port->private_data = NULL;
			port->subscribe = NULL;
			port->unsubscribe = NULL;
			port->use = NULL;
			port->unuse = NULL;
			port->event_input = NULL;
			port->private_free = NULL;
		}
	}

	info.port = port->port;

	snd_seq_set_port_info(port, &info);
	snd_seq_system_client_ev_port_start(client->number, port->port);

	if (copy_to_user(_info, &info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;

	return 0;
}

/* 
 * DELETE PORT ioctl() 
 */
static int snd_seq_ioctl_delete_port(client_t * client, snd_seq_port_info_t * _info)
{
	snd_seq_port_info_t info;
	int err;

	if (in_interrupt())
		return -EBUSY;

	if (!client || !_info)
		return -EINVAL;

	/* set passed parameters */
	if (copy_from_user(&info, _info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;
	
	/* it is not allowed to remove the port for an another client */
	if (info.client != client->number)
		return -EPERM;

	err = snd_seq_delete_port(client, info.port);
	if (err >= 0)
		snd_seq_system_client_ev_port_exit(client->number, info.port);
	return err;
}


/* 
 * GET_PORT_INFO ioctl() (on any client) 
 */
static int snd_seq_ioctl_get_port_info(client_t *client, snd_seq_port_info_t * info)
{
	client_t *cptr;
	client_port_t *port;
	snd_seq_port_info_t _info;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&_info, info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;
	cptr = snd_seq_client_use_ptr(_info.client);
	if (!cptr)
		return -ENXIO;

	port = snd_seq_port_use_ptr(cptr, _info.port);
	if (!port) {
		snd_seq_client_unlock(cptr);
		return -ENOENT;			/* don't change */
	}

	/* get port info */
	snd_seq_get_port_info(port, &_info);
	snd_seq_ports_unlock(cptr);
	snd_seq_client_unlock(cptr);

	if (copy_to_user(info, &_info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;
	return 0;
}


/* 
 * SET_PORT_INFO ioctl() (only ports on this/own client) 
 */
static int snd_seq_ioctl_set_port_info(client_t * client, snd_seq_port_info_t * info)
{
	client_port_t *port;
	snd_seq_port_info_t _info;

	if (!client || !info)
		return -EINVAL;

	if (copy_from_user(&_info, info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;

	if (_info.client != client->number) /* only set our own ports ! */
		return -EPERM;
	port = snd_seq_port_use_ptr(client, _info.port);
	if (port) {
		snd_seq_set_port_info(port, &_info);
		snd_seq_ports_unlock(client);
	}
	return 0;
}


/*
 * port subscription (connection)
 */
#define PERM_RD		(SND_SEQ_PORT_CAP_READ|SND_SEQ_PORT_CAP_SUBS_READ)
#define PERM_WR		(SND_SEQ_PORT_CAP_WRITE|SND_SEQ_PORT_CAP_SUBS_WRITE)

/* check permission for subscription input from sport */
static int check_subscribe_read(client_port_t *sport, client_port_t *dport, snd_seq_port_subscribe_t *info)
{
	if (! check_port_perm(sport, dport->group, PERM_RD))
		return -EPERM;
	if ((info->exclusive && sport->input.list) ||
	    snd_seq_port_is_subscribed(&sport->input, &info->dest))
		return -EBUSY;
	return 0;
}

/* check permission for subscription output to dport */
static int check_subscribe_write(client_port_t *sport, client_port_t *dport, snd_seq_port_subscribe_t *info)
{
	if (! check_port_perm(dport, sport->group, PERM_WR))
		return -EPERM;
	if ((info->exclusive && dport->output.list) ||
	    snd_seq_port_is_subscribed(&dport->output, &info->sender))
		return -EBUSY;
	return 0;
}
  
/* one way subscription: MIDI-in -> this client */
static int subscribe_read(client_port_t *sport, client_port_t *dport, snd_seq_port_subscribe_t *info)
{
	int result;

	if ((result = check_subscribe_read(sport, dport, info)) < 0)
		return result;
	if ((result = snd_seq_port_subscribe(sport, info)) < 0)
		return result;
	result = snd_seq_port_add_subscriber(&sport->input, &info->dest,
					     info->exclusive, info->realtime, 1);
	if (result < 0) {
		snd_seq_port_unsubscribe(sport, info);
		return result;
	}
	result = snd_seq_port_add_subscriber(&dport->itrack, &info->sender,
					     0, 0, 0);
	if (result < 0) {
		snd_seq_port_remove_subscriber(&sport->input, &info->dest);
		snd_seq_port_unsubscribe(sport, info);
		return result;
	}
	return 0;
}

/* one way subscription: this client -> MIDI-out */
static int subscribe_write(client_port_t *sport, client_port_t *dport, snd_seq_port_subscribe_t *info)
{
	int result;

	if ((result = check_subscribe_write(sport, dport, info)) < 0)
		return result;
	if ((result = snd_seq_port_use(dport, info)) < 0)
		return result;
	result = snd_seq_port_add_subscriber(&dport->output, &info->sender,
					     info->exclusive, info->realtime, 1);
	if (result < 0) {
		snd_seq_port_unuse(dport, info);
		return result;
	}
	result = snd_seq_port_add_subscriber(&sport->otrack, &info->dest,
					     0, 0, 0);
	if (result < 0) {
		snd_seq_port_remove_subscriber(&dport->output, &info->dest);
		snd_seq_port_unuse(dport, info);
		return result;
	}
	return 0;
}

/* routing: MIDI-in -> MIDI-out */
static int connect_ports(client_port_t *sport, client_port_t *dport, snd_seq_port_subscribe_t *info)
{
	int result;

	/* check permissions */
	if ((result = check_subscribe_read(sport, dport, info)) < 0)
		return result;
	if ((result = check_subscribe_write(sport, dport, info)) < 0)
		return result;
	if ((result = snd_seq_port_subscribe(sport, info)) < 0)
		return result;
	if ((result = snd_seq_port_use(dport, info)) < 0) {
		snd_seq_port_unsubscribe(sport, info);
		return result;
	}
	result = snd_seq_port_add_subscriber(&sport->input, &info->dest,
					     info->exclusive, info->realtime, 0);
	if (result < 0) {
		snd_seq_port_unsubscribe(sport, info);
		snd_seq_port_unuse(dport, info);
		return result;
	}
	result = snd_seq_port_add_subscriber(&dport->output, &info->sender,
					     info->exclusive, info->realtime, 0);
	if (result < 0) {
		snd_seq_port_remove_subscriber(&sport->input, &info->dest);
		snd_seq_port_unsubscribe(sport, info);
		snd_seq_port_unuse(dport, info);
		return result;
	}
	return 0;
}


/*
 * send an subscription notify event to user client:
 * client must be user client.
 */
int snd_seq_client_notify_subscription(client_t *client, snd_seq_port_subscribe_t *info, int evtype)
{
	snd_seq_event_t event;

	memset(&event, 0, sizeof(event));
	event.flags = SND_SEQ_DEST_DIRECT;
	event.type = evtype;
	event.source.queue = SND_SEQ_ADDRESS_UNKNOWN;
	event.source.client = client->number;
	event.dest.queue = SND_SEQ_ADDRESS_UNKNOWN;
	if (evtype == SND_SEQ_EVENT_PORT_SUBSCRIBED ||
	    evtype == SND_SEQ_EVENT_PORT_UNSUBSCRIBED) {
		event.dest.port = info->sender.port;
		event.dest.client = info->sender.client;
		memcpy(&event.data.addr, &info->dest, sizeof(info->dest));
	} else {
		event.dest.port = info->dest.port;
		event.dest.client = info->dest.client;
		memcpy(&event.data.addr, &info->sender, sizeof(info->sender));
	}

	return snd_seq_deliver_event(client, &event, 0, 0); /* non-atomic */
}


/* 
 * add to port's subscription list IOCTL interface 
 */
static int snd_seq_ioctl_subscribe_port(client_t * client, snd_seq_port_subscribe_t * _subs)
{
	int result = -EINVAL;
	client_t *receiver = NULL, *sender = NULL;
	client_port_t *sport = NULL, *dport = NULL;
	snd_seq_port_subscribe_t subs;

	if (in_interrupt())
		return -EBUSY;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&subs, _subs, sizeof(snd_seq_port_subscribe_t)))
		return -EFAULT;

	if (subs.sender.queue != subs.dest.queue ||
	    subs.sender.queue >= SND_SEQ_MAX_QUEUES)
		return -EINVAL;

	if ((receiver = snd_seq_client_use_ptr(subs.dest.client)) == NULL)
		goto __end;
	if ((sender = snd_seq_client_use_ptr(subs.sender.client)) == NULL)
		goto __end;
	if ((sport = snd_seq_port_use_ptr(sender, subs.sender.port)) == NULL)
		goto __end;
	if ((dport = snd_seq_port_use_ptr(receiver, subs.dest.port)) == NULL)
		goto __end;

	if (client->number == subs.dest.client && receiver != sender) {
		/* one way subscription: MIDI-in -> this client */
		if ((result = subscribe_read(sport, dport, &subs)) >= 0) {
			if (sender->type == USER_CLIENT)
				snd_seq_client_notify_subscription(sender, &subs, SND_SEQ_EVENT_PORT_SUBSCRIBED);
		}
	} else if (client->number == subs.sender.client && receiver != sender) {
		/* one way subscription: this client -> MIDI-out */
		if ((result = subscribe_write(sport, dport, &subs)) >= 0) {
			if (receiver->type == USER_CLIENT)
				snd_seq_client_notify_subscription(receiver, &subs, SND_SEQ_EVENT_PORT_USED);
		}
	} else {
		/* routing: MIDI-in -> MIDI-out */
		result = -EPERM;
		if (check_port_perm(sport, client->group, SND_SEQ_PORT_CAP_NO_EXPORT))
			goto __end;
		if (check_port_perm(dport, client->group, SND_SEQ_PORT_CAP_NO_EXPORT))
			goto __end;
		if ((result = connect_ports(sport, dport, &subs)) >= 0) {
			if (sender->type == USER_CLIENT)
				snd_seq_client_notify_subscription(sender, &subs, SND_SEQ_EVENT_PORT_SUBSCRIBED);
			if (receiver->type == USER_CLIENT)
				snd_seq_client_notify_subscription(receiver, &subs, SND_SEQ_EVENT_PORT_USED);
		}
	}

      __end:
      	if (sport)
		snd_seq_ports_unlock(sender);
	if (dport)
		snd_seq_ports_unlock(receiver);
	if (sender)
		snd_seq_client_unlock(sender);
	if (receiver)
		snd_seq_client_unlock(receiver);
	return result;
}


/* 
 * remove from port's subscription list 
 */
static int snd_seq_ioctl_unsubscribe_port(client_t * client, snd_seq_port_subscribe_t * _subs)
{
	int result = -ENXIO;
	client_t *receiver = NULL, *sender = NULL;
	client_port_t *sport = NULL, *dport = NULL;
	snd_seq_port_subscribe_t subs;

	if (in_interrupt())
		return -EBUSY;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&subs, _subs, sizeof(snd_seq_port_subscribe_t)))
		return -EFAULT;

	if ((receiver = snd_seq_client_use_ptr(subs.dest.client)) == NULL)
		goto __end;
	if ((sender = snd_seq_client_use_ptr(subs.sender.client)) == NULL)
		goto __end;
	if ((sport = snd_seq_port_use_ptr(sender, subs.sender.port)) == NULL)
		goto __end;
	if ((dport = snd_seq_port_use_ptr(receiver, subs.dest.port)) == NULL)
		goto __end;

	if (client->number == subs.dest.client && receiver != sender) {
		/* one way subscription: MIDI-in -> this client */
		if (! snd_seq_port_is_subscribed(&sport->input, &subs.dest)) {
			result = -ENOENT;
			goto __end;
		}
		snd_seq_port_remove_subscriber(&sport->input, &subs.dest);
		snd_seq_port_remove_subscriber(&dport->itrack, &subs.sender);
		snd_seq_port_unsubscribe(sport, &subs);
		if (sender->type == USER_CLIENT)
			snd_seq_client_notify_subscription(sender, &subs, SND_SEQ_EVENT_PORT_UNSUBSCRIBED);
		result = 0;

	} else if (client->number == subs.sender.client && receiver != sender) {
		/* one way subscription: this client -> MIDI-out */
		if (! snd_seq_port_is_subscribed(&dport->output, &subs.sender)) {
			result = -ENOENT;
			goto __end;
		}
		snd_seq_port_remove_subscriber(&dport->output, &subs.sender);
		snd_seq_port_remove_subscriber(&sport->otrack, &subs.dest);
		snd_seq_port_unuse(dport, &subs);
		if (receiver->type == USER_CLIENT)
			snd_seq_client_notify_subscription(receiver, &subs, SND_SEQ_EVENT_PORT_UNUSED);
		result = 0;
		
	} else {
		/* routing: MIDI-in -> MIDI-out */
		if (! snd_seq_port_is_subscribed(&sport->input, &subs.dest) ||
		    ! snd_seq_port_is_subscribed(&dport->output, &subs.sender)) {
			result = -ENOENT;
			goto __end;
		}
		snd_seq_port_remove_subscriber(&sport->input, &subs.dest);
		snd_seq_port_remove_subscriber(&dport->output, &subs.sender);
		snd_seq_port_unsubscribe(sport, &subs);
		snd_seq_port_unuse(dport, &subs);
		if (sender->type == USER_CLIENT)
			snd_seq_client_notify_subscription(sender, &subs, SND_SEQ_EVENT_PORT_UNSUBSCRIBED);
		if (receiver->type == USER_CLIENT)
			snd_seq_client_notify_subscription(receiver, &subs, SND_SEQ_EVENT_PORT_UNUSED);
		result = 0;
	}

      __end:
      	if (sport)
		snd_seq_ports_unlock(sender);
	if (dport)
		snd_seq_ports_unlock(receiver);
	if (sender)
		snd_seq_client_unlock(sender);
	if (receiver)
		snd_seq_client_unlock(receiver);
	return result;
}


/* GET_QUEUE_STATUS ioctl() */
static int snd_seq_ioctl_get_queue_status(client_t * client, snd_seq_queue_status_t * _status)
{
	snd_seq_queue_status_t status;
	queue_t *queue;
	timer_t *tmr;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&status, _status, sizeof(snd_seq_queue_status_t)))
		return -EFAULT;

	queue = queueptr(status.queue);
	if (!queue)
		return -EINVAL;
	memset(&status, 0, sizeof(status));
	status.queue = queue->queue;
	
	tmr = queue->timer;
	status.events = queue->tickq->cells + queue->timeq->cells;

	status.time = snd_seq_timer_get_cur_time(tmr);
	status.tick = snd_seq_timer_get_cur_tick(tmr);

	status.running = tmr->running;

	status.flags = 0;

	if (copy_to_user(_status, &status, sizeof(snd_seq_queue_status_t)))
		return -EFAULT;
	return 0;
}


/* GET_QUEUE_TEMPO ioctl() */
static int snd_seq_ioctl_get_queue_tempo(client_t * client, snd_seq_queue_tempo_t * _tempo)
{
	snd_seq_queue_tempo_t tempo;
	queue_t *queue;
	timer_t *tmr;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&tempo, _tempo, sizeof(snd_seq_queue_tempo_t)))
		return -EFAULT;

	queue = queueptr(tempo.queue);
	if (!queue)
		return -EINVAL;
	memset(&tempo, 0, sizeof(tempo));
	tempo.queue = queue->queue;
	
	tmr = queue->timer;

	tempo.tempo = tmr->tempo;
	tempo.ppq = tmr->ppq;

	if (copy_to_user(_tempo, &tempo, sizeof(snd_seq_queue_tempo_t)))
		return -EFAULT;
	return 0;
}


/* SET_QUEUE_TEMPO ioctl() */
static int snd_seq_ioctl_set_queue_tempo(client_t * client, snd_seq_queue_tempo_t * _tempo)
{
	int result;
	snd_seq_queue_tempo_t tempo;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&tempo, _tempo, sizeof(snd_seq_queue_tempo_t)))
		return -EFAULT;

	if (snd_seq_queue_check_access(tempo.queue, client->number)) {
		result = snd_seq_queue_timer_set_tempo(tempo.queue, client->number, tempo.tempo);
		if (result < 0)
			return result;
		result = snd_seq_queue_timer_set_ppq(tempo.queue, client->number, tempo.ppq);
		if (result < 0)
			return result;
	} else {
		return -EPERM;
	}	

	return 0;
}


/* GET_QUEUE_OWNER ioctl() */
static int snd_seq_ioctl_get_queue_owner(client_t * client, snd_seq_queue_owner_t * _owner)
{
	snd_seq_queue_owner_t owner;
	queue_t *queue;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&owner, _owner, sizeof(snd_seq_queue_owner_t)))
		return -EFAULT;

	queue = queueptr(owner.queue);
	if (!queue)
		return -EINVAL;
	memset(&owner, 0, sizeof(owner));
	owner.queue = queue->queue;
	
	owner.owner = queue->owner;
	owner.locked = queue->locked;

	if (copy_to_user(_owner, &owner, sizeof(snd_seq_queue_owner_t)))
		return -EFAULT;
	return 0;
}


/* SET_QUEUE_OWNER ioctl() */
static int snd_seq_ioctl_set_queue_owner(client_t * client, snd_seq_queue_owner_t * _owner)
{
	snd_seq_queue_owner_t owner;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&owner, _owner, sizeof(snd_seq_queue_owner_t)))
		return -EFAULT;

	if (snd_seq_queue_check_access(owner.queue, client->number)) {
		if (owner.locked)
			snd_seq_queue_use(owner.queue, client->number, 1);
		if (snd_seq_queue_set_locked(owner.queue, client->number, owner.locked) < 0)
			return -EPERM;
		if (snd_seq_queue_set_owner(owner.queue, owner.owner) < 0)
			return -EPERM;
	} else {
		return -EPERM;
	}	

	return 0;
}


/* GET_QUEUE_TIMER ioctl() */
static int snd_seq_ioctl_get_queue_timer(client_t * client, snd_seq_queue_timer_t * _timer)
{
	snd_seq_queue_timer_t timer;
	queue_t *queue;
	timer_t *tmr;

	if (in_interrupt())
		return -EBUSY;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&timer, _timer, sizeof(snd_seq_queue_timer_t)))
		return -EFAULT;

	queue = queueptr(timer.queue);
	if (!queue)
		return -EINVAL;
	tmr = queue->timer;

	memset(&timer, 0, sizeof(timer));
	timer.queue = queue->queue;

	down(&queue->use_mutex);
	timer.type = tmr->type;
	timer.slave = tmr->slave_type;
	timer.number = tmr->timer_no;
	timer.resolution = tmr->resolution;
	timer.midi_client = tmr->midi_client;
	timer.midi_port = tmr->midi_port;
	timer.sync_tick_resolution = tmr->sync_tick_resolution;
	timer.sync_real_resolution = tmr->sync_real_resolution;
	up(&queue->use_mutex);
	
	if (copy_to_user(_timer, &timer, sizeof(snd_seq_queue_timer_t)))
		return -EFAULT;
	return -EINVAL;
}


/* SET_QUEUE_TIMER ioctl() */
static int snd_seq_ioctl_set_queue_timer(client_t * client, snd_seq_queue_timer_t * _timer)
{
	int result = 0;
	snd_seq_queue_timer_t timer;

	if (in_interrupt())
		return -EBUSY;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&timer, _timer, sizeof(snd_seq_queue_timer_t)))
		return -EFAULT;

	if (timer.type < 0 || timer.type >= SND_SEQ_TIMER_MIDI_TICK)
		return -EINVAL;
	if (timer.resolution < 0)
		return -EINVAL;

	if (snd_seq_queue_check_access(timer.queue, client->number)) {
		queue_t *q;
		timer_t *tmr;

		q = queueptr(timer.queue);
		if (q == NULL)
			return -EIO;
		tmr = q->timer;
		down(&q->use_mutex);
		snd_seq_queue_timer_close(timer.queue);
		tmr->type = timer.type;
		tmr->slave_type = timer.slave;
		tmr->timer_no = timer.number;
		tmr->resolution = timer.resolution;
		tmr->midi_client = timer.midi_client;
		tmr->midi_port = timer.midi_port;
		tmr->sync_tick_resolution = timer.sync_tick_resolution;
		tmr->sync_real_resolution = timer.sync_real_resolution;
		result = snd_seq_queue_timer_open(timer.queue);
		up(&q->use_mutex);
	} else {
		return -EPERM;
	}	

	return result;
}


/* GET_QUEUE_SYNC ioctl() */
static int snd_seq_ioctl_get_queue_sync(client_t * client, snd_seq_queue_sync_t * _sync)
{
	snd_seq_queue_sync_t sync;
	queue_t *queue;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&sync, _sync, sizeof(snd_seq_queue_sync_t)))
		return -EFAULT;

	queue = queueptr(sync.queue);
	if (!queue)
		return -EINVAL;
	memset(&sync, 0, sizeof(sync));
	sync.queue = queue->queue;
	
	if (copy_to_user(_sync, &sync, sizeof(snd_seq_queue_sync_t)))
		return -EFAULT;
	return -EINVAL;
}


/* SET_QUEUE_SYNC ioctl() */
static int snd_seq_ioctl_set_queue_sync(client_t * client, snd_seq_queue_sync_t * _sync)
{
	snd_seq_queue_sync_t sync;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&sync, _sync, sizeof(snd_seq_queue_sync_t)))
		return -EFAULT;

	if (snd_seq_queue_check_access(sync.queue, client->number)) {
		/* ... code ... */
		return -EINVAL;
	} else {
		return -EPERM;
	}	

	return 0;
}


/* GET_QUEUE_CLIENT ioctl() */
static int snd_seq_ioctl_get_queue_client(client_t * client, snd_seq_queue_client_t * _info)
{
	snd_seq_queue_client_t info;
	queue_t *queue;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(snd_seq_queue_client_t)))
		return -EFAULT;

	queue = queueptr(info.queue);
	if (!queue)
		return -EINVAL;
	info.client = client->number;
	info.used = test_bit(client->number, &queue->clients_bitmap) ? 1 : 0;

	if (copy_to_user(_info, &info, sizeof(snd_seq_queue_client_t)))
		return -EFAULT;
	return 0;
}


/* SET_QUEUE_CLIENT ioctl() */
static int snd_seq_ioctl_set_queue_client(client_t * client, snd_seq_queue_client_t * _info)
{
	queue_t *queue;
	snd_seq_queue_client_t info;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(snd_seq_queue_client_t)))
		return -EFAULT;

	queue = queueptr(info.queue);
	if (!queue)
		return -EINVAL;

	if (info.used >= 0) {
		snd_seq_queue_use(info.queue, client->number, info.used);
	}

	return snd_seq_ioctl_get_queue_client(client, _info);
}


/* GET_CLIENT_POOL ioctl() */
static int snd_seq_ioctl_get_client_pool(client_t * client, snd_seq_client_pool_t * _info)
{
	snd_seq_client_pool_t info;
	client_t *cptr;

	if (copy_from_user(&info, _info, sizeof(info)))
		return -EFAULT;

	cptr = snd_seq_client_use_ptr(info.client);
	if (!cptr)
		return -ENOENT;
	memset(&info, 0, sizeof(info));
	info.output_pool = cptr->pool_size;
	info.output_room = cptr->pool_room;
	info.output_free = info.output_pool;
	if (cptr->pool)
		info.output_free = snd_seq_unused_cells(cptr->pool);
	if (cptr->type == USER_CLIENT) {
		info.input_pool = cptr->data.user.fifo_pool_size;
		info.input_free = info.input_pool;
		if (cptr->data.user.fifo)
			info.input_free = snd_seq_unused_cells(cptr->data.user.fifo->pool);
	} else {
		info.input_pool = 0;
		info.input_free = 0;
	}
	snd_seq_client_unlock(cptr);
	
	if (copy_to_user(_info, &info, sizeof(info)))
		return -EFAULT;
	return 0;
}

/* SET_CLIENT_POOL ioctl() */
static int snd_seq_ioctl_set_client_pool(client_t * client, snd_seq_client_pool_t * _info)
{
	snd_seq_client_pool_t info;
	int rc;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(info)))
		return -EFAULT;

	if (client->number != info.client)
		return -EINVAL; /* can't change other clients */

	if (info.output_pool >= 1 && info.output_pool <= SND_SEQ_MAX_EVENTS &&
	    info.output_pool != client->pool_size) {
		if (snd_seq_write_pool_allocated(client)) {
			/* remove all existing cells */
			snd_seq_queue_client_leave_cells(client->number);
			snd_seq_pool_done(client->pool);
		}
		client->pool_size = info.output_pool;
		rc = snd_seq_pool_init(client->pool, info.output_pool);
		if (rc < 0)
			return rc;
		client->pool_room = (info.output_pool + 1) / 2;
	}
	if (client->type == USER_CLIENT && client->data.user.fifo != NULL &&
	    info.input_pool >= 1 &&
	    info.input_pool <= SND_SEQ_MAX_CLIENT_EVENTS &&
	    info.input_pool != client->data.user.fifo_pool_size) {
		/* change pool size */
		rc = snd_seq_fifo_resize(client->data.user.fifo, info.input_pool);
		if (rc < 0)
			return rc;
		client->data.user.fifo_pool_size = info.input_pool;
	}
	if (info.output_room >= 1 &&
	    info.output_room <= client->pool_size) {
		client->pool_room  = info.output_room;
	}

	return snd_seq_ioctl_get_client_pool(client, _info);
}


/* RESET_POOL ioctl() */
static int snd_seq_ioctl_reset_pool(client_t * client, snd_seq_reset_pool_t * _info)
{
	snd_seq_reset_pool_t info;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(info)))
		return -EFAULT;

	if (info.reset_output && client->pool_size > 0) {
		switch (info.reset_output) {
		case SND_SEQ_RESET_POOL_ALL:
			snd_seq_queue_client_leave_cells(client->number);
			break;
		default:
			/* FIXME: not implemented yet */
			return -EINVAL;
		}
	}

	if (info.reset_input &&
	    client->type == USER_CLIENT && client->data.user.fifo != NULL &&
	    client->data.user.fifo_pool_size > 0) {
		switch (info.reset_input) {
		case SND_SEQ_RESET_POOL_ALL:
			snd_seq_fifo_clear(client->data.user.fifo);
			break;
		default:
			/* FIXME: not implemented yet */
			return -EINVAL;
		}
	}

	return 0;
}


/* REMOVE_EVENTS ioctl() */
static int snd_seq_ioctl_remove_events(client_t * client,
	snd_seq_remove_events_t *_info)
{
	snd_seq_remove_events_t info;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(info)))
		return -EFAULT;

	/*
	 * Input mostly not implemented XXX.
	 */
	if (info.input) {
		if (info.remove_mode == 0) {
			/*
			 * No restrictions so for a user client we can clear
			 * the whole fifo
			 */
			if (client->type == USER_CLIENT)
				snd_seq_fifo_clear(client->data.user.fifo);

			/* TODO kernel client? */
		}
	}

	if (info.output)
		snd_seq_queue_remove_cells(client->number, &info);

	return 0;
}



/*
 * get subscription info
 */
static int snd_seq_ioctl_get_subscription(client_t *client, snd_seq_port_subscribe_t *_subs)
{
	int result;
	client_t *receiver = NULL, *sender = NULL;
	client_port_t *sport = NULL, *dport = NULL;
	snd_seq_port_subscribe_t subs;
	subscribers_t *p;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&subs, _subs, sizeof(subs)))
		return -EFAULT;

	result = -EINVAL;
	if (client->number == subs.dest.client &&
	    subs.sender.client != subs.dest.client) {
		if ((sender = snd_seq_client_use_ptr(subs.sender.client)) == NULL)
			goto __end;
		if ((sport = snd_seq_port_use_ptr(sender, subs.sender.port)) == NULL)
			goto __end;
		p = snd_seq_port_get_subscription(&sport->input, &subs.dest);
		if (p) {
			result = 0;
			subs.exclusive = sport->input.exclusive;
			subs.realtime = p->realtime;
		}
		snd_seq_subscribers_unlock(&sport->input);
	} else {
		if ((receiver = snd_seq_client_use_ptr(subs.dest.client)) == NULL)
			goto __end;
		if ((dport = snd_seq_port_use_ptr(receiver, subs.dest.port)) == NULL)
			goto __end;
		p = snd_seq_port_get_subscription(&dport->output, &subs.sender);
		if (p) {
			result = 0;
			subs.exclusive = dport->output.exclusive;
			subs.realtime = p->realtime;
		}
		snd_seq_subscribers_unlock(&dport->output);
	}

      __end:
      	if (sport)
		snd_seq_ports_unlock(sender);
	if (dport)
		snd_seq_ports_unlock(receiver);
	if (sender)
		snd_seq_client_unlock(sender);
	if (receiver)
		snd_seq_client_unlock(receiver);
	if (result >= 0) {
		if (copy_to_user(_subs, &subs, sizeof(subs)))
			return -EFAULT;
	}
	return result;
}


/*
 * get subscription info - check only its presence
 */
static int snd_seq_ioctl_query_subs(client_t *client, snd_seq_query_subs_t *_subs)
{
	int result = -ENXIO;
	client_t *cptr = NULL;
	client_port_t *port = NULL;
	snd_seq_query_subs_t subs;
	subscribers_t *p;
	subscribers_group_t *group;
	int i;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&subs, _subs, sizeof(subs)))
		return -EFAULT;

	if ((cptr = snd_seq_client_use_ptr(subs.client)) == NULL)
		goto __end;
	if ((port = snd_seq_port_use_ptr(cptr, subs.port)) == NULL)
		goto __end;

	switch (subs.type) {
	case SND_SEQ_QUERY_SUBS_READ:
		group = &port->input;
		break;
	case SND_SEQ_QUERY_SUBS_WRITE:
		group = &port->output;
		break;
	case SND_SEQ_QUERY_SUBS_READ_TRACK:
		group = &port->itrack;
		break;
	case SND_SEQ_QUERY_SUBS_WRITE_TRACK:
		group = &port->otrack;
		break;
	default:
		goto __end;
	}

	snd_seq_subscribers_lock(group);
	/* search for the subscriber */
	subs.num_subs = group->count;
	p = group->list;
	for (i = 0; i < subs.index; i++) {
		if (p == NULL)
			break;
		p = p->next;
	}
	if (p) {
		/* found! */
		subs.addr = p->addr;
		subs.exclusive = group->exclusive;
		subs.realtime = p->realtime;
		result = 0;
	} else
		result = -ENOENT;
	snd_seq_subscribers_unlock(group);

      __end:
   	if (port)
		snd_seq_ports_unlock(cptr);
	if (cptr)
		snd_seq_client_unlock(cptr);
	if (result >= 0) {
		if (copy_to_user(_subs, &subs, sizeof(subs)))
			return -EFAULT;
	}
	return result;
}


/*
 * query next client
 */
static int snd_seq_ioctl_query_next_client(client_t *client, snd_seq_client_info_t *info)
{
	client_t *cptr = NULL;
	snd_seq_client_info_t _info;
	int len;

	if (copy_from_user(&_info, info, sizeof(_info)))
		return -EFAULT;

	_info.name[sizeof(_info.name)-1] = 0; /* to be sure */
	len = strlen(_info.name);
	/* search for next client */
	_info.client++;
	if (_info.client < 0)
		_info.client = 0;
	for (; _info.client < SND_SEQ_MAX_CLIENTS; _info.client++) {
		cptr = snd_seq_client_use_ptr(_info.client);
		if (cptr) {
			if ((len == 0 || strncmp(_info.name, cptr->name, len) == 0) &&
			    (!*_info.group || strcmp(_info.group, cptr->group) == 0))
				break;
			snd_seq_client_unlock(cptr);
			cptr = NULL;
		}
	}
	if (!cptr)
		return -ENOENT;

	get_client_info(cptr, &_info);
	snd_seq_client_unlock(cptr);

	if (copy_to_user(info, &_info, sizeof(_info)))
		return -EFAULT;
	return 0;
}

/* 
 * query next port
 */
static int snd_seq_ioctl_query_next_port(client_t *client, snd_seq_port_info_t *info)
{
	client_t *cptr;
	client_port_t *port = NULL;
	snd_seq_port_info_t _info;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&_info, info, sizeof(_info)))
		return -EFAULT;
	cptr = snd_seq_client_use_ptr(_info.client);
	if (!cptr)
		return -ENXIO;

	/* search for next port */
	_info.port++;
	port = snd_seq_port_query_nearest(cptr, &_info);
	if (!port) {
		snd_seq_client_unlock(cptr);
		return -ENOENT;
	}

	/* get port info */
	_info.port = port->port;
	snd_seq_get_port_info(port, &_info);
	snd_seq_ports_unlock(cptr);
	snd_seq_client_unlock(cptr);

	if (copy_to_user(info, &_info, sizeof(_info)))
		return -EFAULT;
	return 0;
}

/* -------------------------------------------------------- */

static int snd_seq_do_ioctl(client_t *client, unsigned int cmd, unsigned long arg)
{
	switch (cmd) {

		case SND_SEQ_IOCTL_PVERSION:
			/* return sequencer version number */
			return snd_ioctl_out((long *) arg, SND_SEQ_VERSION);

		case SND_SEQ_IOCTL_CLIENT_ID:
			/* return the id of this client */
			return snd_ioctl_out((long *) arg, client->number);

		case SND_SEQ_IOCTL_SYSTEM_INFO:
			/* return system information */
			return snd_seq_ioctl_system_info(client, (snd_seq_system_info_t *) arg);

		case SND_SEQ_IOCTL_GET_CLIENT_INFO:
			/* return info on specified client */
			return snd_seq_ioctl_get_client_info(client, (snd_seq_client_info_t *) arg);

		case SND_SEQ_IOCTL_SET_CLIENT_INFO:
			/* set info on specified client */
			return snd_seq_ioctl_set_client_info(client, (snd_seq_client_info_t *) arg);

		case SND_SEQ_IOCTL_CREATE_PORT:
			/* create a port for this client */
			return snd_seq_ioctl_create_port(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_DELETE_PORT:
			/* remove a port from this client */
			return snd_seq_ioctl_delete_port(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_GET_PORT_INFO:
			/* get info for specified port */
			return snd_seq_ioctl_get_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_SET_PORT_INFO:
			/* set info for specified port in this client */
			return snd_seq_ioctl_set_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_SUBSCRIBE_PORT:
			/* add to port's subscription list */
			return snd_seq_ioctl_subscribe_port(client, (snd_seq_port_subscribe_t *) arg);

		case SND_SEQ_IOCTL_UNSUBSCRIBE_PORT:
			/* remove from port's subscription list */
			return snd_seq_ioctl_unsubscribe_port(client, (snd_seq_port_subscribe_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_STATUS:
			/* get the status for the specified queue */
			return snd_seq_ioctl_get_queue_status(client, (snd_seq_queue_status_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_TEMPO:
			/* get the tempo for the specified queue */
			return snd_seq_ioctl_get_queue_tempo(client, (snd_seq_queue_tempo_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_TEMPO:
			/* set a tempo for the specified queue */
			return snd_seq_ioctl_set_queue_tempo(client, (snd_seq_queue_tempo_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_OWNER:
			/* get the owner for the specified queue */
			return snd_seq_ioctl_get_queue_owner(client, (snd_seq_queue_owner_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_OWNER:
			/* set an owner for the specified queue */
			return snd_seq_ioctl_set_queue_owner(client, (snd_seq_queue_owner_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_TIMER:
			/* get the timer for the specified queue */
			return snd_seq_ioctl_get_queue_timer(client, (snd_seq_queue_timer_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_TIMER:
			/* set an timer for the specified queue */
			return snd_seq_ioctl_set_queue_timer(client, (snd_seq_queue_timer_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_SYNC:
			/* get the sync source for the specified queue */
			return snd_seq_ioctl_get_queue_sync(client, (snd_seq_queue_sync_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_SYNC:
			/* set an sync source for the specified queue */
			return snd_seq_ioctl_set_queue_sync(client, (snd_seq_queue_sync_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_CLIENT:
			/* get client specific info for specified queue */
			return snd_seq_ioctl_get_queue_client(client, (snd_seq_queue_client_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_CLIENT:
			/* set client specfic info for specified queue */
			return snd_seq_ioctl_set_queue_client(client, (snd_seq_queue_client_t *) arg);

		case SND_SEQ_IOCTL_GET_CLIENT_POOL:
			/* get client pool size */
			return snd_seq_ioctl_get_client_pool(client, (snd_seq_client_pool_t *) arg);

		case SND_SEQ_IOCTL_SET_CLIENT_POOL:
			/* set client pool size */
			return snd_seq_ioctl_set_client_pool(client, (snd_seq_client_pool_t *) arg);

		case SND_SEQ_IOCTL_RESET_POOL:
			/* reset events in client pool */
			return snd_seq_ioctl_reset_pool(client, (snd_seq_reset_pool_t *) arg);

		case SND_SEQ_IOCTL_GET_SUBSCRIPTION:
			/* get subscription info */
			return snd_seq_ioctl_get_subscription(client, (snd_seq_port_subscribe_t *) arg);

		case SND_SEQ_IOCTL_QUERY_NEXT_CLIENT:
			/* query next client */
			return snd_seq_ioctl_query_next_client(client, (snd_seq_client_info_t *)arg);

		case SND_SEQ_IOCTL_QUERY_NEXT_PORT:
			/* query next port */
			return snd_seq_ioctl_query_next_port(client, (snd_seq_port_info_t *)arg);

		case SND_SEQ_IOCTL_REMOVE_EVENTS:
			return snd_seq_ioctl_remove_events(client, (snd_seq_remove_events_t *) arg);

		case SND_SEQ_IOCTL_QUERY_SUBS:
			return snd_seq_ioctl_query_subs(client, (snd_seq_query_subs_t *) arg);

		default:
			snd_printd("seq unknown ioctl() 0x%x (type='%c', number=0x%2x)\n",
				   cmd, _IOC_TYPE(cmd), _IOC_NR(cmd));
	}
	return -ENXIO;
}


static int snd_seq_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	client_t *client = (client_t *) file->private_data;

	if (!client)
		return -ENXIO;
		
	return snd_seq_do_ioctl(client, cmd, arg);
}


/* -------------------------------------------------------- */


/* exported to kernel modules */
int snd_seq_create_kernel_client(snd_card_t *card, int client_index, snd_seq_client_callback_t * callback)
{
	client_t *client;

	if (in_interrupt())
		return -EBUSY;

	if (!callback)
		return -EINVAL;
	if (card && client_index > 7)
		return -EINVAL;
	if (!card && client_index > 63)
		return -EINVAL;
	if (card)
		client_index += 64 + (card->number << 3);

	down(&register_mutex);
	/* empty write queue as default */
	client = seq_create_client1(client_index, 0);
	if (!client) {
		up(&register_mutex);
		return -EBUSY;	/* failure code */
	}
	usage_alloc(&client_usage, 1);

	client->accept_input = callback->allow_output;
	client->accept_output = callback->allow_input;
		
	/* fill client data */
	client->data.kernel.card = card;
	client->data.kernel.private_data = callback->private_data;
	sprintf(client->name, "Client-%d", client->number);

	client->type = KERNEL_CLIENT;
	up(&register_mutex);

	/* make others aware this new client */
	snd_seq_system_client_ev_client_start(client->number);
	
	/* return client number to caller */
	return client->number;
}

/* exported to kernel modules */
int snd_seq_delete_kernel_client(int client)
{
	client_t *ptr;

	if (in_interrupt())
		return -EBUSY;

	ptr = clientptr(client);
	if (!ptr)
		return -EINVAL;

	seq_free_client(ptr);
	snd_kfree(ptr);
	return 0;
}


/* skeleton to enqueue event, called from snd_seq_kernel_client_enqueue
 * and snd_seq_kernel_client_enqueue_blocking
 */
static int kernel_client_enqueue(int client, snd_seq_event_t *ev,
				 struct file *file, int blocking,
				 int atomic, int hop)
{
	client_t *cptr;
	int rc;

	cptr = snd_seq_client_use_ptr(client);
	if (!cptr)
		return -EINVAL;
	
	if (! cptr->accept_output) {
		snd_seq_client_unlock(cptr);
		return -EPERM;
	}

	/* fill in client number */
	ev->source.queue = SND_SEQ_ADDRESS_UNKNOWN;
	ev->source.client = client;

	/* send it */
	rc = snd_seq_client_enqueue_event(cptr, ev, file, blocking, atomic, hop);

	snd_seq_client_unlock(cptr);
	return rc;
}

/*
 * exported, called by kernel clients to enqueue events (w/o blocking)
 *
 * RETURN VALUE: zero if succeed, negative if error
 */
int snd_seq_kernel_client_enqueue(int client, snd_seq_event_t * ev,
				  int atomic, int hop)
{
	return kernel_client_enqueue(client, ev, NULL, 0, atomic, hop);
}

/*
 * exported, called by kernel clients to enqueue events (with blocking)
 *
 * RETURN VALUE: zero if succeed, negative if error
 */
int snd_seq_kernel_client_enqueue_blocking(int client, snd_seq_event_t * ev,
					   struct file *file,
					   int atomic, int hop)
{
	return kernel_client_enqueue(client, ev, file, 1, atomic, hop);
}


/* 
 * exported, called by kernel clients to dispatch events directly to other
 * clients, bypassing the queues.  Event time-stamp will be updated.
 *
 * RETURN VALUE: negative = delivery failed,
 *		 zero, or positive: the number of delivered events
 */
int snd_seq_kernel_client_dispatch(int client, snd_seq_event_t * ev,
				   int atomic, int hop)
{
	client_t *cptr;
	int result;

	cptr = snd_seq_client_use_ptr(client);
	if (!cptr)
		return -EINVAL;

	if (!cptr->accept_output) {
		result = -EPERM;
	} else {
		ev->flags |= SND_SEQ_DEST_DIRECT;
		/* fill in client number */
		ev->source.queue = SND_SEQ_ADDRESS_UNKNOWN;
		ev->source.client = client;

		result = snd_seq_deliver_event(cptr, ev, atomic, hop);
	}
	snd_seq_client_unlock(cptr);
	return result;
}


/*
 * exported, called by kernel clients to perform same functions as with
 * userland ioctl() 
 */
int snd_seq_kernel_client_ctl(int clientid, unsigned int cmd, void *arg)
{
	client_t *client;
	mm_segment_t fs;
	int result;

	client = clientptr(clientid);
	if (!client)
		return -ENXIO;
	fs = snd_enter_user();
	result = snd_seq_do_ioctl(client, cmd, (unsigned long)arg);
	snd_leave_user(fs);
	return result;
}


/* exported (for OSS emulator) */
int snd_seq_kernel_client_write_poll(int clientid, struct file *file, poll_table *wait)
{
	client_t *client;

	client = clientptr(clientid);
	if (!client)
		return -ENXIO;

	if (snd_seq_write_pool_allocated(client)) {
		poll_wait(file, &client->output_sleep, wait);
		if (snd_seq_output_ok(client))
			return 1;
	}

	return 0;
}

/*---------------------------------------------------------------------------*/

/*
 *  /proc interface
 */
static void snd_seq_info_dump_subscribers(snd_info_buffer_t * buffer, subscribers_group_t * group)
{
	subscribers_t *s;

	snd_seq_subscribers_lock(group);
	for (s = group->list; s; s = s->next) {
		snd_iprintf(buffer, "%d:%d:%d",
				s->addr.queue,
				s->addr.client,
				s->addr.port);
		if (s->next)
			snd_iprintf(buffer, ", ");
	}
	snd_seq_subscribers_unlock(group);
}

#define FLAG_PERM_RD(perm) ((perm) & SND_SEQ_PORT_CAP_READ ? ((perm) & SND_SEQ_PORT_CAP_SUBS_READ ? 'R' : 'r') : '-')
#define FLAG_PERM_WR(perm) ((perm) & SND_SEQ_PORT_CAP_WRITE ? ((perm) & SND_SEQ_PORT_CAP_SUBS_WRITE ? 'W' : 'w') : '-')
#define FLAG_PERM_EX(perm) ((perm) & SND_SEQ_PORT_CAP_NO_EXPORT ? '-' : 'e')

static void snd_seq_info_dump_ports(snd_info_buffer_t * buffer, client_port_t * ports)
{
	client_port_t *p = ports;

	while (p) {
#ifdef SHOW_CAPABILITY
		snd_iprintf(buffer, "  Port %3d : \"%s\" (group=%s:%c%c%c) (cap=%c%c%c)\n",
			    p->port, p->name, p->group,
			    FLAG_PERM_RD(p->cap_group),
			    FLAG_PERM_WR(p->cap_group),
			    FLAG_PERM_EX(p->cap_group),
			    FLAG_PERM_RD(p->capability),
			    FLAG_PERM_WR(p->capability),
			    FLAG_PERM_EX(p->capability));
#else
		snd_iprintf(buffer, "  Port %3d : \"%s\"\n",
			    p->port, p->name);
#endif
		if (p->input.list) {
			snd_iprintf(buffer, "    Read Subscribers: ");
			snd_seq_info_dump_subscribers(buffer, &p->input);
			snd_iprintf(buffer, "\n");
		}
		if (p->itrack.list) {
			snd_iprintf(buffer, "    Read Tracking: ");
			snd_seq_info_dump_subscribers(buffer, &p->itrack);
			snd_iprintf(buffer, "\n");
		}
		if (p->output.list) {
			snd_iprintf(buffer, "    Write Subscribers: ");
			snd_seq_info_dump_subscribers(buffer, &p->output);
			snd_iprintf(buffer, "\n");
		}
		if (p->otrack.list) {
			snd_iprintf(buffer, "    Write Tracking: ");
			snd_seq_info_dump_subscribers(buffer, &p->otrack);
			snd_iprintf(buffer, "\n");
		}

		p = p->next;
	}
}


/* exported to seq_info.c */
void snd_seq_info_clients_read(snd_info_buffer_t * buffer, void *private_data)
{
	extern void snd_seq_info_pool(snd_info_buffer_t * buffer, pool_t * pool, char *space);
	int c;
	client_t *client;

	snd_iprintf(buffer, "Client info\n");
	snd_iprintf(buffer, "  cur  clients : %d\n", client_usage.cur);
	snd_iprintf(buffer, "  peak clients : %d\n", client_usage.peak);
	snd_iprintf(buffer, "  max  clients : %d\n", SND_SEQ_MAX_CLIENTS);
	snd_iprintf(buffer, "\n");

	/* list the client table */
	for (c = 0; c < SND_SEQ_MAX_CLIENTS; c++) {
		client = snd_seq_client_use_ptr(c);
		if (!client)
			continue;
		snd_seq_ports_lock(client);
		switch (client->type) {
			case NO_CLIENT:
				/*snd_iprintf(buffer, "Client %3d : Free\n", c); */
				break;

			case USER_CLIENT:
				snd_iprintf(buffer, "Client %3d : \"%s\" [User]\n",
					    c, client->name);
				snd_seq_info_dump_ports(buffer, client->ports);
				if (snd_seq_write_pool_allocated(client)) {
					snd_iprintf(buffer, "  Output pool :\n");
					snd_seq_info_pool(buffer, client->pool, "    ");
				}
				if (client->data.user.fifo->pool) {
					snd_iprintf(buffer, "  Input pool :\n");
					snd_seq_info_pool(buffer, client->data.user.fifo->pool, "    ");
				}
				break;

			case KERNEL_CLIENT:
				snd_iprintf(buffer, "Client %3d : \"%s\" [Kernel]\n",
					    c, client->name);
				snd_seq_info_dump_ports(buffer, client->ports);
				if (snd_seq_write_pool_allocated(client)) {
					snd_iprintf(buffer, "  Output pool :\n");
					snd_seq_info_pool(buffer, client->pool, "    ");
				}
				break;
		}
		snd_seq_ports_unlock(client);
		snd_seq_client_unlock(client);
	}
}


/*---------------------------------------------------------------------------*/


/*
 *  REGISTRATION PART
 */
static snd_minor_t snd_seq_reg =
{
	"sequencer",		/* comment */
	NULL,			/* reserved */
	
	NULL,			/* unregister */
	NULL,			/* lseek */
	snd_seq_read,		/* read */
	snd_seq_write,		/* write */
	snd_seq_open,		/* open */
	snd_seq_release,	/* release */
	snd_seq_poll,		/* poll */
	snd_seq_ioctl,		/* ioctl */
	NULL			/* mmap */
};


/* 
 * register sequencer device 
 */
int snd_sequencer_device_init(void)
{
	int err;

	down(&register_mutex);

	if ((err = snd_register_device(SND_DEVICE_TYPE_SEQUENCER, NULL, 0, &snd_seq_reg, "seq")) < 0) {
		up(&register_mutex);
		return err;
	}
	
	up(&register_mutex);

	return 0;
}



/* 
 * unregister sequencer device 
 */
void snd_sequencer_device_done(void)
{
	snd_unregister_device(SND_DEVICE_TYPE_SEQUENCER, NULL, 0);
}
