/*
 *  Driver for C-Media's CMI8330 soundcards.
 *  Copyright (c) by George Talusan <gstalusan@uwaterloo.ca>
 *    http://www.undergrad.math.uwaterloo.ca/~gstalusa
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * NOTES
 *
 *  The extended registers contain mixer settings which are largely
 *  untapped for the time being.
 *
 *  MPU401 and SPDIF are not supported yet.  I don't have the hardware
 *  to aid in coding and testing, so I won't bother.
 *
 *  To quickly load the module,
 *
 *  modprobe -a snd-card-cmi8330 snd_sbport=0x220 snd_sbirq=5 snd_sbdma8=1
 *    snd_sbdma16=5 snd_wssport=0x530 snd_wssirq=11 snd_wssdma=0
 *
 *  This card has two mixers and two PCM devices.  I've cheesed it such
 *  that recording and playback can be done through the same device.
 *  The driver "magically" routes the capturing to the AD1848 codec,
 *  and playback to the SB16 codec.  This allows for full-duplex mode
 *  to some extent.
 *  The utilities in alsa-utils are aware of both devices, so passing
 *  the appropriate parameters to amixer and alsactl will give you
 *  full control over both mixers.
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE

#include "../include/driver.h"
#include "../include/ad1848.h"
#include "../include/sb.h"
#include "../include/initval.h"

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;
int snd_sbport[SND_CARDS] = SND_DEFAULT_PORT;
int snd_sbirq[SND_CARDS] = SND_DEFAULT_IRQ;
int snd_sbdma8[SND_CARDS] = SND_DEFAULT_DMA;
int snd_sbdma8_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;
int snd_sbdma16[SND_CARDS] = SND_DEFAULT_DMA;
int snd_sbdma16_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;
int snd_wssport[SND_CARDS] = SND_DEFAULT_PORT;
int snd_wssirq[SND_CARDS] = SND_DEFAULT_IRQ;
int snd_wssdma[SND_CARDS] = SND_DEFAULT_DMA;
int snd_wssdma_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;

MODULE_AUTHOR("George Talusan <gstalusan@uwaterloo.ca>");
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for CMI8330 soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string  for CMI8330 soundcard.");
MODULE_PARM(snd_sbport, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_sbport, "Port # for CMI8330 SB driver.");
MODULE_PARM(snd_sbirq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_sbirq, "IRQ # for CMI8330 SB driver.");
MODULE_PARM(snd_sbdma8, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_sbdma8, "DMA8 for CMI8330 SB driver.");
MODULE_PARM(snd_sbdma8_size, __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_sbdma8_size, "DMA8 size in kB for CMI8330 SB driver.");
MODULE_PARM(snd_sbdma16, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_sbdma16, "DMA16 for CMI8330 SB driver.");
MODULE_PARM(snd_sbdma16_size, __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_sbdma16_size, "DMA16 size in kB for CMI8330 SB driver.");
MODULE_PARM(snd_wssport, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_wssport, "Port # for CMI8330 WSS driver.");
MODULE_PARM(snd_wssirq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_wssirq, "IRQ # for CMI8330 WSS driver.");
MODULE_PARM(snd_wssdma, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_wssdma, "DMA for CMI8330 WSS driver.");
MODULE_PARM(snd_wssdma_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_wssdma_size, "DMA size in kB for CMI8330 WSS driver.");

#define CMI8330_RMUX3D    16
#define CMI8330_MUTEMUX   17
#define CMI8330_OUTPUTVOL 18
#define CMI8330_MASTVOL   19
#define CMI8330_LINVOL    20
#define CMI8330_CDINVOL   21
#define CMI8330_WAVVOL    22
#define CMI8330_RECMUX    23
#define CMI8330_WAVGAIN   24
#define CMI8330_LINGAIN   25
#define CMI8330_CDINGAIN  26

static unsigned char snd_cmi8330_image[] =
{
	0x40,			/* 16 - recording mux */
	0x40,			/* 17 - mute mux */
	0x0,			/* 18 - vol */
	0x0,			/* 19 - master volume */
	0x0,			/* 20 - line-in volume */
	0x0,			/* 21 - cd-in volume */
	0x0,			/* 22 - wave volume */
	0x0,			/* 23 - mute/rec mux */
	0x0,			/* 24 - wave rec gain */
	0x0,			/* 25 - line-in rec gain */
	0x0			/* 26 - cd-in rec gain */
};

struct snd_cmi8330_wss {
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_irq_t *irqptr;
	snd_dma_t *dmaptr;
	snd_pcm1_channel_t *playback;
	unsigned short pcm_status_reg;
};

struct snd_cmi8330_sb {
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;
	snd_irq_t *mpuirqptr;
	snd_irq_t *irqptr;
	snd_dma_t *dma8ptr;
	snd_dma_t *dma16ptr;
	snd_pcm1_channel_t *capture;
};

static struct snd_cmi8330 {
	snd_card_t *card;
	struct snd_cmi8330_wss wss;
	struct snd_cmi8330_sb sb;
	unsigned char image[10];
#ifdef CONFIG_ISAPNP
	struct isapnp_dev *cap;
	struct isapnp_dev *play;
#endif
} *snd_cmi8330_cards[SND_CARDS] = SND_DEFAULT_PTR;

#ifdef CONFIG_ISAPNP
static unsigned int snd_cmi8330_pnpids[] =
{
	(ISAPNP_VENDOR('C', 'M', 'I') << 16 | ISAPNP_DEVICE(0x0001)),	/* device */
	(ISAPNP_VENDOR('@', '@', '@') << 16 | ISAPNP_FUNCTION(0x0001)),	/* capture */
	(ISAPNP_VENDOR('@', 'X', '@') << 16 | ISAPNP_FUNCTION(0x0001)),	/* playback */
	0
};
#endif

static void snd_cmi8330_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_cmi8330_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static void snd_cmi8330_wss_interrupt(int irq, void *dev_id,
				      struct pt_regs *regs)
{
	struct snd_cmi8330_wss *acard;
	unsigned char status;

	acard = (struct snd_cmi8330_wss *) dev_id;

	if (acard && acard->pcm) {
		if ((status = inb(acard->pcm_status_reg)) & 0x01) {
			snd_ad1848_interrupt(acard->pcm, status);
		}
	}
}

static void snd_cmi8330_sb_interrupt(int irq, void *dev_id,
				     struct pt_regs *regs)
{
	struct snd_cmi8330_sb *acard;
	sbdsp_t *dsp;
	unsigned long flags;
	unsigned char status;

	acard = (struct snd_cmi8330_sb *) dev_id;

	if (acard && acard->pcm) {
		dsp = (sbdsp_t *) acard->pcm->private_data;

		spin_lock_irqsave(&dsp->mixer.lock, flags);
		status = snd_sb16mixer_read(&dsp->mixer, 0x82);
		spin_unlock_irqrestore(&dsp->mixer.lock, flags);

		if (status & 0x03) {
			snd_sb16dsp_interrupt(acard->pcm, status);
		}
	}
}

static int snd_cmi8330_3d_switch(struct snd_cmi8330 *acard,
				 int w_flag, unsigned int *bitmap)
{
	ad1848_t *codec;
	unsigned long flags;
	unsigned int field;

	codec = (ad1848_t *) acard->wss.pcm->private_data;
	field = (unsigned int) acard->image[CMI8330_RMUX3D];
	spin_lock_irqsave(&codec->reg_lock, flags);

	if (w_flag) {
		if (*bitmap) {	/* turn it on */
			snd_mixer_set_bit(&field, 5, 0);
		} else {	/* turn it off */
			snd_mixer_set_bit(&field, 5, 1);
		}
		snd_ad1848_out(codec, CMI8330_RMUX3D, field);
		acard->image[CMI8330_RMUX3D] = (unsigned char) field;
	} else {
		*bitmap = !snd_mixer_get_bit(&field, 5);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	return 1;
}

static int snd_cmi8330_loud_switch(struct snd_cmi8330 *acard,
				   int w_flag, unsigned int *bitmap)
{
	ad1848_t *codec;
	unsigned long flags;
	unsigned int field;

	codec = (ad1848_t *) acard->wss.pcm->private_data;
	field = (unsigned int) acard->image[CMI8330_MUTEMUX];
	spin_lock_irqsave(&codec->reg_lock, flags);

	if (w_flag) {
		if (*bitmap) {	/* turn it on */
			snd_mixer_set_bit(&field, 6, 0);
		} else {	/* turn it off */
			snd_mixer_set_bit(&field, 6, 1);
		}
		snd_ad1848_out(codec, CMI8330_MUTEMUX, field);
		acard->image[CMI8330_MUTEMUX] = (unsigned char) field;
	} else {
		*bitmap = !snd_mixer_get_bit(&field, 6);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	return 1;
}

static int snd_cmi8330_resources(int dev, struct snd_cmi8330 *acard,
				 snd_card_t * card)
{
	static int possible_irqs[] = {-1};
	static int possible_dmas[] = {-1};
	int err;

	err = snd_register_ioport(card, snd_wssport[dev],  8,
				  "CMI8330/C3D (AD1848)", NULL);
	if (err < 0) {
		snd_printk("CMI8330/C3D (AD1848) could not register ioports\n");
		return err;
	}
	err = snd_register_ioport(card, snd_sbport[dev], 16,
				  "CMI8330/C3D (SB16)", NULL);
	if (err < 0) {
		snd_printk("CMI8330/C3D (SB16) could not register ioports\n");
		return err;
	}
	err = snd_register_interrupt(card, "CMI8330/C3D (AD1848)", snd_wssirq[dev],
			     SND_IRQ_TYPE_ISA, snd_cmi8330_wss_interrupt,
			 &acard->wss, possible_irqs, &acard->wss.irqptr);
	if (err < 0) {
		snd_printk("CMI8330/C3D (AD1848) irq failed\n");
		return err;
	}
	err = snd_register_dma_channel(card, "CMI8330/C3D (AD1848)", snd_wssdma[dev],
		   SND_DMA_TYPE_ISA, snd_wssdma_size[dev], possible_dmas,
				       &acard->wss.dmaptr);
	if (err < 0) {
		snd_printk("CMI8330/C3D (AD1848) dma failed\n");
		return err;
	}
	err = snd_register_interrupt(card, "CMI8330/C3D (SB16)", snd_sbirq[dev],
			      SND_IRQ_TYPE_ISA, snd_cmi8330_sb_interrupt,
			   &acard->sb, possible_irqs, &acard->sb.irqptr);
	if (err < 0) {
		snd_printk("CMI8330/C3D (SB16) irq failed\n");
		return err;
	}
	err = snd_register_dma_channel(card, "CMI8330/C3D (SB16)", snd_sbdma8[dev],
		   SND_DMA_TYPE_ISA, snd_sbdma8_size[dev], possible_dmas,
				       &acard->sb.dma8ptr);
	if (err < 0) {
		snd_printk("CMI8330/C3D (SB16) dma8 failed\n");
		return err;
	}
	err = snd_register_dma_channel(card, "CMI8330/C3D (SB16)", snd_sbdma16[dev],
		  SND_DMA_TYPE_ISA, snd_sbdma16_size[dev], possible_dmas,
				       &acard->sb.dma16ptr);
	if (err < 0) {
		snd_printk("CMI8330/C3D (SB16) dma16 failed\n");
		return err;
	}
	return 0;
}

static int snd_cmi8330_detect(int dev, struct snd_cmi8330 *acard,
			      snd_card_t * card)
{
	snd_pcm_t *pcm;
	ad1848_t *codec_wss;
	sbdsp_t *codec_sb;

	pcm = snd_ad1848_new_device(card, snd_wssport[dev] + 4, acard->wss.irqptr,
				    acard->wss.dmaptr, AD1848_HW_DETECT);
	if (!pcm) {
		snd_printk("CMI8330/C3D (AD1848) device busy??\n");
		return -1;
	}
	codec_wss = (ad1848_t *) pcm->private_data;

	if (codec_wss->hardware != AD1848_HW_CMI8330) {
		snd_printk("CMI8330/C3D (AD1848) not found during probe\n");
		return -1;
	}
	if (snd_pcm_register(pcm, 0) < 0) {
		snd_printk("CMI8330/C3D (AD1848) failed to register PCM\n");
		return -1;
	}
	acard->wss.pcm = pcm;
	acard->wss.pcm_status_reg = codec_wss->port + 2;

	pcm = snd_sb16dsp_new_device(card, snd_sbport[dev], acard->sb.irqptr,
			acard->sb.dma8ptr, acard->sb.dma16ptr, SB_HW_16);
	if (!pcm) {
		snd_printk("CMI8330/C3D (SB16) device busy??\n");
		return -1;
	}
	codec_sb = (sbdsp_t *) pcm->private_data;

	if (codec_sb->hardware != SB_HW_16) {
		snd_printk("CMI8330/C3D (SB16) not found during probe\n");
		return -1;
	}
	if (snd_pcm_register(pcm, 1) < 0) {
		snd_printk("CMI8330/C3D (SB16) failed to register PCM\n");
		return -1;
	}
	acard->sb.pcm = pcm;
	return 0;
}

static int snd_cmi8330_mixer_setup(snd_kmixer_t * mixer,
				   struct snd_cmi8330 *acard)
{
	snd_kmixer_element_t *mixer_3d;
	snd_kmixer_element_t *mixer_loud;
	snd_kmixer_group_t *group;

	/* let's add the stuff to the Output Gain group */
	group = snd_mixer_group_find(mixer, SND_MIXER_GRP_OGAIN, 0);

	mixer_3d = snd_mixer_lib_sw1(mixer, "3D Surround Switch", 0, 1,
		       (snd_mixer_sw1_control_t *) snd_cmi8330_3d_switch,
				     (void *) acard);
	mixer_loud = snd_mixer_lib_sw1(mixer, "Loudness Switch", 0, 1,
		     (snd_mixer_sw1_control_t *) snd_cmi8330_loud_switch,
				       (void *) acard);
	if (!group) {
		snd_printk("CMI8330/C3D (SB16) failed to find mixer group\n");
		return -1;
	}
	if (!mixer_3d || !mixer_loud) {
		snd_printk("CMI8330/C3D (SB16) failed to allocate mixer elements\n");
		return -1;
	}
	if (snd_mixer_group_element_add(mixer, group, mixer_3d) < 0) {
		snd_printk("CMI8330/C3D (SB16) failed to add 3D Switch to group\n");
		return -1;
	}
	if (snd_mixer_group_element_add(mixer, group, mixer_loud) < 0) {
		snd_printk("CMI8330/C3D (SB16) failed to add Loudness Switch to group\n");
		return -1;
	}
	return 0;
}

static int snd_cmi8330_mixer(struct snd_cmi8330 *acard, snd_card_t * card)
{
	sbdsp_t *codec;
	snd_kmixer_t *mixer;

	mixer = snd_ad1848_new_mixer(acard->wss.pcm, 0);

	if (!mixer) {
		snd_printk("CMI8330/C3D (AD1848) mixer not found!\n");
		return -1;
	}
	if (snd_mixer_register(mixer, 0) < 0) {
		snd_printk("CMI8330/C3D (AD1848) mixer register failed\n");
		return -1;
	}
	acard->wss.mixer = mixer;

	codec = (sbdsp_t *) acard->sb.pcm->private_data;
	mixer = snd_sb16dsp_new_mixer(card, &codec->mixer, codec->hardware, 1);

	if (!mixer) {
		snd_printk("CMI8330/C3D (SB16) mixer not found!\n");
		return -1;
	}
	if (snd_cmi8330_mixer_setup(mixer, acard) < 0) {
		snd_printk("CMI8330/C3D (SB16) mixer setup failed!\n");
		return -1;
	}
	if (snd_mixer_register(mixer, 1) < 0) {
		snd_printk("CMI8330/C3D (SB16) mixer register failed\n");
		return -1;
	}
	acard->sb.mixer = mixer;
	return 0;
}

#ifdef CONFIG_ISAPNP
static int snd_cmi8330_isapnp(int dev, struct snd_cmi8330 *acard)
{
	static int idx = 0;
	static struct isapnp_card *card = NULL;
	struct isapnp_dev *pdev;
	unsigned int tmp;

      __again:
	acard->cap = acard->play = NULL;
	while ((tmp = snd_cmi8330_pnpids[idx]) != 0) {
		if ((card = isapnp_find_card(tmp >> 16, tmp & 0xffff, card)))
			break;
		idx += 3;
	}

	if (!card) {
		snd_printk("CMI8330/C3D PnP no device\n");
		return -ENODEV;
	}
	tmp = snd_cmi8330_pnpids[idx + 1];	/* capture */
	acard->cap = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);

	if (!acard->cap) {
		snd_printdd("CMI8330/C3D PnP no logical device\n");
		goto __again;
	}
	tmp = snd_cmi8330_pnpids[idx + 2];	/* playback */
	acard->play = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);

	pdev = acard->cap;
	if (pdev->prepare(pdev)<0)
		return -EAGAIN;
	/* allocate AD1848 resources */
	if (snd_wssport[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[0], snd_wssport[dev], 8);
	if (snd_wssdma[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[0], snd_wssdma[dev], 1);
	if (snd_wssirq[dev] != SND_AUTO_IRQ)
		isapnp_resource_change(&pdev->irq_resource[0], snd_wssirq[dev], 1);

	if (pdev->activate(pdev)<0) {
		snd_printk("CMI8330/C3D (AD1848) PnP configure failure\n");
		return -EBUSY;
	}
	snd_wssport[dev] = pdev->resource[0].start;
	snd_wssdma[dev] = pdev->dma_resource[0].start;
	snd_wssirq[dev] = pdev->irq_resource[0].start;

	if (pdev->activate(pdev)<0)
		return -EAGAIN;

	/* allocate SB16 resources */
	pdev = acard->play;
	if (pdev->prepare(pdev)<0) {
		acard->cap->deactivate(acard->cap);
		return -EAGAIN;
	}
	if (snd_sbport[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[0], snd_sbport[dev], 16);
	if (snd_sbdma8[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[0], snd_sbdma8[dev], 1);
	if (snd_sbdma16[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[1], snd_sbdma16[dev], 1);
	if (snd_sbirq[dev] != SND_AUTO_IRQ)
		isapnp_resource_change(&pdev->irq_resource[0], snd_sbirq[dev], 1);

	if (pdev->activate(pdev)<0) {
		snd_printk("CMI8330/C3D (SB16) PnP configure failure\n");
		acard->cap->deactivate(acard->cap);
		return -EBUSY;
	}
	snd_sbport[dev] = pdev->resource[0].start;
	snd_sbdma8[dev] = pdev->dma_resource[0].start;
	snd_sbdma16[dev] = pdev->dma_resource[1].start;
	snd_sbirq[dev] = pdev->irq_resource[0].start;

	return 0;
}

static void snd_cmi8330_deactivate(struct snd_cmi8330 *acard)
{
	if (acard->cap)
		acard->cap->deactivate(acard->cap);
	if (acard->play)
		acard->play->deactivate(acard->play);
}
#endif

static int snd_cmi8330_probe(int dev, struct snd_cmi8330 *acard)
{
	snd_card_t *card;
	ad1848_t *codec;
	unsigned long flags;
	unsigned char *ptr;
	int i;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_cmi8330_use_inc, snd_cmi8330_use_dec);
	if (!card) {
		snd_printk("CMI8330/C3D could not get a new card\n");
		return -ENOMEM;
	}
	card->type = SND_CARD_TYPE_CMI8330;

#ifdef CONFIG_ISAPNP
	if (snd_cmi8330_isapnp(dev, acard) < 0) {
		snd_printk("CMI8330/C3D PnP detection failed\n");
		snd_card_free(card);
		return -EBUSY;
	}
#endif
	if (snd_cmi8330_resources(dev, acard, card) < 0) {
		snd_printk("CMI8330/C3D resource allocation failed\n");
		snd_card_free(card);
		return -EBUSY;
	}
	if (snd_cmi8330_detect(dev, acard, card) < 0) {
		snd_printk("CMI8330/C3D failed to detect\n");
		snd_card_free(card);
		return -ENODEV;
	}
	codec = (ad1848_t *) acard->wss.pcm->private_data;
	memcpy(&acard->image, &snd_cmi8330_image, sizeof(snd_cmi8330_image));
	ptr = (unsigned char *) &acard->image;

	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_ad1848_out(codec, AD1848_MISC_INFO,	/* switch on MODE2 */
		       codec->image[AD1848_MISC_INFO] |= 0x40);
	snd_ad1848_out(codec, CMI8330_RMUX3D,	/* enable SB16 mixer */
		       acard->image[CMI8330_RMUX3D] |= 0x40);
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	if (snd_cmi8330_mixer(acard, card) < 0) {
		snd_printk("CMI8330/C3D failed to create mixers\n");
		goto __nodev;
	}
	spin_lock_irqsave(&codec->reg_lock, flags);
	for (i = CMI8330_RMUX3D; i <= CMI8330_CDINGAIN; i++) {
		snd_ad1848_out(codec, i, *ptr++);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	/*
	 * KLUDGE ALERT
	 *  route SB16 capture to AD1848 capture
	 *  route AD1848 playback to SB16 playback
	 */
	acard->sb.capture = acard->sb.pcm->capture.private_data;
	acard->sb.pcm->capture.private_data = acard->wss.pcm->capture.private_data;

	acard->wss.playback = acard->wss.pcm->playback.private_data;
	acard->wss.pcm->playback.private_data = acard->sb.pcm->playback.private_data;

	strcpy(card->abbreviation, "CMI8330/C3D");
	strcpy(card->shortname, "C-Media CMI8330/C3D");
	sprintf(card->longname, "%s at 0x%x, irq %i, dma %i",
		acard->wss.pcm->name,
		codec->port,
		acard->wss.irqptr->irq,
		acard->wss.dmaptr->dma);

	if (!snd_card_register(card)) {
		acard->card = card;
		return 0;
	}
	snd_card_free(card);
	return -ENOMEM;

      __nodev:
	snd_card_free(card);
	return -ENXIO;
}

static void snd_cmi8330_free(int dev)
{
	struct snd_cmi8330 *acard;

	acard = snd_cmi8330_cards[dev];

	if (acard) {
		if (acard->card) {
			snd_card_unregister(acard->card);
		}
		if (acard->wss.pcm) {
			/* unkludge */
			if (acard->wss.playback)
				acard->wss.pcm->playback.private_data = acard->wss.playback;
			snd_pcm_unregister(acard->wss.pcm);
		}
		if (acard->wss.mixer) {
			snd_mixer_unregister(acard->wss.mixer);
		}
		if (acard->sb.pcm) {
			/* unkludge */
			if (acard->sb.capture)
				acard->sb.pcm->capture.private_data = acard->sb.capture;
			snd_pcm_unregister(acard->sb.pcm);
		}
		if (acard->sb.mixer) {
			snd_mixer_unregister(acard->sb.mixer);
		}
		if (acard->card) {
			snd_card_free(acard->card);
		}
#ifdef CONFIG_ISAPNP
		snd_cmi8330_deactivate(acard);
#endif
		snd_kfree(acard);
		snd_cmi8330_cards[dev] = NULL;
	}
}

void cleanup_module(void)
{
	int i;

	for (i = 0; i < SND_CARDS; i++) {
		snd_cmi8330_free(i);
	}
}

int init_module(void)
{
	int dev, cards;
	struct snd_cmi8330 *acard;

	for (dev = cards = 0; dev < SND_CARDS && snd_wssport[dev] > 0; dev++) {
		acard = (struct snd_cmi8330 *) snd_kcalloc(sizeof(struct snd_cmi8330),
							   GFP_KERNEL);
		if (!acard) {
			continue;
		}
		if (snd_cmi8330_probe(dev, acard) < 0) {
			snd_kfree(acard);

			if (snd_wssport[dev] == SND_AUTO_PORT) {
				break;
			}
			snd_cmi8330_free(dev);
			snd_printk("CMI8330 #%i not found at 0x%x or device busy\n", dev + 1,
				   snd_wssport[dev]);
			continue;
		}
		snd_cmi8330_cards[dev] = acard;
		cards++;
	}

	if (!cards) {
		snd_printk("CMI8330 not found or device busy\n");
		return -ENODEV;
	}
	return 0;
}
