/*
 *  Copyright (c) 1999 by Uros Bizjak <uros@kss-loka.si>
 *                        Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *  SB16ASP/AWE32 CSP control
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/sb.h"
#include "../../include/sb_csp.h"

/*
 * CSP information (stored on hwdep->private_data)
 */
typedef struct snd_sb_csp {
	sbdsp_t *codec;		/* SB16 DSP */
	int used;		/* usage flag - exclusive */
	int version;		/* CSP version (0x10 - 0x1f) */
	char codec_name[16];	/* name of codec */
	int running;		/* running state */
	unsigned int pcm_format;	/* PCM format */
	int mode;		/* MODE */
	int mono;		/* mono = 1, stereo = 0 */
	int s8bit;		/* 8bit = 1, 16bit sample = 0 */
	int qpos_left;		/* Q-Sound left position */
	int qpos_right;		/* Q-Sound right position */
	struct semaphore access_mutex;	/* locking */
} snd_sb_csp_t;

/*
 * prototypes
 */
static void snd_sb_csp_free(void *private_data);
static int snd_sb_csp_open(snd_hwdep_t *hw, struct file *file);
static int snd_sb_csp_ioctl(snd_hwdep_t *hw, struct file *file, unsigned int cmd, unsigned long arg);
static int load_microcode(snd_sb_csp_t *p, snd_sb_csp_microcode_t *code);
static int snd_sb_csp_release(snd_hwdep_t *hw, struct file *file);

static int command_seq(sbdsp_t *codec, const unsigned char *seq, int size);
static int set_codec_parameter(sbdsp_t *codec, unsigned char par, unsigned char val);
static int set_register(sbdsp_t *codec, unsigned char reg, unsigned char val);
static int read_register(sbdsp_t *codec, unsigned char reg);
static int set_mode_register(sbdsp_t *codec, unsigned char mode);
static int get_version(sbdsp_t *codec);

static int check_version(snd_sb_csp_t *p);
static int snd_sb_csp_detect(sbdsp_t *codec, int *version);
static int snd_sb_csp_load(snd_sb_csp_t *p, const unsigned char *buf, int size, int initblock);
static int snd_sb_csp_start(snd_sb_csp_t *p, int mono);
static int snd_sb_csp_stop(snd_sb_csp_t *p);
static int snd_sb_csp_pause(snd_sb_csp_t *p);
static int snd_sb_csp_restart(snd_sb_csp_t *p);

static int snd_sb_csp_qsound_start(snd_sb_csp_t *p);
static int snd_sb_csp_qsound_pos(snd_sb_csp_t *p, int left, int right);
static int snd_sb_csp_qsound_stop(snd_sb_csp_t *p);


/*
 * Detect CSP chip and create a new instance
 */
snd_hwdep_t *
snd_sb_csp_new_device(sbdsp_t *codec)
{
	snd_sb_csp_t *p;
	int version;
	snd_hwdep_t *hw;

	if (snd_sb_csp_detect(codec, &version))
		return NULL;

	hw = snd_hwdep_new_device(codec->card, "SB-CSP");
	if (hw == NULL)
		return NULL;

	p = snd_kcalloc(sizeof(*p), GFP_KERNEL);
	if (p == NULL) {
		snd_hwdep_free(hw);
		return NULL;
	}

	p->codec = codec;
	p->version = version;
	init_MUTEX(&p->access_mutex);

	hw->type = SND_HWDEP_TYPE_SBCSP;
	hw->private_data = p;
	hw->private_free = snd_sb_csp_free;

	/* operators - only write/ioctl */
	hw->ops.open = snd_sb_csp_open;
	hw->ops.ioctl = snd_sb_csp_ioctl;
	hw->ops.release = snd_sb_csp_release;

	return hw;
}


/*
 * free_private for hwdep instance
 */
static void
snd_sb_csp_free(void *private_data)
{
	snd_sb_csp_t *p = private_data;
	if (p) {
		if (p->running & SND_SB_CSP_ST_RUNNING)
			snd_sb_csp_stop(p);
		snd_kfree(p);
	}
}


/*
 * open the device exclusively
 */
static int
snd_sb_csp_open(snd_hwdep_t *hw, struct file *file)
{
	snd_sb_csp_t *p = hw->private_data;

	down(&p->access_mutex);
	if (p->used) {
		up(&p->access_mutex);
		return -EBUSY;
	}
	p->used++;
	up(&p->access_mutex);

	return 0;
}


/*
 * ioctl for hwdep device:
 */
static int
snd_sb_csp_ioctl(snd_hwdep_t *hw, struct file *file, unsigned int cmd, unsigned long arg)
{
	snd_sb_csp_t *p = hw->private_data;
	snd_sb_csp_info_t info;
	snd_sb_csp_qsound_t qpos;
	int err, mono;

	if (p == NULL)
		return -EINVAL;

	if (check_version(p))
		return -ENODEV;

	down(&p->access_mutex);

	switch (cmd) {
	/* get information */
	case SND_SB_CSP_IOCTL_INFO:
		memcpy(info.codec_name, p->codec_name, sizeof(p->codec_name));
		info.version = p->version;
		info.state = p->running;
		info.pcm_format = p->pcm_format;
		info.csp_mode = p->mode;
		info.channels = (p->mono ? 1 : 2);
		err = copy_to_user((void*)arg, &info, sizeof(info));
		break;

	/* load CSP microcode */
	case SND_SB_CSP_IOCTL_LOAD_CODE:
		if (p->running & SND_SB_CSP_ST_RUNNING)
			err = -EBUSY;
		else
			err = load_microcode(p, (snd_sb_csp_microcode_t*)arg);
		break;

	/* change CSP running state */
	case SND_SB_CSP_IOCTL_START:
		if (copy_from_user(&mono, (void*)arg, sizeof(mono)))
			err = -EFAULT;
		else
			err = snd_sb_csp_start(p, mono);
		break;
	case SND_SB_CSP_IOCTL_STOP:
		err = snd_sb_csp_stop(p);
		break;
	case SND_SB_CSP_IOCTL_PAUSE:
		err = snd_sb_csp_pause(p);
		break;
	case SND_SB_CSP_IOCTL_RESTART:
		err = snd_sb_csp_restart(p);
		break;

	/* QSound stuffs */
	case SND_SB_CSP_IOCTL_QSOUND_START:
		err = snd_sb_csp_qsound_start(p);
		break;
	case SND_SB_CSP_IOCTL_QSOUND_STOP:
		err = snd_sb_csp_qsound_stop(p);
		break;
	case SND_SB_CSP_IOCTL_QSOUND_POS:
		if (copy_from_user(&qpos, (void*)arg, sizeof(qpos)))
			err = -EFAULT;
		else
			err = snd_sb_csp_qsound_pos(p, qpos.left, qpos.right);
		break;
	case SND_SB_CSP_IOCTL_QSOUND_GET_POS:
		qpos.left = p->qpos_left;
		qpos.right = p->qpos_right;
		err = copy_to_user((void*)arg, &qpos, sizeof(qpos));
		break;

	default:
		err = -EINVAL;
		break;
	}

	up(&p->access_mutex);
	return err;
}
 

/*
 * load microcode via ioctl:
 * code is user-space pointer
 */
static int
load_microcode(snd_sb_csp_t *p, snd_sb_csp_microcode_t *code)
{
	snd_sb_csp_mc_header_t info;
	int err;
	int s8bit;

	if (copy_from_user(&info, code, sizeof(info)))
		return -EFAULT;

	switch (info.csp_mode) {
	case SND_SB_CSP_MODE_NONE:
		p->mode = 0; /* just clear */
		return 0;
	case SND_SB_CSP_MODE_QSOUND:
		if (info.pcm_format == SND_PCM_SFMT_S16_LE)
			s8bit = 0;
		else if (info.pcm_format == SND_PCM_SFMT_S8)
			s8bit = 1;
		else
			return -EINVAL;
		break;
	case SND_SB_CSP_MODE_DSP_WRITE:
	case SND_SB_CSP_MODE_DSP_READ:
	case SND_SB_CSP_MODE_DSP_RW:
		if (info.pcm_format == SND_PCM_SFMT_S8 ||
		    info.pcm_format == SND_PCM_SFMT_U8)
			s8bit = 1;
		else
			s8bit = 0;
		/* other format should be checked whether it is accepted */
		break;
	default:
		return -EINVAL;
	}

	/* clear before loading code */
	p->pcm_format = 0;
	p->mode = 0;

	/* load init microcode if it exists */
	if (info.init_size > 0) {
		err = snd_sb_csp_load(p, code->init_code, info.init_size, 1);
		if (err)
			return err;
	}
	/* load main microcode */
	err = snd_sb_csp_load(p, code->main_code, info.main_size, 0);
	if (err)
		return err;

	/* finished loading successfuly */

	/* copy codec_name */
	memset(p->codec_name, 0, sizeof(p->codec_name));
	strncpy(p->codec_name, info.codec_name, sizeof(p->codec_name) - 1);
	p->codec_name[sizeof(p->codec_name) - 1] = 0;

	/* copy parameters */
	p->mode = info.csp_mode;
	p->pcm_format = info.pcm_format;
	p->s8bit = s8bit;
	p->running |= SND_SB_CSP_ST_LOADED;

	return 0;
}


/*
 * release the device
 */
static int
snd_sb_csp_release(snd_hwdep_t *hw, struct file *file)
{
	snd_sb_csp_t *p = hw->private_data;

	down(&p->access_mutex);
	snd_sb_csp_stop(p);
	p->used--;
	up(&p->access_mutex);

	return 0;
}


/*
 * send command sequence to DSP
 */
static int
command_seq(sbdsp_t *codec, const unsigned char *seq, int size)
{
	int i;
	for (i = 0; i < size; i++) {
		if (! snd_sb16dsp_command(codec, seq[i]))
			return -EIO;
	}
	return 0;
}


/*
 * set CSP codec parameter
 */
static int
set_codec_parameter(sbdsp_t *codec, unsigned char par, unsigned char val)
{
	unsigned char dsp_cmd[3];

	dsp_cmd[0] = 0x05;	/* CSP set codec parameter */
	dsp_cmd[1] = val;	/* Parameter value */
	dsp_cmd[2] = par;	/* Parameter */
	command_seq(codec, dsp_cmd, 3);
	snd_sb16dsp_command(codec, 0x03);	/* DSP read? */
	if (snd_sb16dsp_get_byte(codec) != par)
		return -EIO;
	return 0;
}


/*
 * set CSP register
 */
static int
set_register(sbdsp_t *codec, unsigned char reg, unsigned char val)
{
	unsigned char dsp_cmd[3];

	dsp_cmd[0] = 0x0E;	/* CSP set register */
	dsp_cmd[1] = reg;	/* CSP Register */
	dsp_cmd[2] = val;	/* value */
	return command_seq(codec, dsp_cmd, 3);
}


/*
 * read CSP register
 * return < 0 -> error
 */
static int
read_register(sbdsp_t *codec, unsigned char reg)
{
	unsigned char dsp_cmd[2];

	dsp_cmd[0] = 0x0F;	/* CSP read register */
	dsp_cmd[1] = reg;	/* CSP Register */
	command_seq(codec, dsp_cmd, 2);
	return snd_sb16dsp_get_byte(codec);	/* Read DSP value */
}


/*
 * set CSP mode register
 */
static int
set_mode_register(sbdsp_t *codec, unsigned char mode)
{
	unsigned char dsp_cmd[2];

	dsp_cmd[0] = 0x04;	/* CSP set mode register */
	dsp_cmd[1] = mode;	/* mode */
	return command_seq(codec, dsp_cmd, 2);
}

/*
 * Detect CSP
 * return 0 if CSP exists. 
 */
static int
snd_sb_csp_detect(sbdsp_t *codec, int *version)
{
	unsigned char csp_test1, csp_test2;
	unsigned long flags;
	int result = -ENODEV;

	spin_lock_irqsave(&codec->reg_lock, flags);

	set_codec_parameter(codec, 0x00, 0x00);
	set_mode_register(codec, 0xfc);	/* 0xFC = ?? */

	csp_test1 = read_register(codec, 0x83);
	set_register(codec, 0x83, ~csp_test1);
	csp_test2 = read_register(codec, 0x83);
	if (csp_test2 != (csp_test1 ^ 0xFF))
		goto __fail;

	set_register(codec, 0x83, csp_test1);
	csp_test2 = read_register(codec, 0x83);
	if (csp_test2 != csp_test1)
		goto __fail;

	set_mode_register(codec, 0x00);	/* 00 = ?? */


	*version = get_version(codec);
	snd_sb16dsp_reset(codec);	 /* reset DSP after getversion! */
	if (*version >= 0x10 && *version <= 0x1f)
		result = 0; /* valid version id */

__fail:
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}


/*
 * get CSP version number
 */
static int
get_version(sbdsp_t *codec)
{
	unsigned char dsp_cmd[2];

	dsp_cmd[0] = 0x08;	/* SB_DSP_!something! */
	dsp_cmd[1] = 0x03;	/* command # (get ChipID?) */
	command_seq(codec, dsp_cmd, 2);

	return (snd_sb16dsp_get_byte(codec));
}


/*
 * check if the CSP version is valid
 */
static int
check_version(snd_sb_csp_t *p)
{
	if (p->version < 0x10 || p->version > 0x1f) {
		snd_printd("sb_csp: invalid version 0x%d\n", p->version);
		return 1;
	}
	return 0;
}


/*
 * download microcode to CSP.  The buffer is on user-sapce.
 * if initblock=1, load "init" block of microcode. It should be
 * followed by "main" block.
 */
static int
snd_sb_csp_load(snd_sb_csp_t *p, const unsigned char *buf, int size, int initblock)
{
	unsigned char mixdata;
	int i;
	int result = -EIO;
	unsigned long flags;

	spin_lock_irqsave(&p->codec->reg_lock, flags);

	snd_sb16dsp_command(p->codec, 0x01);	/* CSP download command */

	if (snd_sb16dsp_get_byte(p->codec) != 0) {
		snd_printd("sb_csp_load: Download command failed!\n");
		goto __fail;
	}

	/* Send CSP lo_bit (lenght -1) */
	snd_sb16dsp_command(p->codec, (unsigned char)(size - 1));
	/* Send hi_bit */
	snd_sb16dsp_command(p->codec, (unsigned char)(size >> 8));
	/* send microcode sequence */
	for (i = 0; i < size; i++, buf++) {
		unsigned c;
		if (copy_from_user(&c, buf, 1)) {
			result = -EFAULT;
			goto __fail;
		}
		if (! snd_sb16dsp_command(p->codec, c))
			goto __fail;
	}

	/*delay(1);*/	/* wait 1ms */

	if (snd_sb16dsp_get_byte(p->codec) != 0)
		goto __fail;

	if (initblock) {
		snd_sb16dsp_command(p->codec, 0x03);
		if (snd_sb16dsp_get_byte(p->codec) != 0x55) {
			snd_printd("sb_csp_load: No 'main' block to load!\n");
			goto __fail;
		}
	}

	/* Mixer register 0x81 (???) */
	outb(0x81, SBP(p->codec, MIXER_ADDR));
	mixdata = inb(SBP(p->codec, MIXER_DATA));
	if ((mixdata & 0xE0) == 0) {	/* Mask with 11100000 */
		/* Something wrong!! */
		snd_printd("sb_csp_load: Download failed! [mixer reg 0x81: 0x%X]\n", mixdata);
		/* We shall never get there, something must go terribly wrong.
		 * [TODO:] RESET DSP and send a bunch of commands to CSP
		 */
		goto __fail;
	}

	result = 0;

__fail:
	spin_unlock_irqrestore(&p->codec->reg_lock, flags);
	return result;
}


/*
 * start CSP
 */
static int
snd_sb_csp_start(snd_sb_csp_t *p, int mono)
{
	unsigned char s_type;	/* sample type */
	int result = -EIO;
	unsigned long flags;

	if (! (p->running & SND_SB_CSP_ST_LOADED)) {
		snd_printd("sb_csp_start: codec not loaded\n");
		return -ENXIO;
	}
		
	if (p->running & SND_SB_CSP_ST_RUNNING) {
		snd_printd("sb_csp_start: CSP already running\n");
		return -EBUSY;
	}

	/* TODO: slowly fade down PCM volume to avoid crackling */

	spin_lock_irqsave(&p->codec->reg_lock, flags);

	set_mode_register(p->codec, 0xC0);	/* C0 = STOP */
	set_mode_register(p->codec, 0x70);	/* 70 = RUN */

	s_type = 0x00;
	if (mono)
		s_type = 0x11;	/* 000n 000n    (n = 1 if one channel) */
	if (p->s8bit)		/* 00dX 00dX    (d = 1 if 8 bit samples) */
		s_type |= 0x22;

	if (set_codec_parameter(p->codec, 0x81, s_type)) {
		snd_printd("cb_csp_start: Set sample type command failed!\n");
		goto __fail;
	}
	if (set_codec_parameter(p->codec, 0x80, 0x00)) {
		snd_printd("sb_csp_start: Codec start command failed!\n");
		goto __fail;
	}

	p->mono = mono;
	p->running |= SND_SB_CSP_ST_RUNNING;

	result = 0;

__fail:
	/* TODO: slowly fade up PCM volume to saved value */
	spin_unlock_irqrestore(&p->codec->reg_lock, flags);
	return result;
}


/*
 * stop CSP
 */
static int
snd_sb_csp_stop(snd_sb_csp_t *p)
{
	int result;
	unsigned long flags;

	if (! (p->running & SND_SB_CSP_ST_RUNNING))
		return 0;

	if (p->running & SND_SB_CSP_ST_QSOUND) {
		result = snd_sb_csp_qsound_stop(p);
		if (result)
			return result;
	}

	spin_lock_irqsave(&p->codec->reg_lock, flags);
	/* TODO: fade down volume to avoid crackling */
	result = set_mode_register(p->codec, 0xC0);	/* C0 = STOP */
	/* TODO: fade up volume to saved value */
	spin_unlock_irqrestore(&p->codec->reg_lock, flags);

	if (result == 0)
		p->running = SND_SB_CSP_ST_LOADED; /* clear all other flags */
	return result;
}


/*
 * pause CSP codec and hold DMA transfer
 */
static int
snd_sb_csp_pause(snd_sb_csp_t *p)
{
	int result;
	unsigned long flags;

	if (! (p->running & SND_SB_CSP_ST_RUNNING))
		return -EBUSY;

	spin_lock_irqsave(&p->codec->reg_lock, flags);
	result = set_codec_parameter(p->codec, 0x80, 0xFF);
	spin_unlock_irqrestore(&p->codec->reg_lock, flags);
	if (result == 0)
		p->running |= SND_SB_CSP_ST_PAUSED;

	return result;
}


/*
 * restart CSP codec resume DMA transfer
 */
static int
snd_sb_csp_restart(snd_sb_csp_t *p)
{
	int result;
	unsigned long flags;

	if (! (p->running & SND_SB_CSP_ST_PAUSED))
		return -EBUSY;

	spin_lock_irqsave(&p->codec->reg_lock, flags);
	result = set_codec_parameter(p->codec, 0x80, 0x00);
	spin_unlock_irqrestore(&p->codec->reg_lock, flags);
	if (result == 0)
		p->running &= ~SND_SB_CSP_ST_PAUSED;

	return result;
}


/*
 * initialize QSound codec
 */
static int
snd_sb_csp_qsound_start(snd_sb_csp_t *p)
{
	unsigned long flags;

	if (p->mode != SND_SB_CSP_MODE_QSOUND)
		return -ENXIO;
	if (!(p->running & SND_SB_CSP_ST_RUNNING) ||
	    (p->running & SND_SB_CSP_ST_QSOUND))
		return -EBUSY;

	/* center position 0001 0000
	 * set Left channel and
	 * set Right channel
	 */
	spin_lock_irqsave(&p->codec->reg_lock, flags);
	set_codec_parameter(p->codec, 0xE0, 0x01);
	set_codec_parameter(p->codec, 0x00, 0xFF);
	set_codec_parameter(p->codec, 0x01, 0xFF);
	set_codec_parameter(p->codec, 0x00, 0x10);
	set_codec_parameter(p->codec, 0x02, 0x00);
	set_codec_parameter(p->codec, 0x03, 0x00);
	spin_unlock_irqrestore(&p->codec->reg_lock, flags);

	p->running |= SND_SB_CSP_ST_QSOUND;

	return 0;
}


/*
 * set QSound position for left and right channel sample
 * with arguments ranging from 0x00 to 0x20.
 */
static int
snd_sb_csp_qsound_pos(snd_sb_csp_t *p, int left, int right)
{
	unsigned long flags;

	if (p->mode != SND_SB_CSP_MODE_QSOUND)
		return -ENXIO;
	if (!(p->running & SND_SB_CSP_ST_RUNNING) ||
	    !(p->running & SND_SB_CSP_ST_QSOUND))
		return -EBUSY;

	spin_lock_irqsave(&p->codec->reg_lock, flags);
	/* left channel */
	set_codec_parameter(p->codec, 0xE0, 0x01);
	set_codec_parameter(p->codec, 0x00, left);
	set_codec_parameter(p->codec, 0x02, 0x00);
	/* right channel */
	set_codec_parameter(p->codec, 0x0E, 0x01);
	set_codec_parameter(p->codec, 0x00, right);
	set_codec_parameter(p->codec, 0x03, 0x00);
	spin_unlock_irqrestore(&p->codec->reg_lock, flags);

	p->qpos_left = left;
	p->qpos_right = right;

	return 0;
}


/*
 * leave QSound
 */
static int
snd_sb_csp_qsound_stop(snd_sb_csp_t *p)
{
	unsigned long flags;

	if (p->mode != SND_SB_CSP_MODE_QSOUND)
		return -ENXIO;
	if (!(p->running & SND_SB_CSP_ST_RUNNING) ||
	    !(p->running & SND_SB_CSP_ST_QSOUND))
		return -EBUSY;

	spin_lock_irqsave(&p->codec->reg_lock, flags);
	set_codec_parameter(p->codec, 0xE0, 0x01);
	set_codec_parameter(p->codec, 0x00, 0x00);
	set_codec_parameter(p->codec, 0x01, 0x00);
	set_codec_parameter(p->codec, 0xE0, 0x01);
	set_codec_parameter(p->codec, 0x00, 0x00);
	set_codec_parameter(p->codec, 0x01, 0x00);
	spin_unlock_irqrestore(&p->codec->reg_lock, flags);

	p->running &= ~SND_SB_CSP_ST_QSOUND;

	return 0;
}

EXPORT_SYMBOL(snd_sb_csp_new_device);

/*
 *  INIT part
 */

int init_module(void)
{
	return 0;
}

void cleanup_module(void)
{
}
