/*
 *  Interface for OSS sequencer emulation
 *
 *  Copyright (C) 1999 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Changes
 * 19990227   Steve Ratcliffe   Made separate file and merged in latest
 * 				midi emulation.
 */

#include "../../include/driver.h"

#ifdef CONFIG_SND_OSSEMUL

#include "../../include/emu8000.h"
#include "../../include/seq_oss.h"
#include "../../include/seq_midi_emul.h"
#include "emu8000_port.h"
#include "emu8000_equalizer.h"
#include "../../include/seq_midi_event.h"

static int snd_emu8000_open_seq_oss(snd_seq_oss_arg_t *arg, void *closure);
static int snd_emu8000_close_seq_oss(snd_seq_oss_arg_t *arg);
static int snd_emu8000_ioctl_seq_oss(snd_seq_oss_arg_t *arg, unsigned int cmd, unsigned long ioarg);
static int snd_emu8000_load_patch_seq_oss(snd_seq_oss_arg_t *arg, int format, const char *buf, int offs, int count);
static int snd_emu8000_reset_seq_oss(snd_seq_oss_arg_t *arg);
static int snd_emu8000_event_oss_input(snd_seq_event_t *ev, int direct, void *private);
static void reset_port_mode(emu8000_port_t *port, int midi_mode);
static void emuspec_control(emu8000_t *emu, emu8000_port_t *port, int cmd, unsigned char *event);
static void gusspec_control(emu8000_t *emu, emu8000_port_t *port, int cmd, unsigned char *event);
static void fake_event(emu8000_t *emu, emu8000_port_t *port, int ch, int param, int val);

/* operators */
static snd_seq_oss_callback_t oss_callback = {
	snd_emu8000_open_seq_oss,
	snd_emu8000_close_seq_oss,
	snd_emu8000_ioctl_seq_oss,
	snd_emu8000_load_patch_seq_oss,
	snd_emu8000_reset_seq_oss,
};


/*
 * register OSS synth
 */

void
snd_emu8000_init_seq_oss(emu8000_t *emu)
{
	snd_seq_oss_reg_t arg;

	arg.type = SND_OSS_SYNTH_TYPE_SAMPLE;
	arg.subtype = SND_OSS_SAMPLE_TYPE_AWE32;
	arg.nvoices = EMU8000_CHANNELS;
	arg.oper = oss_callback;
	arg.private_data = emu;

	snd_seq_device_register(emu->card, 0, "EMU8000", SND_SEQ_DEV_OSS,
				(void*)&arg, sizeof(arg), &emu->oss_synth);
}


/*
 * unregister
 */
void
snd_emu8000_free_seq_oss(emu8000_t *emu)
{
	if (emu->oss_synth)
		snd_seq_device_unregister(emu->oss_synth);
	emu->oss_synth = NULL;
}


/* use port number as a unique soundfont client number */
#define SF_CLIENT_NO(p)	((p) + 0x1000)

/*
 * open port for OSS sequencer
 */
static int
snd_emu8000_open_seq_oss(snd_seq_oss_arg_t *arg, void *closure)
{
	emu8000_t *emu = closure;
	emu8000_port_t *p;
	snd_seq_port_callback_t callback;

	if (!arg || !emu)
		return -ENXIO;

	down(&emu->register_mutex);

	memset(&callback, 0, sizeof(callback));
	callback.event_input = snd_emu8000_event_oss_input;

	p = snd_emu8000_create_port(emu, "OSS-Sequencer Port",
				    EMU8000_CHANNELS,
				    SND_SEQ_PORT_TYPE_SPECIFIC, &callback);
	if (! p) {
		snd_printk("emu8000: can't create port\n");
		up(&emu->register_mutex);
		return -ENOMEM;
	}

	/* fill the argument data */
	arg->private_data = p;
	arg->addr.client = p->chset.client;
	arg->addr.port = p->chset.port;
	arg->addr.queue = 0;
	arg->addr.channel = 0;
	p->oss_arg = arg;

	reset_port_mode(p, 0);

	snd_emu8000_reset_port(p);

	snd_emu8000_inc_count(emu);

	up(&emu->register_mutex);
	return 0;
}


#define DEFAULT_DRUM_FLAGS	((1<<9) | (1<<25))

/*
 * reset port mode
 */
static void
reset_port_mode(emu8000_port_t *port, int midi_mode)
{
	if (midi_mode) {
		port->port_mode = PORT_MODE_OSS_MIDI;
		port->drum_flags = DEFAULT_DRUM_FLAGS;
		port->volume_atten = 0;
		port->oss_arg->event_passing = SND_SEQ_OSS_PROCESS_KEYPRESS;
	} else {
		port->port_mode = PORT_MODE_OSS_SYNTH;
		port->drum_flags = 0;
		port->volume_atten = 32;
		port->oss_arg->event_passing = SND_SEQ_OSS_PROCESS_EVENTS;
	}
}


/*
 * close port
 */
static int
snd_emu8000_close_seq_oss(snd_seq_oss_arg_t *arg)
{
	emu8000_t *emu;
	emu8000_port_t *p;

	if (!arg || !(p = arg->private_data))
		return -EINVAL;

	emu = p->emu;
	down(&emu->register_mutex);
	snd_emu8000_sounds_off_all(emu, p);
	snd_soundfont_close_check(emu->sflist, SF_CLIENT_NO(p->chset.port));
	snd_seq_event_port_detach(p->chset.client, p->chset.port);
	snd_emu8000_dec_count(emu);

	up(&emu->register_mutex);
	return 0;
}


/*
 * load patch
 */
static int
snd_emu8000_load_patch_seq_oss(snd_seq_oss_arg_t *arg, int format,
			      const char *buf, int offs, int count)
{
	emu8000_t *emu;
	emu8000_port_t *p;
	int rc;

	if (!arg || !(p = arg->private_data))
		return -ENXIO;

	emu = p->emu;
	if (EMU8000_CHECK(emu) < 0)
		return -ENXIO;

	if (format == SND_OSS_GUS_PATCH)
		rc = snd_soundfont_load_guspatch(emu->sflist, buf, count,
						 SF_CLIENT_NO(p->chset.port));
	else if (format == SND_OSS_SOUNDFONT_PATCH) {
		soundfont_patch_info_t patch;
		if (count < sizeof(patch))
			rc = -EINVAL;
		if (copy_from_user(&patch, buf, sizeof(patch)))
			rc = -EFAULT;
		switch (patch.type) {
		case SND_EMU8000_LOAD_CHORUS_FX:
			count -= sizeof(patch);
			buf += sizeof(patch);
			rc = snd_emu8000_load_chorus_fx(emu, patch.optarg,
							buf, count);
			break;

		case SND_EMU8000_LOAD_REVERB_FX:
			count -= sizeof(patch);
			buf += sizeof(patch);
			rc = snd_emu8000_load_reverb_fx(emu, patch.optarg,
							buf, count);
			break;

		default:
			rc = snd_soundfont_load(emu->sflist, buf, count, SF_CLIENT_NO(p->chset.port));
			break;
		}
	} else
		rc = 0;
	/*snd_mutex_up(emu, patch);*/
	return rc;
}


/*
 * ioctl
 */
static int
snd_emu8000_ioctl_seq_oss(snd_seq_oss_arg_t *arg, unsigned int cmd, unsigned long ioarg)
{
	emu8000_port_t *p;
	emu8000_t *emu;

	if (!arg || !(p = arg->private_data))
		return -ENXIO;

	emu = p->emu;
	switch (cmd) {
	case SND_OSS_SNDCTL_SEQ_RESETSAMPLES:
		snd_soundfont_remove_samples(emu->sflist);
		return 0;
			
	case SND_OSS_SNDCTL_SYNTH_MEMAVL:
		return emu->mem_size - snd_soundfont_mem_offset(emu->sflist) * 2;
	}

	return 0;
}


/*
 * reset device
 */
static int
snd_emu8000_reset_seq_oss(snd_seq_oss_arg_t *arg)
{
	emu8000_port_t *p;

	if (! arg || !(p = arg->private_data))
		return -ENXIO;
	snd_emu8000_reset_port(p);
	return 0;
}


/*
 * receive raw events: only SEQ_PRIVATE is accepted.
 */
static int
snd_emu8000_event_oss_input(snd_seq_event_t *ev, int direct, void *private)
{
	emu8000_t *emu;
	emu8000_port_t *p = private;
	unsigned char cmd, *data;

	emu = p->emu;
	if (EMU8000_CHECK(emu) < 0)
		return -ENXIO;

	if (ev->type != SND_SEQ_EVENT_OSS)
		return snd_emu8000_event_input(ev, direct, private);

	data = ev->data.raw8.d;
	/* only SEQ_PRIVATE is accepted */
	if (data[0] != 0xfe)
		return 0;
	cmd = data[2] & _EMU8000_OSS_MODE_VALUE_MASK;
	if (data[2] & _EMU8000_OSS_MODE_FLAG)
		emuspec_control(emu, p, cmd, data);
	else
		gusspec_control(emu, p, cmd, data);
	return 0;
}


/*
 * OSS/AWE driver specific h/w controls
 */
static void
emuspec_control(emu8000_t *emu, emu8000_port_t *port, int cmd, unsigned char *event)
{
	int voice;
	unsigned short p1;
	short p2;
	int i;
	snd_midi_channel_t *chan;

	voice = event[3];
	if (voice < 0 || voice >= port->chset.max_channels)
		chan = NULL;
	else
		chan = &port->chset.channels[voice];

	p1 = *(unsigned short *) &event[4];
	p2 = *(short *) &event[6];

	switch (cmd) {
	case _EMU8000_OSS_DEBUG_MODE:
		break;
	case _EMU8000_OSS_REVERB_MODE:
		emu->reverb_mode = p1;
		snd_emu8000_update_reverb_mode(emu);
		break;

	case _EMU8000_OSS_CHORUS_MODE:
		emu->chorus_mode = p1;
		snd_emu8000_update_chorus_mode(emu);
		break;
		      
	case _EMU8000_OSS_REMOVE_LAST_SAMPLES:
		snd_soundfont_remove_unlocked(emu->sflist);
		break;

	case _EMU8000_OSS_INITIALIZE_CHIP:
		snd_emu8000_init(emu);
		break;

	case _EMU8000_OSS_SEND_EFFECT:
		if (chan)
			snd_emu8000_send_effect_oss(emu, chan, p1, p2);
		break;

	case _EMU8000_OSS_RESET_CHANNEL:
		/*snd_emu8000_channel_init(chset, chan);*/
		break;
		
	case _EMU8000_OSS_TERMINATE_ALL:
		snd_emu8000_terminate_all(emu);
		break;

	case _EMU8000_OSS_TERMINATE_CHANNEL:
		//snd_emu8000_mute_channel(emu, chan);
		break;

	case _EMU8000_OSS_RELEASE_ALL:
		fake_event(emu, port, voice, SND_MCTL_ALL_NOTES_OFF, 0);
		break;
	case _EMU8000_OSS_NOTEOFF_ALL:
		fake_event(emu, port, voice, SND_MCTL_ALL_SOUNDS_OFF, 0);
		break;

	case _EMU8000_OSS_INITIAL_VOLUME:
		if (p2) {
			port->volume_atten = (short)p1;
			snd_emu8000_update_port(emu, port, EMU8000_UPDATE_VOLUME);
		}
		break;

	case _EMU8000_OSS_CHN_PRESSURE:
		if (chan) {
			chan->midi_pressure = p1;
			snd_emu8000_update_channel(emu, chan, EMU8000_UPDATE_FMMOD|EMU8000_UPDATE_FM2FRQ2);
		}
		break;

	case _EMU8000_OSS_CHANNEL_MODE:
		reset_port_mode(port, p1);
		snd_emu8000_reset_port(port);
		break;

	case _EMU8000_OSS_DRUM_CHANNELS:
		port->drum_flags = *(unsigned int*)&event[4];
		for (i = 0; i < port->chset.max_channels; i++) {
			chan = &port->chset.channels[i];
			chan->drum_channel = ((port->drum_flags >> i) & 1) ? 1 : 0;
		}
		break;

	case _EMU8000_OSS_MISC_MODE:
		if (p1 < EMU8000_MD_END)
			port->ctrls[p1] = p2;
		break;

	case _EMU8000_OSS_EQUALIZER:
		emu->bass_level = p1;
		emu->treble_level = p2;
		snd_emu8000_update_equalizer(emu);
		break;

	default:
		break;
	}
}


/*
 * GUS specific h/w controls
 */

#include <linux/ultrasound.h>

static void
gusspec_control(emu8000_t *emu, emu8000_port_t *port, int cmd, unsigned char *event)
{
	int voice;
	unsigned short p1;
	short p2;
	int plong;
	snd_midi_channel_t *chan;

	if (port->port_mode != PORT_MODE_OSS_SYNTH)
		return;
	if (cmd == _GUS_NUMVOICES)
		return;
	voice = event[3];
	if (voice < 0 || voice >= port->chset.max_channels)
		return;

	chan = &port->chset.channels[voice];

	p1 = *(unsigned short *) &event[4];
	p2 = *(short *) &event[6];
	plong = *(int*) &event[4];

	switch (cmd) {
	case _GUS_VOICESAMPLE:
		chan->midi_program = p1;
		return;

	case _GUS_VOICEBALA:
		/* 0 to 15 --> 0 to 127 */
		chan->control[SND_MCTL_MSB_PAN] = (int)p1 << 3;
		snd_emu8000_update_channel(emu, chan, EMU8000_UPDATE_PAN);
		return;

	case _GUS_VOICEVOL:
	case _GUS_VOICEVOL2:
		/* not supported yet */
		return;

	case _GUS_RAMPRANGE:
	case _GUS_RAMPRATE:
	case _GUS_RAMPMODE:
	case _GUS_RAMPON:
	case _GUS_RAMPOFF:
		/* volume ramping not supported */
		return;

	case _GUS_VOLUME_SCALE:
		return;

	case _GUS_VOICE_POS:
		snd_emu8000_send_effect(emu, chan, EMU8000_FX_SAMPLE_START,
					(short)(plong & 0x7fff),
					EMU8000_FX_FLAG_SET);
		snd_emu8000_send_effect(emu, chan, EMU8000_FX_COARSE_SAMPLE_START,
					(plong >> 15) & 0xffff,
					EMU8000_FX_FLAG_SET);
		return;
	}
}


/*
 * send an event to midi emulation
 */
static void
fake_event(emu8000_t *emu, emu8000_port_t *port, int ch, int param, int val)
{
	snd_seq_event_t ev;
	memset(&ev, 0, sizeof(ev));
	ev.type = SND_SEQ_EVENT_CONTROLLER;
	ev.dest.channel = ch;
	ev.data.control.param = param;
	ev.data.control.value = val;
	snd_emu8000_event_input(&ev, 0, port);
}

#endif /* CONFIG_SND_OSSEMUL */
