/*
 *   ALSA sequencer Queue handling
 *   Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __SND_SEQ_QUEUE_H
#define __SND_SEQ_QUEUE_H

#include "seq_memory.h"
#include "seq_prioq.h"
#include "seq_timer.h"

#define SEQ_QUEUE_NO_OWNER (-1)

struct snd_seq_stru_queue {
	int queue;		/* queue number */

	prioq_t	*tickq;		/* midi tick event queue */
	prioq_t	*timeq;		/* real-time event queue */	
	
	timer_t *timer;		/* time keeper for this queue */
	int	owner;		/* client that 'owns' the timer */
	int	locked:1,	/* timer is only accesibble by owner if set */
		klocked:1,	/* kernel lock (after START) */	
		check_again:1,
		check_blocked:1;
	spinlock_t owner_lock;
	spinlock_t check_lock;

	/* clients which uses this queue (bitmap) */
	unsigned int clients_bitmap[SND_SEQ_MAX_CLIENTS/sizeof(unsigned int)];
	unsigned int clients;	/* users of this queue */
	spinlock_t clients_lock;
	struct semaphore use_mutex;
};


/* init queues structure */
void snd_seq_queues_init(int num);

/* delete queues */ 
void snd_seq_queues_delete(void);


/* create new queue (constructor) */
extern queue_t *snd_seq_queue_new(void);

/* delete queue (destructor) */
extern void snd_seq_queue_delete(queue_t **q);

/* notification that client has left the system */
extern void snd_seq_queue_client_termination(int client);

/* final stage */
extern void snd_seq_queue_client_leave(int client);
extern void snd_seq_queue_client_leave_cells(int client);

/* enqueue a event received from one the clients */
extern int snd_seq_enqueue_event(snd_seq_event_cell_t *cell, int atomic);


/* return pointer to queue structure for specified id */
queue_t *queueptr(int queueid);

/* check single queue and dispatch events */
void snd_seq_check_queue(queue_t *q, int atomic);
/* check queues and dispatch events */
void snd_seq_check_queues(int atomic);

/* access to queue's parameters */
extern int snd_seq_queue_check_access(int queueid, int client);
extern int snd_seq_queue_timer_stop(int queueid, int client);
extern int snd_seq_queue_timer_start(int queueid, int client);
extern int snd_seq_queue_timer_continue(int queueid, int client);
extern int snd_seq_queue_timer_set_tempo(int queueid, int client, int tempo);
extern int snd_seq_queue_timer_set_ppq(int queueid, int client,int ppq);
extern int snd_seq_queue_set_owner(int queueid, int client);
extern int snd_seq_queue_set_locked(int queueid, int client, int locked);
extern int snd_seq_queue_timer_open(int queueid);
extern int snd_seq_queue_timer_close(int queueid);
extern int snd_seq_queue_use(int queueid, int client, int use);

#endif
