/*
 *   ALSA sequencer Priority Queue
 *   Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../include/driver.h"
#include "seq_timer.h"
#include "seq_prioq.h"
#include "seq_timer.h"


/* Implementation is a simple linked list for now...

   This priority queue orders the events on timestamp. For events with an
   equeal timestamp the queue behaves as a FIFO. 

   *
   *           +-------+
   *  Head --> | first |
   *           +-------+
   *                 |ptr_l
   *           +-----v-+
   *           |       |
   *           +-------+
   *                 |
   *           +-----v-+
   *           |       |
   *           +-------+
   *                 |
   *           +-----v-+
   *  Tail --> | last  |
   *           +-------+
   *

 */



/* create new prioq (constructor) */
prioq_t *snd_seq_prioq_new(void)
{
	prioq_t *f;

	f = snd_kcalloc(sizeof(prioq_t), GFP_KERNEL);
	if (f == NULL) {
		snd_printd("oops: malloc failed for snd_seq_prioq_new()\n");
		return NULL;
	}
	
	f->lock = SPIN_LOCK_UNLOCKED;
	f->head = NULL;
	f->tail = NULL;
	f->cells = 0;
	
	return f;
}

/* delete prioq (destructor) */
void snd_seq_prioq_delete(prioq_t **fifo)
{
	prioq_t *f = *fifo;
	*fifo = NULL;

	if (f == NULL) {
		snd_printd("oops: snd_seq_prioq_delete() called with NULL prioq\n");
		return;
	}

	/* release resources...*/
	/*....................*/
	
	if (f->cells > 0) {
		/* drain prioQ */
		int max = 0;
		
		while (snd_seq_prioq_avail(f) && max++ < SND_SEQ_MAX_EVENTS)
			snd_seq_cell_free(snd_seq_prioq_cell_out(f));
		if (max >= SND_SEQ_MAX_EVENTS)
			printk("oops: snd_seq_prioq_delete() drain trouble\n");
	}
	
	snd_kfree(f);
}




/* compare timestamp between events */
/* return 1 if a >= b; 0 */
static inline int compare_timestamp(snd_seq_event_t * a, snd_seq_event_t * b)
{
	if ((a->flags & SND_SEQ_TIME_STAMP_MASK) == SND_SEQ_TIME_STAMP_REAL) {
		/* compare ticks */
		return (snd_seq_compare_tick_time(&a->time.tick, &b->time.tick));
	} else {
		/* compare real time */
		return (snd_seq_compare_real_time(&a->time.real, &b->time.real));
	}
}

/* enqueue cell to prioq */
void snd_seq_prioq_cell_in(prioq_t * f, snd_seq_event_cell_t * cell)
{
	snd_seq_event_cell_t *cur, *prev;
	unsigned long flags;

	if (f == NULL) {
		snd_printd("oops: snd_seq_prioq_cell_in() called with NULL prioq\n");
		return;
	}
	if (cell == NULL) {
		snd_printd("oops: snd_seq_prioq_cell_in() called with NULL cell\n");
		return;
	}
	
	spin_lock_irqsave(&f->lock, flags);

	/* check if this element needs to inserted at the end (ie. ordered 
	   data is inserted) This will be very likeley if a sequencer 
	   application or midi file player is feeding us (sequential) data */
	if (f->tail) {
		if (compare_timestamp(&cell->event, &f->tail->event)) {
			/* add new cell to tail of the fifo */
			f->tail->ptr_l = cell;
			f->tail = cell;
			cell->ptr_l = NULL;
			f->cells++;
			spin_unlock_irqrestore(&f->lock, flags);
			return;
		}
	}
	/* traverse list of elements to find the place where the new cell is
	   to be inserted... Note that this is a order n process ! */

	prev = NULL;		/* previous cell */
	cur = f->head;		/* cursor */

	while (cur != NULL) {
		/* compare timestamps */
		if (!compare_timestamp(&cell->event, &cur->event)) {
			/* new cell has earlier schedule time, */
			/* insert it before cursor */

			if (prev != NULL)
				prev->ptr_l = cell;
			cell->ptr_l = cur;

			/* this is the first cell, set head to it */
			if (f->head == cur)
				f->head = cell;

			f->cells++;
			spin_unlock_irqrestore(&f->lock, flags);
			return;
		} else {
			/* new cell has equal or larger schedule time, */
			/* move cursor to next cell */
			prev = cur;
			cur = cur->ptr_l;
		}
	}

	/* reached end of the list */
	if (prev != NULL)
		prev->ptr_l = cell;
	cell->ptr_l = NULL;
	f->tail = cell;
	if (f->head == NULL)
		f->head = cell;
	f->cells++;
	spin_unlock_irqrestore(&f->lock, flags);
}

/* dequeue cell from prioq */
snd_seq_event_cell_t *snd_seq_prioq_cell_out(prioq_t * f)
{
	snd_seq_event_cell_t *cell;
	unsigned long flags;

	if (f == NULL) {
		snd_printd("oops: snd_seq_prioq_cell_in() called with NULL prioq\n");
		return NULL;
	}
	spin_lock_irqsave(&f->lock, flags);

	cell = f->head;
	if (cell) {
		f->head = cell->ptr_l;

		/* reset tail if this was the last element */
		if (f->tail == cell)
			f->tail = NULL;

		cell->ptr_l = NULL;
		f->cells--;
	}

	spin_unlock_irqrestore(&f->lock, flags);
	return cell;
}

/* return number of events available in prioq */
int snd_seq_prioq_avail(prioq_t * f)
{
	if (f == NULL) {
		snd_printd("oops: snd_seq_prioq_cell_in() called with NULL prioq\n");
		return 0;
	}
	return f->cells;
}


/* peek at cell at the head of the prioq */
snd_seq_event_cell_t *snd_seq_prioq_cell_peek(prioq_t * f)
{
	if (f == NULL) {
		snd_printd("oops: snd_seq_prioq_cell_in() called with NULL prioq\n");
		return NULL;
	}
	return f->head;
}


static inline int prioq_match(snd_seq_event_cell_t *cell, int client, int timestamp)
{
	if (cell->event.source.client == client ||
	    cell->event.dest.client == client)
		return 1;
	if (!timestamp)
		return 0;
	switch (cell->event.flags & SND_SEQ_TIME_STAMP_MASK) {
	case SND_SEQ_TIME_STAMP_TICK:
		if (cell->event.time.tick)
			return 1;
		break;
	case SND_SEQ_TIME_STAMP_REAL:
		if (cell->event.time.real.tv_sec ||
		    cell->event.time.real.tv_nsec)
			return 1;
		break;
	}
	return 0;
}

/* remove cells for left client */
void snd_seq_prioq_leave(prioq_t * f, int client, int timestamp)
{
	register snd_seq_event_cell_t *cell, *next;
	unsigned long flags;
	snd_seq_event_cell_t *prev = NULL;
	snd_seq_event_cell_t *freefirst = NULL, *freeprev = NULL, *freenext;

	/* collect all removed cells */
	spin_lock_irqsave(&f->lock, flags);
	cell = f->head;
	while (cell) {
		next = cell->ptr_l;
		if (prioq_match(cell, client, timestamp)) {
			/* remove cell from prioq */
			if (cell == f->head) {
				f->head = cell->ptr_l;
			} else {
				prev->ptr_l = cell->ptr_l;
			}
			if (cell == f->tail)
				f->tail = cell->ptr_l;
			f->cells--;
			/* add cell to free list */
			cell->ptr_l = NULL;
			if (!freefirst) {
				freefirst = cell;
			} else {
				freeprev->ptr_l = cell;
			}
			freeprev = cell;
		} else {
#if 0
			printk("type = %i, source = %i, dest = %i, client = %i\n",
				cell->event.type,
				cell->event.source.client,
				cell->event.dest.client,
				client);
#endif
			prev = cell;
		}
		cell = next;		
	}
	spin_unlock_irqrestore(&f->lock, flags);	

	/* remove selected cells */
	while (freefirst) {
		freenext = freefirst->ptr_l;
		snd_seq_cell_free(freefirst);
		freefirst = freenext;
	}
}
