/*
 *  ALSA sequencer Client Manager
 *  Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *                        Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../include/driver.h"
#include "../../include/minors.h"
#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif

#include "../../include/seq_kernel.h"
#include "seq_clientmgr.h"
#include "seq_memory.h"
#include "seq_queue.h"
#include "seq_timer.h"
#include "seq_info.h"
#include "seq_system.h"
#include "seq_lock.h"

#undef TRACK_SUBSCRIBE

/* Client Manager

 * this module handles the connections of userland and kernel clients
 * 
 */

#define SND_SEQ_LFLG_INPUT	0x0001
#define SND_SEQ_LFLG_OUTPUT	0x0002
#define SND_SEQ_LFLG_OPEN	(SND_SEQ_LFLG_INPUT|SND_SEQ_LFLG_OUTPUT)

#define semaphore_of(fp)	((fp)->f_dentry->d_inode->i_sem)

static spinlock_t clients_lock = SPIN_LOCK_UNLOCKED;
static DECLARE_MUTEX(register_mutex);

static char clienttablock[SND_SEQ_MAX_CLIENTS];
static client_t *clienttab[SND_SEQ_MAX_CLIENTS];
static usage_t client_usage = {0, 0};

static inline unsigned short snd_seq_file_flags(struct file *file)
{
        switch (file->f_mode & (FMODE_READ | FMODE_WRITE)) {
        case FMODE_WRITE:
                return SND_SEQ_LFLG_OUTPUT;
        case FMODE_READ:
                return SND_SEQ_LFLG_INPUT;
        default:
                return SND_SEQ_LFLG_OPEN;
        }
}

static inline int snd_seq_write_pool_allocated(client_t *client)
{
	return snd_seq_total_cells(client->pool) > 0;
}

static inline int snd_seq_output_ok(client_t *client)
{
	return snd_seq_unused_cells(client->pool) >= client->pool_room;
}

/* return pointer to client structure for specified id */
static client_t *clientptr(int clientid)
{
	if (clientid < 0 || clientid >= SND_SEQ_MAX_CLIENTS) {
		snd_printd("Seq: oops. Trying to get pointer to client %d\n", clientid);
		return NULL;
	}
	return clienttab[clientid];
}

extern int snd_seq_client_load[];

client_t *snd_seq_client_use_ptr(int clientid)
{
	unsigned long flags;
	client_t *client;

	spin_lock_irqsave(&clients_lock, flags);
	client = clientptr(clientid);
	if (client)
		goto __lock;
	spin_unlock_irqrestore(&clients_lock, flags);
#ifdef CONFIG_KMOD
	if (!in_interrupt()) {
		if (clientid < 64) {
			int idx;
			char name[32];
			
			for (idx = 0; idx < 64; idx++) {
				if (snd_seq_client_load[idx] < 0)
					break;
				if (snd_seq_client_load[idx] == clientid) {
					sprintf(name, "snd-seq-client-%i", clientid);
					request_module(name);
					break;
				}
			}
		} else if (clientid >= 64 && clientid < 128) {
			snd_request_card((clientid - 64) / 8);
		}
		spin_lock_irqsave(&clients_lock, flags);
		client = clientptr(clientid);
		if (client)
			goto __lock;
		spin_unlock_irqrestore(&clients_lock, flags);
	}
#endif
	return NULL;

      __lock:
	spin_lock(&client->use_lock);
	client->use++;
	spin_unlock(&client->use_lock);
	spin_unlock_irqrestore(&clients_lock, flags);
	return client;
}

void snd_seq_client_unlock(client_t *client)
{
	unsigned long flags;

	if (!client)
		return;
	spin_lock_irqsave(&client->use_lock, flags);
	if (!(--client->use) && waitqueue_active(&client->use_sleep))
	    	wake_up(&client->use_sleep);
	spin_unlock_irqrestore(&client->use_lock, flags);
}

static void usage_alloc(usage_t * res, int num)
{
	res->cur += num;
	if (res->cur > res->peak)
		res->peak = res->cur;
}

static void usage_free(usage_t * res, int num)
{
	res->cur -= num;
}

/* initialise data structures */
void client_init_data(void)
{
	/* zap out the client table */
	memset(&clienttablock, 0, sizeof(clienttablock));
	memset(&clienttab, 0, sizeof(clienttab));
}


static client_t *seq_create_client1(int client_index, int poolsize)
{
	unsigned long flags;
	int c;
	client_t *client;

	/* init client data */
	client = snd_kcalloc(sizeof(client_t), GFP_KERNEL);
	if (!client)
		return NULL;
	client->pool = snd_seq_pool_new();
	if (! client->pool) {
		snd_kfree(client);
		return NULL;
	}
	client->pool_size = poolsize;
	client->pool_room = (poolsize + 1) / 2;
	client->type = NO_CLIENT;
	client->use_lock = SPIN_LOCK_UNLOCKED;
	init_waitqueue_head(&client->use_sleep);
	client->ports_lock = SPIN_LOCK_UNLOCKED;
	init_waitqueue_head(&client->ports_sleep);

	client->output_sleep_lock = SPIN_LOCK_UNLOCKED;
	init_waitqueue_head(&client->output_sleep);
 
	/* find free slot in the client table */
	spin_lock_irqsave(&clients_lock, flags);
	if (client_index < 0) {
		for (c = 128; c < SND_SEQ_MAX_CLIENTS; c++) {
			if (clienttab[c] || clienttablock[c])
				continue;
			clienttab[client->number = c] = client;
			spin_unlock_irqrestore(&clients_lock, flags);
			return client;
		}
	} else {
		if (!clienttab[client_index] && !clienttablock[client_index]) {
			clienttab[client->number = client_index] = client;
			spin_unlock_irqrestore(&clients_lock, flags);
			return client;
		}
	}
	spin_unlock_irqrestore(&clients_lock, flags);
	snd_kfree(client);
	return NULL;	/* no free slot found or busy, return failure code */
}


static int seq_free_client1(client_t *client)
{
	unsigned long flags;

	if (!client)
		return -EINVAL;
	spin_lock_irqsave(&clients_lock, flags);
	clienttablock[client->number] = 1;
	clienttab[client->number] = NULL;
	spin_unlock_irqrestore(&clients_lock, flags);
	spin_lock_irqsave(&client->use_lock, flags);
	while (client->use) {
		snd_seq_sleep_in_lock(&client->use_sleep, &client->use_lock);
	}
	spin_unlock_irqrestore(&client->use_lock, flags);
	snd_seq_queue_client_termination(client->number);
	snd_seq_delete_ports(client);
	snd_seq_queue_client_leave(client->number);
	if (client->pool) {
		snd_seq_pool_done(client->pool);
		snd_seq_pool_delete(&client->pool);
	}
	clienttablock[client->number] = 0;
	return 0;
}


static void seq_free_client(client_t * client)
{
	if (!client)
		return;

	down(&register_mutex);
	switch (client->type) {
	case NO_CLIENT:
		snd_printk("Seq: Trying to free unused client %d\n", client->number);
		break;
	case USER_CLIENT:
	case KERNEL_CLIENT:
		seq_free_client1(client);
		usage_free(&client_usage, 1);
		break;

	default:
		snd_printk("Seq: Trying to free client %d with undefined type = %d\n", client->number, client->type);
	}
	up(&register_mutex);

	snd_seq_system_client_ev_client_exit(client->number);
}



/* -------------------------------------------------------- */

/* create a user client */
static int snd_seq_open(unsigned short minor, int cardnum, int device, struct file *file)
{
	int c, mode;			/* client id */
	client_t *client;
	user_client_t *user;

	down(&register_mutex);
	client = seq_create_client1(-1, SND_SEQ_DEFAULT_EVENTS);
	if (!client) {
		up(&register_mutex);
		return -ENOMEM;	/* failure code */
	}

	mode = snd_seq_file_flags(file);
	if (mode & SND_SEQ_LFLG_INPUT)
		client->accept_input = 1;
	if (mode & SND_SEQ_LFLG_OUTPUT)
		client->accept_output = 1;

	user = &client->data.user;
	user->fifo = NULL;
	user->fifo_pool_size = 0;

	if (mode & SND_SEQ_LFLG_INPUT) {
		user->fifo_pool_size = SND_SEQ_DEFAULT_CLIENT_EVENTS;
		user->fifo = snd_seq_fifo_new(user->fifo_pool_size);
		if (! user->fifo) {
			seq_free_client1(client);
			snd_kfree(client);
			up(&register_mutex);
			return -ENOMEM;
		}
	}

	usage_alloc(&client_usage, 1);
	client->type = USER_CLIENT;
	up(&register_mutex);

	c = client->number;
	(user_client_t *) file->private_data = client;

	/* fill client data */
	user->file = file;
	sprintf(client->name, "Client-%d", c);

	/* make others aware this new client */
	snd_seq_system_client_ev_client_start(c);

	MOD_INC_USE_COUNT;

	return 0;
}

static int snd_seq_release(unsigned short minor, int cardnum, int device, struct file *file)
{
	client_t *client = (client_t *) file->private_data;

	if (client) {
		seq_free_client(client);
		if (client->data.user.fifo)
			snd_seq_fifo_delete(&client->data.user.fifo);
		snd_kfree(client);
	}

	MOD_DEC_USE_COUNT;
	return 0;
}


/* handle client read() */
static long snd_seq_read(struct file *file, char *buf, long count)
{
	client_t *client = (client_t *) file->private_data;
	fifo_t *fifo;
	int err;
	long result = 0;

	if (!(snd_seq_file_flags(file) & SND_SEQ_LFLG_INPUT))
		return -ENXIO;

	if (verify_area(VERIFY_WRITE, buf, count))
		return -EFAULT;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;

	if (!client->accept_input || !(fifo = client->data.user.fifo))
		return -EIO;

	if (fifo->overflow) {
		/* buffer overflow is detected */
		snd_seq_fifo_clear(fifo);
		/* return error code */
		return -ERANGE;
	}

	/* while data available in queue */
	while (count >= sizeof(snd_seq_event_t)) {
		snd_seq_event_cell_t *cell;
		int len;
		int atomic;

		atomic = ((file->f_flags & O_NONBLOCK) || result > 1)? 1: 0;
		cell = snd_seq_fifo_cell_out(fifo, &err, atomic);
		if (cell == NULL)
			break;

		/* check if buffer is big enough for the new event */
		switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
		case SND_SEQ_EVENT_LENGTH_VARIABLE:
			/* handle variable length data */
			len = sizeof(snd_seq_event_t) + cell->event.data.ext.len;
			break;
		case SND_SEQ_EVENT_LENGTH_FIXED:
		default:
			/* normal event, no special action needed */
			len = sizeof(snd_seq_event_t);
			break;
		}
		/* check if buffer is big enough for the new event */
		if (count < len) {
			if (result == 0)
				err = -EINVAL; /* This will never work */

			/* Put the cell back on the fifo */
			snd_seq_fifo_cell_putback(fifo, cell);
			break; /* not enough room for this new event */
		}

		/* Place the cell into the users read buffer */
		switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
		case SND_SEQ_EVENT_LENGTH_VARIABLE:
			/* handle variable length data */
			len = sizeof(snd_seq_event_t) + cell->event.data.ext.len;
			copy_to_user(buf, &cell->event, sizeof(snd_seq_event_t));
			if (cell->event.data.ext.ptr != NULL) {
				copy_to_user(buf + sizeof(snd_seq_event_t), cell->event.data.ext.ptr, cell->event.data.ext.len);
			}
			break;

		case SND_SEQ_EVENT_LENGTH_FIXED:
		default:
			/* normal event, no special action needed */
			len = sizeof(snd_seq_event_t);
			copy_to_user(buf, &cell->event, sizeof(snd_seq_event_t));
			break;
		}
		/* free cell - DO NOT use snd_seq_cell_free here! */
		snd_seq_fifo_cell_free(fifo, cell);

		result += len;
		count -= len;
		buf += len;
	}

	return err ? err : result;
}


/* deliver an event to the specified destination.
   if fill_time is true, the current queue time is copied (for direct passing).
   if filter is non-zero, client filter bitmap is tested.
   */
static int snd_seq_deliver_single_event(snd_seq_event_t *event, int fill_time, int filter, int atomic)
{
	client_t *dest = NULL;
	client_port_t *dest_port = NULL;
	int result = -EINVAL;
	int direct = 0;

	if ((event->flags & SND_SEQ_DEST_MASK) == SND_SEQ_DEST_DIRECT)
		direct = 1;

	dest = snd_seq_client_use_ptr(event->dest.client);
	if (! dest)
		goto __skip;
	if (! dest->accept_input)
		goto __skip;
	if (filter && ! (dest->filter & filter))
		goto __skip;
	if (dest->filter & SND_SEQ_FILTER_USE_EVENT) {
		if (!test_bit(event->type, &dest->event_filter))
			goto __skip;
	}

	dest_port = snd_seq_port_use_ptr(dest, event->dest.port);
	if (! dest_port)
		goto __skip;

	if (fill_time) {
		queue_t *q;
		q = queueptr(event->dest.queue);
		if (q && test_bit(dest->number, &q->clients_bitmap)) {
			switch (event->flags & SND_SEQ_TIME_STAMP_MASK) {
			case SND_SEQ_TIME_STAMP_TICK:
				event->time.tick = snd_seq_timer_get_cur_tick(q->timer);
				break;
			case SND_SEQ_TIME_STAMP_REAL:
				event->time.real = snd_seq_timer_get_cur_time(q->timer);
				break;
  			}
		}
	}
  
	switch (dest->type) {
	case USER_CLIENT:
		if (dest->data.user.fifo == NULL)
			result = 0; /* no read buffer -- ignored */
		else {
			/* LENGTH_VARUSR or LENGTH_VARIPC event will be
			 * converted to LENGTH_VARIABLE in cell_alloc(),
			 * so we don't have to worry about it.
			 */
			result = snd_seq_fifo_event_in(dest->data.user.fifo, event, atomic);
		}
		break;

	case KERNEL_CLIENT:
		if (dest_port->event_input == NULL)
			break;  /* no input? - just ignored */
		if ((event->flags & SND_SEQ_EVENT_LENGTH_MASK) == SND_SEQ_EVENT_LENGTH_VARUSR &&
		    event->type < SND_SEQ_EVENT_NORMAL_CONTROLS) {
			/* copy VARUSR event to VARIABLE event */
			snd_seq_event_t newev;
			char *ptr;
			memcpy(&newev, event, sizeof(newev));
			ptr = snd_seq_ext_malloc(event->data.ext.len, atomic);
			newev.flags &= ~SND_SEQ_EVENT_LENGTH_MASK;
			newev.flags |= SND_SEQ_EVENT_LENGTH_VARIABLE;
			/* copy data to temporary kernel space */
			if (ptr == NULL) {
				result = -ENOMEM;
				break;
			}
			if (copy_from_user(ptr, event->data.ext.ptr, event->data.ext.len)) {
				snd_seq_ext_free(ptr, event->data.ext.len);
				result = -EFAULT;
				break;
			}
			newev.data.ext.ptr = ptr;
			result = dest_port->event_input(&newev, direct, dest_port->private_data);
			/* free temporary buffer */
			snd_seq_ext_free(ptr, event->data.ext.len);
		} else {
			result = dest_port->event_input(event, direct, dest_port->private_data);
		}
		break;

	default:
		break;
	}

  __skip:
	if (dest_port)
		snd_seq_ports_unlock(dest);
	if (dest)
		snd_seq_client_unlock(dest);
	return result;
}


/* deliver an event to the destination port(s).
 * if the event is to subscribers or broadcast, the event is dispatched
 * to multiple targets.
 */
static int snd_seq_deliver_event(client_t *client, snd_seq_event_t *event, int atomic)
{
	int fill_time = 0;

	if (event->dest.queue == SND_SEQ_ADDRESS_SUBSCRIBERS ||
	    event->dest.client == SND_SEQ_ADDRESS_SUBSCRIBERS) {
		/* send the event to all subscribers */
		client_port_t *src_port;
		subscribers_t *subs;

		src_port = snd_seq_port_use_ptr(client, event->source.port);
		if (! src_port)
			return 0; /* skip */
		if (! src_port->input.list ||
		    !(src_port->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION)) {
			snd_seq_ports_unlock(client);
			return 0; /* skip */
		}

		snd_seq_subscribers_lock(&src_port->input);
		for (subs = src_port->input.list; subs; subs = subs->next) {
			event->dest.client = subs->addr.client;
			event->dest.port = subs->addr.port;
			event->dest.queue = subs->addr.queue;
			event->flags &= ~SND_SEQ_TIME_STAMP_MASK;
			event->flags |=
				subs->realtime ? SND_SEQ_TIME_STAMP_REAL : SND_SEQ_TIME_STAMP_TICK;
			snd_seq_deliver_single_event(event, 1, 0, atomic);
		}
		snd_seq_subscribers_unlock(&src_port->input);
		snd_seq_ports_unlock(client);
		return 0;
	}

	if (event->dest.queue == SND_SEQ_ADDRESS_BROADCAST ||
	    event->dest.client == SND_SEQ_ADDRESS_BROADCAST) {
		/* send the event to all clients */
		int dest;
		if (event->dest.queue < SND_SEQ_MAX_QUEUES)
			fill_time = 1;
		for (dest = 0; dest < SND_SEQ_MAX_CLIENTS; dest++) {
			/* don't send to itself */
			if (dest == client->number)
				continue;
			event->dest.client = dest;
			snd_seq_deliver_single_event(event, fill_time, SND_SEQ_FILTER_BROADCAST, atomic);
		}
		return 0;

	}

	/* deliver to the specified port */
	if (event->dest.client < SND_SEQ_MAX_CLIENTS) {
		if ((event->flags & SND_SEQ_DEST_MASK) == SND_SEQ_DEST_DIRECT &&
		    event->dest.queue < SND_SEQ_MAX_QUEUES)
			fill_time = 1;
		snd_seq_deliver_single_event(event, fill_time, 0, atomic);
	} else {
		/* multicast - not supported yet */
		snd_printd("seq: multicast not supported yet.\n");
	}

	return 0;
}


/* dispatch an event cell */
int snd_seq_dispatch_event(snd_seq_event_cell_t *cell, int atomic)
{
	client_t *client;
	unsigned long flags;

	client = snd_seq_client_use_ptr(cell->event.source.client);
	if (! client)
		return -EINVAL;

	snd_seq_deliver_event(client, &cell->event, atomic);
	/* event was processed, or duplicated on fifo.
	   now release the cell */
	snd_seq_cell_free(cell);
	/* if the client is sleeping, wake it up */
	spin_lock_irqsave(&client->output_sleep_lock, flags);
	if (waitqueue_active(&client->output_sleep)) {
		/* has enough space now? */
		if (snd_seq_output_ok(client))
			wake_up(&client->output_sleep);
	}
	spin_unlock_irqrestore(&client->output_sleep_lock, flags);

	snd_seq_client_unlock(client);
	return 0;
}


/* Allocate a cell from client pool and enqueue it to queue:
 * if pool is empty and blocking is TRUE, sleep until a new cell is
 * available.
 *
 * NOTE: VARUSR event is converted to VARIABLE event in seq_event_dup().
 *       Event data is copied from user space to kernel malloced buffer.
 */
static int snd_seq_client_enqueue_single_event(client_t *client, snd_seq_event_t *event, struct file *file, int blocking, int atomic)
{
	unsigned long flags;
	snd_seq_event_cell_t *cell;

	if (! snd_seq_write_pool_allocated(client))
		return -EINVAL; /* queue is not allocated */

	cell = NULL;

	spin_lock_irqsave(&client->output_sleep_lock, flags);
	if (snd_seq_unused_cells(client->pool) > 0)
		cell = snd_seq_event_dup(client->pool, event, atomic);

	/* sleep until pool has enough room */
	while (cell == NULL) {
		if (! blocking) {
			spin_unlock_irqrestore(&client->output_sleep_lock, flags);
			return -EAGAIN;
		}
		/* change semaphore to allow other clients
		   to access device file */
		if (file)
			up(&semaphore_of(file));

		snd_seq_sleep_in_lock(&client->output_sleep, &client->output_sleep_lock);

		/* restore semaphore again */
		if (file)
			down(&semaphore_of(file));
		if (signal_pending(current)) {
			spin_unlock_irqrestore(&client->output_sleep_lock, flags);
			return -EINTR;
		}
		/* try again.. */
		cell = snd_seq_event_dup(client->pool, event, atomic);
	}
	spin_unlock_irqrestore(&client->output_sleep_lock, flags);

	if (snd_seq_enqueue_event(cell, atomic) < 0) {
		snd_seq_cell_free(cell);
		return -ENOMEM;
	}

	return 0;
}


/* Allocate cell from client pool and enqueue it to queue:
 * EVENT_NOTE is decomposed to NOTEON and NOTEOFF events.
 */
static int snd_seq_client_enqueue_event(client_t *client, snd_seq_event_t *ev, struct file *file, int blocking, int atomic)
{
	if (ev->type == SND_SEQ_EVENT_NOTE) {
		snd_seq_event_t offev;
		int rc;

		/* create note-off event */
		memcpy(&offev, ev, sizeof(offev));

		offev.type = SND_SEQ_EVENT_NOTEOFF;
		/* add the dulation time */
		switch (offev.flags & SND_SEQ_TIME_STAMP_MASK) {
		case SND_SEQ_TIME_STAMP_TICK:
			offev.time.tick += ev->data.note.duration;
			break;
		case SND_SEQ_TIME_STAMP_REAL:
			/* unit for duration is ms */
			offev.time.real.tv_nsec +=
			1000000 * (ev->data.note.duration % 1000);
			offev.time.real.tv_sec += ev->data.note.duration / 1000 +
				offev.time.real.tv_nsec % 1000000000;
			offev.time.real.tv_nsec %= 1000000000;
			break;
		}

		/* check for enough space - two cells are necessary! */
		if (! blocking) {
			if (snd_seq_unused_cells(client->pool) < 2)
				return -EAGAIN;
		}

		/* change the event type to note-on */
		ev->type = SND_SEQ_EVENT_NOTEON;

		/* note-on */
		rc = snd_seq_client_enqueue_single_event(client, ev, file, blocking, atomic);
		if (rc < 0)
			return rc;
		/* note-off */
		return snd_seq_client_enqueue_single_event(client, &offev, file, blocking, atomic);
	}
	return snd_seq_client_enqueue_single_event(client, ev, file, blocking, atomic);
}


/* handle write() */
static long snd_seq_write(struct file *file, const char *buf, long count)
{
	client_t *client = (client_t *) file->private_data;
	int written = 0, len;
	int err = 0;
	snd_seq_event_t event;

	if (!(snd_seq_file_flags(file) & SND_SEQ_LFLG_OUTPUT))
		return -ENXIO;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;
		
	if (!client->accept_output || client->pool == NULL)
		return -EIO;

	/* allocate the pool now if the pool is not allocated yet */ 
	if (client->pool_size > 0 && !snd_seq_write_pool_allocated(client)) {
		if (snd_seq_pool_init(client->pool, client->pool_size) < 0)
			return -ENOMEM;
	}

	/* only process whole events */
	while (count >= sizeof(snd_seq_event_t)) {
		int extlen;
		
		/* Read in the event header from the user */
		len = sizeof(event);
		if (copy_from_user(&event, buf, len)) {
			err = -EFAULT;
			break;
		}
		event.source.queue = SND_SEQ_ADDRESS_UNKNOWN;
		event.source.client = client->number;	/* fill in client number */

		/* Check for extension data length */
		switch (event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
		case SND_SEQ_EVENT_LENGTH_VARIABLE:
		case SND_SEQ_EVENT_LENGTH_VARUSR:
			/* Check for invalid extension data length */
			extlen = event.data.ext.len;
			if (extlen < 0) {
				/* back out, will get an error this time or next */
				err = -EINVAL;
				goto error_exit;
			}
			if ((event.flags & SND_SEQ_EVENT_LENGTH_MASK) == SND_SEQ_EVENT_LENGTH_VARIABLE) {
				if (extlen + len > count) {
					/* back out, will get an error this time or next */
					err = -EINVAL;
					goto error_exit;
				}
				/* use VARUSR event */
				event.flags &= ~SND_SEQ_EVENT_LENGTH_MASK;
				event.flags |= SND_SEQ_EVENT_LENGTH_VARUSR;
				/* set user space pointer */
				event.data.ext.ptr = (char*)buf + sizeof(snd_seq_event_t);
				len += extlen;  /* Inc in case of skip */
			}				
			break;
		case SND_SEQ_EVENT_LENGTH_VARIPC:
			err = -EINVAL; /* FIXME: not supported yet.. */
			goto error_exit;
		}

		/* Check if this event can be directly delivered */
		/* NOTE: if dest.queue is special values (subscribers or
		   broadcast), the events are delivered immediately.
		   */
		if ((event.flags & SND_SEQ_DEST_MASK) == SND_SEQ_DEST_DIRECT ||
		    event.dest.queue == SND_SEQ_ADDRESS_SUBSCRIBERS ||
		    event.dest.queue == SND_SEQ_ADDRESS_BROADCAST) {
			snd_seq_deliver_event(client, &event, 0);
		} else {
			/* Not direct, normal queuing */
			queue_t *q;

			/* check the queue */
			q = queueptr(event.dest.queue);
			if (!q || !test_bit(client->number, &q->clients_bitmap))
				goto __skip;

			err = snd_seq_client_enqueue_event(client, &event, file,
							  !(file->f_flags & O_NONBLOCK), 0);
			if (err)
				break;
		}

		/* Update pointers and counts */
	__skip:
		count -= len;
		buf += len;
		written += len;
	}

  error_exit:
	return written ? written : err;
}


static unsigned int snd_seq_poll(struct file *file, poll_table * wait)
{
	client_t *client = (client_t *) file->private_data;
	unsigned int mask = 0;

	/* check client structures are in place */
	if (client == NULL)
		return -EIO;

	if ((snd_seq_file_flags(file) & SND_SEQ_LFLG_INPUT) &&
	    client->data.user.fifo) {

		/* check if data is available in the outqueue */
		if (snd_seq_fifo_poll_wait(client->data.user.fifo, file, wait))
			mask |= POLLIN | POLLRDNORM;
	}

	if (snd_seq_file_flags(file) & SND_SEQ_LFLG_OUTPUT) {

		poll_wait(file, &client->output_sleep, wait);

		/* check if data is available in the pool */
		if (!snd_seq_write_pool_allocated(client) || snd_seq_output_ok(client))
			mask |= POLLOUT | POLLWRNORM;
	}

	return mask;
}


/*-----------------------------------------------------*/


/* SYSTEM_INFO ioctl() */
static int snd_seq_ioctl_system_info(client_t *client, snd_seq_system_info_t * _info)
{
	snd_seq_system_info_t info;

	/* fill the info fields */
	info.queues = SND_SEQ_MAX_QUEUES;
	info.clients = SND_SEQ_MAX_CLIENTS;
	info.ports = 256;	/* fixed limit */
	info.channels = 256;	/* fixed limit */

	if (copy_to_user(_info, &info, sizeof(info)))
		return -EFAULT;
	return 0;
}


/* CLIENT_INFO ioctl() */
static int snd_seq_ioctl_get_client_info(client_t * client, snd_seq_client_info_t * _client_info)
{
	client_t *cptr;
	snd_seq_client_info_t client_info;

	if (copy_from_user(&client_info, _client_info, sizeof(client_info)))
		return -EFAULT;

	/* requested client number */
	cptr = snd_seq_client_use_ptr(client_info.client);
	if (!cptr)
		return -ENOENT;		/* don't change !!! */
	memset(&client_info, 0, sizeof(client_info));
	client_info.client = cptr->number;

	/* fill the info fields */
	client_info.type = cptr->type;
	strcpy(client_info.name, cptr->name);
	client_info.filter = cptr->filter;
	memcpy(client_info.multicast_filter, cptr->multicast_filter, sizeof(client_info.multicast_filter));
	memcpy(client_info.event_filter, cptr->event_filter, sizeof(client_info.event_filter));

	snd_seq_client_unlock(cptr);

	if (copy_to_user(_client_info, &client_info, sizeof(client_info)))
		return -EFAULT;
	return 0;
}


/* CLIENT_INFO ioctl() */
static int snd_seq_ioctl_set_client_info(client_t * client, snd_seq_client_info_t * _client_info)
{
	snd_seq_client_info_t client_info;

	if (copy_from_user(&client_info, _client_info, sizeof(client_info)))
		return -EFAULT;

	/* it is not allowed to set the info fields for an another client */
	if (client->number != client_info.client)
		return -EPERM;
	/* also client type must be set now */
	if (client->type != client_info.type)
		return -EINVAL;

	/* fill the info fields */
	strncpy(client->name, client_info.name, sizeof(client->name)-1);
	client->name[sizeof(client->name)-1] = '\0';
	client->filter = client_info.filter;
	memcpy(client->multicast_filter, client_info.multicast_filter, sizeof(client->multicast_filter));
	memcpy(client->event_filter, client_info.event_filter, sizeof(client->event_filter));

	return 0;
}


/* 
 * CREATE PORT ioctl() 
 */
static int snd_seq_ioctl_create_port(client_t * client, snd_seq_port_info_t * _info)
{
	client_port_t *port;
	snd_seq_port_info_t info;
	snd_seq_port_callback_t *callback;

	if (!client || !_info)
		return -EINVAL;
	port = snd_seq_create_port(client);
	if (!port)
		return -ENOMEM;


	if (copy_from_user(&info, _info, sizeof(snd_seq_port_info_t))) {
		snd_seq_delete_port(client, port->port);
		return -EFAULT;
	}

	/* it is not allowed to create the port for an another client */
	if (info.client != client->number)
		return -EPERM;
	if (client->type == USER_CLIENT && info.kernel) {
		snd_seq_delete_port(client, port->port);
		return -EINVAL;
	}
	if (client->type == KERNEL_CLIENT) {
		if ((callback = info.kernel) != NULL) {
			port->private_data = callback->private_data;
			port->subscribe = callback->subscribe;
			port->unsubscribe = callback->unsubscribe;
			port->use = callback->use;
			port->unuse = callback->unuse;
			port->event_input = callback->event_input;
			port->private_free = callback->private_free;
		} else {
			port->private_data = NULL;
			port->subscribe = NULL;
			port->unsubscribe = NULL;
			port->use = NULL;
			port->unuse = NULL;
			port->event_input = NULL;
			port->private_free = NULL;
		}
	}

	info.port = port->port;

	snd_seq_set_port_info(port, &info);
	snd_seq_system_client_ev_port_start(client->number, port->port);

	if (copy_to_user(_info, &info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;

	return 0;
}

/* 
 * DELETE PORT ioctl() 
 */
static int snd_seq_ioctl_delete_port(client_t * client, snd_seq_port_info_t * _info)
{
	snd_seq_port_info_t info;
	int err;

	if (!client || !_info)
		return -EINVAL;

	/* set passed parameters */
	if (copy_from_user(&info, _info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;
	
	/* it is not allowed to remove the port for an another client */
	if (info.client != client->number)
		return -EPERM;

	err = snd_seq_delete_port(client, info.port);
	if (err >= 0)
		snd_seq_system_client_ev_port_exit(client->number, info.port);
	return err;
}


/* 
 * GET_PORT_INFO ioctl() (on any client) 
 */
static int snd_seq_ioctl_get_port_info(client_t *client, snd_seq_port_info_t * info)
{
	client_t *cptr;
	client_port_t *port;
	snd_seq_port_info_t _info;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&_info, info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;
	cptr = snd_seq_client_use_ptr(_info.client);
	if (!cptr)
		return -ENXIO;

	port = snd_seq_port_use_ptr(cptr, _info.port);
	if (!port) {
		snd_seq_client_unlock(cptr);
		return -ENOENT;			/* don't change */
	}

	/* get port info */
	snd_seq_get_port_info(port, &_info);
	snd_seq_ports_unlock(cptr);
	snd_seq_client_unlock(cptr);

	if (copy_to_user(info, &_info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;
	return 0;
}


/* 
 * SET_PORT_INFO ioctl() (only ports on this/own client) 
 */
static int snd_seq_ioctl_set_port_info(client_t * client, snd_seq_port_info_t * info)
{
	client_port_t *port;
	snd_seq_port_info_t _info;

	if (!client || !info)
		return -EINVAL;

	if (copy_from_user(&_info, info, sizeof(snd_seq_port_info_t)))
		return -EFAULT;

	if (_info.client != client->number) /* only set our own ports ! */
		return -EPERM;
	port = snd_seq_port_use_ptr(client, _info.port);
	if (port) {
		snd_seq_set_port_info(port, &_info);
		snd_seq_ports_unlock(client);
	}
	return 0;
}


/* 
 * add to port's subscription list IOCTL interface 
 */
static int snd_seq_ioctl_subscribe_port(client_t * client, snd_seq_port_subscribe_t * _subs)
{
	int result = -ENXIO;
	client_t *receiver = NULL, *sender = NULL;
	client_port_t *sport = NULL, *dport = NULL;
	snd_seq_port_subscribe_t subs;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&subs, _subs, sizeof(snd_seq_port_subscribe_t)))
		return -EFAULT;

	if (subs.sender.queue != subs.dest.queue)
		return -EINVAL;

	receiver = snd_seq_client_use_ptr(subs.dest.client);
	if (!receiver)
		goto __end;
	sender = snd_seq_client_use_ptr(subs.sender.client);
	if (!sender)
		goto __end;

	sport = snd_seq_port_use_ptr(sender, subs.sender.port);
	if (sport == NULL)
		goto __end;
	dport = snd_seq_port_use_ptr(receiver, subs.dest.port);
	if (dport == NULL)
		goto __end;

	if (sport == dport || subs.sender.client == subs.dest.client) {
		result = -EINVAL;
		goto __end;
	}
	
	if (client->number == subs.dest.client) {

		if (!(sport->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION)) {
			result = -EINVAL;
			goto __end;
		}
		
		if ((result = snd_seq_port_subscribe(sport, &subs)) < 0)
			goto __end;

#ifdef TRACK_SUBSCRIBE
		snd_printd("Add subscription; %d:%d will send to %d:%d via queue %d\n",
			   subs.sender.client, subs.sender.port,
		  	   subs.dest.client, subs.dest.port, subs.dest.queue);
#endif

		result = snd_seq_port_add_subscriber(&sport->input, &subs.dest, subs.exclusive, subs.realtime);
		if (result < 0) {
			snd_seq_port_unsubscribe(sport, &subs);
		} else {
			result = snd_seq_port_add_subscriber(&dport->itrack, &subs.sender, 0, 0);
			if (result < 0) {
				snd_seq_port_remove_subscriber(&sport->output, &subs.dest);
				snd_seq_port_unsubscribe(sport, &subs);
			}
		}

	} else if (client->number == subs.sender.client) {

		if (!(dport->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION)) {
			result = -EINVAL;
			goto __end;
		}
		
		if ((result = snd_seq_port_use(dport, &subs)) < 0)
			goto __end;
	
#ifdef TRACK_SUBSCRIBE
		snd_printd("Add use; %d:%d will use %d:%d via queue %i\n",
			   subs.sender.client, subs.sender.port,
		  	   subs.dest.client, subs.dest.port, subs.dest.queue);
#endif
	
		result = snd_seq_port_add_subscriber(&dport->output, &subs.sender, subs.exclusive, subs.realtime);
		if (result < 0) {
			snd_seq_port_unuse(dport, &subs);
		} else {
			result = snd_seq_port_add_subscriber(&sport->otrack, &subs.dest, 0, 0);
			if (result < 0) {
				snd_seq_port_remove_subscriber(&sport->output, &subs.dest);
				snd_seq_port_unuse(dport, &subs);
			}
		}
	
	} else {

		result = -EPERM;

	}

      __end:
      	if (sport)
		snd_seq_ports_unlock(sender);
	if (dport)
		snd_seq_ports_unlock(receiver);
	if (sender)
		snd_seq_client_unlock(sender);
	if (receiver)
		snd_seq_client_unlock(receiver);
	return result;
}


/* 
 * remove from port's subscription list 
 */
static int snd_seq_ioctl_unsubscribe_port(client_t * client, snd_seq_port_subscribe_t * _subs)
{
	int result = -ENXIO;
	client_t *receiver = NULL, *sender = NULL;
	client_port_t *sport = NULL, *dport = NULL;
	snd_seq_port_subscribe_t subs;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&subs, _subs, sizeof(snd_seq_port_subscribe_t)))
		return -EFAULT;

	receiver = snd_seq_client_use_ptr(subs.dest.client);
	if (!receiver)
		goto __end;

	sender = snd_seq_client_use_ptr(subs.sender.client);
	if (!sender)
		goto __end;

	sport = snd_seq_port_use_ptr(sender, subs.sender.port);
	if (sport == NULL)
		goto __end;
	dport = snd_seq_port_use_ptr(receiver, subs.dest.port);
	if (dport == NULL)
		goto __end;

	if (sport == dport) {
		result = -EINVAL;
		goto __end;
	}
	if (!(dport->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION)) {
		result = -EINVAL;
		goto __end;
	}
	
	if (client->number == subs.dest.client) {

#ifdef TRACK_SUBSCRIBE
		snd_printd("Remove subscription; %d:%d will send to %d:%d via queue %d\n",
			   subs.sender.client, subs.sender.port,
		  	   subs.dest.client, subs.dest.port, subs.dest.queue);
#endif

		snd_seq_port_remove_subscriber(&sport->input, &subs.dest);
		snd_seq_port_remove_subscriber(&dport->itrack, &subs.sender);
		snd_seq_port_unsubscribe(sport, &subs);
		result = 0;

	} else if (client->number == subs.sender.client) {

#ifdef TRACK_SUBSCRIBE
		snd_printd("Remove use; %d:%d will use %d:%d via queue %i\n",
			   subs.sender.client, subs.sender.port,
		  	   subs.dest.client, subs.dest.port, subs.dest.queue);
#endif
	
		snd_seq_port_remove_subscriber(&dport->output, &subs.sender);
		snd_seq_port_remove_subscriber(&sport->otrack, &subs.dest);
		snd_seq_port_unuse(dport, &subs);
		result = 0;
		
	} else {

		result = -EPERM;

	}

      __end:
      	if (sport)
		snd_seq_ports_unlock(sender);
	if (dport)
		snd_seq_ports_unlock(receiver);
	if (sender)
		snd_seq_client_unlock(sender);
	if (receiver)
		snd_seq_client_unlock(receiver);
	return result;
}


/* GET_QUEUE_STATUS ioctl() */
static int snd_seq_ioctl_get_queue_status(client_t * client, snd_seq_queue_status_t * _status)
{
	snd_seq_queue_status_t status;
	queue_t *queue;
	timer_t *tmr;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&status, _status, sizeof(snd_seq_queue_status_t)))
		return -EFAULT;

	queue = queueptr(status.queue);
	if (!queue)
		return -EINVAL;
	memset(&status, 0, sizeof(status));
	status.queue = queue->queue;
	
	tmr = queue->timer;
	status.events = queue->tickq->cells + queue->timeq->cells;

	status.time = snd_seq_timer_get_cur_time(tmr);
	status.tick = snd_seq_timer_get_cur_tick(tmr);

	status.running = tmr->running;

	status.flags = 0;

	if (copy_to_user(_status, &status, sizeof(snd_seq_queue_status_t)))
		return -EFAULT;
	return 0;
}


/* GET_QUEUE_TEMPO ioctl() */
static int snd_seq_ioctl_get_queue_tempo(client_t * client, snd_seq_queue_tempo_t * _tempo)
{
	snd_seq_queue_tempo_t tempo;
	queue_t *queue;
	timer_t *tmr;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&tempo, _tempo, sizeof(snd_seq_queue_tempo_t)))
		return -EFAULT;

	queue = queueptr(tempo.queue);
	if (!queue)
		return -EINVAL;
	memset(&tempo, 0, sizeof(tempo));
	tempo.queue = queue->queue;
	
	tmr = queue->timer;

	tempo.tempo = tmr->tempo;
	tempo.ppq = tmr->ppq;

	if (copy_to_user(_tempo, &tempo, sizeof(snd_seq_queue_tempo_t)))
		return -EFAULT;
	return 0;
}


/* SET_QUEUE_TEMPO ioctl() */
static int snd_seq_ioctl_set_queue_tempo(client_t * client, snd_seq_queue_tempo_t * _tempo)
{
	int result;
	snd_seq_queue_tempo_t tempo;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&tempo, _tempo, sizeof(snd_seq_queue_tempo_t)))
		return -EFAULT;

	if (snd_seq_queue_check_access(tempo.queue, client->number)) {
		result = snd_seq_queue_timer_set_tempo(tempo.queue, client->number, tempo.tempo);
		if (result < 0)
			return result;
		result = snd_seq_queue_timer_set_ppq(tempo.queue, client->number, tempo.ppq);
		if (result < 0)
			return result;
	} else {
		return -EPERM;
	}	

	return 0;
}


/* GET_QUEUE_OWNER ioctl() */
static int snd_seq_ioctl_get_queue_owner(client_t * client, snd_seq_queue_owner_t * _owner)
{
	snd_seq_queue_owner_t owner;
	queue_t *queue;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&owner, _owner, sizeof(snd_seq_queue_owner_t)))
		return -EFAULT;

	queue = queueptr(owner.queue);
	if (!queue)
		return -EINVAL;
	memset(&owner, 0, sizeof(owner));
	owner.queue = queue->queue;
	
	owner.owner = queue->owner;
	owner.locked = queue->locked;

	if (copy_to_user(_owner, &owner, sizeof(snd_seq_queue_owner_t)))
		return -EFAULT;
	return 0;
}


/* SET_QUEUE_OWNER ioctl() */
static int snd_seq_ioctl_set_queue_owner(client_t * client, snd_seq_queue_owner_t * _owner)
{
	snd_seq_queue_owner_t owner;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&owner, _owner, sizeof(snd_seq_queue_owner_t)))
		return -EFAULT;

	if (snd_seq_queue_check_access(owner.queue, client->number)) {
		if (owner.locked)
			snd_seq_queue_use(owner.queue, client->number, 1);
		if (snd_seq_queue_set_locked(owner.queue, client->number, owner.locked) < 0)
			return -EPERM;
		if (snd_seq_queue_set_owner(owner.queue, owner.owner) < 0)
			return -EPERM;
	} else {
		return -EPERM;
	}	

	return 0;
}


/* GET_QUEUE_TIMER ioctl() */
static int snd_seq_ioctl_get_queue_timer(client_t * client, snd_seq_queue_timer_t * _timer)
{
	snd_seq_queue_timer_t timer;
	queue_t *queue;
	timer_t *tmr;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&timer, _timer, sizeof(snd_seq_queue_timer_t)))
		return -EFAULT;

	queue = queueptr(timer.queue);
	if (!queue)
		return -EINVAL;
	tmr = queue->timer;

	memset(&timer, 0, sizeof(timer));
	timer.queue = queue->queue;

	down(&queue->use_mutex);
	timer.type = tmr->type;
	timer.slave = tmr->slave_type;
	timer.number = tmr->timer_no;
	timer.resolution = tmr->resolution;
	timer.midi_client = tmr->midi_client;
	timer.midi_port = tmr->midi_port;
	timer.sync_tick_resolution = tmr->sync_tick_resolution;
	timer.sync_real_resolution = tmr->sync_real_resolution;
	up(&queue->use_mutex);
	
	if (copy_to_user(_timer, &timer, sizeof(snd_seq_queue_timer_t)))
		return -EFAULT;
	return -EINVAL;
}


/* SET_QUEUE_TIMER ioctl() */
static int snd_seq_ioctl_set_queue_timer(client_t * client, snd_seq_queue_timer_t * _timer)
{
	int result = 0;
	snd_seq_queue_timer_t timer;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&timer, _timer, sizeof(snd_seq_queue_timer_t)))
		return -EFAULT;

	if (timer.type < 0 || timer.type >= SND_SEQ_TIMER_MIDI_TICK)
		return -EINVAL;
	if (timer.resolution < 0)
		return -EINVAL;

	if (snd_seq_queue_check_access(timer.queue, client->number)) {
		queue_t *q;
		timer_t *tmr;

		q = queueptr(timer.queue);
		if (q == NULL)
			return -EIO;
		tmr = q->timer;
		down(&q->use_mutex);
		snd_seq_queue_timer_close(timer.queue);
		tmr->type = timer.type;
		tmr->slave_type = timer.slave;
		tmr->timer_no = timer.number;
		tmr->resolution = timer.resolution;
		tmr->midi_client = timer.midi_client;
		tmr->midi_port = timer.midi_port;
		tmr->sync_tick_resolution = timer.sync_tick_resolution;
		tmr->sync_real_resolution = timer.sync_real_resolution;
		result = snd_seq_queue_timer_open(timer.queue);
		up(&q->use_mutex);
	} else {
		return -EPERM;
	}	

	return result;
}


/* GET_QUEUE_SYNC ioctl() */
static int snd_seq_ioctl_get_queue_sync(client_t * client, snd_seq_queue_sync_t * _sync)
{
	snd_seq_queue_sync_t sync;
	queue_t *queue;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&sync, _sync, sizeof(snd_seq_queue_sync_t)))
		return -EFAULT;

	queue = queueptr(sync.queue);
	if (!queue)
		return -EINVAL;
	memset(&sync, 0, sizeof(sync));
	sync.queue = queue->queue;
	
	if (copy_to_user(_sync, &sync, sizeof(snd_seq_queue_sync_t)))
		return -EFAULT;
	return -EINVAL;
}


/* SET_QUEUE_SYNC ioctl() */
static int snd_seq_ioctl_set_queue_sync(client_t * client, snd_seq_queue_sync_t * _sync)
{
	snd_seq_queue_sync_t sync;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&sync, _sync, sizeof(snd_seq_queue_sync_t)))
		return -EFAULT;

	if (snd_seq_queue_check_access(sync.queue, client->number)) {
		/* ... code ... */
		return -EINVAL;
	} else {
		return -EPERM;
	}	

	return 0;
}


/* GET_QUEUE_CLIENT ioctl() */
static int snd_seq_ioctl_get_queue_client(client_t * client, snd_seq_queue_client_t * _info)
{
	snd_seq_queue_client_t info;
	queue_t *queue;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(snd_seq_queue_client_t)))
		return -EFAULT;

	queue = queueptr(info.queue);
	if (!queue)
		return -EINVAL;
	info.client = client->number;
	info.used = test_bit(client->number, &queue->clients_bitmap) ? 1 : 0;

	if (copy_to_user(_info, &info, sizeof(snd_seq_queue_client_t)))
		return -EFAULT;
	return 0;
}


/* SET_QUEUE_CLIENT ioctl() */
static int snd_seq_ioctl_set_queue_client(client_t * client, snd_seq_queue_client_t * _info)
{
	queue_t *queue;
	snd_seq_queue_client_t info;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(snd_seq_queue_client_t)))
		return -EFAULT;

	queue = queueptr(info.queue);
	if (!queue)
		return -EINVAL;

	if (info.used >= 0) {
		snd_seq_queue_use(info.queue, client->number, info.used);
	}

	return snd_seq_ioctl_get_queue_client(client, _info);
}


/* GET_CLIENT_POOL ioctl() */
static int snd_seq_ioctl_get_client_pool(client_t * client, snd_seq_client_pool_t * _info)
{
	snd_seq_client_pool_t info;
	client_t *cptr;

	if (copy_from_user(&info, _info, sizeof(info)))
		return -EFAULT;

	cptr = snd_seq_client_use_ptr(info.client);
	if (!cptr)
		return -ENOENT;
	memset(&info, 0, sizeof(info));
	info.output_pool = cptr->pool_size;
	info.output_room = cptr->pool_room;
	info.output_free = info.output_pool;
	if (cptr->pool)
		info.output_free = snd_seq_unused_cells(cptr->pool);
	if (cptr->type == USER_CLIENT) {
		info.input_pool = cptr->data.user.fifo_pool_size;
		info.input_free = info.input_pool;
		if (cptr->data.user.fifo)
			info.input_free = snd_seq_unused_cells(cptr->data.user.fifo->pool);
	} else {
		info.input_pool = 0;
		info.input_free = 0;
	}
	snd_seq_client_unlock(cptr);
	
	if (copy_to_user(_info, &info, sizeof(info)))
		return -EFAULT;
	return 0;
}

/* SET_CLIENT_POOL ioctl() */
static int snd_seq_ioctl_set_client_pool(client_t * client, snd_seq_client_pool_t * _info)
{
	snd_seq_client_pool_t info;
	int rc;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(info)))
		return -EFAULT;

	if (client->number != info.client)
		return -EINVAL; /* can't change other clients */

	if (info.output_pool >= 1 && info.output_pool <= SND_SEQ_MAX_EVENTS &&
	    info.output_pool != client->pool_size) {
		if (snd_seq_write_pool_allocated(client)) {
			/* remove all existing cells */
			snd_seq_queue_client_leave_cells(client->number);
			snd_seq_pool_done(client->pool);
		}
		client->pool_size = info.output_pool;
		rc = snd_seq_pool_init(client->pool, info.output_pool);
		if (rc < 0)
			return rc;
		client->pool_room = (info.output_pool + 1) / 2;
	}
	if (client->type == USER_CLIENT && client->data.user.fifo != NULL &&
	    info.input_pool >= 1 &&
	    info.input_pool <= SND_SEQ_MAX_CLIENT_EVENTS &&
	    info.input_pool != client->data.user.fifo_pool_size) {
		/* change pool size */
		rc = snd_seq_fifo_resize(client->data.user.fifo, info.input_pool);
		if (rc < 0)
			return rc;
		client->data.user.fifo_pool_size = info.input_pool;
	}
	if (info.output_room >= 1 &&
	    info.output_room <= client->pool_size) {
		client->pool_room  = info.output_room;
	}

	return snd_seq_ioctl_get_client_pool(client, _info);
}


/* RESET_POOL ioctl() */
static int snd_seq_ioctl_reset_pool(client_t * client, snd_seq_reset_pool_t * _info)
{
	snd_seq_reset_pool_t info;

	if (!client)
		return -EINVAL;

	if (copy_from_user(&info, _info, sizeof(info)))
		return -EFAULT;

	if (info.reset_output && client->pool_size > 0) {
		switch (info.reset_output) {
		case SND_SEQ_RESET_POOL_ALL:
			snd_seq_queue_client_leave_cells(client->number);
			break;
		default:
			/* FIXME: not implemented yet */
			return -EINVAL;
		}
	}

	if (info.reset_input &&
	    client->type == USER_CLIENT && client->data.user.fifo != NULL &&
	    client->data.user.fifo_pool_size > 0) {
		switch (info.reset_input) {
		case SND_SEQ_RESET_POOL_ALL:
			snd_seq_fifo_clear(client->data.user.fifo);
			break;
		default:
			/* FIXME: not implemented yet */
			return -EINVAL;
		}
	}

	return 0;
}


static int snd_seq_do_ioctl(client_t *client, unsigned int cmd, unsigned long arg)
{
	switch (cmd) {

		case SND_SEQ_IOCTL_PVERSION:
			/* return sequencer version number */
			return snd_ioctl_out((long *) arg, SND_SEQ_VERSION);

		case SND_SEQ_IOCTL_CLIENT_ID:
			/* return the id of this client */
			return snd_ioctl_out((long *) arg, client->number);

		case SND_SEQ_IOCTL_SYSTEM_INFO:
			/* return system information */
			return snd_seq_ioctl_system_info(client, (snd_seq_system_info_t *) arg);

		case SND_SEQ_IOCTL_GET_CLIENT_INFO:
			/* return info on specified client */
			return snd_seq_ioctl_get_client_info(client, (snd_seq_client_info_t *) arg);

		case SND_SEQ_IOCTL_SET_CLIENT_INFO:
			/* set info on specified client */
			return snd_seq_ioctl_set_client_info(client, (snd_seq_client_info_t *) arg);

		case SND_SEQ_IOCTL_CREATE_PORT:
			/* create a port for this client */
			return snd_seq_ioctl_create_port(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_DELETE_PORT:
			/* remove a port from this client */
			return snd_seq_ioctl_delete_port(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_GET_PORT_INFO:
			/* get info for specified port */
			return snd_seq_ioctl_get_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_SET_PORT_INFO:
			/* set info for specified port in this client */
			return snd_seq_ioctl_set_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_SUBSCRIBE_PORT:
			/* add to port's subscription list */
			return snd_seq_ioctl_subscribe_port(client, (snd_seq_port_subscribe_t *) arg);

		case SND_SEQ_IOCTL_UNSUBSCRIBE_PORT:
			/* remove from port's subscription list */
			return snd_seq_ioctl_unsubscribe_port(client, (snd_seq_port_subscribe_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_STATUS:
			/* get the status for the specified queue */
			return snd_seq_ioctl_get_queue_status(client, (snd_seq_queue_status_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_TEMPO:
			/* get the tempo for the specified queue */
			return snd_seq_ioctl_get_queue_tempo(client, (snd_seq_queue_tempo_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_TEMPO:
			/* set a tempo for the specified queue */
			return snd_seq_ioctl_set_queue_tempo(client, (snd_seq_queue_tempo_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_OWNER:
			/* get the owner for the specified queue */
			return snd_seq_ioctl_get_queue_owner(client, (snd_seq_queue_owner_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_OWNER:
			/* set an owner for the specified queue */
			return snd_seq_ioctl_set_queue_owner(client, (snd_seq_queue_owner_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_TIMER:
			/* get the timer for the specified queue */
			return snd_seq_ioctl_get_queue_timer(client, (snd_seq_queue_timer_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_TIMER:
			/* set an timer for the specified queue */
			return snd_seq_ioctl_set_queue_timer(client, (snd_seq_queue_timer_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_SYNC:
			/* get the sync source for the specified queue */
			return snd_seq_ioctl_get_queue_sync(client, (snd_seq_queue_sync_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_SYNC:
			/* set an sync source for the specified queue */
			return snd_seq_ioctl_set_queue_sync(client, (snd_seq_queue_sync_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_CLIENT:
			/* get client specific info for specified queue */
			return snd_seq_ioctl_get_queue_client(client, (snd_seq_queue_client_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_CLIENT:
			/* set client specfic info for specified queue */
			return snd_seq_ioctl_set_queue_client(client, (snd_seq_queue_client_t *) arg);

		case SND_SEQ_IOCTL_GET_CLIENT_POOL:
			return snd_seq_ioctl_get_client_pool(client, (snd_seq_client_pool_t *) arg);

		case SND_SEQ_IOCTL_SET_CLIENT_POOL:
			return snd_seq_ioctl_set_client_pool(client, (snd_seq_client_pool_t *) arg);

		case SND_SEQ_IOCTL_RESET_POOL:
			return snd_seq_ioctl_reset_pool(client, (snd_seq_reset_pool_t *) arg);

		default:
			snd_printd("seq unknown ioctl() 0x%x (type='%c', number=0x%2x)\n",
				   cmd, _IOC_TYPE(cmd), _IOC_NR(cmd));
	}
	return -ENXIO;
}


static int snd_seq_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	client_t *client = (client_t *) file->private_data;

	if (!client)
		return -ENXIO;
		
	return snd_seq_do_ioctl(client, cmd, arg);
}


/* -------------------------------------------------------- */


/* exported to kernel modules */
int snd_seq_create_kernel_client(snd_card_t *card, int client_index, snd_seq_client_callback_t * callback)
{
	client_t *client;

	if (!callback)
		return -EINVAL;
	if (card && client_index > 7)
		return -EINVAL;
	if (!card && client_index > 63)
		return -EINVAL;
	if (card)
		client_index += 64 + (card->number << 3);

	down(&register_mutex);
	/* empty write queue as default */
	client = seq_create_client1(client_index, 0);
	if (!client) {
		up(&register_mutex);
		return -EBUSY;	/* failure code */
	}
	usage_alloc(&client_usage, 1);

	client->accept_input = callback->allow_output;
	client->accept_output = callback->allow_input;
		
	/* fill client data */
	client->data.kernel.card = card;
	client->data.kernel.private_data = callback->private_data;
	sprintf(client->name, "Client-%d", client->number);

	client->type = KERNEL_CLIENT;
	up(&register_mutex);

	/* make others aware this new client */
	snd_seq_system_client_ev_client_start(client->number);
	
	/* return client number to caller */
	return client->number;
}

/* exported to kernel modules */
int snd_seq_delete_kernel_client(int client)
{
	client_t *ptr;

	ptr = clientptr(client);
	if (!ptr)
		return -EINVAL;

	seq_free_client(ptr);
	snd_kfree(ptr);
	return 0;
}


/* skeleton to enqueue event, called from snd_seq_kernel_client_enqueue
 * and snd_seq_kernel_client_enqueue_blocking
 */
static int kernel_client_enqueue(int client, snd_seq_event_t *ev, struct file *file, int blocking, int atomic)
{
	client_t *cptr;
	int rc;

	cptr = snd_seq_client_use_ptr(client);
	if (!cptr)
		return -EINVAL;

	/* fill in client number */
	ev->source.queue = SND_SEQ_ADDRESS_UNKNOWN;
	ev->source.client = client;

	/* direct event processing */
	if ((ev->flags & SND_SEQ_DEST_MASK) == SND_SEQ_DEST_DIRECT ||
	    ev->dest.queue == SND_SEQ_ADDRESS_SUBSCRIBERS ||
	    ev->dest.queue == SND_SEQ_ADDRESS_BROADCAST) {
		snd_seq_deliver_event(cptr, ev, atomic);
		snd_seq_client_unlock(cptr);
		return 0;
	}

	/* we got one event, copy it into our own data structure and */
	/* feed it to the prioQ */
	if (ev->dest.queue >= SND_SEQ_MAX_QUEUES) {
		snd_seq_client_unlock(cptr);
		return -EINVAL;
	}

	rc = snd_seq_client_enqueue_event(cptr, ev, file, blocking, atomic);

	snd_seq_client_unlock(cptr);
	return rc;
}

/* exported, called by kernel clients to enqueue events (w/o blocking) */
int snd_seq_kernel_client_enqueue(int client, snd_seq_event_t * ev, int atomic)
{
	return kernel_client_enqueue(client, ev, NULL, 0, atomic);
}

/* exported, called by kernel clients to enqueue events (with blocking) */
int snd_seq_kernel_client_enqueue_blocking(int client, snd_seq_event_t * ev, struct file *file, int atomic)
{
	return kernel_client_enqueue(client, ev, file, 1, atomic);
}


/* 
 * exported, called by kernel clients to dispatch events directly to other
 * clients, bypassing the queues. No processing of time stamps is done.
 */
int snd_seq_kernel_client_dispatch(int client, snd_seq_event_t * ev, int atomic)
{
	client_t *cptr;

	cptr = snd_seq_client_use_ptr(client);
	if (!cptr)
		return -EINVAL;

	/* fill in client number */
	ev->source.queue = SND_SEQ_ADDRESS_UNKNOWN;
	ev->source.client = client;

	snd_seq_deliver_event(cptr, ev, atomic);
	snd_seq_client_unlock(cptr);
	return 0;	/* success */
}


/*
 * exported, called by kernel clients to perform same functions as with
 * userland ioctl() 
 */
int snd_seq_kernel_client_ctl(int clientid, unsigned int cmd, void *arg)
{
	client_t *client;
	mm_segment_t fs;
	int result;

	client = clientptr(clientid);
	if (!client)
		return -ENXIO;
	fs = snd_enter_user();
	result = snd_seq_do_ioctl(client, cmd, (unsigned long)arg);
	snd_leave_user(fs);
	return result;
}


/* exported (for OSS emulator) */
int snd_seq_kernel_client_write_poll(int clientid, struct file *file, poll_table *wait)
{
	client_t *client;

	client = clientptr(clientid);
	if (!client)
		return -ENXIO;

	if (snd_seq_write_pool_allocated(client)) {
		poll_wait(file, &client->output_sleep, wait);
		if (snd_seq_output_ok(client))
			return 1;
	}

	return 0;
}

/*---------------------------------------------------------------------------*/

/*
 *  /proc interface
 */
static void snd_seq_info_dump_subscribers(snd_info_buffer_t * buffer, subscribers_group_t * group)
{
	subscribers_t *s;

	snd_seq_subscribers_lock(group);
	for (s = group->list; s; s = s->next) {
		snd_iprintf(buffer, "%d:%d:%d",
				s->addr.queue,
				s->addr.client,
				s->addr.port);
		if (s->next)
			snd_iprintf(buffer, ", ");
	}
	snd_seq_subscribers_unlock(group);
}

static void snd_seq_info_dump_ports(snd_info_buffer_t * buffer, client_port_t * ports)
{
	client_port_t *p = ports;

	while (p) {
#if SHOW_CAPABILITY
		snd_iprintf(buffer, "  Port %3d : \"%s\" (%s,%s,%s,%s)\n",
			    p->port, p->name,
		(p->capability & SND_SEQ_PORT_CAP_MIDI_IN ? "MIDI_in" : ""),
			    (p->capability & SND_SEQ_PORT_CAP_MIDI_OUT ? "MIDI_out" : ""),
		(p->capability & SND_SEQ_PORT_CAP_SYNC_IN ? "Sync_in" : ""),
			    (p->capability & SND_SEQ_PORT_CAP_SYNC_OUT ? "Sync_out" : ""),
			    (p->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION ? "Subscription" : ""));
#else
		snd_iprintf(buffer, "  Port %3d : \"%s\"\n",
			    p->port, p->name);
#endif
		snd_iprintf(buffer, "    Subscribers: ");
		snd_seq_info_dump_subscribers(buffer, &p->input);
		snd_iprintf(buffer, "\n    Subscribers tracking: ");
		snd_seq_info_dump_subscribers(buffer, &p->itrack);
		snd_iprintf(buffer, "\n    Users: ");
		snd_seq_info_dump_subscribers(buffer, &p->output);
		snd_iprintf(buffer, "\n    Users tracking: ");
		snd_seq_info_dump_subscribers(buffer, &p->otrack);
		snd_iprintf(buffer, "\n");

		p = p->next;
	}
}


/* exported to seq_info.c */
void snd_seq_info_clients_read(snd_info_buffer_t * buffer, void *private_data)
{
	extern void snd_seq_info_pool(snd_info_buffer_t * buffer, pool_t * pool, char *space);
	int c;
	client_t *client;

	snd_iprintf(buffer, "Client info\n");
	snd_iprintf(buffer, "  cur  clients : %d\n", client_usage.cur);
	snd_iprintf(buffer, "  peak clients : %d\n", client_usage.peak);
	snd_iprintf(buffer, "  max  clients : %d\n", SND_SEQ_MAX_CLIENTS);
	snd_iprintf(buffer, "\n");

	/* list the client table */
	for (c = 0; c < SND_SEQ_MAX_CLIENTS; c++) {
		client = snd_seq_client_use_ptr(c);
		if (!client)
			continue;
		snd_seq_ports_lock(client);
		switch (client->type) {
			case NO_CLIENT:
				/*snd_iprintf(buffer, "Client %3d : Free\n", c); */
				break;

			case USER_CLIENT:
				snd_iprintf(buffer, "Client %3d : \"%s\" [User]\n",
					    c, client->name);
				snd_seq_info_dump_ports(buffer, client->ports);
				if (snd_seq_write_pool_allocated(client)) {
					snd_iprintf(buffer, "  Output pool :\n");
					snd_seq_info_pool(buffer, client->pool, "    ");
				}
				if (client->data.user.fifo->pool) {
					snd_iprintf(buffer, "  Input pool :\n");
					snd_seq_info_pool(buffer, client->data.user.fifo->pool, "    ");
				}
				break;

			case KERNEL_CLIENT:
				snd_iprintf(buffer, "Client %3d : \"%s\" [Kernel]\n",
					    c, client->name);
				snd_seq_info_dump_ports(buffer, client->ports);
				if (snd_seq_write_pool_allocated(client)) {
					snd_iprintf(buffer, "  Output pool :\n");
					snd_seq_info_pool(buffer, client->pool, "    ");
				}
				break;
		}
		snd_seq_ports_unlock(client);
		snd_seq_client_unlock(client);
	}
}


/*---------------------------------------------------------------------------*/


/*
 *  REGISTRATION PART
 */
static snd_minor_t snd_seq_reg =
{
	"sequencer",		/* comment */
	NULL,			/* reserved */
	
	NULL,			/* unregister */
	NULL,			/* lseek */
	snd_seq_read,		/* read */
	snd_seq_write,		/* write */
	snd_seq_open,		/* open */
	snd_seq_release,	/* release */
	snd_seq_poll,		/* poll */
	snd_seq_ioctl,		/* ioctl */
	NULL			/* mmap */
};


/* 
 * register sequencer device 
 */
int snd_sequencer_device_init(void)
{
	int err;

	down(&register_mutex);

	if ((err = snd_register_device(SND_DEVICE_TYPE_SEQUENCER, NULL, 0, &snd_seq_reg, "seq")) < 0) {
		up(&register_mutex);
		return err;
	}
	
	up(&register_mutex);

	return 0;
}



/* 
 * unregister sequencer device 
 */
void snd_sequencer_device_done(void)
{
	snd_unregister_device(SND_DEVICE_TYPE_SEQUENCER, NULL, 0);
}
