
/*
    Detection routines for OPTi 82C9xx chips/cards.
    Copyright (C) 1998-99 by Massimo Piccioni <piccio@caronte.csr.unibo.it>
    Copyright (C) 1999 by Lorenzo Piccioni

    Part of this code was developed at the Italian Ministry of Air Defence,
    Sixth Division (oh, che pace ...), Rome.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "../include/driver.h"

#ifndef ESUCCESS
#define ESUCCESS	0
#endif	/* ESUCCESS */

#define opti9xx_printd( args... )	snd_printd( "opti9xx: " ##args )

enum chipsets {
	c928 = 0,	c929 = 1,	c924 = 2,	c925 = 3,
	c930 = 4,	c931 = 5,	c933 = 6,	unsupported = c933
};

static struct {
	char *name;
	unsigned char pwd;
	int mc_base;
	char mc_regs;
	char mc_size;
	int pwd_reg;
} chip_info[] = {
	{ "82C928",		0xe2,	0xf8d,	6,	6,	2 },
	{ "82C929",		0xe3,	0xf8d,	6,	6,	2 },
	{ "82C924",		0xe5,	0xf8d,	12,	9,	2 },
	{ "82C925",		0xe5,	0xf8d,	14,	9,	2 },
	{ "82C930",		0xe4,	0xf8f,	12,	1,	0 },
	{ "82C931",		0xe4,	0xf8d,	26,	1,	0 },
	{ "unsupported",	0x00,	0x000,	0,	0,	0 }
};

#define OPTi9XX_CHIP		chip_info[opti9xx_chip].name
#define OPTi9XX_MC_BASE		chip_info[opti9xx_chip].mc_base
#define OPTi9XX_MC_REGS		chip_info[opti9xx_chip].mc_regs
#define OPTi9XX_MC_SIZE		chip_info[opti9xx_chip].mc_size
#define OPTi9XX_PASSWD		chip_info[opti9xx_chip].pwd

#define OPTi9XX_MC(x)		x
#define OPTi92X_MC_REG(x)	OPTi9XX_MC_BASE + OPTi9XX_MC(x - 1)
#define OPTi9XX_PWD_REG		OPTi9XX_MC_BASE + \
				chip_info[opti9xx_chip].pwd_reg

#define OPTi924_MC_INDEX_IREG	OPTi92X_MC_REG(8)
#define OPTi924_MC_DATA_IREG	OPTi92X_MC_REG(9)
#define OPTi924_MC_DREGS	7

#define OPTi93X_MC_INDEX_IREG	opti93x_mc_index
#define OPTi93X_MC_DATA_IREG	OPTi93X_MC_INDEX_IREG + 1
#define OPTi93X_MC_IREGS	2

#define OPTi93X_ENABLE_PWD	0x00
#define OPTi93X_INDEX_BITS	((OPTi93X_MC_INDEX_IREG & (1 << 8)) >> 4)|\
				((OPTi93X_MC_INDEX_IREG & (0xf0)) >> 4)

snd_card_t *opti9xx_card = NULL;
int opti9xx_chip = unsupported;
int opti93x_mc_index = 0xe0e;		/* 0x[ef][0..f]e are valid values */ 

#ifdef CONFIG_ISAPNP
static unsigned int snd_opti9xx_isapnp_table[] = {
	/* c924 */
	c924,
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_DEVICE(0x0924)),    /* device */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x0000)),  /* audio  */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x0001)),  /* game   */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x0002)),  /* mpu401 */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x0003)),  /* cdrom  */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x0004)),  /* modem  */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x0005)),  /* master */
	0,
	/* c931 */
	c931,
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_DEVICE(0x0931)),    /* device */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x9310)),  /* audio  */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x0001)),  /* game   */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0x0002)),  /* mpu401 */
	(ISAPNP_VENDOR('O','P','T')<<16|ISAPNP_FUNCTION(0xffff)),  /* aux0   */
	0,
	0	/* that's all */
};
#endif	/* CONFIG_ISAPNP */


unsigned char snd_opti9xx_read(unsigned int idx)
{
	unsigned long flags;
	unsigned char retval = 0xff;

	if (idx > OPTi9XX_MC_REGS)
		return retval;

	save_flags(flags);
	cli();

	outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
	switch (opti9xx_chip) {
	case c924:
	case c925:
		if (idx > OPTi924_MC_DREGS) {
			outb(idx, OPTi924_MC_INDEX_IREG);
			outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
			retval = inb(OPTi924_MC_DATA_IREG);
			break;
		}
	case c928:
	case c929:
		retval = inb(OPTi92X_MC_REG(idx));
		break;
	case c930:
	case c931:
		outb(idx, OPTi93X_MC_INDEX_IREG);
		outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
		retval = inb(OPTi93X_MC_DATA_IREG);
		break;
	default:	/* uff */
	}

	restore_flags(flags);
	return retval;
}

void snd_opti9xx_write(unsigned int idx, unsigned char value)
{
	unsigned long flags;

	if (idx > OPTi9XX_MC_REGS)
		return;

	save_flags(flags);
	cli();

	outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
	switch (opti9xx_chip) {
	case c924:
	case c925:
		if (idx > OPTi924_MC_DREGS) {
			outb(idx, OPTi924_MC_INDEX_IREG);
			outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
			outb(value, OPTi924_MC_DATA_IREG);
			break;
		}
	case c928:
	case c929:
		outb(value, OPTi92X_MC_REG(idx));
		break;
	case c930:
	case c931:
		outb(idx, OPTi93X_MC_INDEX_IREG);
		outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
		outb(value, OPTi93X_MC_DATA_IREG);
		break;
	default:	/* uff */
	}

	restore_flags(flags);
}


int snd_opti92x_detect_isa(void)
{
	for (opti9xx_chip = c928; opti9xx_chip < c930; opti9xx_chip++) {
		unsigned char value, check;

		if (snd_register_ioport(opti9xx_card,
			OPTi9XX_MC_BASE, OPTi9XX_MC_SIZE,
			"opti92x - mc regs", NULL))
			continue;

		value = snd_opti9xx_read(OPTi9XX_MC(1));
		check = inb(OPTi9XX_MC_BASE + OPTi9XX_MC(1));
		if ((value != 0xff) && (value != check))
			if (value == snd_opti9xx_read(OPTi9XX_MC(1)))
				return ESUCCESS;

		snd_unregister_ioports(opti9xx_card);
	}

	opti9xx_chip = unsupported;
	return -ENODEV;
}

int snd_opti93x_detect_isa(void)
{
	int error = -ENODEV;
	int hb, lb;

	for (hb = 0x00; (hb < 0x02) && error; hb++)
		for (lb = 0x00; (lb < 0x10) && error; lb++) {
			opti93x_mc_index = 0xe0e + (hb << 8) + (lb << 4);

			error = snd_register_ioport(opti9xx_card,
				OPTi93X_MC_INDEX_IREG, OPTi93X_MC_IREGS,
				"opti93x - mc regs", NULL);
		}

	if (error)
		return error;

	/* the 82C931 chip may appear as an 82C930 */
	for (opti9xx_chip = c931; opti9xx_chip >= c930; opti9xx_chip--) {
		unsigned char value, check;
		int i;

		if (snd_register_ioport(opti9xx_card,
			OPTi9XX_MC_BASE, OPTi9XX_MC_SIZE,
			"opti93x - mc regs", NULL))
			continue;

		outb(OPTi9XX_PASSWD, OPTi9XX_PWD_REG);
		outb(OPTi93X_ENABLE_PWD | OPTi93X_INDEX_BITS,
			OPTi9XX_MC_BASE);

		value = snd_opti9xx_read(OPTi9XX_MC(7));
		check = ~value;
		snd_opti9xx_write(OPTi9XX_MC(7), check);

		if (snd_opti9xx_read(OPTi9XX_MC(7)) == check) {
			for (i = 1; i <= OPTi9XX_MC_REGS; i++) {
				value = snd_opti9xx_read(OPTi9XX_MC(i));
				check = snd_opti9xx_read(OPTi9XX_MC(i +
					OPTi9XX_MC_REGS));

				if ((value != 0xff) && (check != 0xff))
					break;
			}

			if (i > OPTi9XX_MC_REGS)
				return ESUCCESS;
		}

		snd_unregister_ioports(opti9xx_card);
		if ((error = snd_register_ioport(opti9xx_card,
			OPTi93X_MC_INDEX_IREG, OPTi93X_MC_IREGS,
			"opti93x - mc regs", NULL)))
			return error;
	}

	opti9xx_chip = unsupported;
	return -ENODEV;
}

int snd_opti9xx_detect_isa(void)
{
	if (snd_opti92x_detect_isa() && snd_opti93x_detect_isa()) {
		opti9xx_printd("cannot find any OPTi based ISA soundcard\n");
		return -ENODEV;
	}

	opti9xx_printd("found an OPTi %s chip (ISA)\n", OPTi9XX_CHIP);
	return ESUCCESS;
}

#ifdef CONFIG_ISAPNP
int snd_opti9xx_detect_pnp(void)
{
	int i;
	unsigned int tmp;

	for (i = 0; (opti9xx_chip = tmp = snd_opti9xx_isapnp_table[i++]);) {
		tmp = snd_opti9xx_isapnp_table[i++];

		if ((isapnp_find_device(tmp >> 16,
			tmp & 0xffff, 0))) {
			opti9xx_printd("found an OPTi %s chip (PnP)\n",
				OPTi9XX_CHIP);
			return ESUCCESS;
		}

		while (snd_opti9xx_isapnp_table[i++]);
	}

	opti9xx_printd("cannot find any OPTi based PnP soundcard\n");
	return -ENODEV;
}
#endif	/* CONFIG_ISAPNP */

int snd_detect_opti9xx(snd_card_t *card, char *rname,
	unsigned int *rtype, unsigned short *rport)
{
	static int found = 0;

	if (found) {
		found = 0;
		snd_unregister_ioports(card);
		return -EBUSY;
	}

	opti9xx_card = card;

#ifdef CONFIG_ISAPNP
	if (snd_opti9xx_detect_pnp())
#endif	/* CONFIG_ISAPNP */
		if (snd_opti9xx_detect_isa()) {
			snd_unregister_ioports(opti9xx_card);
			return -ENODEV;
		}

	sprintf(rname, "OPTi %s based soundcard", OPTi9XX_CHIP);
	*rport = OPTi9XX_MC_BASE;
	*rtype = SND_CARD_TYPE_OPTI9XX;

	found++;
	return ESUCCESS;
}
