/*
    card-azt2320.c - driver for Aztech Systems AZT2320 based soundcards.
    Copyright (C) 1999 by Massimo Piccioni <piccio@caronte.csr.unibo.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
    This driver should provide support for most Aztech AZT2320 based sound
    cards (in PnP mode *only*).
    No docs available for us at Aztech headquarters !!!   Unbelievable ...
    No other help obtained.

    Since I have no idea on how to enable WSS, default Sound Blaster Pro mode
    is used (half-duplex audio).
*/

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/initval.h"
#include "../include/mpu401.h"
#include "../include/opl3.h"
#include "../include/sb.h"

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;		/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;		/* ID for this card */
int snd_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_mpu_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_fm_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_irq[SND_CARDS] = SND_DEFAULT_IRQ;		/* Pnp setup */
int snd_mpu_irq[SND_CARDS] = SND_DEFAULT_IRQ;		/* Pnp setup */
int snd_dma8[SND_CARDS] = SND_DEFAULT_DMA;		/* PnP setup */
int snd_dma8_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */

#ifdef MODULE_PARM
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for azt2320 based soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for azt2320 based soundcard.");
MODULE_PARM(snd_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_port, "Port # for azt2320 driver.");
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port # for azt2320 driver.");
MODULE_PARM(snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_fm_port, "FM port # for azt2320 driver.");
MODULE_PARM(snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_irq, "IRQ # for azt2320 driver.");
MODULE_PARM(snd_mpu_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_irq, "MPU-401 IRQ # for azt2320 driver.");
MODULE_PARM(snd_dma8, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma8, "8-bit DMA # for azt2320 driver.");
MODULE_PARM(snd_dma8_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma8_size, "8-bit DMA size for azt2320 driver.");
#endif	/* MODULE_PARM */

static struct snd_card_azt2320 {
	snd_irq_t *irqptr;
	snd_irq_t *mpuirqptr;
	snd_dma_t *dma8ptr;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;
#ifdef CONFIG_ISAPNP
	struct isapnp_logdev *logdev;
	struct isapnp_logdev *logdevmpu;
#endif	/* CONFIG_ISAPNP */
} *snd_card_azt2320_cards[SND_CARDS] = SND_DEFAULT_PTR;

#ifdef CONFIG_ISAPNP
static unsigned int snd_card_azt2320_pnpids[] = {
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_DEVICE(0x1008),    /* device */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x1008),  /* audio */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x2001),  /* mpu401 */
	0	/* end */
};
#endif	/* CONFIG_ISAPNP */

#define	DRIVER_NAME	"snd-card-azt2320"

#define snd_azt2320_printk(args...)	snd_printk(DRIVER_NAME": " ##args)

static void snd_card_azt2320_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_azt2320_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}

static void snd_card_azt2320_interrupt(int irq, void *dev_id,
							struct pt_regs *regs)
{
	struct snd_card_azt2320 *acard;
	snd_pcm_t *pcm;
	sbdsp_t *codec;

	if (!(acard = (struct snd_card_azt2320 *) dev_id))
		return;

	if (!(pcm = acard->pcm))
		return;

	if (!(codec = (sbdsp_t *) pcm->private_data))
		return;

	if (codec->open8 & SB_OPEN_PCM)
		snd_sb8dsp_interrupt(pcm);
}

static void snd_card_azt2320_mpu_interrupt(int irq, void *dev_id,
							struct pt_regs *regs)
{
	struct snd_card_azt2320 *acard;

	if (!(acard = (struct snd_card_azt2320 *) dev_id))
		return;

	snd_mpu401_uart_interrupt(acard->rmidi);
}

static int snd_card_azt2320_resources(int dev, struct snd_card_azt2320 *acard,
							snd_card_t *card)
{
	int error;
	static int no_alternatives[] = {-1};

	if (snd_register_ioport(card, snd_mpu_port[dev], 2,
			DRIVER_NAME" - MPU-401", NULL) < 0)
		snd_mpu_port[dev] = -1;
	else if (snd_register_interrupt(card,
			DRIVER_NAME" - MPU-401", snd_mpu_irq[dev],
			SND_IRQ_TYPE_ISA, snd_card_azt2320_mpu_interrupt,
			acard, no_alternatives, &acard->mpuirqptr) < 0) {
		snd_unregister_ioports(card);
		snd_mpu_port[dev] = -1;
	}

	if ((error = snd_register_ioport(card, snd_port[dev], 16,
			DRIVER_NAME" - DSP", NULL)) < 0)
		return error;
	if ((error = snd_register_interrupt(card,
			DRIVER_NAME" - DSP", snd_irq[dev],
			SND_IRQ_TYPE_ISA, snd_card_azt2320_interrupt,
			acard, no_alternatives, &acard->irqptr)) < 0)
		return error;
	if ((error = snd_register_dma_channel(card,
			DRIVER_NAME" - DSP", snd_dma8[dev],
			SND_DMA_TYPE_ISA, snd_dma8_size[dev],
			no_alternatives, &acard->dma8ptr)) < 0)
		return error;

	if (snd_register_ioport(card, snd_fm_port[dev], 8,
			DRIVER_NAME" - FM", NULL) < 0)
		snd_fm_port[dev] = -1;

	return 0;
}

#ifdef CONFIG_ISAPNP
static int snd_card_azt2320_isapnp(int dev, struct snd_card_azt2320 *acard)
{
        int i;
        unsigned int tmp;
	struct isapnp_dev *pdev;
	struct isapnp_config cfg;

      	pdev = NULL;
	for (i = 0; (tmp = snd_card_azt2320_pnpids[i]); i += 3)
		if ((pdev = isapnp_find_device(tmp >> 16, tmp & 0xffff, dev)))
			break;

	if (!pdev)
		return -ENODEV;

	tmp = snd_card_azt2320_pnpids[i+1];
	acard->logdev = isapnp_find_logdev(pdev, tmp >> 16, tmp & 0xffff, 0);
	if (!acard->logdev)
		return -ENXIO;

	tmp = snd_card_azt2320_pnpids[i+2];
	acard->logdevmpu = isapnp_find_logdev(pdev, tmp >> 16, tmp & 0xffff, 0);

	if (isapnp_cfg_begin(acard->logdev->dev->csn, acard->logdev->number))
		return -EAGAIN;
	if (isapnp_config_init(&cfg, acard->logdev)) {
		isapnp_cfg_end();
		return -EAGAIN;
	}

	if (snd_port[dev] != SND_AUTO_PORT)
		cfg.port[0] = snd_port[dev];
	if (snd_fm_port[dev] != SND_AUTO_PORT)
		cfg.port[1] = snd_mpu_port[dev];
	if (snd_dma8[dev] != SND_AUTO_DMA)
		cfg.dma[0] = snd_dma8[dev];
	if (snd_irq[dev] != SND_AUTO_IRQ)
		cfg.irq[0] = snd_irq[dev];

	if (isapnp_configure(&cfg)) {
		snd_azt2320_printk("AUDIO isapnp configure failure\n");
		isapnp_cfg_end();
		return -EBUSY;
	}

	snd_port[dev] = cfg.port[0];
	snd_fm_port[dev] = cfg.port[1];
	snd_dma8[dev] = cfg.dma[0];
	snd_irq[dev] = cfg.irq[0];
        isapnp_activate(acard->logdev->number);

	if (!acard->logdevmpu || isapnp_config_init(&cfg, acard->logdevmpu)) {
		isapnp_cfg_end();
		return 0;
	}

	if (snd_mpu_port[dev] != SND_AUTO_PORT)
		cfg.port[0] = snd_mpu_port[dev];
	if (snd_mpu_irq[dev] != SND_AUTO_IRQ)
		cfg.irq[0] = snd_mpu_irq[dev];

	if (isapnp_configure(&cfg)) {
		/* not  fatal error */
		snd_azt2320_printk("MPU-401 isapnp configure failure\n");
		snd_mpu_port[dev] = -1;
		isapnp_deactivate(acard->logdevmpu->number);
		acard->logdevmpu = NULL;
	}
	else {
		snd_mpu_port[dev] = cfg.port[0];
		snd_mpu_irq[dev] = cfg.irq[0];
		isapnp_activate(acard->logdevmpu->number);
	}

	isapnp_cfg_end();
	return 0;
}

static void snd_card_azt2320_deactivate(struct snd_card_azt2320 *acard)
{
	if (!acard->logdev)
		return;

	if (isapnp_cfg_begin(acard->logdev->dev->csn, acard->logdev->number))
		return;
	isapnp_deactivate(acard->logdev->number);
	if (acard->logdevmpu)
		isapnp_deactivate(acard->logdevmpu->number);
	isapnp_cfg_end();
}
#endif	/* CONFIG_ISAPNP */

static int snd_card_azt2320_probe(int dev, struct snd_card_azt2320 *acard)
{
	int error;
	sbdsp_t *codec;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;

	if (!(card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_card_azt2320_use_inc, snd_card_azt2320_use_dec)))
		return -ENOMEM;
	card->type = SND_CARD_TYPE_SB_10;

#ifdef CONFIG_ISAPNP
	if ((error = snd_card_azt2320_isapnp(dev, acard)))
		goto __on_error;
#else
	snd_azt2320_printk("you have to enable PnP support ...\n");
	return -ENOSYS;
#endif	/* CONFIG_ISAPNP */

	if ((error = snd_card_azt2320_resources(dev, acard, card)))
		goto __on_error;

	if (!(pcm = snd_sb8dsp_new_device(card, snd_port[dev],
			acard->irqptr, acard->dma8ptr, SB_HW_AUTO)))
		goto __on_error;

	codec = (sbdsp_t *) pcm->private_data;
	if (!(mixer = snd_sb8dsp_new_mixer(card,
			&codec->mixer, codec->hardware, 0))) {
		snd_pcm_free(pcm);
		goto __on_error;
	}
	if ((error = snd_mixer_register(mixer, 0))) {
		snd_mixer_free(mixer);
		snd_pcm_free(pcm);
		goto __on_error;
	}

	rmidi = NULL;
	if (snd_mpu_port[dev] > 0) {
		if (!(rmidi = snd_mpu401_uart_new_device(card,
				MPU401_HW_AZT2320,
				snd_mpu_port[dev], snd_mpu_irq[dev])))
			snd_azt2320_printk("no MPU-401 device at 0x%x ???\n",
				snd_mpu_port[dev]);
		else if (snd_rawmidi_register(rmidi, 0)) {
			snd_rawmidi_free(rmidi);
			rmidi = NULL;
		}
	}

	synth = NULL;
	if (snd_fm_port[dev] > 0) {
		if (!(synth = snd_opl3_new_device(card,
				snd_fm_port[dev], snd_fm_port[dev] + 2,
				OPL3_HW_AUTO, 0)))
			snd_azt2320_printk("no OPL device at 0x%x-0x%x ?\n",
				snd_fm_port[dev], snd_fm_port[dev] + 2);
		else if (snd_hwdep_register(synth, 0)) {
			snd_hwdep_free(synth);
			synth = NULL;
		}
	}

	if ((error = snd_pcm_register(pcm, 0))) {
		snd_hwdep_unregister(synth);
		snd_rawmidi_unregister(rmidi);
		snd_mixer_unregister(mixer);
		snd_pcm_free(pcm);
		goto __on_error;
	}

	strcpy(card->abbreviation, "AZT2320");
	strcpy(card->shortname, "Aztech AZT2320");
	sprintf(card->longname, "%s soundcard, %s at 0x%x, irq %i, dma %i",
		card->shortname, codec->name, codec->port,
		acard->irqptr->irq, acard->dma8ptr->dma);

	if (!(error = snd_card_register(card))) {
		acard->card = card;
		acard->pcm = pcm;
		acard->mixer = mixer;
		acard->rmidi = rmidi;
		acard->synth = synth;
		return 0;
	}

	snd_hwdep_unregister(synth);
	snd_rawmidi_unregister(rmidi);
	snd_mixer_unregister(mixer);
	snd_pcm_free(pcm);

      __on_error:
	snd_card_free(card);
	return error ? error : -ENXIO;
}

static void snd_card_azt2320_free(int dev)
{
	struct snd_card_azt2320 *acard;

	acard = snd_card_azt2320_cards[dev];
	if (acard) {
		snd_card_azt2320_cards[dev] = NULL;
		snd_card_unregister(acard->card);
		if (acard->synth)
			snd_hwdep_unregister(acard->synth);
		if (acard->rmidi)
			snd_rawmidi_unregister(acard->rmidi);
		if (acard->mixer)
			snd_mixer_unregister(acard->mixer);
		if (acard->pcm)
			snd_pcm_unregister(acard->pcm);
		snd_card_free(acard->card);
#ifdef CONFIG_ISAPNP
		snd_card_azt2320_deactivate(acard);
#endif	/* CONFIG_ISAPNP */
		snd_kfree(acard);
	}
}

int init_module(void)
{
	int cards, dev;
	struct snd_card_azt2320 *acard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		acard = (struct snd_card_azt2320 *) snd_kcalloc(
				sizeof(struct snd_card_azt2320), GFP_KERNEL);
		if (!acard)
			break;

		if (snd_card_azt2320_probe(dev, acard)) {
#ifdef CONFIG_ISAPNP
			snd_card_azt2320_deactivate(acard);
#endif	/* CONFIG_ISAPNP */
			snd_kfree(acard);
			continue;
		}
		snd_card_azt2320_cards[cards++] = acard;
	}

	if (!cards)
		snd_azt2320_printk("no AZT2320 based soundcards found\n");
	return cards ? 0 : -ENODEV;
}

void cleanup_module(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_card_azt2320_free(dev);
}

