%{
/* sieve.l -- sieve lexer
 * Larry Greenfield
 */
/***********************************************************
        Copyright 1999 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of Carnegie Mellon
University not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission.

CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE FOR
ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
******************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h> /* for strdup */
#include "xalloc.h"

#include "tree.h"
#include "sieve-gram.h"

static int start_line;
  
static int tonum(char *c);
static char *fixstr(char *);
static char *mlbuf;
static int mlbufsz, mlcur;
extern int yyerror(char *);
%}

%option yylineno
%option noyywrap

ws		[ \t]+
ident		[a-zA-Z_][a-zA-Z_0-9]*
CRLF		(\r\n|\r|\n)

%state MULTILINE COMMENT

%%
<MULTILINE>^\.{CRLF}	{ BEGIN(INITIAL); 
                          mlbuf[mlcur] = '\0';
                          yylval.sval = mlbuf; return STRING; }
<MULTILINE>^\.\.  { /* dot stuffing! we want one . */ yyless(1); }
<MULTILINE>(.|\n) { if (mlcur == mlbufsz) 
			mlbuf = xrealloc(mlbuf, 1 + (mlbufsz+=1024));
		    mlbuf[mlcur++] = yytext[0]; }
<MULTILINE><<EOF>> { char *buf = NULL;
                    asprintf (&buf,
			      "unexpected end of file in string started on line %d",
			      start_line);
		    yyerror (buf);
		    free (buf);
		    yyterminate(); }
<INITIAL>\/\*      { BEGIN(COMMENT); start_line = yylineno; }
<COMMENT>[^*]*\*\/ { BEGIN(INITIAL); }
<COMMENT>[^*]*[^/] ;
<COMMENT><<EOF>>  { char *buf = NULL;
                    asprintf (&buf,
			      "unexpected end of file in comment started on line %d",
			      start_line);
		    yyerror (buf);
		    free (buf);
		    yyterminate(); }
<INITIAL>text:{ws}?(#.*)?{CRLF}	{
                          BEGIN(MULTILINE);
			  mlcur = 0;
			  mlbufsz = 0;
			  mlbuf = NULL;
                          start_line = yylineno; }
<INITIAL>[0-9]+[KMG]?	{ yylval.nval = tonum(yytext); return NUMBER; }
<INITIAL>if		return IF;
<INITIAL>elsif		return ELSIF;
<INITIAL>else		return ELSE;
<INITIAL>anyof		return ANYOF;
<INITIAL>allof		return ALLOF;
<INITIAL>exists		return EXISTS;
<INITIAL>false		return SFALSE;
<INITIAL>true		return STRUE;
<INITIAL>address	return ADDRESS;
<INITIAL>envelope       return ENVELOPE;
<INITIAL>header		return HEADER;
<INITIAL>not		return NOT;
<INITIAL>size		return SIZE;
<INITIAL>reject		return REJCT;
<INITIAL>fileinto	return FILEINTO;
<INITIAL>redirect	return FORWARD;
<INITIAL>keep		return KEEP;
<INITIAL>require        return REQUIRE;
<INITIAL>stop		return STOP;
<INITIAL>discard	return DISCARD;
<INITIAL>setflag        return SETFLAG;
<INITIAL>addflag        return ADDFLAG;
<INITIAL>removeflag     return REMOVEFLAG;
<INITIAL>mark           return MARK;
<INITIAL>unmark         return UNMARK;
<INITIAL>notify         return NOTIFY;
<INITIAL>denotify       return DENOTIFY;
<INITIAL>:low           return LOW;
<INITIAL>:medium        return MEDIUM;
<INITIAL>:high          return HIGH;
<INITIAL>vacation	return VACATION;
<INITIAL>:days		return DAYS;
<INITIAL>:addresses	return ADDRESSES;
<INITIAL>:subject	return SUBJECT;
<INITIAL>:mime		return MIME;
<INITIAL>:comparator	return COMPARATOR;
<INITIAL>:is		return IS;
<INITIAL>:contains	return CONTAINS;
<INITIAL>:matches	return MATCHES;
<INITIAL>:regex		return REGEX;
<INITIAL>:over		return OVER;
<INITIAL>:under		return UNDER;
<INITIAL>:all		return ALL;
<INITIAL>:localpart	return LOCALPART;
<INITIAL>:domain	return DOMAIN;
<INITIAL>:user		return USER;
<INITIAL>:detail	return DETAIL;
<INITIAL>\"([^"]|\\.)*\"	{ yylval.sval = fixstr(yytext); return STRING; }
<INITIAL>[ \t\n\r] ;	/* ignore whitespace */
<INITIAL>#.* ;		/* ignore comments */
.		return yytext[0];

%%
static int tonum(char *c)
{
  int val = atoi(c);
  switch (c[strlen(c)-1]) {
  case 'K': val *= (1 << 10); break;
  case 'M': val *= (1 << 20); break;
  case 'G': val *= (1 << 30); break;
  default: break;
  }
  return val;
}

static char *fixstr(char *str)
{
    char *r, *s = (char *) xmalloc(sizeof(char) * strlen(str));

    (void) yy_flex_realloc;
    (void) yyunput;

    r = s;
    str++; /* skip open " */
    while (*str != '"') {
	if (*str == '\\')
	    str++;
	*s++ = *str++;
    }
    *s = '\0';
    return r;
}
