/*
* (C) 2000, 2001 David O'Toole $Date: 2001/04/07 23:27:07 $
* $Revision: 1.5 $
* 
* MACHINES, MACHINE INTERFACE, TYPE REGISTRY 
*
* This software is distributed under the terms of the
* GNU General Public License (GPL). Read the included file
* COPYING for more information. 
* 
*/

#ifndef _machine_h_
#define _machine_h_ 

#include <stdlib.h>

#include "octal.h"
#include "pattern.h"

#define MAX_NAME 128
#define MAX_CHANNELS 64
#define MAX_PARAMS 32

enum { OX_OUTPUT_BUFFER_SIZE = sizeof(samp)*4096 }; 
enum { OX_MAX_TYPES = 128 };

enum { OX_MONO = 1, OX_STEREO = 2 };
enum { OX_NOTE_OFF = 0 };   

//
// TYPE/WIDGET OPTIONS
// The strings to match these are kept in machine.c

enum { 	OX_TYPE_NOTE,
		OX_TYPE_VELOCITY, 
		OX_TYPE_GENERIC,
		OX_TYPE_TRIGGER,
		OX_TYPE_WAVE };    

enum { 	OX_WIDGET_SLIDER,
		OX_WIDGET_OPTION, 
		OX_WIDGET_BUTTON 
};

struct _machine_type; 
struct _package;

//
// MACHINE PARAMETERS 
// 
// Machine writers no longer fill this in directly; it wasn't flexible enough.
// Instead they pass a null-terminated array of strings formatted like this:
//
// :name:description:type:widget:min:max:start:
// with the colons at the start and end optional

typedef struct _param_spec { 
  char name[100]; 
  char description[100]; 
  int type;
  int widget;    
  param min, max, start; 

} param_spec;

//
// MACHINES
//

typedef struct _machine {

	struct _machine_type *type;          
	struct _package *pkg; 

	// public fields for access by machine writers

	void *state;           
  	samp *lin,  *rin;  
	samp *lout, *rout;   

	// mixer gui information

	char instance_name[MAX_NAME];   
	int x, y;
 
	// mixer state information

	long frame;  
	int generated;
	param *param_state;
	int channel_state[MAX_CHANNELS];
	int num_channels;

} machine;


// 
// MACHINE TYPES: info about plugin modules loaded
//

typedef struct _machine_type { 
  
	void* so_handle;

	// virtual functions

	int         (*ox_init)(struct _machine_type*);              
	void        (*ox_create)(machine*);   
	void        (*ox_event)(machine*,int,int,param);
	int         (*ox_work)(machine*, int);   
	void        (*ox_destroy)(machine*);
	void        (*ox_desc)(char*,int,param); 
	void        (*ox_channel)(machine*, int, int);        

	// public fields to be filled in by machine writer

	char* short_name;   
	char* long_name; 
	int max_channels;      
	int input_channels;   
	int output_channels; 
	char **param_info;

	// parameter information (filled in from param_info);

	int num_params;
	param_spec param_specs[MAX_PARAMS];  
	

} machine_type;


// 
// OX_API PACKAGE: callback functions for various host features
//

typedef struct _package {
	void* (*alloc)(size_t);          // p_alloc         
	void  (*free)(void*);
	
	// MUSICAL

	param (*text2note)(const char*); // p_text2note     
	const char* (*note2text)(param); // you get the idea  
	freq  (*text2freq)(const char*);   
	freq  (*note2freq)(param);

	// ENVIRONMENT

	int (*get_sr)(void);	// retrieve sampling rate
	         
  
} package; 

// Be sure to change the definition of this next one if you
// add members to the above

void mc_init_global_pkg(void); 

//
// MACHINE REGISTRY
//

int mc_make_registry(const char* machine_dir);
machine_type* mc_get_type(const char* name);
machine_type* mc_get_type_by_index(int x);
int mc_get_num_types();
	
int mc_load_type(machine_type *t, const char *fn);
int mc_unload_type(machine_type *t);


// 
// CREATION, MANIPULATION 
//

machine* mc_create(machine_type *t);
void mc_destroy(machine *m);
void mc_event(machine *m, int track, int ix, param value);
void mc_channels_configure(machine *m, int num);

// work buffer allocator wrappers. don't use these yourself.

void mc_allocate_outbuf(machine*, samp**, samp**);
void mc_free_buf(samp**, samp**); 

//
// PROPERTY INSPECTION SHORTCUTS
// 

int mc_is_stereo_out(machine* m);
int mc_is_stereo_in(machine* m);
int mc_wants_input(machine* m);

//
// MEMORY WRAPPERS
//

void* ox_alloc(size_t);   // possibly calls mlock()    
void* pkg_alloc(size_t);  // wraps ox_alloc for machines 
void  ox_free(void*);    // unlocks mem and frees

#endif 



