/*
   XRMenuCell.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@net-community.com>
   Date: May 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: June 1998

   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <config.h>

#include <AppKit/NSCursor.h>
#include <AppKit/NSApplication.h>
#include <AppKit/NSColor.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSButton.h>
#include <AppKit/NSGraphics.h>
#include <AppKit/NSFont.h>

#include <Foundation/NSException.h>

#include <gnustep/xraw/XRColor.h>
#include <gnustep/xraw/XRScreen.h>
#include <gnustep/xraw/XRMenu.h>
#include <gnustep/xraw/XRMenuCell.h>
#include <gnustep/xraw/XRContext.h>

#include "wraster/wraster.h"

extern void XRDrawString(const char *str, NSRect rect);
extern char* XRCopyStringBuffer(const char *buf);

//
// Backend structure for XRMenuCell
//
typedef struct _XRMenuCell_struct
{
	char *stringValue;
	char *keyEquivStringValue;
	float titleDescender;
	float titleWidth;
	float keyEquivalentWidth;

} XRMenuCell_struct;

#define TSTRING (((XRMenuCell_struct *)be_mi_reserved)->stringValue)
#define KEYV (((XRMenuCell_struct *)be_mi_reserved)->keyEquivStringValue)
#define TDESCENDER (((XRMenuCell_struct *)be_mi_reserved)->titleDescender)
#define TWIDTH (((XRMenuCell_struct *)be_mi_reserved)->titleWidth)
#define KWIDTH (((XRMenuCell_struct *)be_mi_reserved)->keyEquivalentWidth)

static RContext *rc;


@implementation XRMenuCell

- init
{
	[super init];
												// Allocate back-end structure
	be_mi_reserved = malloc(sizeof(XRMenuCell_struct));
	TSTRING = NULL;
	KEYV = NULL;
	rc = RCreateContext([XRContext currentXDisplay], 
						[(XRScreen *)[NSScreen mainScreen] xScreen], NULL);

	return self;
}

- (id) copyWithZone: (NSZone*)z
{
XRMenuCell *cell = [super copyWithZone: z];

	[cell setTitle:contents];
	[cell setKeyEquivalent:[self userKeyEquivalent]];

    return cell;
}

- (void)dealloc
{
	if(TSTRING)										
		free(TSTRING);									// free backend caching
	if(KEYV)
		free(KEYV);
	free(be_mi_reserved);								// free the back-end
														// structure
	[super dealloc];
}

- (void)setTitle:(NSString *)aString
{
	[super setTitle:aString];

	if(TSTRING)
		free(TSTRING);									// free cache
	TSTRING = NULL;
														
	TSTRING = XRCopyStringBuffer([aString cString]);	// cache cell contents
	TWIDTH = [cell_font widthOfString:aString];			// cache str width
	TDESCENDER = [cell_font descender];					
}

- (void)setKeyEquivalent:(NSString *)key
{
NSString* keyEquiv;

	[super setKeyEquivalent:key];
	if ((keyEquiv = [self userKeyEquivalent]))			// cache key equivalent
		{												// and layout info if 
		if ([keyEquiv cStringLength] > 0)				// key equiv exists
			{
			KEYV = XRCopyStringBuffer([keyEquiv cString]);	// cache key equiv
			KWIDTH = [cell_font widthOfString:keyEquiv];	// cache str width
			}
		}
}

- (void)setFont:(NSFont *)fontObject
{
NSString* keyEquiv;

	[super setFont:fontObject];

	TWIDTH = [cell_font widthOfString:[self title]];
	TDESCENDER = [cell_font descender];
	if ((keyEquiv = [self userKeyEquivalent]))			// cache key equivalent
		{												// and layout info if 
		if ([keyEquiv cStringLength] > 0)				// key equiv exists
			{
			KEYV = XRCopyStringBuffer([keyEquiv cString]);	// cache key equiv
			KWIDTH = [cell_font widthOfString:keyEquiv];	// cache str width
			}
		}
}

- (void)drawInteriorWithFrame:(NSRect)cellFrame
                       inView:(NSView*)view
	       backgroundGray:(float)backgroundGray
{
  NSRect rect = cellFrame;
  NSPoint point;

  [cell_font set];				// set drawing font
						// calc the title's origin
//  point.y = rect.origin.y - (rect.size.height / 2) - TDESCENDER - 2;
  point.y = rect.origin.y;
  point.x = rect.origin.x + xDist;
  rect.origin = point;

  if ([self imagePosition] == NSImageLeft)
    rect.origin.x += 4 + RIGHT_IMAGE_WIDTH;
  XRDrawString(TSTRING, rect);			// draw the title

  rect.size.width = 15;				// calc image rect
  rect.size.height = cellFrame.size.height;
  if ([self imagePosition] == NSImageLeft)
    rect.origin.x = cellFrame.origin.x + 2;
  else
    rect.origin.x = cellFrame.origin.x + cellFrame.size.width 
					- (4 + RIGHT_IMAGE_WIDTH);
  rect.origin.y = cellFrame.origin.y;

  if (hasSubmenu)				// if submenu draw aroow image
    {
      NSImage* image = (backgroundGray == NSWhite ? 
		  [NSImage imageNamed:@"common_RightH"] : 
		  [NSImage imageNamed:@"common_Right"]);

      [super _drawImage:image inFrame:rect];
    }
  else if (KEYV)
    {
      // if cell has a key equivalent draw it's text
      if (KWIDTH < cellFrame.size.width)
	point.x = cellFrame.origin.x + cellFrame.size.width - (5 + KWIDTH);
      else
	point.x = cellFrame.origin.x + xDist;
      rect.origin = point;
      rect.origin.y++;
      XRDrawString(KEYV, rect);
    }
  else if ([self image])
    {
      NSImage* image = [self image];

      [super _drawImage: image inFrame: rect];
    }
}

- (void)drawWithFrame:(NSRect)cellFrame
               inView:(NSView *)controlView
{
NSRect rect = cellFrame;
BOOL isEnabled = [self isEnabled];
float backgroundGray = NSLightGray;
float titleGray;

RImage *menuGradient;
RXImage *mySplatch;
RColor from, to;
RColor from2, to2;

	from2.red = 0xff;
	from2.green = 0xff;
	from2.blue = 0xff;
	from2.alpha = 0xff;
	to2.red = 0xff;
	to2.green = 0xff;
	to2.blue = 0xff;
	to2.alpha = 0xff;

	from.red = 0xc0;
	from.green = 0xc0;
	from.blue = 0xc0;
	from.alpha = 0xff;
	to.red = 0x90;
	to.green = 0x90;
	to.blue = 0x90;
	to.alpha = 0xff;

/*
    	from.red = 0xaa;
    	from.green = 0xaa;
    	from.blue = 0xaa;
    	from.alpha = 0xff;
    	to.red = 0xaa;
    	to.green = 0xaa;
    	to.blue = 0xaa;
    	to.alpha = 0xff;
*/

	if (cell_highlighted && isEnabled) 
		{
        menuGradient = RRenderGradient((int)cellFrame.size.width, 	
										(int)cellFrame.size.height,
                        				&from2, &to2, RGRD_HORIZONTAL);

        RBevelImage(menuGradient, RBEV_RAISED2);

        mySplatch = RConvertImage2(rc, menuGradient);

        XRDrawXImage(mySplatch->image, cellFrame);

		RDestroyImage(menuGradient);
		RDestroyXImage(rc, mySplatch);

		backgroundGray = NSWhite;
/*
		backgroundGray = NSWhite;
		[[NSColor colorWithCalibratedWhite:backgroundGray alpha:1.0] set];
		NSRectFill(cellFrame);						// clear the cell frame
		NSDrawButton (cellFrame, cellFrame);
*/
		}
	else
		{
		if ((int) cellFrame.size.width != 0)
			{
        	menuGradient = RRenderGradient((int)cellFrame.size.width, 
											(int)cellFrame.size.height,
                        					&from, &to, RGRD_HORIZONTAL);

			RBevelImage(menuGradient, RBEV_RAISED2);

			mySplatch = RConvertImage2(rc, menuGradient);

			XRDrawXImage(mySplatch->image, cellFrame);

			RDestroyImage(menuGradient);
			RDestroyXImage(rc, mySplatch);
			[[XRContext currentContext] flush];
			}
		}

#ifdef FAR_DEBUG
	fprintf (stderr,
	"XRMenuCell drawWithFrame origin (%1.2f, %1.2f), size (%1.2f, %1.2f)\n",
			cellFrame.origin.x, cellFrame.origin.y, 
			cellFrame.size.width, cellFrame.size.height);
#endif																		

	titleGray = isEnabled ? NSBlack : NSDarkGray;	// set title gray
	[[NSColor colorWithCalibratedWhite:titleGray alpha:1.0] set];

	cellFrame.origin.x += 1;								// calc interior
	cellFrame.origin.y += 2;
	cellFrame.size.width -= 3;
	cellFrame.size.height -= 3;

	[self drawInteriorWithFrame:cellFrame					// draw interior
                       	 inView:controlView
             			 backgroundGray:backgroundGray];
}

- (Class) classForCoder: aCoder
{
  if ([self class] == [XRMenuCell class])
    return [super class];
  return [self class];
}

@end
