/* 
   XRCursor.m

   NSCursor for GNUstep GUI X/RAW Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Scott Christley <scottc@net-community.com>
   Date: March 1996
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>

#include <AppKit/NSApplication.h>

#include <gnustep/xraw/XRCursor.h>
#include <gnustep/xraw/XRContext.h>
#include <gnustep/xraw/XRWindow.h>
#include <gnustep/xraw/XRColor.h>


//
// Backend structure for XRCursor
//
typedef struct _XRCursor_struct
{
  Cursor cursor;
  BOOL is_valid;
} XRCursor_struct;

#define XCURSOR (((XRCursor_struct *)be_cursor_reserved)->cursor)
#define XVALID (((XRCursor_struct *)be_cursor_reserved)->is_valid)

static char xraw_blank_cursor_bits [] = {
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

//
// Class variables
//
static NSCursor *xraw_blank_cursor = nil;
static NSCursor *xraw_hidden_cursor = nil;



@implementation XRCursor

+ (void)initialize
{
	if (self == [XRCursor class])
		[self setVersion:1];								// Initial version
}

//
// X Window private methods
//
- (void)setXCursor:(Cursor)cursor
{
	XCURSOR = cursor;
	XVALID = YES;
}

- (Cursor)xCursor
{
  // If the xCursor isn't valid
  // then try to create it from the image
  if (!XVALID)
    {
#if 0
      NSApplication *theApp = [NSApplication sharedApplication];
      NSWindow *w = [theApp mainWindow];
      Pixmap shape, mask;
      XColor black, white;
      Window win;

      if (w)
        win = [(XRWindow *)w xWindow];
      else
        {
          XRContext* context = (XRContext *)[XRContext currentContext];
          win = [context xAppRootWindow];
        }

      shape = XCreatePixmapFromBitmapData(xDisplay, win,
					  cursor_bits, 16, 16, 1, 0, 1);
      mask = XCreatePixmapFromBitmapData(xDisplay, win,
					 cursor_bits, 16, 16, 1, 0, 1);
      black = [(XRColor *)[NSColor blackColor] xColor];
      white = [(XRColor *)[NSColor whiteColor] xColor];

      XCURSOR = XCreatePixmapCursor(xDisplay, shape, mask, 
				    &white, &black, 0, 0);
      XVALID = YES;
#endif
    }

  return XCURSOR;
}

- (BOOL)xCursorIsValid
{
	return XVALID;
}

//
// Class methods
//

//
// Setting the Cursor
//
+ (void)hide
{
	if (!xraw_blank_cursor)							// Create the blank cursor 
		{											// if necessary
		Display *xDisplay = [XRContext currentXDisplay];
		NSApplication *theApp = [NSApplication sharedApplication];
		NSWindow *w = [theApp mainWindow];
		Pixmap shape, mask;
		XColor black, white;
		Cursor c;
        Window win;

        if (w)
            win = [(XRWindow *)w xWindow];
        else
            {
            XRContext* context = (XRContext *)[XRContext currentContext];
            win = [context xAppRootWindow];
            }
	
		xraw_blank_cursor = [[NSCursor alloc] initWithImage: nil];
	
		shape = XCreatePixmapFromBitmapData(xDisplay, win,
											xraw_blank_cursor_bits, 
											16, 16, 1, 0, 1);
		mask = XCreatePixmapFromBitmapData(xDisplay, win,
											xraw_blank_cursor_bits, 
											16, 16, 1, 0, 1);
		black = [(XRColor *)[NSColor blackColor] xColor];
		white = [(XRColor *)[NSColor whiteColor] xColor];
	
		c = XCreatePixmapCursor(xDisplay, shape, mask, &white, &black, 0, 0);
		[(XRCursor *)xraw_blank_cursor setXCursor: c];
		}
													// If the cursor is already 
	if (xraw_blank_cursor == [self currentCursor])	// hidden then don't hide 
		return;										// again.

	xraw_hidden_cursor = [self currentCursor];		// Save the current cursor 
	[xraw_blank_cursor set];						// and set the blank cursor
}

+ (void)unhide
{
	if (!xraw_blank_cursor)							// if no blank cursor then 
		return;										// we couldn't have hidden
	
	if (xraw_blank_cursor != [self currentCursor])	// If cursor isn't hidden 
		return;										// then just return
	
	[xraw_hidden_cursor set];						// Set back to the current
}													// cursor

//
// Getting the Cursor
//
+ (NSCursor *)arrowCursor
{
NSCursor *cur;
Cursor c;											// Create the standard left 
													// pointer arrow
	cur = [[[NSCursor alloc] initWithImage: nil] autorelease];
	c = XCreateFontCursor([XRContext currentXDisplay], XC_left_ptr);
	[(XRCursor *)cur setXCursor: c];
	
	return cur;
}

+ (NSCursor *)IBeamCursor
{
NSCursor *cur;
Cursor c;
													// Create standard I beam
	cur = [[[NSCursor alloc] initWithImage: nil] autorelease];
	c = XCreateFontCursor([XRContext currentXDisplay], XC_xterm);
	[(XRCursor *)cur setXCursor: c];
	
	return cur;
}

//
// Instance methods
//

//
// Initializing a New NSCursor Object
//
- (id)initWithImage:(NSImage *)newImage
{
	[super initWithImage: newImage];
															// Allocate backend
	be_cursor_reserved = malloc(sizeof(XRCursor_struct));	// structure
	XCURSOR = 0;
	XVALID = NO;
	
	return self;
}

- (void)dealloc
{
	[super dealloc];
	free(be_cursor_reserved);						// Free backend structure
}

@end

//
// Backend Methods
//
@implementation XRCursor (GNUstepBackend)

+ (void)currentCursorHasChanged
{
NSCursor *c = [self currentCursor];
BOOL isValidXCursor = [(XRCursor *)c xCursorIsValid];

	if (isValidXCursor)
		{
		Cursor xc = [(XRCursor *)c xCursor];
		NSApplication *theApp = [NSApplication sharedApplication];
		NSWindow *w = [theApp mainWindow];
        Window win;

        if (w)
            win = [(XRWindow *)w xWindow];
        else
            {
            XRContext* context = (XRContext *)[XRContext currentContext];
            win = [context xAppRootWindow];
            }

		XDefineCursor([XRContext currentXDisplay],win, xc);
		}
}

- (Class) classForCoder: aCoder
{
  if ([self class] == [XRCursor class])
    return [super class];
  return [self class];
}

@end
