/*
   XRBitmapImageRep.m

   NSBitmapImageRep for GNUstep GUI X/RAW Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Adam Fedor <fedor@colorado.edu>
   Author:  Scott Christley <scottc@net-community.com>
   Date: Feb 1996
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998

   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <stdlib.h>
#include <tiff.h>

#include <Foundation/NSLock.h>
#include <Foundation/NSData.h>

#include <gnustep/xraw/XRBitmapImageRep.h>
#include <gnustep/xraw/XRWindow.h>
#include <gnustep/xraw/XRContext.h>
#include <gnustep/xraw/XRScreen.h>
#include <gnustep/xraw/XRColor.h>

//
// Backend structure for XRBitmapImageRep
//
typedef struct _XRBitmapImageRep_struct
{
	BOOL is_cached;
	XImage *xImage;

} XRBitmapImageRep_struct;

#define XRCACHE (((XRBitmapImageRep_struct *)back_end_reserved)->is_cached)
#define XIMAGE (((XRBitmapImageRep_struct *)back_end_reserved)->xImage)



@implementation XRBitmapImageRep

//
// Class methods
//
+ (void) initialize
{
	if (self == [XRBitmapImageRep class])
		[self registerImageRepClass: [self class]];			// Register ourself
}

//
// Instance methods
//

//
// Initialization
//
- (id)initWithBitmapDataPlanes:(unsigned char **)planes
					pixelsWide:(int)width
					pixelsHigh:(int)height
					bitsPerSample:(int)bps
					samplesPerPixel:(int)spp
					hasAlpha:(BOOL)alpha
					isPlanar:(BOOL)isPlanar
					colorSpaceName:(NSString *)colorSpaceName
					bytesPerRow:(int)rowBytes
					bitsPerPixel:(int)pixelBits;
{
	[super initWithBitmapDataPlanes:planes
						pixelsWide:width
						pixelsHigh:height
						bitsPerSample:bps
						samplesPerPixel:spp
						hasAlpha:alpha
						isPlanar:isPlanar
						colorSpaceName:colorSpaceName
						bytesPerRow:rowBytes
						bitsPerPixel:pixelBits];

											// Allocate the  back-end structure
	back_end_reserved = malloc(sizeof(XRBitmapImageRep_struct));
	XRCACHE = NO;

	return self;
}

- (void)dealloc
{
	if (XRCACHE)							// free ximage cache
		XDestroyImage(XIMAGE);

	free(back_end_reserved);				// Free the back-end structure

	[super dealloc];
}

//
// RGB color image with color values separate
// Use colorimage postscript operator
//
- (BOOL)drawColorSeparate
{
	return YES;
}

//
// RGB color image with color values not separate
// Use colorimage postscript operator
//
- (BOOL) drawColorContig
{
  uint32 row, col, i;
  Colormap colorMap;
  Display *xDisplay;
  int xscreen;
  unsigned char *bData, alpha;
  unsigned short *cPtr, bg;
  XColor c1;

  if (!XRCACHE)				// if image is not yet cached
    {					// create an ximage cache
      xDisplay = [XRContext currentXDisplay];
      xscreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
      colorMap = XDefaultColormap(xDisplay, xscreen);

										      // get ptr to image rep data
      bData = [(NSBitmapImageRep *)self bitmapData];

										      // grab a default XImage
      XIMAGE = XGetImage(xDisplay, RootWindow(xDisplay, xscreen),
			0, 0, (int)size.width, (int)size.height,
			AllPlanes, ZPixmap);

										      // def image background color
      bg = [(XRColor *)[NSColor lightGrayColor] xColor].red / 257;

      // convert our internal bitmap data into an XImage
      if (numColors == 4)
	{
	  // CMYK, RGBA ?
	  for (row = 0; row < size.height; row++)
	    {
	      for (col = 0; col < size.width; col++)
		{
		  // Create XColor from image data
		  c1.red = (unsigned short)((char)*bData++);
		  c1.green = (unsigned short)((char)*bData++);
		  c1.blue = (unsigned short)((char)*bData++);
		  alpha = 255 - (unsigned short)((char)*bData++);

		  // mix in alpha to produce RGB out
		  c1.red = ((bg * alpha) + (c1.red * 257));
		  c1.green = ((bg * alpha) + (c1.green * 257));
		  c1.blue = ((bg * alpha) + (c1.blue * 257));

		  // alloc colors and put in ximage
		  XAllocColor(xDisplay, colorMap, &c1);
		  XPutPixel(XIMAGE, col, row, c1.pixel);
		}
	    }
	}
      else
	{
	  // RGB, Grayscale ?
	  cPtr = &c1.red;
	  for (row = 0; row < size.height; row++)
	    {
	      for (col = 0; col < size.width; col++)
		{
		  for (i = 0; i < numColors; i++)
		    {
		      // Create XColor from image data
		      cPtr[i] = (unsigned short)(257 * (char)*bData++);
		      XAllocColor(xDisplay, colorMap, &c1);
		      XPutPixel(XIMAGE, col, row, c1.pixel);
		    }
		}
	    }
	}
      XRCACHE = YES;				// ximage can be drawn
    }
  return XRCACHE;
}

- (BOOL)draw
{
  NSDebugLog(@"XRBitmapImageRep: draw\n");
  NSDebugLog(@"XRBitmapImageRep: color space %s\n", [_colorSpace cString]);
  NSDebugLog(@"XRBitmapImageRep: size %f %f\n", size.width, size.height);
  NSDebugLog(@"%d %d %d %d\n",bitsPerSample,_pixelsWide,_pixelsHigh,hasAlpha);
  NSDebugLog(@"%d %d %d %d\n",bytesPerRow,numColors,bitsPerPixel,compression);

	if ([_colorSpace compare: NSDeviceRGBColorSpace] == NSOrderedSame)
		{
		if (_isPlanar)
			return [self drawColorSeparate];
		else
	 		return [self drawColorContig];
    	}

	return NO;
}

- (XImage *)xImage				{ return XIMAGE; }

- (Class) classForCoder: aCoder
{
  if ([self class] == [XRBitmapImageRep class])
    return [super class];
  return [self class];
}

@end
