// This is -*- C++ -*-
// $Id: Permutation.cpp,v 1.2 1999/03/16 18:29:24 alstrup Exp $

/* 
 * Permutation.cpp
 *
 * Copyright (C) 1998 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <assert.h>
#include <string>
#include <iostream>
#include "Random.h"
#include "Permutation.h"

Permutation::Permutation()
{ }

Permutation::Permutation(size_t N)
{
  set_identity(N);
}

Permutation::Permutation(const Permutation& p) :
  perm_(p.perm_), cyclerep_(p.cyclerep_)
{ }

Permutation&
Permutation::operator=(const Permutation& p)
{
  perm_ = p.perm_;
  cyclerep_ = p.cyclerep_;
  return *this;
}

void
Permutation::initialize(size_t N, size_t* d)
{
  char* seen = new char[N];
  memset(seen, 0, N*sizeof(char));

  cyclerep_.clear();
  perm_.clear();
  perm_.reserve(N);
  for(size_t i=0; i<N; ++i) {
    
    // Die if we are being fed something that isn't a permutation on 0...(N-1)
    assert(d[i] < N);
    assert(! seen[d[i]]);
   
    seen[d[i]] = 1;
    perm_.add(d[i]);
  }
  delete [] seen;
}

size_t
Permutation::orbit_size(size_t i) const
{
  size_t count=0;
  size_t j=i;
  do {
    ++count;
    j = map(j);
  } while (j != i);
  return count;
}

// The order of a permutation = the lcm of the size of the orbits
size_t
Permutation::order() const
{
  calc_cycles();
  size_t lcm = 1;
  for(size_t i=0; i<cyclerep_.size(); ++i) {
    size_t os = orbit_size((cyclerep_.data())[i]);
    
    // Euclid is the man
    size_t a = lcm < os ? lcm : os;
    size_t b = lcm < os ? os : lcm;
    size_t r;
    while ((r = b % a) != 0) {
      b = a;
      a = r;
    }

    lcm = lcm*os/a;
  }

  return lcm;
}

void
Permutation::set_identity(size_t N)
{
  assert(N);
  perm_.clear();
  perm_.reserve(N);
  for(size_t i=0; i<N; ++i)
    perm_.add(i);
  cyclerep_ = perm_;
}

void
Permutation::set_shift(size_t N, int k)
{
  assert(N);
  perm_.clear();
  perm_.reserve(N);
  while (k < 0) k += N;
  for(size_t i=0; i<N; ++i)
    perm_.add((i+k) % N);
  
  cyclerep_.clear();
}

// The reverse permutation (1 2 3 .. N -> N .. 3 2 1) is pretty much the
// worst case in terms of our cyclerep_ notation, as it consists of N/2
// disjoint transpositions.  Ugh.
void
Permutation::set_reverse(size_t N)
{
  assert(N);
  perm_.clear();
  perm_.reserve(N);
  for(size_t i=0; i<N; ++i)
    perm_.add(N-1-i);

  cyclerep_.clear();
}

void
Permutation::scramble_explicit(Random& rnd)
{
  if (size() < 2) return;
  size_t* d = perm_.data();
  for(size_t i=0; i<size()-1; ++i) {
    size_t j = rnd.random_integer(i,size()-1);
    if (i != j) {
      size_t t = d[i];
      d[i] = d[j];
      d[j] = t;
    }
  }
  cyclerep_.clear();
}

void
Permutation::precompose(const Permutation& p)
{
  assert(size() == p.size());

  size_t* tmp = new size_t[size()];
  for(size_t i=0; i<size(); ++i)
    tmp[i] = map(p.map(i));

  size_t* d = perm_.data();
  for(size_t i=0; i<size(); ++i)
    d[i] = tmp[i];

  delete [] tmp;
  cyclerep_.clear();
}

void
Permutation::postcompose(const Permutation& p)
{
  assert(size() == p.size());

  size_t* tmp = new size_t[size()];
  for(size_t i=0; i<size(); ++i)
    tmp[i] = p.map(map(i));

  size_t* d = perm_.data();
  for(size_t i=0; i<size(); ++i)
    d[i] = tmp[i];

  delete [] tmp;
  cyclerep_.clear();
}

void
Permutation::invert()
{
  size_t* tmp = new size_t[size()];
  for(size_t i=0; i<size(); ++i)
    tmp[map(i)] = i;

  size_t* d = perm_.data();
  for(size_t i=0; i<size(); ++i)
    d[i] = tmp[i];

  delete [] tmp;
  // cyclerep_ is invariant under inversion...
}

void
Permutation::spew(ostream& out) const
{
  for(size_t i=0; i<size(); ++i)
    out << i << " => " << map(i) << endl;
}

void
Permutation::spew_cycles(ostream& out) const
{
  calc_cycles();
  bool id = true;
  for(size_t i=0; i<cyclerep_.size(); ++i) {
    size_t start = (cyclerep_.data())[i];
    if (map(start) != start) { // don't display singleton cycles
      id = false;
      size_t j = start;
      out << "( ";
      do {
	out << j << " ";
	j = map(j);
      } while (j != start);
      out << ")";
    }
  }
  if (id)
    out << "(identity)";
}

// For a random permutation, the average # of cycles is very small,
// even for permutations on very large sets.  (It would be interesting
// to know the asymptotics of this number as N -> infinity.  If there
// are any group theorists or combinatorics people out there reading
// this who know the answer, drop me an e-mail at trow@emccta.com and
// let me know...
void
Permutation::calc_cycles() const
{
  if (cyclerep_.size())
    return;

  RCArray<size_t> proxy;
  // Better to use char than bool, since bool is the size of an int on
  // some architectures...
  char* seen = new char[size()];
  memset(seen, 0, size()*sizeof(char)); // sizeof(char)==1 in ANSI, no?
  for(size_t i=0; i<size(); ++i) {
    if (! seen[i]) {
      proxy.add(i);
      size_t j=i;
      do {
	seen[j] = 1;
	j = map(j);
      } while (! seen[j]);
    }
  }
  delete [] seen;

  // It isn't nice to fool the compiler!
  ((Permutation*)this)->cyclerep_ = proxy;
}


// $Id: Permutation.cpp,v 1.2 1999/03/16 18:29:24 alstrup Exp $

