/* -*- Mode: c++ -*-
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#include <VrFileSink.h>
#include <VrSigSource.h>
#include <VrScopeSink.h>
#include <VrAudioSink.h>
#include <VrNullSink.h>
#include <VrGuppiSource.h>
#include <VrFakeGuppiSource.h>
#include <VrRealFIRfilter.h>
#include <VrComplexFIRfilter.h>
#include <VrQuadratureDemod.h>
#include <VrConnect.h>
#include <VrComplex.h>
#include <VrMultiTask.h>
#include <VrPerfGraph.h>

int main(void) {

  int gupRate;
  int CFIRdecimate;
  int quadRate;
  int RFIRdecimate;
  int audioRate;
  int cTaps;
  double seconds = 5;

  cTaps = 400;
  gupRate = 33000000;
  CFIRdecimate = 825;
  quadRate = gupRate / CFIRdecimate;
  RFIRdecimate = 5;
  audioRate = quadRate / RFIRdecimate;

  const int numChannels=4;
  float center_freq[] = {10370400, 10355400, 10340400, 10960500};
  float gain[] = {2.0, 2.0, 2.0, 2.0};
  int num_taps[] = {cTaps, cTaps, cTaps, cTaps};

  cerr << "GuPPI Sampling Rate: " << gupRate << endl;
  cerr << "Complex FIR decimation factor: " << CFIRdecimate << endl;
  cerr << "QuadDemod Sampling Rate: " << quadRate << endl;
  cerr << "Real FIR decimation factor: " << RFIRdecimate << endl;
  cerr << "Audio Sampling Rate: " << audioRate << endl;

  // Create Modules
  VrGuppiSource<char>* source = new VrGuppiSource<char>();
  VrComplexFIRfilter<char>* channel_filter =
    new VrComplexFIRfilter<char>(numChannels, CFIRdecimate, 
					     num_taps, center_freq, gain);

  VrQuadratureDemod<float>* demod[numChannels];
  VrRealFIRfilter<float,short>* if_filter[numChannels];
  VrNullSink<short>* sink[numChannels];

  VrMultiTask *multi = new VrMultiTask();
#ifdef PERFMON
  VrPerfGraph *g=new VrPerfGraph("Multi-channel FM audio receive");
  g->add("source",source);
  g->add("channel",channel_filter);
#endif /* PERFMON */

  int i;
  for (i = 0; i < numChannels; i++)
    {
      demod[i] = new VrQuadratureDemod<float>(0.0);
      if_filter[i] = new VrRealFIRfilter<float,short>(RFIRdecimate,4000.0,20,1.0);
      sink[i] = new VrNullSink<short>();

      multi->add(sink[i]);
      // Connect Modules

      CONNECT(sink[i], if_filter[i], audioRate, 16);
      CONNECT(if_filter[i], demod[i], audioRate, 32);
      CONNECTN(demod[i], channel_filter, i, quadRate, 64);

#ifdef PERFMON
      g->add("demod",demod[i]);
      g->add("filter",if_filter[i]);
      g->add("sink",sink[i]);
#endif /* PERFMON */

    }

  CONNECT(channel_filter, source, gupRate, 8);

  // Start System
  multi->start();
  while (1)
	multi->process();
#ifdef PERFMON
  g->outputGraph(seconds);
#endif /* PERFMON */
  cerr << endl << "Running time: "<<seconds<<endl;

#ifdef PERFMON
  cerr << "Total samples generated by module:\n";
  cerr << "VrGuPPiSource: " << source->getTotalSamples() << endl;
  cerr << "VrComplexFIRfilter: " << channel_filter->getTotalSamples() << endl;
  cerr << "VrQuadratureDemod: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << demod[i]->getTotalSamples();
  cerr << endl << "VrRealFIRfilter: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << if_filter[i]->getTotalSamples();
  cerr << endl << "VrScopeSink: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << sink[i]->getTotalSamples();

  cerr << endl << "\nTotal cycles taken by each module:\n"; 
  cerr << "VrGuPPiSource: " << source->getTotalCycles() << endl;
  cerr << "VrComplexFIRfilter: " << channel_filter->getTotalCycles() << endl;
  cerr << "VrQuadratureDemod: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << demod[i]->getTotalCycles();
  cerr << endl << "VrRealFIRfilter: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << if_filter[i]->getTotalCycles();
  cerr << endl << "VrScopeSink: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << sink[i]->getTotalCycles();

  cerr << endl << "\nTotal US cycles taken by each module:\n";
  cerr << "VrGuPPiSource: " << source->getUSCycles() << endl;
  cerr << "VrComplexFIRfilter: " << channel_filter->getUSCycles() << endl;
  cerr << "VrQuadratureDemod: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << demod[i]->getUSCycles();
  cerr << endl << "VrRealFIRfilter: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << if_filter[i]->getUSCycles();
  cerr << endl << "VrScopeSink: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << sink[i]->getUSCycles();
  cerr << endl << "TOTAL: " << endl;
  int tot=demod[0]->getUSCycles();
  for(i=0;i<numChannels;i++) {cerr << "\t" << (sink[i]->getUSCycles()+sink[i]->getTotalCycles()); tot+=(sink[i]->getUSCycles()+sink[i]->getTotalCycles()-demod[i]->getUSCycles());}
  cerr << endl << "GRAND TOTAL: " << tot << endl;

  cerr << endl << "\nAverage cycles/output sample taken by each module:\n";
  cerr << "VrGuPPiSource: " << source->getCyclesPerSample() << endl;
  cerr << "VrComplexFIRfilter: " << channel_filter->getCyclesPerSample() << endl;
  cerr << "VrQuadratureDemod: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << demod[i]->getCyclesPerSample();
  cerr << endl << "VrRealFIRfilter: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << if_filter[i]->getCyclesPerSample();
  cerr << endl << "VrScopeSink: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << sink[i]->getCyclesPerSample();

  cerr << endl;
  /*<< "\nAverage upstream cycles/output sample taken by each module:\n";
  cerr << "VrComplexFIRfilter: " << channel_filter->getUSCyclesPerOutputSample() << endl;
  cerr << "VrQuadratureDemod: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << demod[i]->getUSCyclesPerOutputSample();
  cerr << endl << "VrRealFIRfilter: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << if_filter[i]->getUSCyclesPerOutputSample();
  cerr << endl << "VrScopeSink: " << endl;
  for(i=0;i<numChannels;i++) cerr << "\t" << sink[i]->getUSCyclesPerOutputSample();
  cerr << endl;
  */
#endif
}
