#include "Multichannel.h"
#include "Channel.h"
#include <qlayout.h>
#include <qapplication.h>
#include <qtimer.h>
#include <VrAR5000Source.h>
#include <VrComplexFIRfilter.h>
#include <VrComplexCascadefilter.h>

#include <VrQFFTSink.h>
#include <VrQuadratureDemod.h>
#include <VrRealFIRfilter.h>
#include <VrAudioSink.h>
#include <VrMultiTask.h>
#include <VrZeroSource.h>
#include <stdiostream.h>


const int Multichannel::numChannels;

const int gupPages = 20;
const int gupRate = 33000000;
const int CFIRdecimate = 825;

//const int CFIRdecimate2 = 1;

const int quadRate = gupRate / CFIRdecimate;
// const int quadRate = gupRate / CFIRdecimate / CFIRdecimate2;
const int RFIRdecimate = 5;

const int audioRate = quadRate / RFIRdecimate;
const float RxFrequency = 49.5;
const float center_freq[] = {10370400, 10355400, 10340400, 10960500};
// const float center_freq[] = {10370400, 10340400, 10310400, 10960500};
// const float gain[] = {1.0, 1.0, 1.0, 1.0};
const float gain[] = {2.0, 2.0, 2.0, 2.0};
const int num_taps[] = {400, 400, 400, 400};
const int FFTSize = 2048;
const float FFTIncr = 0.25;

Multichannel::Multichannel(QWidget *p = 0, const char *name = 0) :
  QWidget(p, name),
  left(0), right(0)
{
  source = new VrAR5000Source<char>(gupRate);
  source->setRxFrequency(RxFrequency);

  chan_filter = new VrComplexFIRfilter<char>(numChannels, CFIRdecimate, 
					     num_taps, center_freq, gain);

  for (int i = 0; i < numChannels; i++)
    {
      //      cascade_filter[i] = new VrComplexCascadefilter<complex>(CFIRdecimate2,
      //							      50,
      //							      0,
      //							      7500,
      //							      1.0);
      chan_FFT[i] = new VrQFFTSink<complex>(FFTSize, FFTIncr);
    }

  zero = new VrZeroSource<short>();

  left_demod = new VrQuadratureDemod<float>(22000);
  right_demod = new VrQuadratureDemod<float>(22000);

  left_if_filter = new VrRealFIRfilter<float,short>(RFIRdecimate, audioRate/2,
						     20, 1.0);
  right_if_filter = new VrRealFIRfilter<float,short>(RFIRdecimate, audioRate/2,
						     20, 1.0);

  audio = new VrAudioSink<short>();


  for (int i = 0; i < numChannels; i++)
    {
      CONNECTN(chan_FFT[i], chan_filter, i, quadRate, 64);
      // CONNECT(chan_FFT[i], cascade_filter[i], quadRate, 64);
      // CONNECTN(cascade_filter[i], chan_filter, i, gupRate/CFIRdecimate, 64);
    }

  CONNECT(chan_filter, source, gupRate, 8);

  CONNECT(left_if_filter, left_demod, quadRate, 32);
  CONNECT(right_if_filter, right_demod, quadRate, 32);

  CONNECT(audio, zero, audioRate, 16);
  CONNECT(audio, zero, audioRate, 16);

  mtask = new VrMultiTask();
  for (int i = 0; i < numChannels; i++)
    mtask->add(chan_FFT[i]);
  mtask->add(audio);
  mtask->setup();

  QBoxLayout *layout = new QVBoxLayout(this);
  
  channel[0] = new Channel("2-way Radio\nChannel A", this);
  layout->addWidget(channel[0]);
  
  channel[1] = new Channel("2-way Radio\nChannel B", this);
  // channel[1] = new Channel("2-way Radio\nChannel C", this);
  layout->addWidget(channel[1]);

  channel[2] = new Channel("2-way Radio\nChannel C", this);
  // channel[2] = new Channel("2-way Radio\nChannel E", this);
  layout->addWidget(channel[2]);

  channel[3] = new Channel("Cordless Phone", this);
  layout->addWidget(channel[3]);
  
  layout->freeze();

  timer = new QTimer(this);


  for (int i = 0; i < numChannels; i++)
    {
      connect(channel[i], SIGNAL(leftClicked()), SLOT(leftClicked()));
      connect(channel[i], SIGNAL(rightClicked()), SLOT(rightClicked()));
    }

  connect(timer, SIGNAL(timeout()), SLOT(doWork()));

  timer->start(10);
}

Multichannel::~Multichannel()
{
  delete source;
  delete chan_filter;
  for (int i = 0; i < numChannels; i++)
    {
      // delete cascade_filter[i];
      delete chan_FFT[i];
    }
  delete left_demod;
  delete left_if_filter;
  delete right_demod;
  delete right_if_filter;
  delete audio;
  delete mtask;
  delete timer;
}

void
Multichannel::leftClicked()
{
  Channel *chan = (Channel *)sender();

  if (chan != left)
    {
      if (left != 0)
	left->setLeftOn(false);
      left = chan;
    }
  else
    left = 0;

  updateAudio();
}

void
Multichannel::rightClicked()
{
  Channel *chan = (Channel *)(sender());

  if (chan != right)
    {
      if (right != 0)
	right->setRightOn(false);
      right = chan;
    }
  else
    right = 0;

  updateAudio();
}

void
Multichannel::updateAudio()
{
  DISCONNECTALLINPUTS(left_demod);
  DISCONNECTALLINPUTS(right_demod);
  DISCONNECTALLINPUTS(audio);

  if (left == 0)
    {
      CONNECT(audio, zero, audioRate, 16);
    }
  else
    {
      CONNECT(audio, left_if_filter, audioRate, 16);
      for (int i = 0; i < numChannels; i++)
	{
	  if (left == channel[i])
	    {
	      CONNECTN(left_demod, chan_filter, i, quadRate, 64);
	      // CONNECT(left_demod, cascade_filter[i], quadRate, 64);
	    }
	}
    }

  if (right == 0)
    {
      CONNECT(audio, zero, audioRate, 16);
    }
  else
    {
      CONNECT(audio, right_if_filter, audioRate, 16);
      for (int i = 0; i < numChannels; i++)
	{
	  if (right == channel[i])
	    {
	      CONNECTN(right_demod, chan_filter, i, quadRate, 64);
	      // CONNECT(right_demod, cascade_filter[i], quadRate, 64);
	    }
	}
    }

  mtask->setup();
}

void
Multichannel::doWork()
{
  mtask->start(1);
  for (int i = 0; i < numChannels; i++)
    channel[i]->setData(chan_FFT[i]->numPoints, 
			chan_FFT[i]->x, chan_FFT[i]->y);
}

int
main(int argc, char **argv)
{
  QApplication a(argc, argv);

  Multichannel m;

  a.setMainWidget(&m);
  m.show();
  a.exec();
}
